#!/bin/ksh
#
# Creates an ESR Weblogic domain in silent mode.
# Note that the Weblogic bin directory needs to be in the system PATH for
# this script to run (WL_HOME/weblogic81/common/bin).
#
# This script takes the Weblogic template filename and domain name as optional 
# inputs.  If domain name is not specified, "wls_domain" name will then be used
# for a domain name.  If template filename is not specified, "wls.jar" will be 
# used as the default.
#

#
# Set the working home directory to the current directory.
#
WORK_HOME=`pwd`

#
# Set the template home.
#
TEMPLATE_HOME=${WORK_HOME}

#
# Set the script input filename and log file.
#
SCRIPT_INPUT_FILENAME=installConfig.txt
SCRIPT_LOG_FILENAME=installConfig.log

#
# Set the domain template filename to the first argument or use the default.
# For ESR this is typically ESRDomain.jar.  The default is wls.jar
#
TEMPLATE_FILENAME=$1
if [ "$TEMPLATE_FILENAME" = "" ]; then
  TEMPLATE_FILENAME=wls.jar
  TEMPLATE_HOME=${WL_HOME}/common/templates/domains 
fi

#
# Set the domain name to the second argument or use the default. 
#
DOMAIN_NAME=$2
if [ "$DOMAIN_NAME" = "" ]; then
  DOMAIN_NAME=wls_domain
fi

#
# Set the domain location to the third argument or use the default. 
#
DOMAIN_LOC=$3
if [ "$DOMAIN_LOC" = "" ]; then
  DOMAIN_LOC=${BEA_HOME}/user_projects/domains
fi

#
# Ensure the domain template JAR file is in the template home directory.
#
if [ ! -f ${TEMPLATE_HOME}/${TEMPLATE_FILENAME} ]; then
  echo "The Domain template file ${TEMPLATE_FILENAME} could not be found in ${TEMPLATE_HOME}."
  exit 1
fi

#
# Inform the user that we are creating the domain.
#
echo Creating Weblogic domain from template ${TEMPLATE_HOME}/${TEMPLATE_FILENAME} using silent input script: ${SCRIPT_INPUT_FILENAME}.

#
# Delete any previously generated log file
#
if [ -f ${WORK_HOME}/${SCRIPT_LOG_FILENAME} ]; then
  rm ${WORK_HOME}/${SCRIPT_LOG_FILENAME}
fi

#
# Create the script input file.
# Note that we are creating the input file from within this script so all
# directories can be absoluate paths in the input file.  This is required
# for the config script to work properly.
#
cat <<EOF >${WORK_HOME}/${SCRIPT_INPUT_FILENAME}
/////////////////////////////////////////////////////////////////////////////////////////
// This is an example of a Configuration Wizard silent script. This example uses the
// Common Services DatSvc Domain template to show how to open, modify, and write out
// a domain in silent mode. Please note that many values used in this script are subject
// to change based on your WebLogic installation and the template you are using.
//
// Usage:
//      <Windows> = config.cmd -mode=silent -silent_script=<silent script>
//      <Unix>    = ./config.sh -mode=silent -silent_script=<silent script>
//
// Where:
//      <silent script> should be the full path to the silent script.
/////////////////////////////////////////////////////////////////////////////////////////

/////////////////////////////////////////////////////////////////////////////////////////
//Read in a domain template.
/////////////////////////////////////////////////////////////////////////////////////////

read template from
    "${TEMPLATE_HOME}/${TEMPLATE_FILENAME}";

find User "weblogic" as u;
set u.password "weblogic";

/////////////////////////////////////////////////////////////////////////////////////////
//Write out the domain.
/////////////////////////////////////////////////////////////////////////////////////////
set OverwriteDomain "true";
write domain to "${DOMAIN_LOC}/${DOMAIN_NAME}";


/////////////////////////////////////////////////////////////////////////////////////////
//Close domain template to indicate completion of work.
/////////////////////////////////////////////////////////////////////////////////////////
close template;
EOF

#
# Execute the Weblogic configuration script to create the domain.
# The config script will run the Weblogic configuration wizard in silent mode.
# Note that the config.sh script comes with Weblogic and should be in the
# system PATH.
#
config.sh -mode=silent -silent_script=${WORK_HOME}/${SCRIPT_INPUT_FILENAME} -log=${WORK_HOME}/${SCRIPT_LOG_FILENAME}

#
# Clean up log file and silent script
#
if [ -f ${WORK_HOME}/${SCRIPT_LOG_FILENAME} ]; then
  rm ${WORK_HOME}/${SCRIPT_INPUT_FILENAME}
  rm ${WORK_HOME}/${SCRIPT_LOG_FILENAME}
fi

#
# Clean up temporary directory created by the configuration wizard
#
if [ -d ${DOMAIN_LOC}/${DOMAIN_NAME} ]; then
  echo "Delete temp directory ${BEA_HOME}/user_projects/domains/${DOMAIN_NAME}/cfgwiz_donotdelete"  
  rm -rf ${DOMAIN_LOC}/${DOMAIN_NAME}/_cfgwiz_donotdelete
fi

#
# Check if the ext_components exists to copy extension jar over
#
if [ ! -d ${BEA_HOME}/ext_components ]; then
  echo "Creating ${BEA_HOME}/ext_components "
  mkdir ${BEA_HOME}/ext_components
fi

#
# Copying the antlr jar to the ext_components
#
if [ ! -f {BEA_HOME}/ext_components/antlr-2.7.6rc1.jar ]; then
  if [ -f ${WORK_HOME}/ext_components/antlr-2.7.6rc1.jar ]; then
     echo "Copying ${WORK_HOME}/ext_components/antlr-2.7.6rc1.jar to ${BEA_HOME}/ext_components/antlr-2.7.6rc1.jar" 
     cp ${WORK_HOME}/ext_components/antlr-2.7.6rc1.jar ${BEA_HOME}/ext_components/antlr-2.7.6rc1.jar
  fi
fi
