#!/bin/sh
# ****************************************************************************
# This script is used to start a managed WebLogic Server for the domain in the
# current working directory.  This script reads in the SERVER_NAME and
# ADMIN_URL as positional parameters, sets the SERVER_NAME variable, then
# starts the server.
#
# Other variables that startWLS takes are:
#
# WLS_USER       - cleartext user for server startup
# WLS_PW         - cleartext password for server startup
# PRODUCTION_MODE      - Set to true for production mode servers, false for
#                  development mode
# JAVA_OPTIONS   - Java command-line options for running the server. (These
#                  will be tagged on to the end of the JAVA_VM and MEM_ARGS)
# JAVA_VM        - The java arg specifying the VM to run.  (i.e. -server,
#                  -hotspot, etc.)
# MEM_ARGS       - The variable to override the standard memory arguments
#                  passed to java
#
# For additional information, refer to the WebLogic Server Administration Guide
# (http://e-docs.bea.com/wls/docs81/ConsoleHelp/startstop.html).
# ****************************************************************************

# Set common environment variables such as WEBLOGIC_CLASSPATH
if [ "$DOMAIN_HOME" == "" ] ; then
   # If the Domain_Home is not found, assume that the setEnv.sh is in the current directory
    if [ -x ./setEnv.sh ] ; then
        . ./setEnv.sh
    else
      echo ./setEnv.sh is not executable or not found.
      exit 1
    fi
else
    if [ -x ${DOMAIN_HOME}/setEnv.sh ] ; then
        . ${DOMAIN_HOME}/setEnv.sh
    else
      echo ${DOMAIN_HOME}/setEnv.sh is not executable or not found.
      exit 1
    fi
fi

# Set SERVER_NAME to the name of the server you wish to start up.
SERVER_NAME="MS1"
MEM_ARGS="-Xms1g -Xmx1g -XX:NewSize=256m -XX:MaxPermSize=256m -XX:SurvivorRatio=5 -XX:+UseParallelGC -XX:+UseCMSCompactAtFullCollection -XX:CMSFullGCsBeforeCompaction=0 "

usage()
{
  echo "Need to set SERVER_NAME and ADMIN_URL environment variables or specify"
  echo "them in command line:"
  echo 'Usage: ./startManagedWebLogic.sh [SERVER_NAME] [ADMIN_URL]'
  echo "for example:"
  echo './startManagedWebLogic.sh managedserver1 http://DNS   app1.aac.DNS   :7001'
  exit 1
}

# Check for variables SERVER_NAME and ADMIN_URL
# SERVER_NAME and ADMIN_URL must by specified before starting a managed server,
# detailed information can be found at http://e-docs.bea.com/wls/docs81/adminguide/startstop.html.
if [ ${#} = 0 ]; then
  if [ "x${SERVER_NAME}" = "x" -o "x${ADMIN_URL}" = "x" ]; then
    usage
  fi
elif [ ${#} = 1 ]; then
  SERVER_NAME=${1}
  if [ "x${ADMIN_URL}" = "x" ]; then
    usage
  fi
elif [ ${#} = 2 ]; then
  SERVER_NAME=${1}
  ADMIN_URL=${2}
else
    usage
fi

# Start WebLogic server
# Reset number of open file descriptors in the current process
# This function is defined in commEnv.sh
resetFd


"$JAVA_HOME/bin/java"   -Dweblogic.Name=${SERVER_NAME}           \
  ${JAVA_VM} ${MEM_ARGS} ${JAVA_OPTIONS}                         \
  -Dweblogic.management.server=${ADMIN_URL}                      \
  -Djava.security.policy="${WL_HOME}/server/lib/weblogic.policy" \
  -cp ${WEBLOGIC_CLASSPATH}                                      \
  weblogic.Server
