/********************************************************************
 * Copyright  2004 VHA. All rights reserved
 ********************************************************************/
// Package
package gov.va.med.fw.util;

// Java classes
import java.io.BufferedInputStream;
import java.io.BufferedOutputStream;
import java.io.FileOutputStream;
import java.io.FileInputStream;
import java.io.OutputStream;
import java.io.InputStream;
import java.io.IOException;
import java.io.File;

// Library classes
import org.apache.commons.io.FilenameUtils;

/**
 * Provides convinient methods to perform IO operations. Extends
 * from <code>org.apache.commons.io.FileUtils</code> to provide 
 * additional methods to access OutputStream and InputStream of 
 * a file even if a file is not yet created under sub directories.  
 *
 * Project: Framework</br>
 * Created on: 1:25:02 PM </br>
 *
 * @author DNS   LEV
 */
public class FileUtils extends org.apache.commons.io.FileUtils {

	/**
	 * A default constructor
	 */
	protected FileUtils() {
		super();
	}
	
	/** Returns an output stream of a file specified by its name. 
	 * A file name may include its whole path even if a path doesn't exist
	 * 
	 * @param name A file name
	 * @return An output stream of a file
	 * @throws IOException In case of errors creating a file
	 */
	public static OutputStream getOutputStream( String name ) throws IOException {
		return getOutputStream( getFile( name ) );
	}

	/** Returns an output stream of a file specified by its handler.
	 * Creates a file if it 
	 * 
	 * @param file A file to get an output stream
	 * @return An output stream of a file
	 * @throws IOException In case of errors creating a file
	 */
	public static OutputStream getOutputStream( File file ) throws IOException {
		
		if( !file.exists() ) {
			file.createNewFile();
		}
		return new BufferedOutputStream( new FileOutputStream( file ) );
	}
	
	/** Returns an input stream of a file specified by its name. 
	 * A file name may include its whole path even if a path doesn't exist
	 * 
	 * @param name A file name
	 * @return An input stream of a file
	 * @throws IOException In case of errors creating a file
	 */
	public static InputStream getInputStream( String name ) throws IOException {
		return getInputStream( getFile( name ) );
	}

	/** Returns an input stream of a file specified by its handler.
	 * Creates a file if it 
	 * 
	 * @param file A file to get an output stream
	 * @return An input stream of a file
	 * @throws IOException In case of errors creating a file
	 */
	public static InputStream getInputStream( File file ) throws IOException {
		
		if( !file.exists() ) {
			file.createNewFile();
		}
		return new BufferedInputStream( new FileInputStream( file ) );
	}
	
	/** Returns a file handle.  Creates a directory as a file 
	 * parent's handle if neccessary
	 * 
	 * @pparam name A file name including its path
	 * @return A created file in its directories if specified
	 * @throws IOException In case of errors creating a file
	 */
	public static File getFile( String name ) throws IOException {
		
		// Creates a file first before attempting to use it
		String fileName = FilenameUtils.getName( name );
		
		// create a path
		String path = FilenameUtils.getPathNoEndSeparator( name );
		File parent = null;
		if( StringUtils.isNotBlank( path ) ) { // not null, not empty, not blank
			parent = new File( path );
			if( !parent.exists() ) {
				parent.mkdirs();
			}
		}

		// Create a file with a parent path if requested
		return parent != null ? new File( parent, fileName ) : new File( fileName );
	}
}