/********************************************************************
 Copyright  2005 VHA. All rights reserved
 ********************************************************************/
package gov.va.med.fw.util;

// Java Classes
import java.util.Collection;
import java.util.Iterator;
import java.util.ArrayList;
import java.util.List;

/**
 * This class provides generic helper methods for collections.
 *
 * @author Andrew Pach
 * @version 3.0
 */
public class CollectionUtils extends org.apache.commons.collections.CollectionUtils
{
    /**
     * Gets the first item in a collection that matches the specific class type.
     * If no object matches the specific class type, then null is returned.
     *
     * @param collection The collection of objects to check.
     * @param classType The class type of the object to get.
     * @return The first Object that matches the class type.
     */
    public static Object getObjectOfClassType(Collection collection, Class classType)
    {
        if (collection != null)
        {
            for (Iterator iterator = collection.iterator(); iterator.hasNext();)
            {
                Object object = iterator.next();
                if (classType.isAssignableFrom(object.getClass()))
                {
                    return object;
                }
            }
        }
        return null;
    }

    /**
     * Gets the list of objects in the passed in collection that match the specific class type.
     * If no objects matches the specific class type, then an empty list is returned.
     *
     * @param collection The collection of objects to check.
     * @param classType The class type of the objects to get.
     * @return The list of Objects that match the class type.
     */
    public static List getObjectsOfClassType(Collection collection, Class classType)
    {
        ArrayList list = new ArrayList();
        if (collection != null)
        {
            for (Iterator iterator = collection.iterator(); iterator.hasNext();)
            {
                Object object = iterator.next();
                if (classType.isAssignableFrom(object.getClass()))
                {
                    list.add(object);
                }
            }
        }
        return list;
    }

    /**
     * Removes the first object of the specified class type from the collection.
     *
     * @param collection The collection of objects to remove the object from.
     * @param classType The class type of the object to remove.
     * @return The object that was removed or null if an object of the specified class type didn't
     * exist in the collection).
     */
    public static Object removeObjectOfClassType(Collection collection, Class classType)
    {
        Object objectToRemove = CollectionUtils.getObjectOfClassType(collection, classType);
        if (objectToRemove != null)
        {
            collection.remove(objectToRemove);
            return objectToRemove;
        }
        else
        {
            return null;
        }
    }

    /**
     * Removes the all objects of the specified class type from the collection.
     *
     * @param collection The collection of objects to remove the objects from.
     * @param classType The class type of the objects to remove.
     * @return The list of objects that were removed or an empty list if no objects of the specified class
     * existed in the collection).
     */
    public static List removeObjectsOfClassType(Collection collection, Class classType)
    {
        List objectsToRemove = CollectionUtils.getObjectsOfClassType(collection, classType);
        if (objectsToRemove.size() > 0)
        {
            for (Iterator iterator = objectsToRemove.iterator(); iterator.hasNext();)
            {
                Object objectToRemove = iterator.next();
                collection.remove(objectToRemove);
            }
        }
        return objectsToRemove;
    }
}
