/********************************************************************
 * Copyright  2006 VHA. All rights reserved
 ********************************************************************/


package gov.va.med.fw.service.transaction;

import org.apache.commons.lang.Validate;
import org.springframework.transaction.support.TransactionSynchronization;
import org.springframework.transaction.support.TransactionSynchronizationManager;

import gov.va.med.fw.cache.CacheManager;
import gov.va.med.fw.service.AbstractComponent;

/**
 * Provides global cleanup of CacheManager contents just prior to commit.
 * 
 * Created May 17, 2006 11:42:43 AM
 * @author DNS   BOHMEG
 */
public class TransactionSynchronizationForCacheManager extends AbstractComponent
		implements TransactionSynchronization {
	public static final String CACHE_AUTOCLEANUP_VIA_TRANSACTION_SYNCHRONIZATION =
		TransactionSynchronizationForCacheManager.class.getName() + ".autoCleanup";
	
	/**
	 * An instance of cacheManager
	 */
	private CacheManager cacheManager = null;
	

	/* (non-Javadoc)
	 * @see org.springframework.transaction.support.TransactionSynchronization#suspend()
	 */
	public void suspend() {
		if(TransactionSynchronizationManager.hasResource(CACHE_AUTOCLEANUP_VIA_TRANSACTION_SYNCHRONIZATION)) {
			TransactionSynchronizationManager.unbindResource(CACHE_AUTOCLEANUP_VIA_TRANSACTION_SYNCHRONIZATION);
		}
	}

	/* (non-Javadoc)
	 * @see org.springframework.transaction.support.TransactionSynchronization#resume()
	 */
	public void resume() {
		if(!TransactionSynchronizationManager.hasResource(CACHE_AUTOCLEANUP_VIA_TRANSACTION_SYNCHRONIZATION)) {
			TransactionSynchronizationManager.bindResource(CACHE_AUTOCLEANUP_VIA_TRANSACTION_SYNCHRONIZATION, Boolean.TRUE.toString());
		}
	}

	/* (non-Javadoc)
	 * @see org.springframework.transaction.support.TransactionSynchronization#beforeCommit(boolean)
	 */
	public void beforeCommit(boolean readOnly) {
		// TODO Auto-generated method stub

	}

	/* (non-Javadoc)
	 * @see org.springframework.transaction.support.TransactionSynchronization#beforeCompletion()
	 */
	public void beforeCompletion() {
		String txName = TransactionSynchronizationManager.getCurrentTransactionName();
		if( txName != null ) {
			this.getCacheManager().removeItem( txName ); // removes "bucket" of storage that was assigned to the current TX
			if(TransactionSynchronizationManager.hasResource(CACHE_AUTOCLEANUP_VIA_TRANSACTION_SYNCHRONIZATION)) {
				TransactionSynchronizationManager.unbindResource(CACHE_AUTOCLEANUP_VIA_TRANSACTION_SYNCHRONIZATION);
			}
		}		
	}

	/* (non-Javadoc)
	 * @see org.springframework.transaction.support.TransactionSynchronization#afterCompletion(int)
	 */
	public void afterCompletion(int status) {
		// TODO Auto-generated method stub

	}

	public void afterPropertiesSet() {
		Validate.notNull( this.cacheManager, "A cacheManager is required" );
	}

	/**
	 * @return Returns the cacheManager.
	 */
	public CacheManager getCacheManager() {
		return cacheManager;
	}

	/**
	 * @param cacheManager The cacheManager to set.
	 */
	public void setCacheManager(CacheManager cacheManager) {
		this.cacheManager = cacheManager;
	}	
}
