/********************************************************************
 * Copyright  2004 VHA. All rights reserved
 ********************************************************************/
package gov.va.med.fw.security.jaas;

import javax.security.auth.spi.LoginModule;
import javax.security.auth.Subject;
import javax.security.auth.callback.CallbackHandler;
import java.util.Map;

/**
 * AbstractLoginModule to provide some basic functionality and helper methods
 * for the subclasses.
 * 
 * @author DNS   mansog
 * @date Mar 22, 2005
 */
public abstract class AbstractLoginModule implements LoginModule {
    private static final String OPTION_TRYFIRSTPASS = "tryFirstPass";
    private static final String OPTION_USEFIRSTPASS = "useFirstPass";
    private static final String OPTION_STOREPASS = "storePass";
    private static final String OPTION_CLEARPASS = "clearPass";
    private static final String OPTION_DEBUG = "debug";
    private static final String TRUE = "true";

    protected Subject subject;
    protected CallbackHandler callbackHandler;
    protected Map sharedState;
    protected Map options;
    private boolean debug = false;
    private boolean tryFirstPass = false;
    private boolean storePass = false;
    private boolean useFirstPass = false;
    private boolean clearPass = false;

    /*
     * (non-Javadoc)
     * 
     * @see javax.security.auth.spi.LoginModule#initialize(javax.security.auth.Subject,
     *      javax.security.auth.callback.CallbackHandler, java.util.Map,
     *      java.util.Map)
     */
    public void initialize(Subject subject, CallbackHandler callbackHandler,
            Map sharedState, Map options) {
        this.subject = subject;
        this.callbackHandler = callbackHandler;
        this.sharedState = sharedState;
        this.options = options;
        debug = getOptionAsBoolean(OPTION_DEBUG);
        tryFirstPass = getOptionAsBoolean(OPTION_TRYFIRSTPASS);
        useFirstPass = getOptionAsBoolean(OPTION_USEFIRSTPASS);
        storePass = getOptionAsBoolean(OPTION_STOREPASS);
        clearPass = getOptionAsBoolean(OPTION_CLEARPASS);
    }

    /**
     * Getter method for CallbackHandler
     * 
     * @return CallbackHandler
     */
    public CallbackHandler getCallbackHandler() {
        return callbackHandler;
    }

    /**
     * @return true if debug is on, false otherwise
     */
    public boolean isDebug() {
        return debug;
    }

    /**
     * @return Map of Options
     */
    public Map getOptions() {
        return options;
    }

    /**
     * @return Map of a SharedState
     */
    public Map getSharedState() {
        return sharedState;
    }

    /**
     * @return Subject
     */
    public Subject getSubject() {
        return subject;
    }

    /**
     * Get the option using specified key
     * 
     * @param key
     * @return option value
     */
    protected Object getOption(String key) {
        return options.get(key);
    }

    /**
     * Get the option using specified key as String
     * 
     * @param key
     * @return option value as String
     */
    protected String getOptionAsString(String key) {
        Object val = getOption(key);
        return val == null ? null : val.toString();
    }

    /**
     * Get the option using specified key and converts it to boolean
     * 
     * @param key
     * @return boolean
     */
    protected boolean getOptionAsBoolean(String key) {
        return TRUE.equalsIgnoreCase((String) options.get(key));
    }

    /**
     * @return true if clearPass option is true, false otherwise.
     */
    public boolean isClearPass() {
        return clearPass;
    }

    /**
     * @return true if storePass option is true, false otherwise.
     */
    public boolean isStorePass() {
        return storePass;
    }

    /**
     * @return true if tryFirstPass option is true, false otherwise.
     */
    public boolean isTryFirstPass() {
        return tryFirstPass;
    }

    /**
     * @return true if useFirstPass option is true, false otherwise.
     */
    public boolean isUseFirstPass() {
        return useFirstPass;
    }
}