/********************************************************************
 * Copyright  2004 VHA. All rights reserved
 ********************************************************************/
package gov.va.med.fw.rule;

// Java classes
import java.util.Map;

// Library classes
import org.apache.commons.lang.Validate;

// Framework classes
import gov.va.med.fw.service.AbstractComponent;

// EDB classes

/**
 * Project: Framework
 * @author DNS   LEV
 * @version 1.0
 */
public class RuleFlowService extends AbstractComponent implements RuleFlow {

    /**
     * A rule service manager that manages irl files 
     * and rule flow names
     */
    private RuleFlowManager ruleManager = null;
    
    /**
     * Default constructor
     */
    public RuleFlowService() {
        super();
    }
    
    /**
     * @see org.springframework.beans.factory.InitializingBean#afterPropertiesSet()
     */
    public void afterPropertiesSet() throws Exception {
        super.afterPropertiesSet();

        if( ruleManager == null ) {
            throw new RuleConfigurationException( "A rule manager must be configured for this service" );
        }
    }
    
    /**
     * @return Returns the ruleManager.
     */
    public RuleFlowManager getRuleManager() {
        return ruleManager;
    }
    
    /**
     * @param ruleManager The ruleManager to set.
     */
    public void setRuleManager(RuleFlowManager ruleManager) {
        this.ruleManager = ruleManager;
    }
    
    /**
    * @see gov.va.med.fw.rule.RuleFlow#execute(gov.va.med.fw.rule.RuleParameters, gov.va.med.fw.rule.RuleDataAware)
    */
    public Object execute( RuleParameters params, RuleDataAware ruleData ) throws RuleException {
       Validate.notNull( params, "Rule parameters must not be null");
       Validate.notNull( ruleData, "A rule data must not be null");

       // Initialize parameters with data aware object and then
       // return the parameters.
       params.initializeRuleParmeters( ruleData );
       
       // Use the initialized parameters for firing rules.
       return this.ruleManager.fireRule( params.getRuleFlowName(), params ); 
    }
    
    /**
    * @see gov.va.med.fw.rule.RuleFlow#execute(gov.va.med.fw.rule.RuleParameters, java.util.Map)
    */
    public Object execute( RuleParameters params, Map ruleData ) throws RuleException {
       Validate.notNull( params, "Rule parameters must not be null");
       Validate.notEmpty( ruleData, "Rule data map must not be null or empty");

       params.initializeRuleParmeters( ruleData );
       
       // Use the initialized parameters for firing rules.
       return this.ruleManager.fireRule( params.getRuleFlowName(), params ); 
    }
    
    /**
    * @see gov.va.med.fw.rule.RuleFlow#execute(gov.va.med.fw.rule.RuleParameter, gov.va.med.fw.rule.RuleDataAware)
    */
    public Object execute( RuleParameter param, RuleDataAware data ) throws RuleException {

       Validate.notNull( param, "A rule parameter must not be null");
       Validate.notNull( data, "A rule data must not be null");
       param.setIncomingData( data.getIncomingData() );
       param.setPristineData( data.getPristineData() );
       param.setResultData( data.getResultData() );
       param.setRuleDataAware( data );
       return this.ruleManager.fireRule( param.getRuleFlowName(), param ); 
    }
}