/*******************************************************************************
 * Copyright  2004 VHA. All rights reserved
 ******************************************************************************/
package gov.va.med.fw.rule;

// Java classes
import java.util.Map;
import java.util.Iterator;
import java.util.Collection;

// Library classes
import org.apache.commons.lang.Validate;
import org.apache.commons.lang.builder.ToStringBuilder;
import org.apache.commons.lang.builder.ToStringStyle;

// EDB classes

/**
 * Provides a wrapper to contain a list of rule parameters. Each rule parameter
 * must have a unique input and output name so that ILOG can bind each rule
 * paramter to a unique parameter name in ILOG rule engine agenda. Project:
 * Framework
 * 
 * @author DNS   LEV
 * @version 1.0
 */
public abstract class AbstractRuleParameters extends AbstractRuleParameter 
	implements RuleParameters {

    private static final long serialVersionUID = -1370346746640850300L;
	private Map ruleParameters = null;
	
	/**
	 * A default constructor
	 */
	protected AbstractRuleParameters() {
		super();
	}

	/**
	 * Check if the rule parameter map is properly set
	 * 
	 * @see org.springframework.beans.factory.InitializingBean#afterPropertiesSet()
	 */
	public void afterPropertiesSet() throws Exception {
		super.afterPropertiesSet();
		Validate.notNull( ruleParameters, "Parameter Factory must be set" );
	}
	
	/**
	 * Returns a map containing rule parameters
	 * 
	 * @return Returns the parameters.
	 */
	public Map getRuleParameters() {
		return ruleParameters;
	}

   /**
    * Returns a map containing rule parameters
    * 
    * @return Returns the parameters.
    */
   public void setRuleParameters( Map parameters ) {
      ruleParameters = parameters;
   }

	/**
	 * Returns a rule parameter class by its name
	 * 
	 * @see gov.va.med.fw.rule.RuleParameters#getRuleParameter(java.lang.String)
	 */
	public RuleParameter getRuleParameter(String name) {
		
		Map ruleParameters = this.getRuleParameters();
		Validate.notEmpty( ruleParameters, "A rule parameter map must not be null" );

		Object param = ruleParameters.get(name);
		return (param instanceof RuleParameter) ? (RuleParameter) param : null;
	}

	/**
	 * Returns a string representation of the object in multiple line.
	 * 
	 * @return A contextual string representation of the object
	 */
	public String toString() {
		ToStringBuilder builder = new ToStringBuilder(this, ToStringStyle.MULTI_LINE_STYLE);
		Collection params = getRuleParameters().values();
		for (Iterator i = params.iterator(); i.hasNext();) {
			Object param = i.next();
			builder.append(param.getClass().getName(), param);
		}
		return builder.toString();
	}

	/**
	 * @see java.lang.Object#finalize()
	 */
	protected void finalize() throws Throwable {
		super.finalize();
		this.ruleParameters.clear();
		this.ruleParameters = null;
	}
}