/********************************************************************
 * Copyright  2004 VHA. All rights reserved
 ********************************************************************/
// Package
package gov.va.med.fw.report;

// Library classes
import org.apache.commons.lang.ClassUtils;
import org.apache.commons.lang.Validate;

import gov.va.med.fw.report.data.ReportData;
import gov.va.med.fw.service.AbstractComponent;
import gov.va.med.fw.service.ServiceDescriptor;
import gov.va.med.fw.service.ServiceException;
import gov.va.med.fw.service.trigger.TriggerEvent;
import gov.va.med.fw.service.trigger.TriggerIdentity;
import gov.va.med.fw.service.trigger.TriggerableService;

/**
 * Trigger an event to process a report generation process asynchronously
 * This implementation uses TriggerRouter to trigger a JMS message encapsulating
 * ReportConfiguration to generate a report on the consumer side of a JMS message
 *
 * Project: Framework</br>
 * Created on: 4:52:06 PM </br>
 *
 * @author DNS   LEV
 */
public class AsynchronousReportService extends AbstractComponent implements ReportService {

	/**
	 * An instance of serialVersionUID
	 */
	private static final long serialVersionUID = 1L;

	/**
	 * An instance of an event router
	 */
	private TriggerableService triggerService = null;
	
	/**
	 * An instance of triggerEventName
	 */
	private String triggerEventName = null;
	
	/**
	 * A default constructor
	 */
	public AsynchronousReportService() {
		super();
	}

	/**
	 * @see gov.va.med.fw.service.AbstractComponent#afterPropertiesSet()
	 */
	public void afterPropertiesSet() throws Exception {
		super.afterPropertiesSet();
		Validate.notNull( this.triggerService, "A trigger service is required" );
		
		// Set a default trigger event name
		if( triggerEventName == null ) {
			StringBuffer tmp = new StringBuffer( ClassUtils.getShortClassName( this.getClass().getName() ) );
			tmp.append( "." )
				.append( ClassUtils.getShortClassName( ServiceDescriptor.class.getName() ) );
			
			// Build a default name "AsynchronousReportService.ServiceDescriptor"
			triggerEventName = tmp.toString(); 
		}
	}

	/**
	 * @see gov.va.med.fw.report.ReportService#generateReport(gov.va.med.fw.report.ReportConfiguration)
	 */
	public void generateReport(ReportConfiguration configuration) throws ServiceException {
		this.triggerService.trigger( new TriggerEvent( this.triggerEventName, new Object[]{configuration},
				TriggerEvent.createTriggerIdentity(TriggerIdentity.DestinationType.REPORT)) );
	}

	/**
	 * @see gov.va.med.fw.report.ReportService#generateReport(gov.va.med.fw.report.ReportConfiguration, gov.va.med.fw.report.data.ReportData)
	 */
	public void generateReport(ReportConfiguration configuration, ReportData data) throws ServiceException {
		this.triggerService.trigger( new TriggerEvent( this.triggerEventName, new Object[]{configuration,data},
				TriggerEvent.createTriggerIdentity(TriggerIdentity.DestinationType.REPORT)) );
	}

	/**
	 * @param triggerEventName The triggerEventName to set.
	 */
	public void setTriggerEventName(String triggerEventName) {
		this.triggerEventName = triggerEventName;
	}
	
	/**
	 * @param triggerService The triggerService to set.
	 */
	public void setTriggerService(TriggerableService triggerService) {
		this.triggerService = triggerService;
	}
}