/********************************************************************
 * Copyright  2004 VHA. All rights reserved
 ********************************************************************/
package gov.va.med.fw.persistent.hibernate;

// Java classes

// Library classes
import java.sql.SQLException;
import java.util.Map;

import org.hibernate.CallbackException;
import org.hibernate.HibernateException;
import org.hibernate.Session;
import org.springframework.orm.hibernate3.HibernateCallback;

import gov.va.med.fw.persistent.DAOException;

// Framework classes

// EDB classes

/**
 * Implements HibernateCallback interface to provide more flexibilities to a
 * caller to perform any persistent operation using Hibernate session. This
 * abstract class is intented to allow derived classes to be configured as a
 * spring managed bean so that its dependencies and resources can be injected
 * through an application context file in xml format.
 * 
 * Project: Framework
 * 
 * @author DNS   LEV
 * @version 1.0
 */
public abstract class AbstractDAOAction implements HibernateCallback {
	private Map contextData;

	/**
	 * A default constructor
	 */
	protected AbstractDAOAction() {
		super();
	}

	protected AbstractDAOAction(Map contextData) {
		super();
		this.contextData = contextData;
	}

	/**
	 * Provides a call back method to perform any persistent operation
	 * 
	 * @see org.springframework.orm.hibernate3.HibernateCallback#doInHibernate(org.hibernate.Session)
	 */
	public Object doInHibernate(Session session) throws HibernateException,
			SQLException {
		try {
			return execute(session);
		} catch(DAOException e) {
			throw new CallbackException(e);
		}
	}

	/**
	 * Execute one or more persistent operation in the specific session
	 * 
	 * @param session
	 *            A session in which this call back is triggered
	 * @return Any return value from a persistent operation.
	 */
	protected abstract Object execute(Session session) throws DAOException;

	/**
	 * @return Returns the contextData.
	 */
	public Map getContextData() {
		return contextData;
	}

	/**
	 * @param contextData The contextData to set.
	 */
	public void setContextData(Map contextData) {
		this.contextData = contextData;
	}
}
