/*******************************************************************************
 * Copyright  2005 VHA. All rights reserved
 ******************************************************************************/

package gov.va.med.fw.persistent;

import gov.va.med.fw.service.ServiceException;

/**
 * An exception that is thrown when a search yields more records than allowed.
 * 
 * @author Andrew Pach
 * @version 3.0
 */
public class MaxRecordsExceededException extends ServiceException {
	/**
	 * An instance of serialVersionUID
	 */
	private static final long serialVersionUID = -1150179356480523951L;

	/**
	 * The total number of records that match the search criteria.
	 */
	private int totalRecords = -1;

	/**
	 * The record limit.
	 */
	private int recordLimit = -1;

	/**
	 * An exception with the total records and the record limit.
	 * 
	 * @param totalRecords
	 *           The total records that would be returned from the search.
	 * @param recordLimit
	 *           The maximum number of records that are allowed to be retrieved.
	 */
	public MaxRecordsExceededException(int totalRecords, int recordLimit) {
		super();
		this.totalRecords = totalRecords;
		this.recordLimit = recordLimit;
	}

	public MaxRecordsExceededException(int recordLimit, String message) {
		super(message);
		this.recordLimit = recordLimit;
	}
	
	/**
	 * An exception with a specific message added.
	 * 
	 * @param totalRecords
	 *           The total records that would be returned from the search.
	 * @param recordLimit
	 *           The maximum number of records that are allowed to be retrieved.
	 * @param msg
	 *           An exception message
	 */
	public MaxRecordsExceededException(int totalRecords, int recordLimit,
			String msg) {
		super(msg);
		this.totalRecords = totalRecords;
		this.recordLimit = recordLimit;
	}

	/**
	 * An exception with a specific parent exception and a message added.
	 * 
	 * @param totalRecords
	 *           The total records that would be returned from the search.
	 * @param recordLimit
	 *           The maximum number of records that are allowed to be retrieved.
	 * @param e
	 *           A parent exception in a chain
	 * @param msg
	 *           An exception message
	 */
	public MaxRecordsExceededException(int totalRecords, int recordLimit,
			String msg, Throwable e) {
		super(msg, e);
		this.totalRecords = totalRecords;
		this.recordLimit = recordLimit;
	}

	/**
	 * Gets the total records.
	 * 
	 * @return The total records.
	 */
	public int getTotalRecords() {
		return totalRecords;
	}

	/**
	 * Gets the record limit.
	 * 
	 * @return The record limit.
	 */
	public int getRecordLimit() {
		return recordLimit;
	}
}