/********************************************************************
 * Copyright  2004 VHA. All rights reserved
 ********************************************************************/

package gov.va.med.fw.hl7.segment;

// Java classes
import java.util.List;

// Framework classes
import gov.va.med.fw.hl7.InvalidSegmentException;
import gov.va.med.fw.hl7.Segment;
import gov.va.med.fw.hl7.constants.SegmentConstants;

/**
 * A ZRD segment containing information about batch trailer
 * 
 * @author Alex Yoon
 * @version 1.0 
 */

public class ZRD extends Segment {

    /**
	 * An instance of serialVersionUID
	 */
	private static final long serialVersionUID = 3216031412986346485L;

	/**
     * Constructs an empty segment. This constructor is ususally usd by derived
     * segment classes to populate its own segment data.
     */
    public ZRD() {
        super();
        setName(SegmentConstants.ZRD);
    }

    /**
     * Constructs a segment of elements from a raw data string. This constructor
     * is usually used to construct an inbound message from a raw message's
     * segment data.
     * 
     * @param data
     *            A raw data string
     * @throws InvalidSegmentException
     *             Thrown if failed to construct a segment due to missing
     *             parameters or invalid segment data to parse.
     */
    public ZRD(String data) throws InvalidSegmentException {
        super(data);
    }

    /**
     * Constructs a segment of elements from a raw data string, an array of
     * valid encoding character strings, and a element delimiter string. This
     * constructor is usually used to construct an inbound message from a raw
     * message's segment data, a specific encoding characters, and a field
     * delimiter.
     * 
     * @param encodingChars
     *            An array of encoding characters
     * @param elementDelimiter
     *            A field delimiter
     * @param segmentData
     *            A raw data string
     * @throws InvalidSegmentException
     *             Thrown if failed to construct a segment due to missing
     *             parameters or invalid segment data to parse.
     */
    public ZRD(String[] encodingChars, String elementDelimiter,
            String segmentData) throws InvalidSegmentException {
        super(encodingChars, elementDelimiter, segmentData);
    }

    /**
     * Constructs a segment from a segment name and a list of elements. This
     * constructor is usually used to construct an outbound message from a list
     * of specific segment elements, and a segment name.
     * 
     * @param name
     *            A segment name
     * @param elements
     *            A list of segment elements
     * @throws InvalidSegmentException
     *             Thrown if failed to construct a segment due to missing
     *             parameters or invalid segment data to parse.
     */
    public ZRD(String name, List elements) throws InvalidSegmentException {
        super(name, elements);
    }

    /**
     * Constructs a segment from an array of encoding characters, a field
     * delimiter, a segment name, and a list of field data. This constructor is
     * usually used to construct an outbound message from a list of specific
     * segment elements, element delimiter, and encoding characters.
     * 
     * @param encodingChars
     *            An array of encoding characters
     * @param elementDelimiter
     *            A field delimiter
     * @param segmentName
     *            A segment name
     * @param segmentElements
     *            A list of field data
     * @throws InvalidSegmentException
     *             Thrown if failed to construct a segment due to missing
     *             parameters or invalid segment data to parse.
     */
    public ZRD(String[] encodingChars, String elementDelimiter,
            String segmentName, List segmentElements)
            throws InvalidSegmentException {
        super(encodingChars, elementDelimiter, segmentName, segmentElements);
    }

    /**
     * 
     * Sequential number
     * 
     * @param input
     *            A value of SetID
     */
    public void setSetID(String input) {
        setElement(input, 0);
    }

    /**
     * 
     * Sequential number
     * 
     * @return String A value of SetID
     */
    public String getSetID() {
        return getElement(0);
    }

    /**
     * 
     * RATED DISABILITIES (VA) (.01) (VISTA Messaging HL7)
     * 
     * @param input
     *            A value of DisabilityCondition
     */
    public void setDisabilityCondition(String input) {
        setElement(input, 1);
    }

    /**
     * 
     * RATED DISABILITIES (VA) (.01) (VISTA Messaging HL7)
     * 
     * @return String A value of DisabilityCondition
     */
    public String getDisabilityCondition() {
        return getElement(1);
    }

    /**
     * 
     * DISABILITY % (2)
     * 
     * @param input
     *            A value of DisabilityPercentage
     */
    public void setDisabilityPercentage(String input) {
        setElement(input, 2);
    }

    /**
     * 
     * DISABILITY % (2)
     * 
     * @return String A value of DisabilityPercentage
     */
    public String getDisabilityPercentage() {
        return getElement(2);
    }

    /**
     * 
     * SERVICE CONNECTED (3)
     * 
     * @param input
     *            A value of ServiceConnected
     */
    public void setServiceConnected(String input) {
        setElement(input, 3);
    }

    /**
     * 
     * SERVICE CONNECTED (3)
     * 
     * @return String A value of ServiceConnected
     */
    public String getServiceConnected() {
        return getElement(3);
    }
    
    /**
     * 
     * Diagnostic Extremity (Bilateral factor)
     * 
     * @param input
     *            A value of DiagnosticExtremity
     */
    public void setDiagnosticExtremity (String input) {
        setElement(input, 11);
    }

    /**
     * 
     * Diagnostic Extremity (Bilateral factor)
     * 
     * @return String A value of DiagnosticExtremity
     */
    public String getDiagnosticExtremity() {
        return getElement(11);
    }    
    
    /**
     * 
     * Rated Disability original effective date - format is MMDDYYYY
     * 
     * @param input
     *            A value of RatedDisabilityOrigEffectiveDate
     */
    public void setRatedDisabilityOrigEffectiveDate (String input) {
        setElement(input, 12);
    }

    /**
     * 
     * Rated Disability original effective date - format is MMDDYYYY
     * 
     * @return String A value of RatedDisabilityOrigEffectiveDate
     */
    public String getRatedDisabilityOrigEffectiveDate() {
        return getElement(12);
    }  
    
    /**
     * 
     * Rated Disability current effective date - format is MMDDYYYY
     * 
     * @param input
     *            A value of RatedDisabilityCurrentEffectiveDate
     */
    public void setRatedDisabilityCurrentEffectiveDate (String input) {
        setElement(input, 13); 
    }

    /**
     * 
     * Rated Disability current effective date - format is MMDDYYYY
     * 
     * @return String A value of RatedDisabilityCurrentEffectiveDate
     */
    public String getRatedDisabilityCurrentEffectiveDate() {
        return getElement(13);
    }     
}