/********************************************************************
 * Copyright  2004 VHA. All rights reserved
 ********************************************************************/

package gov.va.med.fw.hl7.segment;

// Java classes
import java.util.List;

// Framework classes
import gov.va.med.fw.hl7.InvalidSegmentException;
import gov.va.med.fw.hl7.Segment;
import gov.va.med.fw.hl7.constants.SegmentConstants;

/**
 * A ZFE segment containing information about batch trailer
 * 
 * @author Alex Yoon
 * @version 1.0
 */

public class ZFE extends Segment {

    /**
	 * An instance of serialVersionUID
	 */
	private static final long serialVersionUID = 7959401926351789747L;

	/**
     * Constructs an empty segment. This constructor is ususally usd by derived
     * segment classes to populate its own segment data.
     */
    public ZFE() {
        super();
        setName(SegmentConstants.ZFE);
    }

    /**
     * Constructs a segment of elements from a raw data string. This constructor
     * is usually used to construct an inbound message from a raw message's
     * segment data.
     * 
     * @param data
     *            A raw data string
     * @throws InvalidSegmentException
     *             Thrown if failed to construct a segment due to missing
     *             parameters or invalid segment data to parse.
     */
    public ZFE(String data) throws InvalidSegmentException {
        super(data);
    }

    /**
     * Constructs a segment of elements from a raw data string, an array of
     * valid encoding character strings, and a element delimiter string. This
     * constructor is usually used to construct an inbound message from a raw
     * message's segment data, a specific encoding characters, and a field
     * delimiter.
     * 
     * @param encodingChars
     *            An array of encoding characters
     * @param elementDelimiter
     *            A field delimiter
     * @param segmentData
     *            A raw data string
     * @throws InvalidSegmentException
     *             Thrown if failed to construct a segment due to missing
     *             parameters or invalid segment data to parse.
     */
    public ZFE(String[] encodingChars, String elementDelimiter,
            String segmentData) throws InvalidSegmentException {
        super(encodingChars, elementDelimiter, segmentData);
    }

    /**
     * Constructs a segment from a segment name and a list of elements. This
     * constructor is usually used to construct an outbound message from a list
     * of specific segment elements, and a segment name.
     * 
     * @param name
     *            A segment name
     * @param elements
     *            A list of segment elements
     * @throws InvalidSegmentException
     *             Thrown if failed to construct a segment due to missing
     *             parameters or invalid segment data to parse.
     */
    public ZFE(String name, List elements) throws InvalidSegmentException {
        super(name, elements);
    }

    /**
     * Constructs a segment from an array of encoding characters, a field
     * delimiter, a segment name, and a list of field data. This constructor is
     * usually used to construct an outbound message from a list of specific
     * segment elements, element delimiter, and encoding characters.
     * 
     * @param encodingChars
     *            An array of encoding characters
     * @param elementDelimiter
     *            A field delimiter
     * @param segmentName
     *            A segment name
     * @param segmentElements
     *            A list of field data
     * @throws InvalidSegmentException
     *             Thrown if failed to construct a segment due to missing
     *             parameters or invalid segment data to parse.
     */
    public ZFE(String[] encodingChars, String elementDelimiter,
            String segmentName, List segmentElements)
            throws InvalidSegmentException {
        super(encodingChars, elementDelimiter, segmentName, segmentElements);
    }

    /**
     * 
     * Sequential Number
     * 
     * @param input
     *            A value of SetID
     */
    public void setSetID(String input) {
        setElement(input, 0);
    }

    /**
     * 
     * Sequential Number
     * 
     * @return String A value of SetID
     */
    public String getSetID() {
        return getElement(0);
    }

    /**
     * 
     * TREATMENT TYPE CODE field (.095)
     * 
     * @param input
     *            A value of TreatmentCode
     */
    public void setTreatmentCode(String input) {
        setElement(input, 1);
    }

    /**
     * 
     * TREATMENT TYPE CODE field (.095)
     * 
     * @return String A value of TreatmentCode
     */
    public String getTreatmentCode() {
        return getElement(1);
    }

    /**
     * 
     * FEE PROGRAM field (.03)
     * 
     * @param input
     *            A value of FeeBasisProgram
     */
    public void setFeeBasisProgram(String input) {
        setElement(input, 2);
    }

    /**
     * 
     * FEE PROGRAM field (.03)
     * 
     * @return String A value of FeeBasisProgram
     */
    public String getFeeBasisProgram() {
        return getElement(2);
    }

    /**
     * 
     * AUTHORIZATION FROM DATE field (.01) (IMPRECISE)
     * 
     * @param input
     *            A value of AuthorizationFromDate
     */
    public void setAuthorizationFromDate(String input) {
        setElement(input, 3);
    }

    /**
     * 
     * AUTHORIZATION FROM DATE field (.01) (IMPRECISE)
     * 
     * @return String A value of AuthorizationFromDate
     */
    public String getAuthorizationFromDate() {
        return getElement(3);
    }

    /**
     * 
     * AUTHORIZATION TO DATE field (.02) (IMPRECISE)
     * 
     * @param input
     *            A value of AuthorizationToDate
     */
    public void setAuthorizationToDate(String input) {
        setElement(input, 4);
    }

    /**
     * 
     * AUTHORIZATION TO DATE field (.02) (IMPRECISE)
     * 
     * @return String A value of AuthorizationToDate
     */
    public String getAuthorizationToDate() {
        return getElement(4);
    }
}