/********************************************************************
 * Copyright  2004 VHA. All rights reserved
 ********************************************************************/

package gov.va.med.fw.hl7.segment;

// Java classes
import java.util.List;

// Framework classes
import gov.va.med.fw.hl7.InvalidSegmentException;
import gov.va.med.fw.hl7.Segment;
import gov.va.med.fw.hl7.constants.SegmentConstants;

/**
 * A ZDP segment containing information about batch trailer
 * 
 * @author Alex Yoon
 * @version 1.0
 */

public class ZDP extends Segment {

    /**
	 * An instance of serialVersionUID
	 */
	private static final long serialVersionUID = 319499040129586373L;

	/**
     * Constructs an empty segment. This constructor is ususally usd by derived
     * segment classes to populate its own segment data.
     */
    public ZDP() {
        super();
        setName(SegmentConstants.ZDP);
    }

    /**
     * Constructs a segment of elements from a raw data string. This constructor
     * is usually used to construct an inbound message from a raw message's
     * segment data.
     * 
     * @param data
     *            A raw data string
     * @throws InvalidSegmentException
     *             Thrown if failed to construct a segment due to missing
     *             parameters or invalid segment data to parse.
     */
    public ZDP(String data) throws InvalidSegmentException {
        super(data);
    }

    /**
     * Constructs a segment of elements from a raw data string, an array of
     * valid encoding character strings, and a element delimiter string. This
     * constructor is usually used to construct an inbound message from a raw
     * message's segment data, a specific encoding characters, and a field
     * delimiter.
     * 
     * @param encodingChars
     *            An array of encoding characters
     * @param elementDelimiter
     *            A field delimiter
     * @param segmentData
     *            A raw data string
     * @throws InvalidSegmentException
     *             Thrown if failed to construct a segment due to missing
     *             parameters or invalid segment data to parse.
     */
    public ZDP(String[] encodingChars, String elementDelimiter,
            String segmentData) throws InvalidSegmentException {
        super(encodingChars, elementDelimiter, segmentData);
    }

    /**
     * Constructs a segment from a segment name and a list of elements. This
     * constructor is usually used to construct an outbound message from a list
     * of specific segment elements, and a segment name.
     * 
     * @param name
     *            A segment name
     * @param elements
     *            A list of segment elements
     * @throws InvalidSegmentException
     *             Thrown if failed to construct a segment due to missing
     *             parameters or invalid segment data to parse.
     */
    public ZDP(String name, List elements) throws InvalidSegmentException {
        super(name, elements);
    }

    /**
     * Constructs a segment from an array of encoding characters, a field
     * delimiter, a segment name, and a list of field data. This constructor is
     * usually used to construct an outbound message from a list of specific
     * segment elements, element delimiter, and encoding characters.
     * 
     * @param encodingChars
     *            An array of encoding characters
     * @param elementDelimiter
     *            A field delimiter
     * @param segmentName
     *            A segment name
     * @param segmentElements
     *            A list of field data
     * @throws InvalidSegmentException
     *             Thrown if failed to construct a segment due to missing
     *             parameters or invalid segment data to parse.
     */
    public ZDP(String[] encodingChars, String elementDelimiter,
            String segmentName, List segmentElements)
            throws InvalidSegmentException {
        super(encodingChars, elementDelimiter, segmentName, segmentElements);
    }

    /**
     * 
     * SEQUENTIAL NUMBER
     * 
     * @param input
     *            A value of SetID
     */
    public void setSetID(String input) {
        setElement(input, 0);
    }

    /**
     * 
     * SEQUENTIAL NUMBER
     * 
     * @return String A value of SetID
     */
    public String getSetID() {
        return getElement(0);
    }

    /**
     * 
     * NAME (.01)
     * 
     * @param input
     *            A value of Name
     */
    public void setDependentName(String input) {
        setElement(input, 1);
    }

    /**
     * 
     * NAME (.01)
     * 
     * @return String A value of Name
     */
    public String getDependentName() {
        return getElement(1);
    }

    /**
     * 
     * SEX (.02)
     * 
     * @param input
     *            A value of Sex
     */
    public void setSex(String input) {
        setElement(input, 2);
    }

    /**
     * 
     * SEX (.02)
     * 
     * @return String A value of Sex
     */
    public String getSex() {
        return getElement(2);
    }

    /**
     * 
     * DATE OF BIRTH (.03) (Imprecise)
     * 
     * @param input
     *            A value of BirthDate
     */
    public void setBirthDate(String input) {
        setElement(input, 3);
    }

    /**
     * 
     * DATE OF BIRTH (.03) (Imprecise)
     * 
     * @return String A value of BirthDate
     */
    public String getBirthDate() {
        return getElement(3);
    }

    /**
     * 
     * SOCIAL SECURITY NUMBER (.09)
     * 
     * @param input
     *            A value of SSN
     */
    public void setSSN(String input) {
        setElement(input, 4);
    }

    /**
     * 
     * SOCIAL SECURITY NUMBER (.09)
     * 
     * @return String A value of SSN
     */
    public String getSSN() {
        return getElement(4);
    }

    /**
     * 
     * RELATIONSHIP (File# 408.12) (.02)
     * 
     * @param input
     *            A value of Relationship
     */
    public void setRelationship(String input) {
        setElement(input, 5);
    }

    /**
     * 
     * RELATIONSHIP (File# 408.12) (.02)
     * 
     * @return String A value of Relationship
     */
    public String getRelationship() {
        return getElement(5);
    }

    /**
     * 
     * IEN of entry in File #408.12
     * 
     * @param input
     *            A value of InternalEntryNumber
     */
    public void setInternalEntryNumber(String input) {
        setElement(input, 6);
    }

    /**
     * 
     * IEN of entry in File #408.12
     * 
     * @return String A value of InternalEntryNumber
     */
    public String getInternalEntryNumber() {
        return getElement(6);
    }

    /**
     * 
     * No corresponding VISTA field at this time.
     * 
     * @param input
     *            A value of SpouseMaidenName
     */
    public void setSpouseMaidenName(String input) {
        setElement(input, 7);
    }

    /**
     * 
     * No corresponding VISTA field at this time.
     * 
     * @return String A value of SpouseMaidenName
     */
    public String getSpouseMaidenName() {
        return getElement(7);
    }

    /**
     * 
     * EFFECTIVE DATE (MULTIPLE 408.1275), FIELD .01 (Imprecise) OR 0000
     * 
     * @param input
     *            A value of DependentDate
     */
    public void setDependentDate(String input) {
        setElement(input, 8);
    }

    /**
     * 
     * EFFECTIVE DATE (MULTIPLE 408.1275), FIELD .01 (Imprecise) OR 0000
     * 
     * @return String A value of DependentDate
     */
    public String getDependentDate() {
        return getElement(8);
    }
    
    /**
     * Pseudo SSN Reason - Veteran
     * 
     */
    public void setPseudoSSNReason(String input) {
        setElement(input, 9);
    }     
    /**
     * Pseudo SSN Reason - Veteran
     * 
     * @return String value of PseudoSSNReason
     */
    public String getPseudoSSNReason() {
        return getElement(9);
    }
    
    /**
     * InactivationDate
     * 
     * @param input
     */
    public void setInactivationDate(String input) {
        setElement(input, 10);
    }     
    /**
     * InactivationDate
     * 
     * @return String value of InactivationDate
     */
    public String getInactivationDate() {
        return getElement(10);
    }   
    
    /**
     * SSNVerificationStatus
     * 
     * @param input
     */
    public void setSSNVerificationStatus(String input) {
        setElement(input, 11);
    }     
    /**
     * SSNVerificationStatus
     * 
     * @return String value of InactivationDate
     */
    public String getSSNVerificationStatus () {
        return getElement(11);
    }            
 
    /**
     * SPOUSE ADDRESS (1.2, 1.3, 1.4, 1.5, 1.6, 1.7, (x.x) 
     * 
     * @param input
     */
    public void setAddress(String input) {
        setElement(input, 12);
    }  
    
    /**
     * SPOUSE ADDRESS
     * 
     * @return String value of Address
     */
    public String getAddress() {
        return getElement(12);
    }            
    
    /**
     * SPOUSE HOME PHONE NUMBER (1.8)
     * 
     * @param input
     */
    public void setHomePhone(String input) {
        setElement(input, 13);
    }  
    
    /**
     * SPOUSE HOME PHONE NUMBER
     * 
     * @return String value of HomePhone
     */
    public String getHomePhone() {
        return getElement(13);
    }            
   
}