/********************************************************************
 * Copyright  2004 VHA. All rights reserved
 ********************************************************************/

package gov.va.med.fw.hl7.segment;

// Java classes
import java.util.List;

// Framework classes
import gov.va.med.fw.hl7.InvalidSegmentException;
import gov.va.med.fw.hl7.Segment;
import gov.va.med.fw.hl7.constants.SegmentConstants;

/**
 * A ZCT segment containing information about batch trailer
 * 
 * @author Alex Yoon
 * @version 1.0
 */

public class ZCT extends Segment {

    /**
	 * An instance of serialVersionUID
	 */
	private static final long serialVersionUID = -3647606501673598665L;

	/**
     * Constructs an empty segment. This constructor is ususally usd by derived
     * segment classes to populate its own segment data.
     */
    public ZCT() {
        super();
        setName(SegmentConstants.ZCT);
    }

    /**
     * Constructs a segment of elements from a raw data string. This constructor
     * is usually used to construct an inbound message from a raw message's
     * segment data.
     * 
     * @param data
     *            A raw data string
     * @throws InvalidSegmentException
     *             Thrown if failed to construct a segment due to missing
     *             parameters or invalid segment data to parse.
     */
    public ZCT(String data) throws InvalidSegmentException {
        super(data);
    }

    /**
     * Constructs a segment of elements from a raw data string, an array of
     * valid encoding character strings, and a element delimiter string. This
     * constructor is usually used to construct an inbound message from a raw
     * message's segment data, a specific encoding characters, and a field
     * delimiter.
     * 
     * @param encodingChars
     *            An array of encoding characters
     * @param elementDelimiter
     *            A field delimiter
     * @param segmentData
     *            A raw data string
     * @throws InvalidSegmentException
     *             Thrown if failed to construct a segment due to missing
     *             parameters or invalid segment data to parse.
     */
    public ZCT(String[] encodingChars, String elementDelimiter,
            String segmentData) throws InvalidSegmentException {
        super(encodingChars, elementDelimiter, segmentData);
    }

    /**
     * Constructs a segment from a segment name and a list of elements. This
     * constructor is usually used to construct an outbound message from a list
     * of specific segment elements, and a segment name.
     * 
     * @param name
     *            A segment name
     * @param elements
     *            A list of segment elements
     * @throws InvalidSegmentException
     *             Thrown if failed to construct a segment due to missing
     *             parameters or invalid segment data to parse.
     */
    public ZCT(String name, List elements) throws InvalidSegmentException {
        super(name, elements);
    }

    /**
     * Constructs a segment from an array of encoding characters, a field
     * delimiter, a segment name, and a list of field data. This constructor is
     * usually used to construct an outbound message from a list of specific
     * segment elements, element delimiter, and encoding characters.
     * 
     * @param encodingChars
     *            An array of encoding characters
     * @param elementDelimiter
     *            A field delimiter
     * @param segmentName
     *            A segment name
     * @param segmentElements
     *            A list of field data
     * @throws InvalidSegmentException
     *             Thrown if failed to construct a segment due to missing
     *             parameters or invalid segment data to parse.
     */
    public ZCT(String[] encodingChars, String elementDelimiter,
            String segmentName, List segmentElements)
            throws InvalidSegmentException {
        super(encodingChars, elementDelimiter, segmentName, segmentElements);
    }

    /**
     * 
     * SEQUENTIAL NUMBER
     * 
     * @param input
     *            A value of SetID
     */
    public void setSetID(String input) {
        setElement(input, 0);
    }

    /**
     * 
     * SEQUENTIAL NUMBER
     * 
     * @return String A value of SetID
     */
    public String getSetID() {
        return getElement(0);
    }

    /**
     * 
     * 1=NOK, 2=2nd NOK, 3=e-contact, 4=2nd e-contact, 5=designee (See table
     * 6.19.1 below)
     * 
     * @param input
     *            A value of ContactType
     */
    public void setContactType(String input) {
        setElement(input, 1);
    }

    /**
     * 
     * 1=NOK, 2=2nd NOK, 3=e-contact, 4=2nd e-contact, 5=designee (See table
     * 6.19.1 below)
     * 
     * @return String A value of ContactType
     */
    public String getContactType() {
        return getElement(1);
    }

    /**
     * 
     * NAME of contact person
     * 
     * @param input
     *            A value of ContactName
     */
    public void setContactName(String input) {
        setElement(input, 2);
    }

    /**
     * 
     * NAME of contact person
     * 
     * @return String A value of ContactName
     */
    public String getContactName() {
        return getElement(2);
    }

    /**
     * 
     * RELATIONSHIP TO PATIENT of contact person
     * 
     * @param input
     *            A value of ContactRelationship
     */
    public void setContactRelationship(String input) {
        setElement(input, 3);
    }

    /**
     * 
     * RELATIONSHIP TO PATIENT of contact person
     * 
     * @return String A value of ContactRelationship
     */
    public String getContactRelationship() {
        return getElement(3);
    }

    /**
     * 
     * ADDRESS of contact person
     * 
     * @param input
     *            A value of ContactAddress
     */
    public void setContactAddress(String input) {
        setElement(input, 4);
    }

    /**
     * 
     * ADDRESS of contact person
     * 
     * @return String A value of ContactAddress
     */
    public String getContactAddress() {
        return getElement(4);
    }

    /**
     * 
     * PHONE NUMBER of contact person
     * 
     * @param input
     *            A value of ContactPhoneNumber
     */
    public void setContactPhoneNumber(String input) {
        setElement(input, 5);
    }

    /**
     * 
     * PHONE NUMBER of contact person
     * 
     * @return String A value of ContactPhoneNumber
     */
    public String getContactPhoneNumber() {
        return getElement(5);
    }

    /**
     * 
     * WORK PHONE NUMBER of contact person
     * 
     * @param input
     *            A value of ContactWorkPhoneNumber
     */
    public void setContactWorkPhoneNumber(String input) {
        setElement(input, 6);
    }

    /**
     * 
     * WORK PHONE NUMBER of contact person
     * 
     * @return String A value of ContactWorkPhoneNumber
     */
    public String getContactWorkPhoneNumber() {
        return getElement(6);
    }

    /**
     * 
     * ADDRESS SAME AS PATIENT'S
     * 
     * @param input
     *            A value of ContactAddressSameAsNOK
     */
    public void setContactAddressSameAsNOK(String input) {
        setElement(input, 7);
    }

    /**
     * 
     * ADDRESS SAME AS PATIENT'S
     * 
     * @return String A value of ContactAddressSameAsNOK
     */
    public String getContactAddressSameAsNOK() {
        return getElement(7);
    }

    /**
     * 
     * SAME AS NOK?
     * 
     * @param input
     *            A value of ContactPersonSameAsNOK
     */
    public void setContactPersonSameAsNOK(String input) {
        setElement(input, 8);
    }

    /**
     * 
     * SAME AS NOK?
     * 
     * @return String A value of ContactPersonSameAsNOK
     */
    public String getContactPersonSameAsNOK() {
        return getElement(8);
    }
    
   
    /**
     * Contact Date/Time last updated. Needed for every contact type
     * 
     * @param input
     */
    public void setLastUpdateDate(String input) {
        setElement(input, 9);
    }    
    /**
     * Contact Date/Time last updated. Needed for every contact type
     * 
     * @return LastUpdateDate
     */
    public String getLastUpdateDate()
    {
        return getElement(9);
    }
}