/********************************************************************
 * Copyright  2004 VHA. All rights reserved
 ********************************************************************/

package gov.va.med.fw.hl7.segment;

// Java classes
import java.util.List;

// Framework classes
import gov.va.med.fw.hl7.InvalidSegmentException;
import gov.va.med.fw.hl7.Segment;
import gov.va.med.fw.hl7.constants.SegmentConstants;

/**
 * A ZCD segment containing information about batch trailer
 * 
 * @author Alex Yoon
 * @version 1.0
 */

public class ZCD extends Segment {

    /**
	 * An instance of serialVersionUID
	 */
	private static final long serialVersionUID = 2397771658461756159L;

	/**
     * Constructs an empty segment. This constructor is ususally usd by derived
     * segment classes to populate its own segment data.
     */
    public ZCD() {
        super();
        setName(SegmentConstants.ZCD);
    }

    /**
     * Constructs a segment of elements from a raw data string. This constructor
     * is usually used to construct an inbound message from a raw message's
     * segment data.
     * 
     * @param data
     *            A raw data string
     * @throws InvalidSegmentException
     *             Thrown if failed to construct a segment due to missing
     *             parameters or invalid segment data to parse.
     */
    public ZCD(String data) throws InvalidSegmentException {
        super(data);
    }

    /**
     * Constructs a segment of elements from a raw data string, an array of
     * valid encoding character strings, and a element delimiter string. This
     * constructor is usually used to construct an inbound message from a raw
     * message's segment data, a specific encoding characters, and a field
     * delimiter.
     * 
     * @param encodingChars
     *            An array of encoding characters
     * @param elementDelimiter
     *            A field delimiter
     * @param segmentData
     *            A raw data string
     * @throws InvalidSegmentException
     *             Thrown if failed to construct a segment due to missing
     *             parameters or invalid segment data to parse.
     */
    public ZCD(String[] encodingChars, String elementDelimiter,
            String segmentData) throws InvalidSegmentException {
        super(encodingChars, elementDelimiter, segmentData);
    }

    /**
     * Constructs a segment from a segment name and a list of elements. This
     * constructor is usually used to construct an outbound message from a list
     * of specific segment elements, and a segment name.
     * 
     * @param name
     *            A segment name
     * @param elements
     *            A list of segment elements
     * @throws InvalidSegmentException
     *             Thrown if failed to construct a segment due to missing
     *             parameters or invalid segment data to parse.
     */
    public ZCD(String name, List elements) throws InvalidSegmentException {
        super(name, elements);
    }

    /**
     * Constructs a segment from an array of encoding characters, a field
     * delimiter, a segment name, and a list of field data. This constructor is
     * usually used to construct an outbound message from a list of specific
     * segment elements, element delimiter, and encoding characters.
     * 
     * @param encodingChars
     *            An array of encoding characters
     * @param elementDelimiter
     *            A field delimiter
     * @param segmentName
     *            A segment name
     * @param segmentElements
     *            A list of field data
     * @throws InvalidSegmentException
     *             Thrown if failed to construct a segment due to missing
     *             parameters or invalid segment data to parse.
     */
    public ZCD(String[] encodingChars, String elementDelimiter,
            String segmentName, List segmentElements)
            throws InvalidSegmentException {
        super(encodingChars, elementDelimiter, segmentName, segmentElements);
    }

    /**
     * 
     * SEQUENTIAL NUMBER
     * 
     * @param input
     *            A value of SetID
     */
    public void setSetID(String input) {
        setElement(input, 0);
    }

    /**
     * 
     * SEQUENTIAL NUMBER
     * 
     * @return String A value of SetID
     */
    public String getSetID() {
        return getElement(0);
    }

    /**
     * 
     * REVIEW DATE (.394) (Precise) (VISTA Messaging HL7)
     * 
     * @param input
     *            A value of ReviewDate
     */
    public void setReviewDate(String input) {
        setElement(input, 1);
    }

    /**
     * 
     * REVIEW DATE (.394) (Precise) (VISTA Messaging HL7)
     * 
     * @return String A value of ReviewDate
     */
    public String getReviewDate() {
        return getElement(1);
    }

    /**
     * 
     * DECIDED BY (.391) (VISTA Messaging HL7)
     * 
     * @param input
     *            A value of DecidedBy
     */
    public void setDecidedBy(String input) {
        setElement(input, 2);
    }

    /**
     * 
     * DECIDED BY (.391) (VISTA Messaging HL7)
     * 
     * @return String A value of DecidedBy
     */
    public String getDecidedBy() {
        return getElement(2);
    }

    /**
     * 
     * FACILITY MAKING DETERMINATION (.393) (VISTA Messaging HL7)
     * 
     * @param input
     *            A value of DeterminationFacility
     */
    public void setDeterminationFacility(String input) {
        setElement(input, 3);
    }

    /**
     * 
     * FACILITY MAKING DETERMINATION (.393) (VISTA Messaging HL7)
     * 
     * @return String A value of DeterminationFacility
     */
    public String getDeterminationFacility() {
        return getElement(3);
    }

    /**
     * 
     * DATE OF DECISION (.392) (Precise) (VISTA Messaging HL7)
     * 
     * @param input
     *            A value of DecisionDate
     */
    public void setDecisionDate(String input) {
        setElement(input, 4);
    }

    /**
     * 
     * DATE OF DECISION (.392) (Precise) (VISTA Messaging HL7)
     * 
     * @return String A value of DecisionDate
     */
    public String getDecisionDate() {
        return getElement(4);
    }

    /**
     * 
     * METHOD OF DETERMINATION (.395) (VISTA Messaging HL7)
     * 
     * @param input
     *            A value of DeterminationMethod
     */
    public void setDeterminationMethod(String input) {
        setElement(input, 5);
    }

    /**
     * 
     * METHOD OF DETERMINATION (.395) (VISTA Messaging HL7)
     * 
     * @return String A value of DeterminationMethod
     */
    public String getDeterminationMethod() {
        return getElement(5);
    }

    /**
     * 
     * DIAGNOSIS field (.396)
     * 
     * @param input
     *            A value of Diagnosis
     */
    public void setDiagnosis(String input) {
        setElement(input, 6);
    }

    /**
     * 
     * DIAGNOSIS field (.396)
     * 
     * @return String A value of Diagnosis
     */
    public String getDiagnosis() {
        return getElement(6);
    }

    /**
     * 
     * PROCEDURE field (.397)
     * 
     * @param input
     *            A value of Procedure
     */
    public void setProcedure(String input) {
        setElement(input, 7);
    }

    /**
     * 
     * PROCEDURE field (.397)
     * 
     * @return String A value of Procedure
     */
    public String getProcedure() {
        return getElement(7);
    }

    /**
     * 
     * AFFECTED EXTREMITY (#2.397)(1)
     * 
     * @param input
     *            A value of AffectedExtremity
     */
    public void setAffectedExtremity(String input) {
        setElement(input, 8);
    }

    /**
     * 
     * AFFECTED EXTREMITY (#2.397)(1)
     * 
     * @return String A value of AffectedExtremity
     */
    public String getAffectedExtremity() {
        return getElement(8);
    }

    /**
     * 
     * CONDITION (#2.398) (.01)
     * 
     * @param input
     *            A value of Condition
     */
    public void setCondition(String input) {
        setElement(input, 9);
    }

    /**
     * 
     * CONDITION (#2.398) (.01)
     * 
     * @return String A value of Condition
     */
    public String getCondition() {
        return getElement(9);
    }

    /**
     * 
     * SCORE (#2.398) (1)
     * 
     * @param input
     *            A value of Score
     */
    public void setScore(String input) {
        setElement(input, 10);
    }

    /**
     * 
     * SCORE (#2.398) (1)
     * 
     * @return String A value of Score
     */
    public String getScore() {
        return getElement(10);
    }

    /**
     * 
     * VETERAN CATASTROPHICALLY DISABLED (.39) (VISTA Messaging HL7)
     * 
     * @param input
     *            A value of CatastrophicallyDisabled
     */
    public void setCatastrophicallyDisabled(String input) {
        setElement(input, 11);
    }

    /**
     * 
     * VETERAN CATASTROPHICALLY DISABLED (.39) (VISTA Messaging HL7)
     * 
     * @return String A value of CatastrophicallyDisabled
     */
    public String getCatastrophicallyDisabled() {
        return getElement(11);
    }

    /**
     * 
     * PERMANENT INDICATOR (#2.398) (2)
     * 
     * @param input
     *            A value of PermanentIndicator
     */
    public void setPermanentIndicator(String input) {
        setElement(input, 12);
    }

    /**
     * 
     * PERMANENT INDICATOR (#2.398) (2)
     * 
     * @return String A value of PermanentIndicator
     */
    public String getPermanentIndicator() {
        return getElement(12);
    }
    
    /** Date Veteran requested CD evaluation (Precise) (VISTA Messaging HL7)
    * 
    * @param input
    *            A value of VeteranRequestedCDReviewDate
    */
   public void setVeteranRequestedCDReviewDate(String input) {
       setElement(input, 13);
   }

   /**
    * 
    * VeteranRequestedCDReviewDate (Precise) (VISTA Messaging HL7)
    * 
    * @return String A value of VeteranRequestedCDReviewDate
    */
   public String getVeteranRequestedCDReviewDate() {
       return getElement(13);
   }    
    
   
   /** Date facility initiated review (Precise) (VISTA Messaging HL7)
    * 
    * @param input
    *            A value of FacilityInitiatedCDReviewDate
    */
   public void setFacilityInitiatedCDReviewDate(String input) {
       setElement(input, 14);
   }

   /**
    * 
    * Date facility initiated review (Precise) (VISTA Messaging HL7)
    * 
    * @return String A value of FacilityInitiatedCDReviewDate
    */
   public String getFacilityInitiatedCDReviewDate() {
       return getElement(14);
   } 
   
   /** Date Veteran was notified (Precise) (VISTA Messaging HL7)
    * 
    * @param input
    *            A value of VeteranWasNotifiedCDDate
    */
   public void setVeteranWasNotifiedCDDate(String input) {
       setElement(input, 15);
   }

   /**
    * 
    * Date Veteran was notified (Precise) (VISTA Messaging HL7)
    * 
    * @return String A value of VeteranWasNotifiedCDDate
    */
   public String getVeteranWasNotifiedCDDate() {
       return getElement(15);
   } 
   
   /**
    * 
    * Descriptor field (#2.401)
    * 
    * @param input
    *            A value of Descriptor
    */
   public void setDescriptor(String input) {
       setElement(input, 16);
   }

   /**
    * 
    * Descriptor field (#2.401)
    * 
    * @return String A value of Descriptor
    */
   public String getDescriptor() {
       return getElement(16);
   }

    
}