/********************************************************************
 * Copyright  2004 VHA. All rights reserved
 ********************************************************************/

package gov.va.med.fw.hl7.segment;

// Java classes
import java.util.List;

// Framework classes
import gov.va.med.fw.hl7.InvalidSegmentException;
import gov.va.med.fw.hl7.Segment;
import gov.va.med.fw.hl7.constants.SegmentConstants;

/**
 * A MFI segment containing information about batch trailer
 * 
 * @author Alex Yoon
 * @version 1.0
 */

public class MFI extends Segment {

    /**
	 * An instance of serialVersionUID
	 */
	private static final long serialVersionUID = 5901890592354160720L;

	/**
     * Constructs an empty segment. This constructor is ususally usd by derived
     * segment classes to populate its own segment data.
     */
    public MFI() {
        super();
        setName(SegmentConstants.MFI);
    }

    /**
     * Constructs a segment of elements from a raw data string. This constructor
     * is usually used to construct an inbound message from a raw message's
     * segment data.
     * 
     * @param data
     *            A raw data string
     * @throws InvalidSegmentException
     *             Thrown if failed to construct a segment due to missing
     *             parameters or invalid segment data to parse.
     */
    public MFI(String data) throws InvalidSegmentException {
        super(data);
    }

    /**
     * Constructs a segment of elements from a raw data string, an array of
     * valid encoding character strings, and a element delimiter string. This
     * constructor is usually used to construct an inbound message from a raw
     * message's segment data, a specific encoding characters, and a field
     * delimiter.
     * 
     * @param encodingChars
     *            An array of encoding characters
     * @param elementDelimiter
     *            A field delimiter
     * @param segmentData
     *            A raw data string
     * @throws InvalidSegmentException
     *             Thrown if failed to construct a segment due to missing
     *             parameters or invalid segment data to parse.
     */
    public MFI(String[] encodingChars, String elementDelimiter,
            String segmentData) throws InvalidSegmentException {
        super(encodingChars, elementDelimiter, segmentData);
    }

    /**
     * Constructs a segment from a segment name and a list of elements. This
     * constructor is usually used to construct an outbound message from a list
     * of specific segment elements, and a segment name.
     * 
     * @param name
     *            A segment name
     * @param elements
     *            A list of segment elements
     * @throws InvalidSegmentException
     *             Thrown if failed to construct a segment due to missing
     *             parameters or invalid segment data to parse.
     */
    public MFI(String name, List elements) throws InvalidSegmentException {
        super(name, elements);
    }

    /**
     * Constructs a segment from an array of encoding characters, a field
     * delimiter, a segment name, and a list of field data. This constructor is
     * usually used to construct an outbound message from a list of specific
     * segment elements, element delimiter, and encoding characters.
     * 
     * @param encodingChars
     *            An array of encoding characters
     * @param elementDelimiter
     *            A field delimiter
     * @param segmentName
     *            A segment name
     * @param segmentElements
     *            A list of field data
     * @throws InvalidSegmentException
     *             Thrown if failed to construct a segment due to missing
     *             parameters or invalid segment data to parse.
     */
    public MFI(String[] encodingChars, String elementDelimiter,
            String segmentName, List segmentElements)
            throws InvalidSegmentException {
        super(encodingChars, elementDelimiter, segmentName, segmentElements);
    }

    /**
     * 
     * 5 COMPONENTS: 1  Segment (e.g., ZEG, LOC, Z..) 2  Segment Name (e.g.,
     * ENROLLMENT GROUP THRESHOLD UPDATE) 3  HL70175 4  File Number from which
     * data is derived (ie., 742060) 5  File Name from which data is derived
     * (ie., ENROLLMENT GROUP THRESHOLD )
     * 
     * @param input
     *            A value of MasterFileID
     */
    public void setMasterFileID(String input) {
        setElement(input, 0);
    }

    /**
     * 
     * 5 COMPONENTS: 1  Segment (e.g., ZEG, LOC, Z..) 2  Segment Name (e.g.,
     * ENROLLMENT GROUP THRESHOLD UPDATE) 3  HL70175 4  File Number from which
     * data is derived (ie., 742060) 5  File Name from which data is derived
     * (ie., ENROLLMENT GROUP THRESHOLD )
     * 
     * @return String A value of MasterFileID
     */
    public String getMasterFileID() {
        return getElement(0);
    }

    /**
     * 
     * NOT USED
     * 
     * @param input
     *            A value of MasterFileApplicationID
     */
    public void setMasterFileApplicationID(String input) {
        setElement(input, 1);
    }

    /**
     * 
     * NOT USED
     * 
     * @return String A value of MasterFileApplicationID
     */
    public String getMasterFileApplicationID() {
        return getElement(1);
    }

    /**
     * 
     * REP OR UPD
     * 
     * @param input
     *            A value of EventCode
     */
    public void setEventCode(String input) {
        setElement(input, 2);
    }

    /**
     * 
     * REP OR UPD
     * 
     * @return String A value of EventCode
     */
    public String getEventCode() {
        return getElement(2);
    }

    /**
     * 
     * Entered Date Time
     * 
     * @param input
     *            A value of EnteredDatetime
     */
    public void setEnteredDate(String input) {
        setElement(input, 3);
    }

    /**
     * 
     * Entered Date Time
     * 
     * @return String A value of EnteredDatetime
     */
    public String getEnteredDate() {
        return getElement(3);
    }

    /**
     * 
     * NOT USED
     * 
     * @param input
     *            A value of EffectiveDatetime
     */
    public void setEffectiveDate(String input) {
        setElement(input, 4);
    }

    /**
     * 
     * NOT USED
     * 
     * @return String A value of EffectiveDatetime
     */
    public String getEffectiveDate() {
        return getElement(4);
    }

    /**
     * 
     * INDICATES HOW THE ACK IS TO BE SENT  VALUE WILL BE: AL - ALWAYS
     * 
     * @param input
     *            A value of ResponseLevelCode
     */
    public void setResponseLevelCode(String input) {
        setElement(input, 5);
    }

    /**
     * 
     * INDICATES HOW THE ACK IS TO BE SENT  VALUE WILL BE: AL - ALWAYS
     * 
     * @return String A value of ResponseLevelCode
     */
    public String getResponseLevelCode() {
        return getElement(5);
    }
}