/********************************************************************
 * Copyright  2004 VHA. All rights reserved
 ********************************************************************/
package gov.va.med.esr.common.rule.service;

import gov.va.med.esr.common.model.financials.PatientVisitSummary;
import gov.va.med.esr.common.model.lookup.VAFacility;
import gov.va.med.esr.common.model.person.Person;
import gov.va.med.esr.common.util.RuleAbstractTestCase;

import java.util.Date;

/**
 * @author DNS   MANSOG
 * @version 1.0
 */
public class ProcessPatientVisitSummaryTest extends RuleAbstractTestCase {
    public ProcessPatientVisitSummaryTest(String testName) throws Exception {
        super(testName);
        // super.setDefaultRollback(false);
    }
    
    public void testAddVisitData() throws Exception {
        Integer incomeYear = new Integer(2003);

        VAFacility facility = getVaFacility(VAFacility.CODE_ALBUQUERQUE);
        Person person = savePerson(buildSimplePerson());

        // Clone the pristine to have a working version
        Person incoming = (Person) person.clone();
        Person onFile = getPersonService().getPerson(
                person.getPersonEntityKey());
        buildPatientVisitData(incomeYear, facility, incoming);
        try {
            Person result = invokeFinancialRuleService(incomeYear, facility,
                    incoming, onFile);

            PatientVisitSummary inVisitSummary = incoming
                    .getPatientVisitSummary(incomeYear, facility);
            PatientVisitSummary resultVisitSummary = result
                    .getPatientVisitSummary(incomeYear, facility);
            assertNotNull("Result PatientVisitSummary should not be null",
                    resultVisitSummary);
            assertEquals(inVisitSummary.getInpatientDays(), resultVisitSummary
                    .getInpatientDays());
            assertEquals(inVisitSummary.getOutpatientDays(), resultVisitSummary
                    .getOutpatientDays());
            assertEquals(inVisitSummary.getLastVisitDate(), resultVisitSummary
                    .getLastVisitDate());
            assertEquals(inVisitSummary.getFacilityVisited(),
                    resultVisitSummary.getFacilityVisited());
            assertEquals(inVisitSummary.getIncomeYear(), resultVisitSummary
                    .getIncomeYear());

        } catch (Exception e) {
            this.fail("Failed due to exception ", e);
        }
    }

    public void testDeleteExistingVisitData() throws Exception {
        Integer incomeYear = new Integer(2003);

        VAFacility facility = getVaFacility(VAFacility.CODE_ALBUQUERQUE);
        Person p = buildSimplePerson();
        buildPatientVisitData(incomeYear, facility, p);
        Person person = savePerson(p);

        // Clone the pristine to have a working version
        Person incoming = (Person) person.clone();
        Person onFile = getPersonService().getPerson(
                person.getPersonEntityKey());
        PatientVisitSummary vs = incoming.getPatientVisitSummary(incomeYear,
                facility);
        assertNotNull("PatientVisitSummary should not be null", vs);
        assertEquals(facility, vs.getFacilityVisited());
        assertEquals(incomeYear, vs.getIncomeYear());

        incoming.setPatientVisitSummary(incomeYear, facility, null);
        try {
            Person result = invokeFinancialRuleService(incomeYear, facility,
                    incoming, onFile);

            PatientVisitSummary resultVisitSummary = result
                    .getPatientVisitSummary(incomeYear, facility);
            assertNull("Result PatientVisitSummary should be null",
                    resultVisitSummary);

        } catch (Exception e) {
            this.fail("Failed due to exception ", e);
        }
    }

    public void testUpdateExistingVisitData() throws Exception {
        Integer incomeYear = new Integer(2003);

        VAFacility facility = getVaFacility(VAFacility.CODE_ALBUQUERQUE);
        Person p = buildSimplePerson();
        buildPatientVisitData(incomeYear, facility, p);
        Person person = savePerson(p);

        // Clone the pristine to have a working version
        Person incoming = (Person) person.clone();
        Person onFile = getPersonService().getPerson(
                person.getPersonEntityKey());
        PatientVisitSummary vs = incoming.getPatientVisitSummary(incomeYear,
                facility);
        assertNotNull("PatientVisitSummary should not be null", vs);
        assertEquals(facility, vs.getFacilityVisited());
        assertEquals(incomeYear, vs.getIncomeYear());

        Integer newInpatientDays = increment(vs.getInpatientDays());
        vs.setInpatientDays(newInpatientDays);
        Integer newOutpatientDays = increment(vs.getOutpatientDays());
        vs.setOutpatientDays(newOutpatientDays);
        Date newLastVisitDate = new Date();
        vs.setLastVisitDate(newLastVisitDate);
        try {
            Person result = invokeFinancialRuleService(incomeYear, facility,
                    incoming, onFile);

            PatientVisitSummary inVisitSummary = incoming
                    .getPatientVisitSummary(incomeYear, facility);
            PatientVisitSummary resultVisitSummary = result
                    .getPatientVisitSummary(incomeYear, facility);
            assertNotNull("Result PatientVisitSummary should not be null",
                    resultVisitSummary);
            assertEquals(newInpatientDays, resultVisitSummary
                    .getInpatientDays());
            assertEquals(newOutpatientDays, resultVisitSummary
                    .getOutpatientDays());
            assertEquals(newLastVisitDate, resultVisitSummary
                    .getLastVisitDate());
            assertEquals(inVisitSummary.getInpatientDays(), resultVisitSummary
                    .getInpatientDays());
            assertEquals(inVisitSummary.getOutpatientDays(), resultVisitSummary
                    .getOutpatientDays());
            assertEquals(inVisitSummary.getLastVisitDate(), resultVisitSummary
                    .getLastVisitDate());
            assertEquals(inVisitSummary.getFacilityVisited(),
                    resultVisitSummary.getFacilityVisited());
            assertEquals(inVisitSummary.getIncomeYear(), resultVisitSummary
                    .getIncomeYear());

        } catch (Exception e) {
            this.fail("Failed due to exception ", e);
        }
    }

    protected PatientVisitSummary buildPatientVisitData(Integer incomeYear,
            VAFacility facility, Person person) throws Exception {

        PatientVisitSummary visitSummary = person.getPatientVisitSummary(
                incomeYear, facility);
        if (visitSummary == null) {
            visitSummary = createPatientVisitSummary(incomeYear);
            visitSummary.setFacilityVisited(facility);
            person.setPatientVisitSummary(incomeYear, facility, visitSummary);
        }
        return visitSummary;
    }

    protected Integer increment(Integer value) {
        return value != null ? new Integer(value.intValue() + 1) : new Integer(
                0);
    }

    protected Person invokeFinancialRuleService(Integer incomeYear,
            VAFacility facility, Person incoming, Person onFile)
            throws Exception {
        return getFinancialInfoRuleService().processFinancialInfo(incomeYear,
                facility, incoming, onFile);
    }
}