/********************************************************************
 * Copyright  2004 EDS. All rights reserved
 ********************************************************************/
package gov.va.med.esr.common.rule.service;

// Java classes
import java.math.BigDecimal;

import gov.va.med.esr.common.model.lookup.VAFacility;
import gov.va.med.esr.common.model.person.Person;
import gov.va.med.esr.common.model.workload.WorkflowCaseInfo;
import gov.va.med.esr.common.model.ee.VerificationInfo;
import gov.va.med.esr.common.model.ee.ServiceConnectionAward;
import gov.va.med.esr.common.model.person.id.PersonEntityKey;
import gov.va.med.esr.common.util.RuleAbstractTestCase;
import gov.va.med.fw.validation.ValidationMessages;

/**
 * @author Carlos Ruiz
 * @version 1.0
 */
public class ProcessIneligibilityServiceTest extends RuleAbstractTestCase {
	
	
	/**
	 * Constructor for ProcessIneligibilityTest.
	 * @param arg0
	 */
	public ProcessIneligibilityServiceTest(String arg0) throws Exception  {
		super(arg0);					
	}
    
    public void customSetUp() throws Exception {
		super.customSetUp();
    
    }
	
		
	/**
	 * SCAndVACheckAmountAreZero If a messages ineligibility date is not NULL 
	 * and a messages reason of ineligibility is not NULL 
	 * and a messages SC percentage is 0 or NULL 
	 * and a messages VA Check amount is 0 or NULL 
	 * then set a veterans ineligibility reason to a messages ineligibility reason 
	 * and set a veterans VARODecision to a messages VARODecision 
	 * and set a veterans ineligibility date to a messages ineligibility date 
	 */
	public void testSCAndVACheckAmountAreZero() throws Exception {
        Person onFile =  buildSimplePerson();
        Person incoming = this.savePerson(onFile);     
        this.attachIneligibleFactor(incoming);
        this.getEnrollmentRuleService().manageIneligibility(incoming, onFile);
        assertNotNull( onFile.getIneligibilityFactor() );     
        assertFalse( onFile.getVeteran().booleanValue() );        
	}
    
    public void testSCAndVACheckAmountAreZeroFromMessage() throws Exception {
        Person onFile =  buildSimplePerson();
        onFile.setServiceConnectionAward(new ServiceConnectionAward());
        onFile.getServiceConnectionAward().setServiceConnectedIndicator(Boolean.FALSE);
        Person incoming = this.savePerson(onFile);     
        this.attachIneligibleFactor(incoming);
        this.getEnrollmentRuleService().processIneligibility(incoming, onFile, null, new VerificationInfo(),false);
        assertNotNull( onFile.getIneligibilityFactor() );        
    }    
	
	/**
	 * IneligibilityDatasIgnored If a messages ineligibility date is not NULL
	 * and a messages reason of ineligibility is not NULL and a veterans
	 * eligibility status is VERIFIED then generate bulletin with the
	 * description HEC is reviewing the data
	 */
	public void testIneligibilityDatasIgnored() throws Exception {
		// For Messages, WORKS 10/26/2005
        Person onFile =  buildSimplePerson();
        this.setEligibilityStatusToVerified(onFile);
        this.attachVerifiedEnrollment(onFile);
        Person incoming = this.savePerson(onFile);     
        this.attachIneligibleFactorFromSite(incoming);        
        
        VAFacility sendingFacility = this.getLookupService().getVaFacilityByStationNumber(incoming.getSendingFacility());
       
        this.getEnrollmentRuleService().processIneligibility(incoming, onFile,sendingFacility, new VerificationInfo(), false);
        assertNull( onFile.getIneligibilityFactor() );        
	}
    
    public void testIneligibilityDataWithSCandCheck() throws Exception {
        // For Messages, WORKS 10/26/2005
        Person onFile =  buildSimplePerson();
        Person incoming = this.savePerson(onFile);
        this.attachMonetaryBenefitAward(incoming);
        this.attachServiceConnectionAward(incoming, new Integer(1));
        this.attachIneligibleFactorFromSite(incoming);
       
        this.getEnrollmentRuleService().processIneligibility(incoming, onFile, null, new VerificationInfo(),false);
        assertNull( onFile.getIneligibilityFactor() );        
    }       
	

	/**
	 * IneligibilityDataIsMismatched If a messages ineligibility date is not
	 * NULL and a messages reason of ineligibility is not NULL and a messages
	 * A&A is set to YES or a messages house bound is set to YES or a messages
	 * VA pension is set to YES and a veterans eligibility status is PENDING
	 * VERIFICATION or PENDING RE-VERIFICATION then create a new workload case
	 * with a description of data mismatch and a reason set to ineligible data
	 * inconsistent with the VA pension
	 */
	public void testIneligibilityDataIsMismatchedTest() throws Exception {
		
        
        Person onFile =  buildSimplePerson();
        this.setEligibilityStatusToPendingVerification(onFile);
        this.attachVerifiedEnrollment(onFile); // to make existing person
        Person incoming = this.savePerson(onFile);     
        this.attachIneligibleFactorFromSite(incoming);
        this.attachHousebound(incoming);
       
        VerificationInfo vi = new VerificationInfo();
        WorkflowCaseInfo workflowCaseInfo = new WorkflowCaseInfo();
        workflowCaseInfo.setErrorMessage("test");
        workflowCaseInfo.setMessageID("fsdf");
        vi.setWorkflowCaseInfo(workflowCaseInfo);
        this.getEnrollmentRuleService().processIneligibility(incoming, onFile, null, vi,false);
        assertNull( onFile.getIneligibilityFactor() );              
	}

    public void testIneligibilityDateIsDeletedTest() throws Exception {
        
        Person onFile =  buildSimplePerson();
        this.setEligibilityStatusToPendingVerification(onFile);
        this.attachIneligibleFactor(onFile);        
        Person incoming = this.savePerson(onFile);     
        incoming.setIneligibilityFactor(null);

        onFile = this.getPersonService().getPerson((PersonEntityKey)incoming.getEntityKey());
        this.getEnrollmentRuleService().manageIneligibility(incoming, onFile);
        assertNull( onFile.getIneligibilityFactor() );       
    }    

    public void testSCAndVACheckAmountAreGreaterThanZero() throws Exception {
        Person onFile =  buildSimplePerson();
        Person incoming = this.savePerson(onFile);     
        this.attachIneligibleFactor(incoming);
        this.attachMonetaryBenefitAward(incoming);
        incoming.getMonetaryBenefitAward().setCheckAmount(new BigDecimal(250));
        this.attachServiceConnectionAward(incoming, new Integer(1));
        this.getEnrollmentRuleService().manageIneligibility(incoming, onFile);
        assertNotNull( onFile.getIneligibilityFactor() );        
    }
    
    public void testHL7SCAndVACheckAmountAreGreaterThanZero() throws Exception {
        Person onFile =  buildSimplePerson();
        Person incoming = this.savePerson(onFile);     
        this.attachIneligibleFactor(incoming);
        this.attachMonetaryBenefitAward(incoming);
        incoming.getMonetaryBenefitAward().setCheckAmount(new BigDecimal(250));
        this.attachServiceConnectionAward(incoming, new Integer(1));
        this.getEnrollmentRuleService().processIneligibility(incoming, onFile, null, new VerificationInfo(),true);
        assertNull( onFile.getIneligibilityFactor() );        
    }
    
    public void testHL7SCAndVACheckAmountAreGreaterThanZeroValidation() throws Exception {
        Person onFile =  buildSimplePerson();
        Person incoming = this.savePerson(onFile);     
        this.attachIneligibleFactor(incoming);
        this.attachMonetaryBenefitAward(incoming);
        incoming.getMonetaryBenefitAward().setCheckAmount(new BigDecimal(250));
        this.attachServiceConnectionAward(incoming, new Integer(1));
        ValidationMessages messages = this.getRuleValidationService().validateIneligibility(incoming, false);
        if( messages != null && !messages.isEmpty() ) {
            assertTrue(messages.getCount() == 1); 
         }
        else
         fail();   
    }      
}
