/********************************************************************
 * Copyright  2004 VHA. All rights reserved
 ********************************************************************/
package gov.va.med.esr.common.rule.service;

import gov.va.med.esr.common.infra.ImpreciseDate;
import gov.va.med.esr.common.model.financials.DependentFinancials;
import gov.va.med.esr.common.model.financials.FinancialStatement;
import gov.va.med.esr.common.model.financials.Hardship;
import gov.va.med.esr.common.model.financials.IncomeTest;
import gov.va.med.esr.common.model.financials.Income;
import gov.va.med.esr.common.model.financials.IncomeTestStatus;
import gov.va.med.esr.common.model.financials.SpouseFinancials;
import gov.va.med.esr.common.model.lookup.AACIndicator;
import gov.va.med.esr.common.model.lookup.ExpenseType;
import gov.va.med.esr.common.model.lookup.EnrollmentPrioritySubGroup;
import gov.va.med.esr.common.model.lookup.EnrollmentStatus;
import gov.va.med.esr.common.model.lookup.EligibilityType;
import gov.va.med.esr.common.model.lookup.IncomeTestSource;
import gov.va.med.esr.common.model.lookup.IncomeTestType;
import gov.va.med.esr.common.model.lookup.MeansTestStatus;
import gov.va.med.esr.common.model.lookup.EnrollmentPriorityGroup;
import gov.va.med.esr.common.model.lookup.EligibilityStatus;
import gov.va.med.esr.common.model.lookup.IncomeType;
//import org.easymock.MockControl; 
//import gov.va.med.esr.service.impl.MockEEInterface;
//import gov.va.med.esr.service.impl.MockEE;
import gov.va.med.esr.common.model.lookup.VAFacility;
import gov.va.med.esr.common.model.lookup.EnrollmentOverrideReason;
import gov.va.med.esr.common.model.person.Person;
import gov.va.med.esr.common.model.ee.Eligibility;
import gov.va.med.esr.common.model.ee.VerificationInfo;
import gov.va.med.esr.common.model.ee.EnrollmentOverride;
import gov.va.med.esr.common.model.person.SSN;
import gov.va.med.esr.common.model.person.id.PersonIdEntityKeyImpl;
import gov.va.med.esr.common.model.party.Address;
import gov.va.med.esr.common.rule.data.FinancialInputData;
import gov.va.med.esr.common.rule.parameter.FinancialInputParameter;
import gov.va.med.esr.service.EEResultInfo;
import gov.va.med.esr.service.impl.FinancialsHelperServiceImpl;
import gov.va.med.fw.service.ServiceException;
import gov.va.med.fw.util.DateUtils;
import gov.va.med.fw.rule.SimpleRuleMethod;
import gov.va.med.fw.util.builder.BuilderException;
import gov.va.med.esr.common.builder.entity.ImpreciseDateBuilder;

import java.math.BigDecimal;
import java.util.Calendar;
import java.util.Date;

/**
 * Financials Test
 * 
 * @author DNS   MANSOG
 * @version 1.0
 */
public class ProcessFinancialTest extends AbstractFinancialTest {
	/**
	 * Constructor for ProcessFinancialTest.
	 * 
	 * @param testName
	 * @throws Exception
	 */
	public ProcessFinancialTest(String testName) throws Exception {
		super(testName);
		// super.setDefaultRollback(false);
	}
	
    public void testFinancialStatementSpouseYNInd() throws Exception {
        PersonIdEntityKeyImpl personKey = new PersonIdEntityKeyImpl(new BigDecimal("373964392"));
        Person onFile = this.getPersonService().getPerson(personKey);
        Integer incomeYear = 2013;

        FinancialStatement stmt = onFile.getFinancialStatement(incomeYear);
        stmt.setContributedToSpouseInd(Boolean.TRUE);
        assertNotNull(stmt.getContributedToSpouseInd());

        Person result = this.savePerson(onFile);
        this.setComplete();
        assertNotNull(result.getFinancialStatement(incomeYear).getContributedToSpouseInd());
    }	
	
    
    public void testValidSSn() throws Exception {
        assertFalse(SSN.isValidSSN("000000000"));
        assertFalse(SSN.isValidSSN("111111111"));
        assertFalse(SSN.isValidSSN("222222222"));
        assertFalse(SSN.isValidSSN("333333333"));
        assertFalse(SSN.isValidSSN("444444444"));
        assertFalse(SSN.isValidSSN("555555555"));
        assertFalse(SSN.isValidSSN("666666666"));
        assertFalse(SSN.isValidSSN("777777777"));
        assertFalse(SSN.isValidSSN("888888888"));
        assertFalse(SSN.isValidSSN("999999999"));
        assertFalse(SSN.isValidSSN(null));
        assertFalse(SSN.isValidSSN("A2342B344"));
        assertFalse(SSN.isValidSSN("123456789"));
        assertFalse(SSN.isValidSSN("000672389"));
        assertFalse(SSN.isValidSSN("000006789"));
        assertFalse(SSN.isValidSSN("845645789"));
        assertFalse(SSN.isValidSSN("945606789"));
        assertFalse(SSN.isValidSSN("235006789"));
        assertFalse(SSN.isValidSSN("729999999"));
        assertTrue(SSN.isValidSSN("728999999"));
        assertTrue(SSN.isValidSSN("234349999"));
        assertTrue(SSN.isValidSSN("728099999"));
        assertTrue(SSN.isValidSSN("728909999"));
        assertTrue(SSN.isValidSSN("728999900"));
    }

    public void testIVMTest() throws Exception {
        Integer incomeYear = new Integer(2003);
        VAFacility facility = getVaFacility(VAFacility.CODE_ALBUQUERQUE);

        Person p = buildSimplePerson();
        IncomeTestSource src = getTestSource(IncomeTestSource.CODE_VAMC);
        IncomeTestType typ = getTestType(IncomeTestType.CODE_MEANS_TEST);
        MeansTestStatus status = getMeansTestStatus(MeansTestStatus.MT_STATUS_MT_COPAY_EXEMPT);
        IncomeTest test = buildIncomeTestData(incomeYear, p, typ, src, status, facility);
        test.setGmtAddress(createAddress());

        Person person = savePerson(p);

        // Clone the pristine to have a working version
        Person incoming = (Person) person.clone();
        Person onFile = getPersonService().getPerson(
                person.getPersonEntityKey());

        IncomeTestSource testSource = getTestSource(IncomeTestSource.CODE_IVM);
        IncomeTestType testType = getTestType(IncomeTestType.CODE_MEANS_TEST);
        MeansTestStatus testStatus = getMeansTestStatus(MeansTestStatus.MT_STATUS_GMT_COPAY_REQUIRED);

        VAFacility ivmFacility = getVaFacility(VAFacility.CODE_MPI_TRIGGER);
        IncomeTest incomingTest = buildIncomeTestData(incomeYear, incoming, testType, testSource, testStatus, ivmFacility);
        incomingTest.setGmtAddress(null);

        try {
            Person result = invokeFinancialRuleService(incomeYear, facility,
                    incoming, onFile);

            IncomeTest incomingIncomeTest = incoming.getIncomeTest(incomeYear);
            IncomeTest resultIncomeTest = result.getIncomeTest(incomeYear);
            assertNotNull("Result IncomeTest should not be null",
                    resultIncomeTest);
            
            assertEquals(incomingIncomeTest.getSource(), resultIncomeTest
                    .getSource());
            assertEquals(incomingIncomeTest.getStatus(), resultIncomeTest
                    .getStatus());
            assertEquals(testStatus, resultIncomeTest.getStatus());
            assertEquals(testType, resultIncomeTest.getType());
            assertEquals(testSource, resultIncomeTest.getSource());
            assertEquals(ivmFacility, resultIncomeTest.getSiteConductingTest());
            assertEquals(incomingIncomeTest.getSiteConductingTest(),
                    resultIncomeTest.getSiteConductingTest());
            assertNotNull(resultIncomeTest.getGmtAddress());

        } catch (Exception e) {
            this.fail("Failed due to exception ", e);
        }
    }

   public void testBadIVMTest() throws Exception {
        Integer incomeYear = new Integer(2003);
        VAFacility facility = getVaFacility(VAFacility.CODE_ALBUQUERQUE);

        Person p = buildSimplePerson();
        IncomeTestSource src = getTestSource(IncomeTestSource.CODE_VAMC);
        IncomeTestType typ = getTestType(IncomeTestType.CODE_MEANS_TEST);
        MeansTestStatus status = getMeansTestStatus(MeansTestStatus.MT_STATUS_MT_COPAY_EXEMPT);
        IncomeTest test = buildIncomeTestData(incomeYear, p, typ, src, status, facility);
        test.setGmtAddress(createAddress());

        Person person = savePerson(p);

        // Clone the pristine to have a working version
        Person incoming = (Person) person.clone();
        Person onFile = getPersonService().getPerson(
                person.getPersonEntityKey());

        IncomeTestSource testSource = getTestSource(IncomeTestSource.CODE_IVM);
        IncomeTestType testType = getTestType(IncomeTestType.CODE_MEANS_TEST);
        MeansTestStatus testStatus = getMeansTestStatus(MeansTestStatus.MT_STATUS_GMT_COPAY_REQUIRED);

        VAFacility ivmFacility = getVaFacility(VAFacility.CODE_MPI_TRIGGER);
        IncomeTest incomingTest = buildIncomeTestData(incomeYear, incoming, testType, testSource, testStatus, ivmFacility);
        incomingTest.setGmtAddress(null);
    	SpouseFinancials newSf = createSpouseFinancials();

     
        try {
            Person result = invokeFinancialRuleService(incomeYear, facility,
                    incoming, onFile);

            IncomeTest incomingIncomeTest = incoming.getIncomeTest(incomeYear);
            IncomeTest resultIncomeTest = result.getIncomeTest(incomeYear);
            assertNotNull("Result IncomeTest should not be null",
                    resultIncomeTest);
            
            assertEquals(incomingIncomeTest.getSource(), resultIncomeTest
                    .getSource());
            assertEquals(incomingIncomeTest.getStatus(), resultIncomeTest
                    .getStatus());
            assertEquals(testStatus, resultIncomeTest.getStatus());
            assertEquals(testType, resultIncomeTest.getType());
            assertEquals(testSource, resultIncomeTest.getSource());
            assertEquals(ivmFacility, resultIncomeTest.getSiteConductingTest());
            assertEquals(incomingIncomeTest.getSiteConductingTest(),
                    resultIncomeTest.getSiteConductingTest());
            assertNotNull(resultIncomeTest.getGmtAddress());

        } catch (Exception e) {
            this.fail("Failed due to exception ", e);
        }
    }    
    
   
   
    public void testAdjustedMedicalExpense() throws Exception {
        Integer incomeYear = new Integer(2003);
        Person person = savePerson(buildSimplePerson());

        IncomeTestSource testSource = getTestSource(IncomeTestSource.CODE_HEC);
        IncomeTestType testType = getTestType(IncomeTestType.CODE_MEANS_TEST);
        MeansTestStatus testStatus = getMeansTestStatus(MeansTestStatus.MT_STATUS_MT_COPAY_EXEMPT);

        buildIncomeTestData(incomeYear, person, testType, testSource, testStatus, getHECFacility());

        FinancialStatement stmt = buildFinancialStatementData(incomeYear, person);
        setExpense(stmt, 7517, ExpenseType.EXPENSE_TYPE_ADJUSTED_MEDICAL);

        // Clone the pristine to have a working version
        Person incoming = (Person) person.clone();
        Person onFile = getPersonService().getPerson(person.getPersonEntityKey());
        Person pristine = (Person) person.clone();

        FinancialInputData data = new FinancialInputData(incoming, onFile, pristine, incomeYear,
                getHECFacility(), true);
        FinancialInputParameter fip = buildFinancialInputParameter(data);

        BigDecimal medExp = fip.getCalculatedNonReimbursedMedicalExpense();
        assertNotNull("Med Exp can not be null.", medExp);        
    }
    public void testHasNoPriorIYTestOrAllPriorTestsMTCopayRequired() throws Exception {
    	boolean flag = hasNoPriorIYTestOrAllPriorTestsMTCopayRequired(MeansTestStatus.MT_STATUS_MT_COPAY_REQUIRED, false);
        assertTrue("expected true", flag);    	
    }
    public void testHasNoPriorIYTestOrAllPriorTestsMTCopayRequiredNeg1() throws Exception {
    	boolean flag = hasNoPriorIYTestOrAllPriorTestsMTCopayRequired(MeansTestStatus.MT_STATUS_EXEMPT, false);
        assertFalse("expected false", flag);    	
    }
    public void testHasNoPriorIYTestOrAllPriorTestsMTCopayRequiredNoTest() throws Exception {
    	boolean flag = hasNoPriorIYTestOrAllPriorTestsMTCopayRequired(null, true);
        assertTrue("expected true", flag);    	
    }        
    private boolean hasNoPriorIYTestOrAllPriorTestsMTCopayRequired(MeansTestStatus.Code status, boolean noTest) throws Exception {
        Integer incomeYear = new Integer(2012);
        Person person = savePerson(buildSimplePerson());

        IncomeTestSource testSource = getTestSource(IncomeTestSource.CODE_HEC);
        IncomeTestType testType = getTestType(IncomeTestType.CODE_MEANS_TEST);
        if (!noTest) {
        	MeansTestStatus testStatus = getMeansTestStatus(status);        	
        	buildIncomeTestData(incomeYear, person, testType, testSource, testStatus, getHECFacility());        	
        }

        FinancialStatement stmt = buildFinancialStatementData(incomeYear, person);
        setExpense(stmt, 7517, ExpenseType.EXPENSE_TYPE_ADJUSTED_MEDICAL);

        // Clone the pristine to have a working version
        Person incoming = (Person) person.clone();
        Person onFile = getPersonService().getPerson(person.getPersonEntityKey());
        Person pristine = (Person) person.clone();

        FinancialInputData data = new FinancialInputData(incoming, onFile, pristine, incomeYear,
                getHECFacility(), true);
        FinancialInputParameter fip = buildFinancialInputParameter(data);

        return fip.hasNoPriorIYTestOrAllPriorTestsMTCopayRequired();
        
    }    
	public void testProcessFinancialInfo() throws Exception {
		Integer incomeYear = new Integer(2003);
		VAFacility facility = getVaFacility(VAFacility.CODE_ALBUQUERQUE);
		Person person = savePerson(buildSimplePerson());
		// Clone the pristine to have a working version
		Person incoming = (Person) person.clone();
		Person onFile = getPersonService().getPerson(
				person.getPersonEntityKey());

		IncomeTestSource testSource = getTestSource(IncomeTestSource.CODE_VAMC);
		IncomeTestType testType = getTestType(IncomeTestType.CODE_MEANS_TEST);
		MeansTestStatus testStatus = getMeansTestStatus(MeansTestStatus.MT_STATUS_MT_COPAY_EXEMPT);

		buildIncomeTestData(incomeYear, incoming, testType, testSource,
				testStatus, facility);

		buildFinancialStatementData(incomeYear, incoming);

		try {
			Person result = invokeFinancialRuleService(incomeYear, facility,
					incoming, onFile);

			IncomeTest incomingIncomeTest = incoming.getIncomeTest(incomeYear);
			IncomeTest resultIncomeTest = result.getIncomeTest(incomeYear);

			assertNotNull("Result IncomeTest should not be null",
					resultIncomeTest);
			assertEquals(incomingIncomeTest.getSource(), resultIncomeTest
					.getSource());
			assertEquals(incomingIncomeTest.getCompletedDate(),
					resultIncomeTest.getCompletedDate());
			assertEquals(incomingIncomeTest.getEffectiveDate(),
					resultIncomeTest.getEffectiveDate());
			assertEquals(incomingIncomeTest.getStatus(), resultIncomeTest
					.getStatus());
			assertEquals(testStatus, resultIncomeTest.getStatus());
			assertEquals(testType, resultIncomeTest.getType());
			assertEquals(testSource, resultIncomeTest.getSource());
			assertEquals(facility, resultIncomeTest.getSiteConductingTest());
			assertEquals(incomingIncomeTest.getSiteConductingTest(),
					resultIncomeTest.getSiteConductingTest());

			FinancialStatement incomingStmt = incoming
					.getFinancialStatement(incomeYear);
			FinancialStatement resultStmt = result
					.getFinancialStatement(incomeYear);
			assertNotNull("Result FinancialStatement should not be null",
					resultStmt);
			assertEquals(incomingStmt.getNumberOfDependentChildren(),
					resultStmt.getNumberOfDependentChildren());
			assertEquals(incomingStmt.getMarriedLastCalendarYear(), resultStmt
					.getMarriedLastCalendarYear());
			assertNotNull(resultStmt.getSpouseFinancials());
			assertNotNull(resultStmt.getDependentFinancials());
			assertEquals(resultStmt.getDependentFinancials().size(), 2);
            
		} catch (Exception e) {
			this.fail("Failed due to exception ", e);
		}
	}

    /**
     * Test for sites with support for new Spouse Contributed y/n indicator
     * 
     * @throws Exception
     */	
	public void testProcessFinancialStatementForDeployedSites() throws Exception {
		Integer incomeYear = new Integer(2003);
		VAFacility facility = getVaFacility(VAFacility.CODE_ALBUQUERQUE);
		IncomeTestSource testSource = getTestSource(IncomeTestSource.CODE_VAMC);
		IncomeTestType testType = getTestType(IncomeTestType.CODE_MEANS_TEST);
		MeansTestStatus testStatus = getMeansTestStatus(MeansTestStatus.MT_STATUS_MT_COPAY_EXEMPT);

		Person person = buildSimplePerson();
		buildIncomeTestData(incomeYear, person, testType, testSource,
				testStatus, facility);
		person = savePerson(person);

		// Clone the pristine to have a working version
		Person incoming = (Person) person.clone();
		
		Person onFile = getPersonService().getPerson(
				person.getPersonEntityKey());
		buildFinancialStatementData(incomeYear, incoming);
		incoming.getFinancialStatement(incomeYear).getActiveSpouseFinancials().setLivedWithPatient(Boolean.FALSE);
		incoming.getFinancialStatement(incomeYear).setMarriedLastCalendarYear(Boolean.TRUE);
        incoming.getFinancialStatement(incomeYear).setContributionToSpouse(null);		
        incoming.getFinancialStatement(incomeYear).setContributedToSpouseInd(Boolean.TRUE);		
        assertNull(onFile.getFinancialStatement(incomeYear));        
		
		try {
			Person result = invokeFinancialRuleService(incomeYear, facility,
					incoming, onFile);

			FinancialStatement incomingStmt = incoming
					.getFinancialStatement(incomeYear);
			FinancialStatement resultStmt = result
					.getFinancialStatement(incomeYear);
			assertNotNull("Result FinancialStatement should not be null",
					resultStmt);
			assertEquals(incomingStmt.getNumberOfDependentChildren(),
					resultStmt.getNumberOfDependentChildren());
			assertEquals(incomingStmt.getMarriedLastCalendarYear(), resultStmt
					.getMarriedLastCalendarYear());
			assertNotNull(resultStmt.getSpouseFinancials());
			assertNotNull(resultStmt.getDependentFinancials());
			assertEquals(resultStmt.getDependentFinancials().size(), 2);
			assertTrue(resultStmt.getContributedToSpouseInd());
		} catch (Exception e) {
			this.fail("Failed due to exception ", e);
		}
	}
	
	/**
	 * Test for sites without support for new Spouse Contributed y/n indicator
	 * 
	 * @throws Exception
	 */
    public void testProcessFinancialStatementForNonDeployedSite() throws Exception {
        Integer incomeYear = new Integer(2003);
        VAFacility facility = getVaFacility(VAFacility.CODE_ALBUQUERQUE);
        IncomeTestSource testSource = getTestSource(IncomeTestSource.CODE_VAMC);
        IncomeTestType testType = getTestType(IncomeTestType.CODE_MEANS_TEST);
        MeansTestStatus testStatus = getMeansTestStatus(MeansTestStatus.MT_STATUS_MT_COPAY_EXEMPT);

        Person person = buildSimplePerson();
        buildIncomeTestData(incomeYear, person, testType, testSource,
                testStatus, facility);
        person = savePerson(person);

        // Clone the pristine to have a working version
        Person incoming = (Person) person.clone();
        
        Person onFile = getPersonService().getPerson(
                person.getPersonEntityKey());
        buildFinancialStatementData(incomeYear, incoming);
        
        assertNull(onFile.getFinancialStatement(incomeYear));        
        
        try {
            Person result = invokeFinancialRuleService(incomeYear, facility,
                    incoming, onFile);

            FinancialStatement incomingStmt = incoming
                    .getFinancialStatement(incomeYear);
            FinancialStatement resultStmt = result
                    .getFinancialStatement(incomeYear);
            assertNotNull("Result FinancialStatement should not be null",
                    resultStmt);
            assertEquals(incomingStmt.getNumberOfDependentChildren(),
                    resultStmt.getNumberOfDependentChildren());
            assertEquals(incomingStmt.getMarriedLastCalendarYear(), resultStmt
                    .getMarriedLastCalendarYear());
            assertNotNull(resultStmt.getSpouseFinancials());
            assertNotNull(resultStmt.getDependentFinancials());
            assertEquals(resultStmt.getDependentFinancials().size(), 2);
            assertNull(resultStmt.getContributedToSpouseInd());
        } catch (Exception e) {
            this.fail("Failed due to exception ", e);
        }
    }	

	public void testProcessFinancialStatementDelete() throws Exception {
		Integer incomeYear = new Integer(2003);
		VAFacility facility = getVaFacility(VAFacility.CODE_ALBUQUERQUE);
		IncomeTestSource testSource = getTestSource(IncomeTestSource.CODE_VAMC);
		IncomeTestType testType = getTestType(IncomeTestType.CODE_MEANS_TEST);
		MeansTestStatus testStatus = getMeansTestStatus(MeansTestStatus.MT_STATUS_MT_COPAY_EXEMPT);

		Person person = buildSimplePerson();
		buildIncomeTestData(incomeYear, person, testType, testSource,
				testStatus, facility);
		buildFinancialStatementData(incomeYear, person);
		person = savePerson(person);

		// Clone the pristine to have a working version
		Person incoming = (Person) person.clone();
		Person onFile = getPersonService().getPerson(
				person.getPersonEntityKey());
		incoming.setFinancialStatement(incomeYear, null);
		try {
			Person result = invokeFinancialRuleService(incomeYear, facility,
					incoming, onFile);

			FinancialStatement resultStmt = result
					.getFinancialStatement(incomeYear);
			assertNull("Result FinancialStatement should be null", resultStmt);

		} catch (Exception e) {
			this.fail("Failed due to exception ", e);
		}
	}

	public void testProcessFinancialStatementRemoveSpouse() throws Exception {

        
		Integer incomeYear = new Integer(2003);
		VAFacility facility = getVaFacility(VAFacility.CODE_ALBUQUERQUE);
		IncomeTestSource testSource = getTestSource(IncomeTestSource.CODE_VAMC);
		IncomeTestType testType = getTestType(IncomeTestType.CODE_MEANS_TEST);
		MeansTestStatus testStatus = getMeansTestStatus(MeansTestStatus.MT_STATUS_MT_COPAY_EXEMPT);

		Person person = buildSimplePerson();
		buildIncomeTestData(incomeYear, person, testType, testSource,
				testStatus, facility);
		buildFinancialStatementData(incomeYear, person);
		person = savePerson(person);

		// Clone the pristine to have a working version
		Person incoming = (Person) person.clone();
		Person onFile = getPersonService().getPerson(
				person.getPersonEntityKey());
		FinancialStatement inStmt = incoming.getFinancialStatement(incomeYear);
		inStmt.removeSpouseFinancials();
		try {
			Person result = invokeFinancialRuleService(incomeYear, facility,
					incoming, onFile);

			FinancialStatement incomingStmt = incoming
					.getFinancialStatement(incomeYear);
			FinancialStatement resultStmt = result
					.getFinancialStatement(incomeYear);
			assertNotNull("Result FinancialStatement should not be null",
					resultStmt);
			assertEquals(incomingStmt.getNumberOfDependentChildren(),
					resultStmt.getNumberOfDependentChildren());
			assertEquals(incomingStmt.getMarriedLastCalendarYear(), resultStmt
					.getMarriedLastCalendarYear());
			assertNull(resultStmt.getSpouseFinancials());

		} catch (Exception e) {
			this.fail("Failed due to exception ", e);
		}
        
	}

	public void testProcessFinancialStatementUpdateSpouse() throws Exception {

        
		Integer incomeYear = new Integer(2003);
		VAFacility facility = getVaFacility(VAFacility.CODE_ALBUQUERQUE);
		IncomeTestSource testSource = getTestSource(IncomeTestSource.CODE_VAMC);
		IncomeTestType testType = getTestType(IncomeTestType.CODE_MEANS_TEST);
		MeansTestStatus testStatus = getMeansTestStatus(MeansTestStatus.MT_STATUS_MT_COPAY_EXEMPT);

		Person person = buildSimplePerson();
		buildIncomeTestData(incomeYear, person, testType, testSource,
				testStatus, facility);
		buildFinancialStatementData(incomeYear, person);
		person = savePerson(person);

		// Clone the pristine to have a working version
		Person incoming = (Person) person.clone();
		Person onFile = getPersonService().getPerson(
				person.getPersonEntityKey());
		FinancialStatement inStmt = incoming.getFinancialStatement(incomeYear);
		SpouseFinancials sf = inStmt.getActiveSpouseFinancials();
		SpouseFinancials newSf = createSpouseFinancials();
		getCopyService().copy(sf.getReportedOn(), newSf.getReportedOn());
		try {
			Person result = invokeFinancialRuleService(incomeYear, facility,
					incoming, onFile);

			FinancialStatement incomingStmt = incoming
					.getFinancialStatement(incomeYear);
			FinancialStatement resultStmt = result
					.getFinancialStatement(incomeYear);
			assertNotNull("Result FinancialStatement should not be null",
					resultStmt);
			assertEquals(incomingStmt.getNumberOfDependentChildren(),
					resultStmt.getNumberOfDependentChildren());
			assertEquals(incomingStmt.getMarriedLastCalendarYear(), resultStmt
					.getMarriedLastCalendarYear());
			assertNotNull(resultStmt.getSpouseFinancials());

			FinancialStatement origStmt = person
					.getFinancialStatement(incomeYear);
			assertEquals(origStmt.getActiveSpouseFinancials().getReportedOn()
					.getName(), resultStmt.getActiveSpouseFinancials()
					.getReportedOn().getName());
			assertEquals(origStmt.getActiveSpouseFinancials().getReportedOn()
					.getOfficialSsn(), resultStmt.getActiveSpouseFinancials()
					.getReportedOn().getOfficialSsn());

		} catch (Exception e) {
			this.fail("Failed due to exception ", e);
		}
        
	}

	public void testProcessFinancialStatementRemoveDependents()
			throws Exception {
		Integer incomeYear = new Integer(2003);
		VAFacility facility = getVaFacility(VAFacility.CODE_ALBUQUERQUE);
		IncomeTestSource testSource = getTestSource(IncomeTestSource.CODE_VAMC);
		IncomeTestType testType = getTestType(IncomeTestType.CODE_MEANS_TEST);
		MeansTestStatus testStatus = getMeansTestStatus(MeansTestStatus.MT_STATUS_MT_COPAY_EXEMPT);

		Person person = buildSimplePerson();
		buildIncomeTestData(incomeYear, person, testType, testSource,
				testStatus, facility);
		buildFinancialStatementData(incomeYear, person);
		person = savePerson(person);

		// Clone the pristine to have a working version
		Person incoming = (Person) person.clone();
		Person onFile = getPersonService().getPerson(
				person.getPersonEntityKey());
		FinancialStatement inStmt = incoming.getFinancialStatement(incomeYear);
		inStmt.removeAllDependentFinancials();
		try {
			Person result = invokeFinancialRuleService(incomeYear, facility,
					incoming, onFile);

			FinancialStatement incomingStmt = incoming
					.getFinancialStatement(incomeYear);
			FinancialStatement resultStmt = result
					.getFinancialStatement(incomeYear);
			assertNotNull("Result FinancialStatement should not be null",
					resultStmt);
			assertEquals(incomingStmt.getNumberOfDependentChildren(),
					resultStmt.getNumberOfDependentChildren());
			assertEquals(incomingStmt.getMarriedLastCalendarYear(), resultStmt
					.getMarriedLastCalendarYear());
			assertNotNull(resultStmt.getSpouseFinancials());
			assertNotNull(resultStmt.getDependentFinancials());
			assertEquals(resultStmt.getDependentFinancials().size(), 0);
		} catch (Exception e) {
			this.fail("Failed due to exception ", e);
		}
	}

	public void testProcessFinancialStatementUpdateDependents()
			throws Exception {
		Integer incomeYear = new Integer(2003);
		VAFacility facility = getVaFacility(VAFacility.CODE_ALBUQUERQUE);
		IncomeTestSource testSource = getTestSource(IncomeTestSource.CODE_VAMC);
		IncomeTestType testType = getTestType(IncomeTestType.CODE_MEANS_TEST);
		MeansTestStatus testStatus = getMeansTestStatus(MeansTestStatus.MT_STATUS_MT_COPAY_EXEMPT);

		Person person = buildSimplePerson();
		buildIncomeTestData(incomeYear, person, testType, testSource,
				testStatus, facility);
		buildFinancialStatementData(incomeYear, person);
		person = savePerson(person);

		// Clone the pristine to have a working version
		Person incoming = (Person) person.clone();
		Person onFile = getPersonService().getPerson(
				person.getPersonEntityKey());
		FinancialStatement inStmt = incoming.getFinancialStatement(incomeYear);
		DependentFinancials df = (DependentFinancials) inStmt
				.getDependentFinancials().iterator().next();
		DependentFinancials newDf = createDependentFinancials();
		getCopyService().copy(df.getReportedOn(), newDf.getReportedOn());
		inStmt.addDependentFinancials(newDf);
		inStmt.removeDependentFinancials(df);
		try {
			Person result = invokeFinancialRuleService(incomeYear, facility,
					incoming, onFile);

			FinancialStatement incomingStmt = incoming
					.getFinancialStatement(incomeYear);
			FinancialStatement resultStmt = result
					.getFinancialStatement(incomeYear);
			assertNotNull("Result FinancialStatement should not be null",
					resultStmt);
			assertEquals(incomingStmt.getNumberOfDependentChildren(),
					resultStmt.getNumberOfDependentChildren());
			assertEquals(incomingStmt.getMarriedLastCalendarYear(), resultStmt
					.getMarriedLastCalendarYear());
			assertNotNull(resultStmt.getSpouseFinancials());
			assertNotNull(resultStmt.getDependentFinancials());
			assertEquals(resultStmt.getDependentFinancials().size(), 2);
		} catch (Exception e) {
			this.fail("Failed due to exception ", e);
		}
	}

    public void testProcessIncomeTest() throws Exception {
		Integer incomeYear = new Integer(2003);
		VAFacility facility = getVaFacility(VAFacility.CODE_ALBUQUERQUE);

		Person person = savePerson(buildSimplePerson());

		// Clone the pristine to have a working version
		Person incoming = (Person) person.clone();
		Person onFile = getPersonService().getPerson(
				person.getPersonEntityKey());

		IncomeTestSource testSource = getTestSource(IncomeTestSource.CODE_VAMC);
		IncomeTestType testType = getTestType(IncomeTestType.CODE_MEANS_TEST);
		MeansTestStatus testStatus = getMeansTestStatus(MeansTestStatus.MT_STATUS_MT_COPAY_EXEMPT);

        IncomeTest inTest = buildIncomeTestData(incomeYear, incoming, testType, testSource, testStatus, facility);
        inTest.setLastEditedDate(new Date());

		try {
			Person result = invokeFinancialRuleService(incomeYear, facility,
					incoming, onFile);

			IncomeTest incomingIncomeTest = incoming.getIncomeTest(incomeYear);
			IncomeTest resultIncomeTest = result.getIncomeTest(incomeYear);
			assertNotNull("Result IncomeTest should not be null",
					resultIncomeTest);
			assertEquals(incomingIncomeTest.getSource(), resultIncomeTest
					.getSource());
			assertEquals(incomingIncomeTest.getCompletedDate(),
					resultIncomeTest.getCompletedDate());
			assertEquals(incomingIncomeTest.getEffectiveDate(),
					resultIncomeTest.getEffectiveDate());
			assertEquals(incomingIncomeTest.getStatus(), resultIncomeTest
					.getStatus());
			assertEquals(testStatus, resultIncomeTest.getStatus());
			assertEquals(testType, resultIncomeTest.getType());
			assertEquals(testSource, resultIncomeTest.getSource());
			assertEquals(facility, resultIncomeTest.getSiteConductingTest());
			assertEquals(incomingIncomeTest.getSiteConductingTest(),
					resultIncomeTest.getSiteConductingTest());
            assertEquals(resultIncomeTest.getLastEditedDate(), incomingIncomeTest.getLastEditedDate());

		} catch (Exception e) {
			this.fail("Failed due to exception ", e);
		}
	}


	public void testProcessIncomeTestNonP8() throws Exception {
		Integer incomeYear = new Integer(2007);
		VAFacility facility = getVaFacility(VAFacility.CODE_ALBUQUERQUE);

		Person p = buildSimplePerson();
		IncomeTestSource src = getTestSource(IncomeTestSource.CODE_VAMC);
		IncomeTestType typ = getTestType(IncomeTestType.CODE_MEANS_TEST);
		MeansTestStatus status = getMeansTestStatus(MeansTestStatus.MT_STATUS_MT_COPAY_REQUIRED);

		Date lastEditDate = new Date();
		IncomeTest tst = buildIncomeTestData(incomeYear, p, typ, src, status,
				facility);
		tst.setLastEditedDate(lastEditDate);
		tst.setDiscloseFinancialInformation(Boolean.TRUE);
		p.setFinancialStatement(new Integer(2008),this.createFinancialStatement());
		Person person = savePerson(p);

		// Clone the pristine to have a working version
		Person incoming = (Person) person.clone();
		Person onFile = getPersonService().getPerson(
				person.getPersonEntityKey());

		IncomeTestSource testSource = getTestSource(IncomeTestSource.CODE_VAMC);
		IncomeTestType testType = getTestType(IncomeTestType.CODE_MEANS_TEST);
		MeansTestStatus testStatus = getMeansTestStatus(MeansTestStatus.MT_STATUS_MT_COPAY_REQUIRED);
		facility = getVaFacility(VAFacility.CODE_CONNECTICUT_HCS);
		
		// NOTICE FUTURE DATE SET HERE
		IncomeTest incomingTest = buildIncomeTestData(new Integer(2008), incoming,
				testType, testSource, testStatus, facility, this.getDateOnly(new Date()),
				new Date(), this.getDateOnly(this.getDate(2009, 9, 1)));
		incomingTest.setDiscloseFinancialInformation(Boolean.TRUE);
		incomingTest.setThresholdA(new BigDecimal(800));
		incomingTest.setGmtThresholdAmount(new BigDecimal(775));	
		
		incoming.setFinancialStatement(new Integer(2008),this.createFinancialStatement());


		
		try {
			Person result = invokeFinancialRuleService(new Integer(2008), facility,
					incoming, onFile);

			IncomeTest incomingIncomeTest = incoming.getIncomeTest(incomeYear);
			IncomeTest resultIncomeTest = result.getIncomeTest(new Integer(2008));
			assertNotNull("Result IncomeTest should not be null",
					resultIncomeTest);

			// This should be over ridden by the incoming (should not become
			// future test)
			assertEquals("source not same", incomingIncomeTest.getSource(), resultIncomeTest
					.getSource());
			assertEquals("status as expected",MeansTestStatus.MT_STATUS_MT_COPAY_REQUIRED.getCode(), resultIncomeTest
					.getStatus().getCode());
			assertEquals("status not equal", testStatus, resultIncomeTest.getStatus());
			assertEquals("type not as expected", testType, resultIncomeTest.getType());
			assertEquals("source not same", testSource, resultIncomeTest.getSource());

			assertFalse( "test future",resultIncomeTest.getFutureTest().booleanValue());			
		} catch (Exception e) {
			this.fail("Failed due to exception ", e);
		}
	}	
	
	
	public void test9366ThrowsEnrollmentException() throws Exception {
		
		VAFacility facility = getVaFacility(VAFacility.CODE_ALBUQUERQUE);

		Person onFile = buildSimplePerson();
		makeRejectEGTPersonForIncomeTesting(onFile);
		
		IncomeTestSource src = getTestSource(IncomeTestSource.CODE_VAMC);
		IncomeTestType typ = getTestType(IncomeTestType.CODE_MEANS_TEST);
		MeansTestStatus status = getMeansTestStatus(MeansTestStatus.MT_STATUS_MT_COPAY_REQUIRED);
	    Address adrOld = this.createAddress();		
		// Build the test
		Integer incomeYear = new Integer(2007);
		Date lastEditDate = getDate(2008,10,1);
		IncomeTest tst = buildIncomeTestData(incomeYear, onFile, typ, src, status,
				facility);
		tst.setLastEditedDate(lastEditDate);
		tst.setPrimaryIncomeTest(Boolean.TRUE);
		tst.setDiscloseFinancialInformation(Boolean.TRUE);
		tst.setEffectiveDate(getDate(2008,10,11));

		tst.setGmtAddress(adrOld);
		tst.setThresholdA(new BigDecimal(700));
		tst.setGmtThresholdAmount(new BigDecimal(770));	

		Person incoming = (Person)savePerson(onFile).clone();
		incoming.removeAllIncomeTests();
		
		IncomeTestSource testSource = getTestSource(IncomeTestSource.CODE_VAMC);
		IncomeTestType testType = getTestType(IncomeTestType.CODE_MEANS_TEST);
		MeansTestStatus testStatus = getMeansTestStatus(MeansTestStatus.MT_STATUS_MT_COPAY_REQUIRED);
		//VAFacility.CODE_ALBUQUERQUE
		//VAFacility fac = getVaFacility(VAFacility.CODE_CONNECTICUT_HCS);
		VAFacility fac = getVaFacility(VAFacility.CODE_ALBUQUERQUE);
 
		IncomeTest incomingTest = buildIncomeTestData(new Integer(2008), incoming,
				testType, testSource, testStatus, facility, this.getDateOnly(new Date()),
				new Date(), this.getDateOnly(this.getDate(2009, 9, 1)));
		incomingTest.setDiscloseFinancialInformation(Boolean.TRUE);
		// The income is hardcoded in builder for statement: $789
		incomingTest.setThresholdA(new BigDecimal(700));
		incomingTest.setGmtThresholdAmount(new BigDecimal(770));	
		// TODO fix code so that a missing statement does not get null pointer error.
		incoming.setFinancialStatement(new Integer(2008),this.createFinancialStatement());
		incomingTest.setNetIncome(new BigDecimal(789));
		incomingTest.setSiteConductingTest(fac);
		incomingTest.setPrimaryIncomeTest(Boolean.FALSE);

		// Prepare for submitting to Process Z07 use case
        this.setEligibilityStatusToVerified(incoming);
        incoming.getEligibilityVerification().setVerificationFacility(
        		this.getLookupService().getVaFacilityByCode(VAFacility.CODE_HEC.getCode()));
        
        this.addPermanentAddress(incoming, Calendar.getInstance().getTime(),
                null);
        VerificationInfo vi = new VerificationInfo();
        vi.setSiteNumber(incoming.getEligibilityVerification()
                .getVerificationFacility().getCode());

        Person updated = this.getMessagingService().processZ07(incoming,fac, vi);
        assertTrue(updated.getEligibilityVerification().getEligibilityStatus()
                .getCode().equals(EligibilityStatus.CODE_VERIFIED.getName()));
        EnrollmentStatus enstatus = updated.getEnrollmentDetermination().getEnrollmentStatus();
        assertTrue(enstatus.getCode().equals(EnrollmentStatus.CODE_VERIFIED.getName()));

	}
	
	
	public void test9366Existing2008Test() throws Exception {

		VAFacility facility = getVaFacility(VAFacility.CODE_ALBUQUERQUE);
		Person onFile = buildSimplePerson();

		IncomeTest oldTst = buildIncomeTestData(new Integer(2007), onFile, getTestType(IncomeTestType.CODE_MEANS_TEST),
				getTestSource(IncomeTestSource.CODE_VAMC), 
				getMeansTestStatus(MeansTestStatus.MT_STATUS_MT_COPAY_REQUIRED),
				facility);
		oldTst.setLastEditedDate(getDate(2008,1,1));
		oldTst.setDiscloseFinancialInformation(Boolean.TRUE);
		oldTst.setEffectiveDate(getDate(2008,10,11));
		
		makeRejectEGTPersonForIncomeTesting(onFile);

		Integer incomeYear = new Integer(2008);        
		IncomeTestSource src = getTestSource(IncomeTestSource.CODE_VAMC);
		IncomeTestType typ = getTestType(IncomeTestType.CODE_MEANS_TEST);
		MeansTestStatus status = getMeansTestStatus(MeansTestStatus.MT_STATUS_MT_COPAY_REQUIRED);

		Date lastEditDate = getDate(2008,10,1);
		IncomeTest tst = buildIncomeTestData(incomeYear, onFile, typ, src, status,
				facility);
		tst.setLastEditedDate(lastEditDate);
		tst.setDiscloseFinancialInformation(Boolean.TRUE);
		tst.setEffectiveDate(getDate(2008,10,11));
		tst.setThresholdA(new BigDecimal(800));
		tst.setGmtThresholdAmount(new BigDecimal(775));	
		// TODO fix code so that a missing statement does not get null pointer error.
		onFile.setFinancialStatement(new Integer(2008),this.createFinancialStatement());

		Person incoming = (Person)savePerson(onFile).clone();
		incoming.removeAllIncomeTests();
		
		IncomeTestSource testSource = getTestSource(IncomeTestSource.CODE_VAMC);
		IncomeTestType testType = getTestType(IncomeTestType.CODE_MEANS_TEST);
		MeansTestStatus testStatus = getMeansTestStatus(MeansTestStatus.MT_STATUS_MT_COPAY_REQUIRED);
		//VAFacility.CODE_ALBUQUERQUE
		//VAFacility fac = getVaFacility(VAFacility.CODE_CONNECTICUT_HCS);
		VAFacility fac = getVaFacility(VAFacility.CODE_ALBUQUERQUE);
		
		/// NOTE has future date
		IncomeTest incomingTest = buildIncomeTestData(new Integer(2008), incoming,
				testType, testSource, testStatus, facility, this.getDateOnly(new Date()),
				new Date(), this.getDateOnly(this.getDate(2009, 7, 1)));
		incomingTest.setDiscloseFinancialInformation(Boolean.TRUE);
		incomingTest.setFutureTest(Boolean.FALSE);
		// The income is hardcoded in builder for statement: $789
		// Theory: Most likely the SQA data was not satisfying income for P8
		// This causes 2008 test to be made future since it offers no benefit 
		incomingTest.setThresholdA(new BigDecimal(800));
		incomingTest.setGmtThresholdAmount(new BigDecimal(775));	
		// TODO fix code so that a missing statement does not get null pointer error.
		incoming.setFinancialStatement(new Integer(2008),this.createFinancialStatement());
		
		incomingTest.setSiteConductingTest(fac);
        this.setEligibilityStatusToVerified(incoming);
        incoming.getEligibilityVerification().setVerificationFacility(
        		this.getLookupService().getVaFacilityByCode(VAFacility.CODE_HEC.getCode()));
        
        this.addPermanentAddress(incoming, Calendar.getInstance().getTime(),
                null);
        VerificationInfo vi = new VerificationInfo();
        vi.setSiteNumber(incoming.getEligibilityVerification()
                .getVerificationFacility().getCode());

        Person updated = this.getMessagingService().processZ07(incoming,
                fac, vi);
        IncomeTest res = updated.getIncomeTest(incomeYear);
        assertFalse( "should not be future",res.getFutureTest().booleanValue());
        assertTrue(updated.getEligibilityVerification().getEligibilityStatus()
                .getCode().equals(EligibilityStatus.CODE_VERIFIED.getName()));
        EnrollmentStatus enstatus = updated.getEnrollmentDetermination().getEnrollmentStatus();
        assertTrue(enstatus.getCode().equals(EnrollmentStatus.CODE_VERIFIED.getName()));

	}  	
	
	public void test9366New2008Test() throws Exception {
		VAFacility facility = getVaFacility(VAFacility.CODE_ALBUQUERQUE);
		Person onFile = buildSimplePerson();

		IncomeTest oldTst = buildIncomeTestData(new Integer(2007), onFile, getTestType(IncomeTestType.CODE_MEANS_TEST),
				getTestSource(IncomeTestSource.CODE_VAMC), 
				getMeansTestStatus(MeansTestStatus.MT_STATUS_MT_COPAY_REQUIRED),
				facility);
		oldTst.setLastEditedDate(getDate(2008,1,1));
		oldTst.setDiscloseFinancialInformation(Boolean.TRUE);
		oldTst.setEffectiveDate(getDate(2008,10,11));
		
		this.makeRejectEGTPersonForIncomeTesting(onFile);
		
        Person incoming = (Person)savePerson(onFile).clone();
		incoming.removeAllIncomeTests();
		incoming.setFinancialStatement(new Integer(2008),this.createFinancialStatement());		
		IncomeTestSource testSource = getTestSource(IncomeTestSource.CODE_VAMC);
		IncomeTestType testType = getTestType(IncomeTestType.CODE_MEANS_TEST);
		MeansTestStatus testStatus = getMeansTestStatus(MeansTestStatus.MT_STATUS_MT_COPAY_REQUIRED);
		VAFacility fac = getVaFacility(VAFacility.CODE_ALBUQUERQUE);
		
		IncomeTest incomingTest = buildIncomeTestData(new Integer(2008), incoming,
				testType, testSource, testStatus, facility, this.getDateOnly(new Date()),
				new Date(), this.getDateOnly(this.getDate(2009, 6, 1)));
		incomingTest.setDiscloseFinancialInformation(Boolean.TRUE);
		incomingTest.setFutureTest(Boolean.FALSE);
		incomingTest.setThresholdA(new BigDecimal(800));
		incomingTest.setGmtThresholdAmount(new BigDecimal(775));	
		incomingTest.setSiteConductingTest(fac);
        this.setEligibilityStatusToVerified(incoming);
        incoming.getEligibilityVerification().setVerificationFacility(
        		this.getLookupService().getVaFacilityByCode(VAFacility.CODE_HEC.getCode()));
        
        this.addPermanentAddress(incoming, Calendar.getInstance().getTime(),
                null);
        VerificationInfo vi = new VerificationInfo();
        vi.setSiteNumber(incoming.getEligibilityVerification()
                .getVerificationFacility().getCode());

        Person updated = this.getMessagingService().processZ07(incoming,
                fac, vi);
        IncomeTest res = updated.getIncomeTest(new Integer(2008));
        assertFalse( "should not be future",res.getFutureTest().booleanValue());
        assertTrue(updated.getEligibilityVerification().getEligibilityStatus()
                .getCode().equals(EligibilityStatus.CODE_VERIFIED.getName()));
        EnrollmentStatus enstatus = updated.getEnrollmentDetermination().getEnrollmentStatus();
        assertTrue(enstatus.getCode().equals(EnrollmentStatus.CODE_VERIFIED.getName()));

	}  	

	
	
	public void test8672Existing2008andReceiveExempt() throws Exception {
		VAFacility facility = getVaFacility(VAFacility.CODE_ALBUQUERQUE);
		Person onFile = buildSimplePerson();
		IncomeTest oldTst = buildIncomeTestData(new Integer(2007),
				onFile,
				getTestType(IncomeTestType.CODE_MEANS_TEST),
				getTestSource(IncomeTestSource.CODE_VAMC), 
				getMeansTestStatus(MeansTestStatus.MT_STATUS_GMT_COPAY_REQUIRED),
				facility);
		oldTst.setLastEditedDate(getDate(2008,1,1));
		oldTst.setDiscloseFinancialInformation(Boolean.TRUE);
		oldTst.setEffectiveDate(getDate(2008,11,20));
		oldTst.setPrimaryIncomeTest(Boolean.TRUE);
		
		this.makeRejectEGTPersonForIncomeTesting(onFile);

		Integer incomeYear2008 = new Integer(2008);        
		IncomeTestSource src = getTestSource(IncomeTestSource.CODE_VAMC);
		IncomeTestType typ = getTestType(IncomeTestType.CODE_MEANS_TEST);
		MeansTestStatus status = getMeansTestStatus(MeansTestStatus.MT_STATUS_MT_COPAY_REQUIRED);

		Date lastEditDate = getDate(2008,10,1);
		IncomeTest tst = buildIncomeTestData(incomeYear2008, onFile, typ, src, status,
				facility);
		tst.setFutureTest(Boolean.TRUE);
		tst.setLastEditedDate(lastEditDate);
		tst.setPrimaryIncomeTest(Boolean.FALSE);		
		tst.setDiscloseFinancialInformation(Boolean.TRUE);
		tst.setEffectiveDate(getDate(2009,11,20));
		tst.setThresholdA(new BigDecimal(400));
		tst.setGmtThresholdAmount(new BigDecimal(475));	
		// TODO fix code so that a missing statement does not get null pointer error.
		onFile.setFinancialStatement(new Integer(2008),this.createFinancialStatement());
		
        Person incoming = (Person)savePerson(onFile).clone();
		incoming.removeAllIncomeTests();
	
		IncomeTestSource testSource = getTestSource(IncomeTestSource.CODE_VAMC);
		IncomeTestType testType = getTestType(IncomeTestType.CODE_MEANS_TEST);
		MeansTestStatus testStatus = getMeansTestStatus(MeansTestStatus.MT_STATUS_MT_COPAY_EXEMPT);
		//VAFacility.CODE_ALBUQUERQUE
		//VAFacility fac = getVaFacility(VAFacility.CODE_CONNECTICUT_HCS);
		VAFacility fac = getVaFacility(VAFacility.CODE_ALBUQUERQUE);
		IncomeTest incomingTest = buildIncomeTestData(incomeYear2008, incoming,
				testType, testSource, testStatus, facility, this.getDateOnly(new Date()),
				new Date(), this.getDateOnly(this.getDate(2009, 1, 1)));
		incomingTest.setDiscloseFinancialInformation(Boolean.TRUE);
		incomingTest.setFutureTest(Boolean.FALSE);
		incomingTest.setPrimaryIncomeTest(Boolean.FALSE);
		incomingTest.setThresholdA(new BigDecimal(1800));
		incomingTest.setGmtThresholdAmount(new BigDecimal(1775));	
		incoming.setFinancialStatement(new Integer(2008),this.createFinancialStatement());
		
		incomingTest.setSiteConductingTest(fac);
        this.setEligibilityStatusToVerified(incoming);
        incoming.getEligibilityVerification().setVerificationFacility(
        		this.getLookupService().getVaFacilityByCode(VAFacility.CODE_HEC.getCode()));
        
        this.addPermanentAddress(incoming, Calendar.getInstance().getTime(),
                null);
        VerificationInfo vi = new VerificationInfo();
        vi.setSiteNumber(incoming.getEligibilityVerification()
                .getVerificationFacility().getCode());

        Person updated = this.getMessagingService().processZ07(incoming,
                fac, vi);
        IncomeTest current = this.getPersonHelperService().getCurrentIncomeTest(updated);
        assertTrue("current test is not 2008",current.getIncomeYear().equals(incomeYear2008));
        IncomeTest res = updated.getIncomeTest(incomeYear2008);
        String code = res.getStatus().getCode();
        assertTrue("should be exempt",MeansTestStatus.MT_STATUS_MT_COPAY_EXEMPT.getCode().equals(code));
        assertFalse( "should not be future",res.getFutureTest().booleanValue());
        assertTrue(updated.getEligibilityVerification().getEligibilityStatus()
                .getCode().equals(EligibilityStatus.CODE_VERIFIED.getName()));
        EnrollmentStatus enstatus = updated.getEnrollmentDetermination().getEnrollmentStatus();
        assertTrue(enstatus.getCode().equals(EnrollmentStatus.CODE_VERIFIED.getName()));

	}  	
	public void test8672forNew2008_9584() throws Exception {
		Person onFile = buildSimplePerson();
		IncomeTest oldTst = this.build_MT_IncomeTest(onFile,new Integer(2007), MeansTestStatus.MT_STATUS_MT_COPAY_REQUIRED);
		oldTst.setLastEditedDate(getDate(2008,1,1));
		oldTst.setDiscloseFinancialInformation(Boolean.TRUE);
		oldTst.setEffectiveDate(getDate(2008,10,11));
		this.makeRejectEGTPersonForIncomeTesting(onFile);
		Integer incomeYear = new Integer(2008);        
        
        Person incoming = (Person)savePerson(onFile).clone();
		incoming.removeAllIncomeTests();
		incoming.setFinancialStatement(new Integer(2008),this.createFinancialStatement());		

		VAFacility fac = getVaFacility(VAFacility.CODE_ALBUQUERQUE);
		IncomeTest incomingTest = buildIncoming_MT_IncomeTest(incoming, new Integer(2008),
				MeansTestStatus.MT_STATUS_PENDING_ADJUDICATION, this.getDate(2009, 6, 1));
		incomingTest.setDiscloseFinancialInformation(Boolean.TRUE);
		incomingTest.setFutureTest(Boolean.FALSE);
		incomingTest.setThresholdA(new BigDecimal(800));
		incomingTest.setGmtThresholdAmount(new BigDecimal(775));	
		incomingTest.setSiteConductingTest(fac);
        
        VerificationInfo vi = new VerificationInfo();
        vi.setSiteNumber(incoming.getEligibilityVerification()
                .getVerificationFacility().getCode());

        Person updated = this.getMessagingService().processZ07(incoming,
                fac, vi);
        IncomeTest res = updated.getIncomeTest(incomeYear);
        assertFalse( "should not be future",res.getFutureTest().booleanValue());
        assertTrue(updated.getEligibilityVerification().getEligibilityStatus()
                .getCode().equals(EligibilityStatus.CODE_VERIFIED.getName()));
        EnrollmentStatus enstatus = updated.getEnrollmentDetermination().getEnrollmentStatus();
        assertTrue(enstatus.getCode().equals(EnrollmentStatus.CODE_VERIFIED.getName()));

	}  	
	public void test8672New2008() throws Exception {
		Person onFile = buildSimplePerson();
		this.makeRejectEGTPersonForIncomeTesting(onFile);
        Person incoming = (Person)savePerson(onFile).clone();

		incoming.setFinancialStatement(new Integer(2008),this.createFinancialStatement());		

		VAFacility fac = getVaFacility(VAFacility.CODE_ALBUQUERQUE);
		IncomeTest incomingTest = buildIncoming_MT_IncomeTest(incoming, new Integer(2008),
				MeansTestStatus.MT_STATUS_MT_COPAY_EXEMPT, this.getDate(2009, 7, 1));
		incomingTest.setDiscloseFinancialInformation(Boolean.TRUE);
		incomingTest.setFutureTest(Boolean.TRUE);
		incomingTest.setThresholdA(new BigDecimal(800));
		incomingTest.setGmtThresholdAmount(new BigDecimal(775));	
		incomingTest.setSiteConductingTest(fac);
        
        VerificationInfo vi = new VerificationInfo();
        vi.setSiteNumber(incoming.getEligibilityVerification()
                .getVerificationFacility().getCode());

        Person updated = this.getMessagingService().processZ07(incoming,
                fac, vi);
        IncomeTest res = updated.getIncomeTest(new Integer(2008));
        assertFalse( "should not be future",res.getFutureTest().booleanValue());
        assertTrue(updated.getEligibilityVerification().getEligibilityStatus()
                .getCode().equals(EligibilityStatus.CODE_VERIFIED.getName()));
        EnrollmentStatus enstatus = updated.getEnrollmentDetermination().getEnrollmentStatus();
        assertTrue(enstatus.getCode().equals(EnrollmentStatus.CODE_VERIFIED.getName()));

	}  		
	
	public void test8672forNew2008_LTC() throws Exception {
		Person onFile = buildSimplePerson();
		IncomeTest oldTst = this.build_MT_IncomeTest(onFile,new Integer(2007), MeansTestStatus.MT_STATUS_MT_COPAY_REQUIRED);
		oldTst.setLastEditedDate(getDate(2008,1,1));
		oldTst.setDiscloseFinancialInformation(Boolean.TRUE);
		oldTst.setEffectiveDate(getDate(2008,10,11));
		this.makeRejectEGTPersonForIncomeTesting(onFile);
		Integer incomeYear = new Integer(2008);        
        
        Person incoming = (Person)savePerson(onFile).clone();
		incoming.removeAllIncomeTests();
		incoming.setFinancialStatement(new Integer(2008),this.createFinancialStatement());		

		VAFacility fac = getVaFacility(VAFacility.CODE_ALBUQUERQUE);

		IncomeTest incomeTest = this.createIncomeTest(
	                IncomeTestType.CODE_LTC_CO_PAY_EXEMPTION_TEST,
	                MeansTestStatus.MT_STATUS_EXEMPT, incomeYear);
        incoming.setIncomeTest(incomeYear, incomeTest);
        
		
		incoming.getIncomeTest(incomeYear).setSiteConductingTest(fac);
		incoming.getIncomeTest(incomeYear).setIncomeYear(incomeYear);
        
        VerificationInfo vi = new VerificationInfo();
        vi.setSiteNumber(incoming.getEligibilityVerification()
                .getVerificationFacility().getCode());

        Person updated = this.getMessagingService().processZ07(incoming,
                fac, vi);
        IncomeTest res = updated.getIncomeTest(incomeYear);

        assertTrue(updated.getEligibilityVerification().getEligibilityStatus()
                .getCode().equals(EligibilityStatus.CODE_VERIFIED.getName()));
        EnrollmentStatus enstatus = updated.getEnrollmentDetermination().getEnrollmentStatus();
        assertTrue(enstatus.getCode().equals(EnrollmentStatus.CODE_PENDING_MEANS_TEST_REQUIRED.getName()));

	}  	
	

	public void testEffectiveDate() {
		Date date = new Date();
		Date testEffectiveDate = null;
		Calendar calendar = DateUtils.createCalendar(date);
		int year = calendar.get(Calendar.YEAR);
		int month = calendar.get(Calendar.MONTH);
		int day = calendar.get(Calendar.DATE);
		calendar.clear();
		calendar.set(year,month,day);            
		testEffectiveDate = calendar.getTime();     
		SimpleRuleMethod method = new SimpleRuleMethod();

		System.out.println("date =" + date.toString());		
		System.out.println("effective date =" + testEffectiveDate.toString());
		System.out.println("isAfter: is testEffectiveDate after date =" + method.isAfter(testEffectiveDate, date));
		System.out.println("isAfter: is date after testEffectiveDate =" + method.isAfter(date, testEffectiveDate));
		System.out.println("isAfterIgnoreTime: is testEffectiveDate after date =" + DateUtils.isAfterIgnoreTime(testEffectiveDate, date));		
		System.out.println("isAfterIgnoreTime: is date after testEffectiveDate =" + DateUtils.isAfterIgnoreTime(date, testEffectiveDate));		
		date = DateUtils.getTomorrowDate();
		System.out.println("date =" + date.toString());		
		System.out.println("effective date =" + testEffectiveDate.toString());
		System.out.println("isAfter: is testEffectiveDate after date =" + method.isAfter(testEffectiveDate, date));
		System.out.println("isAfter: is date after testEffectiveDate =" + method.isAfter(date, testEffectiveDate));
		System.out.println("isAfterIgnoreTime: is testEffectiveDate after date =" + DateUtils.isAfterIgnoreTime(testEffectiveDate, date ));		
		System.out.println("isAfterIgnoreTime: is date after testEffectiveDate =" + DateUtils.isAfterIgnoreTime(date, testEffectiveDate));		
	
	
	}
	
	/**
     Person updated = this.getFinancialsService().updateFinancialAssessment(
                new Integer(2005), (Person) incoming.clone()); 

	 */
	public void test8672fromUI() throws Exception {
		Person onFile = buildSimplePerson();
		Integer incomeYear2007 = new Integer(2007);		
		IncomeTest oldTst = this.build_MT_IncomeTest(onFile,incomeYear2007, MeansTestStatus.MT_STATUS_MT_COPAY_EXEMPT);
		oldTst.setLastEditedDate(getDate(2008,1,1));
		oldTst.setDiscloseFinancialInformation(Boolean.TRUE);
		oldTst.setEffectiveDate(getDate(2008,10,11));
		this.makeRejectEGTPersonForIncomeTesting(onFile);
        this.setEnrollmentStatus(EnrollmentStatus.CODE_VERIFIED.getCode(), onFile);
        onFile.getEnrollmentDetermination().setPriorityGroup(this.getLookupService().getEnrollmentPriorityGroupByCode(
                EnrollmentPriorityGroup.GROUP_5.getCode()));
        onFile.getEnrollmentDetermination().setPrioritySubGroup(null);
		
	       
		Integer incomeYear2008 = new Integer(2008);        
		IncomeTestSource src = getTestSource(IncomeTestSource.CODE_VAMC);
		IncomeTestType typ = getTestType(IncomeTestType.CODE_MEANS_TEST);
		MeansTestStatus status = getMeansTestStatus(MeansTestStatus.MT_STATUS_MT_COPAY_REQUIRED);
		
		VAFacility facility = getVaFacility(VAFacility.CODE_ALBUQUERQUE);
		Date lastEditDate = getDate(2008,10,1);
		IncomeTest tst = buildIncomeTestData(incomeYear2008, onFile, typ, src, status,	facility);
		tst.setFutureTest(Boolean.TRUE);
		tst.setLastEditedDate(lastEditDate);
		tst.setPrimaryIncomeTest(Boolean.TRUE);		
		tst.setDiscloseFinancialInformation(Boolean.TRUE);
		tst.setEffectiveDate(getDate(2009,10,11));
		tst.setThresholdA(new BigDecimal(1400));
		tst.setGmtThresholdAmount(new BigDecimal(1475));	
		// TODO fix code so that a missing statement does not get null pointer error.
		onFile.setFinancialStatement(new Integer(2008),this.createFinancialStatement());		

		
		Person incoming = (Person)savePerson(onFile).clone();

		incoming.setFinancialStatement(new Integer(2008),this.createFinancialStatement());		
		incoming.getFinancialStatement(new Integer(2008)).setDebt(null);
		IncomeType it = this.getLookupService().getIncomeTypeByCode(IncomeType.INCOME_TYPE_TOTAL_INCOME_FROM_EMPLOYMENT.getCode());
		Income i = new Income();
		i.setAmount(new BigDecimal(40000));
		incoming.getFinancialStatement(new Integer(2008)).setIncome(it, i);

		IncomeTest incomingTest = incoming.getIncomeTest(new Integer(2008));
		IncomeTestStatus stat = incomingTest.getIncomeTestStatus();
		MeansTestStatus mtStatus = 
			this.getLookupService().getMeansTestStatusByCode(MeansTestStatus.MT_STATUS_GMT_COPAY_REQUIRED.getCode());
		incomingTest.removeAllStatuses();
		incomingTest.setIncomeTestStatus(getTestType(IncomeTestType.CODE_MEANS_TEST), mtStatus);
		VAFacility fac = getVaFacility(VAFacility.CODE_HEC);
		incomingTest.setDiscloseFinancialInformation(Boolean.TRUE);
		incomingTest.setAgreesToPayDeductible(Boolean.TRUE);
		incomingTest.setFutureTest(Boolean.FALSE);
		incomingTest.setThresholdA(new BigDecimal(800));
		incomingTest.setGmtThresholdAmount(new BigDecimal(810));	

		// NOTE - issue with R305 where check of future date not consistent. Should be calendar day, ignoring time
		incomingTest.setEffectiveDate(new Date());

		
		
		//		incomingTest.setThresholdA(new BigDecimal(800));
//		incomingTest.setGmtThresholdAmount(new BigDecimal(875));
//		incomingTest.setNetIncome(new BigDecimal(1000));
		incomingTest.setSiteConductingTest(fac);
		incomingTest.setSendForAdjudication(Boolean.TRUE);

		incoming.removeAllIncomeTests();		
		incoming.setIncomeTest(incomeYear2008, incomingTest);
		Person updated = this.getFinancialsService().updateFinancialAssessment( new Integer(2008), incoming); 
       IncomeTest current = this.getPersonHelperService().getCurrentIncomeTest(updated);
		System.out.println("current test is =" + current.getIncomeYear().toString());       
       assertTrue( "should be 2007 was" + current.getIncomeYear().toString(),incomeYear2007.equals(current.getIncomeYear()));
        IncomeTest res = updated.getIncomeTest(incomeYear2008);
		System.out.println("2008 effective date =" + res.getEffectiveDate().toString());        
        assertTrue( "should be future",res.getFutureTest().booleanValue());
 
        
        assertTrue(updated.getEligibilityVerification().getEligibilityStatus()
                .getCode().equals(EligibilityStatus.CODE_VERIFIED.getName()));
        EnrollmentStatus enstatus = updated.getEnrollmentDetermination().getEnrollmentStatus();
        assertTrue(enstatus.getCode().equals(EnrollmentStatus.CODE_VERIFIED.getName()));

	}  	
		
	
	
	public void test8672forNew2008_9584_Exempt() throws Exception {
		Person onFile = buildSimplePerson();
		IncomeTest oldTst = this.build_MT_IncomeTest(onFile,new Integer(2007), MeansTestStatus.MT_STATUS_MT_COPAY_EXEMPT);
		oldTst.setLastEditedDate(getDate(2008,1,1));
		oldTst.setDiscloseFinancialInformation(Boolean.TRUE);
		oldTst.setEffectiveDate(getDate(2008,10,11));
		this.makeRejectEGTPersonForIncomeTesting(onFile);
        this.setEnrollmentStatus(EnrollmentStatus.CODE_VERIFIED.getCode(), onFile);
        onFile.getEnrollmentDetermination().setPriorityGroup(this.getLookupService().getEnrollmentPriorityGroupByCode(
                EnrollmentPriorityGroup.GROUP_5.getCode()));
        onFile.getEnrollmentDetermination().setPrioritySubGroup(null);
		
		Integer incomeYear2008 = new Integer(2008);        
		IncomeTestSource src = getTestSource(IncomeTestSource.CODE_VAMC);
		IncomeTestType typ = getTestType(IncomeTestType.CODE_MEANS_TEST);
		MeansTestStatus status = getMeansTestStatus(MeansTestStatus.MT_STATUS_MT_COPAY_REQUIRED);
		
		VAFacility facility = getVaFacility(VAFacility.CODE_ALBUQUERQUE);
		Date lastEditDate = getDate(2008,10,1);
		IncomeTest tst = buildIncomeTestData(incomeYear2008, onFile, typ, src, status,
				facility);
		tst.setFutureTest(Boolean.TRUE);
		tst.setLastEditedDate(lastEditDate);
		tst.setPrimaryIncomeTest(Boolean.FALSE);		
		tst.setDiscloseFinancialInformation(Boolean.TRUE);
		tst.setEffectiveDate(getDate(2009,10,11));
		tst.setThresholdA(new BigDecimal(1400));
		tst.setGmtThresholdAmount(new BigDecimal(1475));	
		// TODO fix code so that a missing statement does not get null pointer error.
		onFile.setFinancialStatement(new Integer(2008),this.createFinancialStatement());		
		
		
        Person incoming = (Person)savePerson(onFile).clone();
		incoming.removeAllIncomeTests();
		incoming.setFinancialStatement(new Integer(2008),this.createFinancialStatement());		

		VAFacility fac = getVaFacility(VAFacility.CODE_ALBUQUERQUE);
		IncomeTest incomingTest = buildIncoming_MT_IncomeTest(incoming, new Integer(2008),
//				MeansTestStatus.MT_STATUS_GMT_COPAY_REQUIRED, this.getDate(2009, 6, 1));
				MeansTestStatus.MT_STATUS_GMT_COPAY_REQUIRED, new Date());
		incomingTest.setLastEditedDate(new Date());		
		incomingTest.setDiscloseFinancialInformation(Boolean.TRUE);
		incomingTest.setFutureTest(Boolean.FALSE);
		incomingTest.setThresholdA(new BigDecimal(800));
		incomingTest.setGmtThresholdAmount(new BigDecimal(875));
		incomingTest.setNetIncome(new BigDecimal(880));
		incomingTest.setSiteConductingTest(fac);
        
        VerificationInfo vi = new VerificationInfo();
        vi.setSiteNumber(incoming.getEligibilityVerification()
                .getVerificationFacility().getCode());
		Integer incomeYear2007 = new Integer(2007);
        Person updated = this.getMessagingService().processZ07(incoming,
                fac, vi);
        IncomeTest res = updated.getIncomeTest(incomeYear2008);
		System.out.println("2008 effective date =" + res.getEffectiveDate().toString());        
        assertTrue( "should be future",res.getFutureTest().booleanValue());
        IncomeTest current = this.getPersonHelperService().getCurrentIncomeTest(updated);
        assertTrue( "should be 2007",incomeYear2007.equals(current.getIncomeYear()));
		System.out.println("current test is =" + current.getIncomeYear().toString());        
        assertTrue(updated.getEligibilityVerification().getEligibilityStatus()
                .getCode().equals(EligibilityStatus.CODE_VERIFIED.getName()));
        EnrollmentStatus enstatus = updated.getEnrollmentDetermination().getEnrollmentStatus();
        assertTrue(enstatus.getCode().equals(EnrollmentStatus.CODE_VERIFIED.getName()));

	}  	
		
	
	private IncomeTest build_MT_IncomeTest(Person person, Integer year, MeansTestStatus.Code code) throws Exception {
		VAFacility facility = getVaFacility(VAFacility.CODE_ALBUQUERQUE);
		IncomeTest test = buildIncomeTestData(year,
				person,
				getTestType(IncomeTestType.CODE_MEANS_TEST),
				getTestSource(IncomeTestSource.CODE_VAMC), 
				getMeansTestStatus(code),
				facility);
		test.setFutureTest(Boolean.FALSE);
		
		return test;
	}
	
	private IncomeTest buildIncoming_MT_IncomeTest(Person person, Integer year, MeansTestStatus.Code code, Date effectiveDate) throws Exception {
		VAFacility facility = getVaFacility(VAFacility.CODE_ALBUQUERQUE);
		IncomeTest test = buildIncomeTestData(year,
				person,
				getTestType(IncomeTestType.CODE_MEANS_TEST),
				getTestSource(IncomeTestSource.CODE_VAMC), 
				getMeansTestStatus(code),
				facility, this.getDateOnly(new Date()),new Date(), effectiveDate);
		test.setFutureTest(Boolean.FALSE);		
		return test;
	}
	
	private void makeRejectEGTPersonForIncomeTesting(Person person) throws Exception {
		this.attachApplication(person);
		person.getApplication().setApplicationDate(getDate(2009,0,1));

		this.attachVerifiedEnrollment(person);
        this.setEnrollmentStatus(EnrollmentStatus.CODE_VERIFIED.getCode(), 
        		person);
        person.getEnrollmentDetermination().setPriorityGroup(this.getLookupService().getEnrollmentPriorityGroupByCode(
                EnrollmentPriorityGroup.GROUP_8.getCode()));
        person.getEnrollmentDetermination().setPrioritySubGroup(
                this.getLookupService().getEnrollmentPrioritySubGroupByCode(EnrollmentPrioritySubGroup.SUBGROUP_C.getCode()));

        Eligibility e = new Eligibility();
	    e.setType(this.getLookupService().getEligibilityTypeByCode(EligibilityType.NSC.getCode()));
	    person.getEnrollmentDetermination().setPrimaryEligiblity(e);
        setEligibilityStatusToVerified(person);
        person.getEligibilityVerification().setVerificationFacility(
        		this.getLookupService().getVaFacilityByCode(VAFacility.CODE_HEC.getCode()));
        this.addPermanentAddress(person, Calendar.getInstance().getTime(),
                null);
	}

	/**
	 * Problem description from CQ for 8373:
	 * I had a vet set up in both Chey6 & Dayt20 (WISDENX One One). The vet was
	 * NSC with a current HEC means test on file. In the Chey6 account, I set
	 * the Eligible for Medicaid to 'YES' and transmitted the Z07. ESR correctly
	 * uploaded the data and changed his MT status to No Longer Reqd and the Rx
	 * Test is now primary.
	 * 
	 * The Z10 back to the sites: ZMT^1^""^""
	 * ZMT^2^20081203^M^29000^^^^^0^20081203134600-0600^^1^^^^""^2^3^1^^^""^^^20090224081446-0600^M^^^^1
	 * 
	 * I checked the sites and noticed that Dayt20 had the Rx Test primary and
	 * the MT is NLR. When I checked in Chey6, the Rx Test is primary, however,
	 * the MT is totally gone. I suspect it is due to the ZMT^1 segment we are
	 * sending (ZMT^1^""^"") because double quotes are considered deletes. I am
	 * unsure as to why the MT was deleted at one site and not the other since
	 * both were sent the exact same thing.
	 * 
WISDENX,ONE ONE

From what I see, here is the scenario:
Vet had a HEC owned 2007 MT Copay Exempt test on ESR.  In Dayt20, I added Elig for Medicaid to the vet.  Here are the ZMT segments from the Z07:
ZMT^1^20081203^N^29000^12000^""^""^34117^0^200812031346-0500^""^1^""^""^""^""^1^3^^^""^^^^20081211093529-0500^A^^37150^^1
ZMT^2^20081203^M^29000^12000^""^""^^0^200812031346-0500^^1^^^""^""^2^3^^^""^^^^20081211093529-0500^M^^^^1
ZMT^4^^^^^^^^^^^^^^^^^^^^^


ESR then shows Rx Copay Test as primary with MT status = No Longer Reqd and here is the Z10 data sent to both Chey6 & Dayt20:
ZMT^1^""^""
ZMT^2^20081203^M^29000^^^^^0^20081203134600-0600^^1^^^^""^2^3^1^^^""^^^20081211083615-0600^M^^^^1
ZMT^4^""^""



WISX,GMTADD

This is a vet I did in June with the same scenario, except I added the Elig for Medicaid on the UI

Vet had a 2008 MT Copay Exempt on ESR (Eff Dt = 5/5/2009 and Site Conducting Test Dayt20).  On the UI, I added Eligible for Medicaid which caused ESR to calculate Rx Test as primary (and MT set to No Longer Reqd).  Here is the Z10 sent to both Chey6 & Dayt20:

ZMT^1^""^""
ZMT^2^20090505^M^25000^^^^^440.95^20090505133600-0500^^0^^^^""^2^4^1^^^988^^^20090605131924-0500^M^^^^1
ZMT^4^""^""

Here is the very next Z07 sent from Dayt20:
ZMT^1^20090505^N^25000^50000^""^""^29402^440.95^200905051336-0500^""^0^""^""^""^""^1^1^^^""^988^^^20090605141937-0500^A^^39550^^1
ZMT^2^20090505^M^25000^50000^""^""^^440.95^200905051336-0500^^0^^^""^""^2^1^^^""^988^^^20090605141937-0500^M^^^^1
ZMT^4^^^^^^^^^^^^^^^^^^^^^

And the next Z07 from Chey6 (which is ignored because site sending Z07 is not equal to the Site Conducting Test):
ZMT^1^^^^^^^^^^^^^^^^^^^^^
ZMT^2^20090505^M^25000^""^""^""^^440.95^200905051336-0500^^0^^^""^""^2^4^^^""^988^^^20090605131924-0500^M^^^^1
ZMT^4^^^^^^^^^^^^^^^^^^^^^

	 * 
	 * @throws Exception
	 */
	public void test8373TestDelete() throws Exception {
		VAFacility facility = getVaFacility(VAFacility.CODE_ALBUQUERQUE);
		Person onFile = buildSimplePerson();
		
		// Make a current MT 
		IncomeTest oldTst = this.build_MT_IncomeTest(onFile,new Integer(2008), MeansTestStatus.MT_STATUS_MT_COPAY_EXEMPT);
		oldTst.setLastEditedDate(getDate(2008,9,1));
		oldTst.setDiscloseFinancialInformation(Boolean.TRUE);
		oldTst.setEffectiveDate(getDate(2009,1,20));
		oldTst.setPrimaryIncomeTest(Boolean.TRUE);
		
		makeRejectEGTPersonForIncomeTesting(onFile);

        this.setEnrollmentStatus(EnrollmentStatus.CODE_VERIFIED.getCode(), onFile);
        onFile.getEnrollmentDetermination().setPriorityGroup(this.getLookupService().getEnrollmentPriorityGroupByCode(
                EnrollmentPriorityGroup.GROUP_8.getCode()));
        onFile.getEnrollmentDetermination().setPrioritySubGroup(
                this.getLookupService().getEnrollmentPrioritySubGroupByCode(EnrollmentPrioritySubGroup.SUBGROUP_C.getCode()));

		onFile.setFinancialStatement(new Integer(2008),this.createFinancialStatement());
		this.attachPharmacyCopayTestWithNonExempt(onFile);
		
        Person incoming = (Person)savePerson(onFile).clone();
        
		IncomeTestSource testSource = getTestSource(IncomeTestSource.CODE_VAMC);
		IncomeTestType testType = getTestType(IncomeTestType.CODE_MEANS_TEST);
		MeansTestStatus testStatus = getMeansTestStatus(MeansTestStatus.MT_STATUS_MT_COPAY_EXEMPT);

		// Handy place to set set future effective date
		IncomeTest incomingTest = buildIncomeTestData(new Integer(2008), incoming,
				testType, testSource, testStatus, facility, this.getDateOnly(new Date()),
				new Date(), this.getDateOnly(this.getDate(2009, 7, 1)));
		incomingTest.setDiscloseFinancialInformation(Boolean.TRUE);
        
        
        
        VerificationInfo vi = new VerificationInfo();
        vi.setSiteNumber(incoming.getEligibilityVerification()
                .getVerificationFacility().getCode());

        // Add Medicare
        this.attachMedicare(incoming);
        this.attachMedicaidFactor(incoming);
        
        Person updated = this.getMessagingService().processZ07(incoming,facility, vi);
        
        IncomeTest current = this.getPersonHelperService().getCurrentIncomeTest(updated);
        assertTrue("current test is not 2008",current.getIncomeYear().equals(new Integer(2008)));
        IncomeTest res = updated.getIncomeTest(new Integer(2008));
        String code = res.getStatus().getCode();
        assertTrue("should be no longer required",MeansTestStatus.MT_STATUS_NO_LONGER_REQUIRED.getCode().equals(code));
        assertFalse( "should not be future",res.getFutureTest().booleanValue());
        assertTrue(updated.getEligibilityVerification().getEligibilityStatus()
                .getCode().equals(EligibilityStatus.CODE_VERIFIED.getName()));
        EnrollmentStatus enstatus = updated.getEnrollmentDetermination().getEnrollmentStatus();
        assertTrue(enstatus.getCode().equals(EnrollmentStatus.CODE_VERIFIED.getName()));

	}  	
	
		
	public void test8672TestPrimaryonexisting2008For9644() throws Exception {
		VAFacility facility = getVaFacility(VAFacility.CODE_ALBUQUERQUE);
		Person onFile = buildSimplePerson();
		
		IncomeTest oldTst = this.build_MT_IncomeTest(onFile,new Integer(2007), MeansTestStatus.MT_STATUS_GMT_COPAY_REQUIRED);
		oldTst.setLastEditedDate(getDate(2008,1,1));
		oldTst.setDiscloseFinancialInformation(Boolean.TRUE);
		oldTst.setEffectiveDate(getDate(2008,11,20));
		oldTst.setPrimaryIncomeTest(Boolean.TRUE);
		
		makeRejectEGTPersonForIncomeTesting(onFile);

        this.setEnrollmentStatus(EnrollmentStatus.CODE_VERIFIED.getCode(), onFile);
        onFile.getEnrollmentDetermination().setPriorityGroup(this.getLookupService().getEnrollmentPriorityGroupByCode(
                EnrollmentPriorityGroup.GROUP_8.getCode()));
        onFile.getEnrollmentDetermination().setPrioritySubGroup(
                this.getLookupService().getEnrollmentPrioritySubGroupByCode(EnrollmentPrioritySubGroup.SUBGROUP_C.getCode()));

		Integer incomeYear2008 = new Integer(2008);        
		IncomeTestSource src = getTestSource(IncomeTestSource.CODE_VAMC);
		IncomeTestType typ = getTestType(IncomeTestType.CODE_MEANS_TEST);
		MeansTestStatus status = getMeansTestStatus(MeansTestStatus.MT_STATUS_GMT_COPAY_REQUIRED);

		Date lastEditDate = getDate(2008,10,1);
		IncomeTest tst = buildIncomeTestData(incomeYear2008, onFile, typ, src, status,
				facility);
		tst.setFutureTest(Boolean.FALSE);
		tst.setLastEditedDate(lastEditDate);
		tst.setPrimaryIncomeTest(Boolean.FALSE);		
		tst.setDiscloseFinancialInformation(Boolean.TRUE);
		tst.setEffectiveDate(getDate(2009,1,20));
		tst.setThresholdA(new BigDecimal(800));
		tst.setGmtThresholdAmount(new BigDecimal(875));	

		onFile.setFinancialStatement(new Integer(2008),this.createFinancialStatement());
		
        Person incoming = (Person)savePerson(onFile).clone();
		incoming.removeAllIncomeTests();
		IncomeTestSource testSource = getTestSource(IncomeTestSource.CODE_VAMC);
		IncomeTestType testType = getTestType(IncomeTestType.CODE_MEANS_TEST);
		MeansTestStatus testStatus = getMeansTestStatus(MeansTestStatus.MT_STATUS_MT_COPAY_REQUIRED);
		VAFacility fac = getVaFacility(VAFacility.CODE_ALBUQUERQUE);
		IncomeTest incomingTest = buildIncomeTestData(incomeYear2008, incoming,
				testType, testSource, testStatus, facility, this.getDateOnly(new Date()),
				new Date(), this.getDateOnly(this.getDate(2009, 9, 1)));
		incomingTest.setDiscloseFinancialInformation(Boolean.TRUE);
		incomingTest.setFutureTest(Boolean.FALSE);
		incomingTest.setPrimaryIncomeTest(Boolean.FALSE);
		incomingTest.setLastEditedDate(new Date());

		incomingTest.setThresholdA(new BigDecimal(400));
		incomingTest.setGmtThresholdAmount(new BigDecimal(475));	

		incoming.setFinancialStatement(new Integer(2008),this.createFinancialStatement());
		
		incomingTest.setSiteConductingTest(fac);
        this.setEligibilityStatusToVerified(incoming);
        incoming.getEligibilityVerification().setVerificationFacility(
        		this.getLookupService().getVaFacilityByCode(VAFacility.CODE_HEC.getCode()));
        
        this.addPermanentAddress(incoming, Calendar.getInstance().getTime(),
                null);
        VerificationInfo vi = new VerificationInfo();
        vi.setSiteNumber(incoming.getEligibilityVerification()
                .getVerificationFacility().getCode());

        Person updated = this.getMessagingService().processZ07(incoming,
                fac, vi);
        IncomeTest current = this.getPersonHelperService().getCurrentIncomeTest(updated);
        assertTrue("current test is not 2008",current.getIncomeYear().equals(incomeYear2008));
        IncomeTest res = updated.getIncomeTest(incomeYear2008);
        String code = res.getStatus().getCode();
        assertTrue("should be GMT Copay Required",MeansTestStatus.MT_STATUS_GMT_COPAY_REQUIRED.getCode().equals(code));
        assertFalse( "should not be future",res.getFutureTest().booleanValue());
        assertTrue(updated.getEligibilityVerification().getEligibilityStatus()
                .getCode().equals(EligibilityStatus.CODE_VERIFIED.getName()));
        EnrollmentStatus enstatus = updated.getEnrollmentDetermination().getEnrollmentStatus();
        assertTrue(enstatus.getCode().equals(EnrollmentStatus.CODE_VERIFIED.getName()));

	}  	

	
	
	
	
	
	public void testProcessIncomeTestNewBecomePrimary() throws Exception {
		Integer incomeYear = new Integer(2007);
		VAFacility facility = getVaFacility(VAFacility.CODE_ALBUQUERQUE);

		Person p = buildSimplePerson();
		IncomeTestSource src = getTestSource(IncomeTestSource.CODE_VAMC);
		IncomeTestType typ = getTestType(IncomeTestType.CODE_MEANS_TEST);
		MeansTestStatus status = getMeansTestStatus(MeansTestStatus.MT_STATUS_GMT_COPAY_REQUIRED);

		Date lastEditDate = new Date();
		IncomeTest tst = buildIncomeTestData(incomeYear, p, typ, src, status,
				facility);
		tst.setLastEditedDate(lastEditDate);
		tst.setDiscloseFinancialInformation(Boolean.FALSE);

		Person person = savePerson(p);

		// Clone the pristine to have a working version
		Person incoming = (Person) person.clone();
		Person onFile = getPersonService().getPerson(
				person.getPersonEntityKey());

		IncomeTestSource testSource = getTestSource(IncomeTestSource.CODE_VAMC);
		IncomeTestType testType = getTestType(IncomeTestType.CODE_MEANS_TEST);
		MeansTestStatus testStatus = getMeansTestStatus(MeansTestStatus.MT_STATUS_MT_COPAY_EXEMPT);

		// Handy place to set set future effective date
		IncomeTest incomingTest = buildIncomeTestData(new Integer(2008), incoming,
				testType, testSource, testStatus, facility, this.getDateOnly(new Date()),
				new Date(), this.getDateOnly(this.getDate(2009, 6, 1)));
		incomingTest.setDiscloseFinancialInformation(Boolean.TRUE);
		// making above cond different will make it primary test

		try {
			Person result = invokeFinancialRuleService(new Integer(2008), facility,
					incoming, onFile);

			IncomeTest incomingIncomeTest = incoming.getIncomeTest(incomeYear);
			IncomeTest resultIncomeTest = result.getIncomeTest(new Integer(2008));
			assertNotNull("Result IncomeTest should not be null",
					resultIncomeTest);

			// This should be over ridden by the incoming (should not become
			// future test)
			assertEquals(incomingIncomeTest.getSource(), resultIncomeTest
					.getSource());
			assertEquals(MeansTestStatus.MT_STATUS_MT_COPAY_EXEMPT.getCode(), resultIncomeTest
					.getStatus().getCode());
			assertEquals(testStatus, resultIncomeTest.getStatus());
			assertEquals(testType, resultIncomeTest.getType());
			assertEquals(testSource, resultIncomeTest.getSource());
			assertEquals(facility, resultIncomeTest.getSiteConductingTest());
			assertEquals(incomingIncomeTest.getSiteConductingTest(),
					resultIncomeTest.getSiteConductingTest());
			assertFalse( resultIncomeTest.getFutureTest().booleanValue());			
		} catch (Exception e) {
			this.fail("Failed due to exception ", e);
		}
	}
	
	public void testDateBuilder() throws Exception  {
		Date date = this.build("20090709103418-0500");
		assertNotNull( date);
		
		Date date2 = this.build("20090709123418-0500");
		assertNotNull( date);
	}
	 public Date build(String value) throws BuilderException   {
	        if(value == null)
	        {
	            return null;
	        }
	        else
	        {
	        	ImpreciseDateBuilder b = new ImpreciseDateBuilder();
	            ImpreciseDate ouptut = (ImpreciseDate)b.build(value);

	            if(ouptut.getDate() == null)
	            {
	                throw new BuilderException("Invalid date");
	            }
	            else
	            {
	                return ouptut.getDate();
	            }
	        }
	    }
	
    public void testProcessIncomeTestBecomeFuture() throws Exception {
        Integer prevYear = new Integer(2007);
        VAFacility prevFacility = getVaFacility(VAFacility.CODE_CONNECTICUT_HCS);

        Person p = buildSimplePerson();
        IncomeTestSource src = getTestSource(IncomeTestSource.CODE_VAMC);
        IncomeTestType typ = getTestType(IncomeTestType.CODE_MEANS_TEST);
        MeansTestStatus status = getMeansTestStatus(MeansTestStatus.MT_STATUS_GMT_COPAY_REQUIRED);

        Date lastEditDate = this.getDateOnly(new Date());
        Date completedDate = this.getDateOnly(new Date());
        Date effectiveDate = this.getDateOnly(new Date());

        IncomeTest tst = buildIncomeTestData(prevYear, p, typ, src, status,
                prevFacility);
        tst.setLastEditedDate(lastEditDate);
        tst.setEffectiveDate(effectiveDate);
        tst.setCompletedDate(completedDate);

        Person person = savePerson(p);

        // Clone the pristine to have a working version
        Person incoming = (Person) person.clone();
        Person onFile = getPersonService().getPerson(
                person.getPersonEntityKey());

        Integer incomeYear = new Integer(2008);
        VAFacility facility = getVaFacility(VAFacility.CODE_ALBUQUERQUE);

        IncomeTestSource testSource = getTestSource(IncomeTestSource.CODE_VAMC);
        IncomeTestType testType = getTestType(IncomeTestType.CODE_MEANS_TEST);
        MeansTestStatus testStatus = getMeansTestStatus(MeansTestStatus.MT_STATUS_GMT_COPAY_REQUIRED);

        buildIncomeTestData(incomeYear, incoming, testType, testSource, testStatus, facility);
       

        try {
            Person result = invokeFinancialRuleService(incomeYear, facility,
                    incoming, onFile);

            IncomeTest incomingIncomeTest = incoming.getIncomeTest(incomeYear);
            IncomeTest resultIncomeTest = result.getIncomeTest(incomeYear);

            assertNotNull("Incoming IncomeTest should not be null",
                    incomingIncomeTest);
            assertNotNull("Result IncomeTest should not be null",
                    resultIncomeTest);
            assertEquals(resultIncomeTest.getEffectiveDate(), addYearToDate(effectiveDate));
            assertEquals(resultIncomeTest.getEffectiveDate(), addYearToDate(effectiveDate));
        } catch (Exception e) {
            this.fail("Failed due to exception ", e);
        }
    }

    public void testEffectiveImmediately1() throws Exception {
        effectiveImmediatelyTest(IncomeTestType.CODE_MEANS_TEST, MeansTestStatus.MT_STATUS_NO_LONGER_REQUIRED, MeansTestStatus.MT_STATUS_GMT_COPAY_REQUIRED);
    }

    public void testEffectiveImmediately1_1() throws Exception {
        effectiveImmediatelyTest(IncomeTestType.CODE_MEANS_TEST, MeansTestStatus.MT_STATUS_NO_LONGER_REQUIRED, MeansTestStatus.MT_STATUS_MT_COPAY_EXEMPT);
    }

    public void testEffectiveImmediately1_2() throws Exception {
        effectiveImmediatelyTest(IncomeTestType.CODE_MEANS_TEST, MeansTestStatus.MT_STATUS_NO_LONGER_REQUIRED, MeansTestStatus.MT_STATUS_MT_COPAY_REQUIRED);
    }

    public void testEffectiveImmediately1_3() throws Exception {
        effectiveImmediatelyTest(IncomeTestType.CODE_MEANS_TEST, MeansTestStatus.MT_STATUS_NO_LONGER_REQUIRED, MeansTestStatus.MT_STATUS_PENDING_ADJUDICATION);
    }

    public void testEffectiveImmediately2() throws Exception {
        effectiveImmediatelyTest(IncomeTestType.CODE_MEANS_TEST, MeansTestStatus.MT_STATUS_GMT_COPAY_REQUIRED, MeansTestStatus.MT_STATUS_GMT_COPAY_REQUIRED, MeansTestStatus.MT_STATUS_NON_EXEMPT, MeansTestStatus.MT_STATUS_EXEMPT);
    }
    
    public void testEffectiveImmediately3() throws Exception {
        effectiveImmediatelyTest(IncomeTestType.CODE_MEANS_TEST, MeansTestStatus.MT_STATUS_GMT_COPAY_REQUIRED, MeansTestStatus.MT_STATUS_MT_COPAY_EXEMPT);
    }
    
    public void testEffectiveImmediately3_1() throws Exception {
        effectiveImmediatelyTest(IncomeTestType.CODE_MEANS_TEST, MeansTestStatus.MT_STATUS_MT_COPAY_REQUIRED, MeansTestStatus.MT_STATUS_MT_COPAY_EXEMPT);
    }
    
    public void testEffectiveImmediately3_2() throws Exception {
        effectiveImmediatelyTest(IncomeTestType.CODE_MEANS_TEST, MeansTestStatus.MT_STATUS_PENDING_ADJUDICATION, MeansTestStatus.MT_STATUS_MT_COPAY_EXEMPT);
    }

    public void testEffectiveImmediately4() throws Exception {
        effectiveImmediatelyTest(IncomeTestType.CODE_MEANS_TEST, MeansTestStatus.MT_STATUS_MT_COPAY_REQUIRED, MeansTestStatus.MT_STATUS_GMT_COPAY_REQUIRED);
    }

    public void testEffectiveImmediately5() throws Exception {        
        MeansTestStatus.Code currentMTCode = MeansTestStatus.MT_STATUS_MT_COPAY_REQUIRED;
        MeansTestStatus.Code incomingMTCode = MeansTestStatus.MT_STATUS_PENDING_ADJUDICATION;
        
        BigDecimal currentGMTThreshold = null;
        BigDecimal currentMTThreshold = null;
        BigDecimal incomingGMTThreshold = new BigDecimal(10000);
        BigDecimal incomingMTThreshold = new BigDecimal(5000);
        
        effectiveImmediatelyTest(IncomeTestType.CODE_MEANS_TEST, currentMTCode, incomingMTCode, currentGMTThreshold, incomingGMTThreshold, currentMTThreshold, incomingMTThreshold);
    }
    
    public void testEffectiveImmediately6() throws Exception {
        MeansTestStatus.Code currentMTCode = MeansTestStatus.MT_STATUS_PENDING_ADJUDICATION;
        MeansTestStatus.Code incomingMTCode = MeansTestStatus.MT_STATUS_PENDING_ADJUDICATION;
        
        BigDecimal currentGMTThreshold = new BigDecimal(8000);
        BigDecimal currentMTThreshold = new BigDecimal(10000);
        BigDecimal incomingGMTThreshold = new BigDecimal(10000);
        BigDecimal incomingMTThreshold = new BigDecimal(5000);
        
        effectiveImmediatelyTest(IncomeTestType.CODE_MEANS_TEST, currentMTCode, incomingMTCode, currentGMTThreshold, incomingGMTThreshold, currentMTThreshold, incomingMTThreshold);
    }

    public void testEffectiveImmediately7() throws Exception {
        effectiveImmediatelyTest(IncomeTestType.CODE_CO_PAY_EXEMPTION_TEST, MeansTestStatus.MT_STATUS_INCOMPLETE_RX_COPAY_ONLY, MeansTestStatus.MT_STATUS_EXEMPT);
    }
    
    public void testEffectiveImmediately8() throws Exception {
        effectiveImmediatelyTest(IncomeTestType.CODE_CO_PAY_EXEMPTION_TEST, MeansTestStatus.MT_STATUS_INCOMPLETE_RX_COPAY_ONLY, MeansTestStatus.MT_STATUS_NON_EXEMPT);
    }

    public void testEffectiveImmediately9() throws Exception {
        effectiveImmediatelyTest(IncomeTestType.CODE_CO_PAY_EXEMPTION_TEST, MeansTestStatus.MT_STATUS_NON_EXEMPT, MeansTestStatus.MT_STATUS_EXEMPT);
    }

    private void effectiveImmediatelyTest(IncomeTestType.Code itTypeCode, MeansTestStatus.Code currentMTStatus, MeansTestStatus.Code incomingMTStatus) throws Exception {
        effectiveImmediatelyTest(itTypeCode, currentMTStatus, incomingMTStatus, null, null);       
    }
    
    private void effectiveImmediatelyTest(IncomeTestType.Code itTypeCode, 
            MeansTestStatus.Code currentMTStatus, 
            MeansTestStatus.Code incomingMTStatus, 
            MeansTestStatus.Code currentRxStatus, MeansTestStatus.Code incomingRxStatus) throws Exception {
        
        effectiveImmediatelyTest(itTypeCode, currentMTStatus, incomingMTStatus, currentRxStatus, incomingRxStatus, null, null, null, null);               
    }
    private void effectiveImmediatelyTest(IncomeTestType.Code itTypeCode, 
            MeansTestStatus.Code currentMTStatus, 
            MeansTestStatus.Code incomingMTStatus, 
            BigDecimal currentGMTThreshold,
            BigDecimal incomingGMTThreshold, 
            BigDecimal currentMTThreshold, BigDecimal incomingMTThreshold) throws Exception {
        
        effectiveImmediatelyTest(itTypeCode, currentMTStatus, incomingMTStatus, null, null, currentGMTThreshold, incomingGMTThreshold, currentMTThreshold, incomingMTThreshold);               
    }
    
    private void effectiveImmediatelyTest(IncomeTestType.Code itTypeCode, 
                                            MeansTestStatus.Code currentMTStatus, 
                                            MeansTestStatus.Code incomingMTStatus, 
                                            MeansTestStatus.Code currentRxStatus, 
                                            MeansTestStatus.Code incomingRxStatus, 
                                            BigDecimal currentGMTThreshold, 
                                            BigDecimal incomingGMTThreshold, 
                                            BigDecimal currentMTThreshold, 
                                            BigDecimal incomingMTThreshold) throws Exception {
        Integer prevYear = new Integer(2002);
        VAFacility prevFacility = getVaFacility(VAFacility.CODE_CONNECTICUT_HCS);

        Person p = buildSimplePerson();
        IncomeTestSource src = getTestSource(IncomeTestSource.CODE_HEC);
        IncomeTestType typ = getTestType(itTypeCode);
        MeansTestStatus status = getMeansTestStatus(currentMTStatus);

        Date lastEditDate = new Date();
        Date completedDate = new Date();
        Date effectiveDate = new Date();

        IncomeTest tst = buildIncomeTestData(prevYear, p, typ, src, status, prevFacility);     
        setRxCopayStatus(tst, currentRxStatus);
        tst.setGmtThresholdAmount(currentGMTThreshold);
        tst.setThresholdA(currentMTThreshold);

        tst.setLastEditedDate(lastEditDate);
        tst.setEffectiveDate(effectiveDate);
        tst.setCompletedDate(completedDate);

        Person person = savePerson(p);

        // Clone the pristine to have a working version
        Person incoming = (Person) person.clone();
        Person onFile = getPersonService().getPerson(
                person.getPersonEntityKey());

        Integer incomeYear = new Integer(2003);
        VAFacility facility = getVaFacility(VAFacility.CODE_ALBUQUERQUE);

        IncomeTestSource testSource = getTestSource(IncomeTestSource.CODE_VAMC);
        IncomeTestType testType = getTestType(itTypeCode);
        MeansTestStatus testStatus = getMeansTestStatus(incomingMTStatus);

        IncomeTest incomingTest = buildIncomeTestData(incomeYear, incoming, testType, testSource, testStatus, facility);
        setRxCopayStatus(incomingTest, incomingRxStatus);
        incomingTest.setGmtThresholdAmount(incomingGMTThreshold);
        incomingTest.setThresholdA(incomingMTThreshold);

        try {
            Person result = invokeFinancialRuleService(incomeYear, facility,
                    incoming, onFile);

            IncomeTest incomingIncomeTest = incoming.getIncomeTest(incomeYear);
            IncomeTest resultIncomeTest = result.getIncomeTest(incomeYear);

            assertNotNull("Incoming IncomeTest should not be null", incomingIncomeTest);
            assertNotNull("Result IncomeTest should not be null", resultIncomeTest);
            assertEquals(incomingIncomeTest.getCompletedDate(), resultIncomeTest.getCompletedDate());
            assertEquals(incomingIncomeTest.getEffectiveDate(), resultIncomeTest.getEffectiveDate());

        } catch (Exception e) {
            this.fail("Failed due to exception ", e);
        }
    }

    private void setRxCopayStatus(IncomeTest test, MeansTestStatus.Code status) throws Exception {
        if (status != null) {
            IncomeTestType rxType = getTestType(IncomeTestType.CODE_CO_PAY_EXEMPTION_TEST);
            MeansTestStatus rxStatus = getMeansTestStatus(status);
            setIncomeTestStatusProps(test, rxType, rxStatus);
        }
    }

	public void testProcessIncomeTestDuplicate() throws Exception {
		Integer incomeYear = new Integer(2003);
		VAFacility facility = getVaFacility(VAFacility.CODE_ALBUQUERQUE);

		Person p = buildSimplePerson();
		IncomeTestSource src = getTestSource(IncomeTestSource.CODE_VAMC);
		IncomeTestType typ = getTestType(IncomeTestType.CODE_MEANS_TEST);
		MeansTestStatus status = getMeansTestStatus(MeansTestStatus.MT_STATUS_MT_COPAY_EXEMPT);

		Date lastEditDate = new Date();
		IncomeTest tst = buildIncomeTestData(incomeYear, p, typ, src, status,
				facility);
		tst.setLastEditedDate(lastEditDate);

		Person person = savePerson(p);

		// Clone the pristine to have a working version
		Person incoming = (Person) person.clone();
		Person onFile = getPersonService().getPerson(
				person.getPersonEntityKey());

		buildIncomeTestData(incomeYear, incoming, typ, src, status, facility,
				new Date(), lastEditDate, new Date());

		try {
			Person result = invokeFinancialRuleService(incomeYear, facility,
					incoming, onFile);

			IncomeTest incomingIncomeTest = incoming.getIncomeTest(incomeYear);
			IncomeTest resultIncomeTest = result.getIncomeTest(incomeYear);
			assertNotNull("Incoming IncomeTest should not be null",
					incomingIncomeTest);
			assertNotNull("Result IncomeTest should not be null",
					resultIncomeTest);

			assertEquals(tst.getCompletedDate(), resultIncomeTest
					.getCompletedDate());
			assertEquals(tst.getEffectiveDate(), resultIncomeTest
					.getEffectiveDate());
		} catch (Exception e) {
			this.fail("Failed due to exception ", e);
		}
	}

	public void testProcessIncomeTestDiffSitesSourceVAMC() throws Exception {
		Integer incomeYear = new Integer(2003);
		VAFacility facility = getVaFacility(VAFacility.CODE_ALBUQUERQUE);
		// Use diff site for site conducting test so that fails the test
		VAFacility siteConductingTest = getVaFacility(VAFacility.CODE_CONNECTICUT_HCS);

		IncomeTestSource testSource = getTestSource(IncomeTestSource.CODE_VAMC);
		IncomeTestType testType = getTestType(IncomeTestType.CODE_MEANS_TEST);
		MeansTestStatus testStatus = getMeansTestStatus(MeansTestStatus.MT_STATUS_PENDING_ADJUDICATION);

		processInvalidIncomeTest(siteConductingTest, incomeYear, facility,
				testType, testSource, testStatus);
	}

	public void testProcessIncomeTestSameSitesSourceNonVAMC() throws Exception {
		Integer incomeYear = new Integer(2003);
		VAFacility facility = getVaFacility(VAFacility.CODE_ALBUQUERQUE);

		IncomeTestSource testSource = getTestSource(IncomeTestSource.CODE_OTHER);
		IncomeTestType testType = getTestType(IncomeTestType.CODE_MEANS_TEST);
		MeansTestStatus testStatus = getMeansTestStatus(MeansTestStatus.MT_STATUS_PENDING_ADJUDICATION);

		processInvalidIncomeTest(incomeYear, facility, testType, testSource,
				testStatus);
	}

	public void testProcessIncompleteIncomeTestDiffSitesSourceNonVAMC()
			throws Exception {
		Integer incomeYear = new Integer(2003);
		VAFacility facility = getVaFacility(VAFacility.CODE_ALBUQUERQUE);
		// Use diff site for site conducting test so that fails the test
		VAFacility siteConductingTest = getVaFacility(VAFacility.CODE_CONNECTICUT_HCS);

		IncomeTestSource testSource = getTestSource(IncomeTestSource.CODE_HEC);
		IncomeTestType testType = getTestType(IncomeTestType.CODE_MEANS_TEST);
		MeansTestStatus testStatus = getMeansTestStatus(MeansTestStatus.MT_STATUS_GMT_COPAY_REQUIRED);

		processInvalidIncomeTest(siteConductingTest, incomeYear, facility,
				testType, testSource, testStatus);
	}

	public void testProcessIncompleteIncomeTestNullCompletionDate()
			throws Exception {
		Integer incomeYear = new Integer(2003);
		VAFacility facility = getVaFacility(VAFacility.CODE_ALBUQUERQUE);

		IncomeTestSource testSource = getLookupService()
				.getIncomeTestSourceByCode(IncomeTestSource.CODE_VAMC.getName());
		IncomeTestType testType = getTestType(IncomeTestType.CODE_MEANS_TEST);
		MeansTestStatus testStatus = getMeansTestStatus(MeansTestStatus.MT_STATUS_MT_COPAY_REQUIRED);
		processInvalidIncomeTest(incomeYear, facility, testType, testSource,
				testStatus, null, new Date(), new Date());
	}

	public void testProcessIncompleteIncomeTestWrongMeansTest()
			throws Exception {
		Integer incomeYear = new Integer(2003);
		VAFacility facility = getVaFacility(VAFacility.CODE_ALBUQUERQUE);

		IncomeTestSource testSource = getLookupService()
				.getIncomeTestSourceByCode(IncomeTestSource.CODE_VAMC.getName());
		IncomeTestType testType = getTestType(IncomeTestType.CODE_MEANS_TEST);
		MeansTestStatus testStatus = getMeansTestStatus(MeansTestStatus.MT_STATUS_NP_LONGER_APPLICABLE_RX_COPAY_ONLY);
		processInvalidIncomeTest(incomeYear, facility, testType, testSource,
				testStatus);
	}

	public void testProcessIncompleteIncomeTestWrongPharmacyCopayTest()
			throws Exception {
		Integer incomeYear = new Integer(2003);
		VAFacility facility = getVaFacility(VAFacility.CODE_ALBUQUERQUE);

		IncomeTestSource testSource = getLookupService()
				.getIncomeTestSourceByCode(IncomeTestSource.CODE_VAMC.getName());
		IncomeTestType testType = getTestType(IncomeTestType.CODE_CO_PAY_EXEMPTION_TEST);
		MeansTestStatus testStatus = getMeansTestStatus(MeansTestStatus.MT_STATUS_GMT_COPAY_REQUIRED);
		processInvalidIncomeTest(incomeYear, facility, testType, testSource,
				testStatus);
	}

	public void testProcessIncomeTestDeletePrimary() throws Exception {
		Integer incomeYear = new Integer(2003);
		VAFacility facility = getVaFacility(VAFacility.CODE_ALBUQUERQUE);

		IncomeTestSource src = getTestSource(IncomeTestSource.CODE_VAMC);
		IncomeTestType typ = getTestType(IncomeTestType.CODE_MEANS_TEST);
		MeansTestStatus status = getMeansTestStatus(MeansTestStatus.MT_STATUS_MT_COPAY_EXEMPT);

		deleteIncomeTest(incomeYear, facility, typ, status, src);

	}

	public void testProcessIncomeTestDeletePrimaryInvalidConds()
			throws Exception {
		Integer incomeYear = new Integer(2003);
		VAFacility facility = getVaFacility(VAFacility.CODE_ALBUQUERQUE);

		IncomeTestSource src = getTestSource(IncomeTestSource.CODE_VAMC);
		IncomeTestType typ = getTestType(IncomeTestType.CODE_MEANS_TEST);
		MeansTestStatus status = getMeansTestStatus(MeansTestStatus.MT_STATUS_MT_COPAY_EXEMPT);

		VAFacility sendingFacility = getVaFacility(VAFacility.CODE_NEW_BEDFORD_CBOC);
		deleteIncomeTest(sendingFacility, incomeYear, facility, typ, status,
				src);

	}

	public void testProcessLTC4() throws Exception {
		Integer incomeYear = new Integer(2003);
		VAFacility facility = getVaFacility(VAFacility.CODE_ALBUQUERQUE);

		Person person = savePerson(buildSimplePerson());

		// Clone the pristine to have a working version
		Person incoming = (Person) person.clone();
		Person onFile = getPersonService().getPerson(
				person.getPersonEntityKey());

		IncomeTestSource testSource = getTestSource(IncomeTestSource.CODE_VAMC);
		IncomeTestType testType = getTestType(IncomeTestType.CODE_MEANS_TEST);
		MeansTestStatus testStatus = getMeansTestStatus(MeansTestStatus.MT_STATUS_MT_COPAY_EXEMPT);

		IncomeTest tst = buildIncomeTestData(incomeYear, incoming, testType,
				testSource, testStatus, facility);

		addTestStatus(tst, IncomeTestType.CODE_LTC_CO_PAY_TEST,
				MeansTestStatus.MT_STATUS_EXEMPT_LTC_COPAY_EXEMPTION_TEST);

		try {
			Person result = invokeFinancialRuleService(incomeYear, facility,
					incoming, onFile);

			IncomeTest incomingIncomeTest = incoming.getIncomeTest(incomeYear);
			IncomeTest resultIncomeTest = result.getIncomeTest(incomeYear);
			assertNotNull("Result IncomeTest should not be null",
					resultIncomeTest);
			assertEquals(incomingIncomeTest.getSource(), resultIncomeTest
					.getSource());
			assertEquals(incomingIncomeTest.getCompletedDate(),
					resultIncomeTest.getCompletedDate());
			assertEquals(incomingIncomeTest.getEffectiveDate(),
					resultIncomeTest.getEffectiveDate());
			assertEquals(incomingIncomeTest.getStatus(), resultIncomeTest
					.getStatus());
			assertEquals(testStatus, resultIncomeTest.getStatus());
			assertEquals(testType, resultIncomeTest.getType());

			assertEquals(testSource, resultIncomeTest.getSource());
			assertEquals(facility, resultIncomeTest.getSiteConductingTest());
			assertEquals(incomingIncomeTest.getSiteConductingTest(),
					resultIncomeTest.getSiteConductingTest());

			IncomeTestStatus ltcStatus = resultIncomeTest
					.getIncomeTestStatusOfType(IncomeTestType.CODE_LTC_CO_PAY_TEST);
			assertNotNull(ltcStatus);
			assertEquals(
					MeansTestStatus.MT_STATUS_EXEMPT_LTC_COPAY_EXEMPTION_TEST
							.getName(), ltcStatus.getStatus().getCode());
		} catch (Exception e) {
			this.fail("Failed due to exception ", e);
		}
	}

	public void testLTC4Delete() throws Exception {
		Integer incomeYear = new Integer(2003);
		VAFacility facility = getVaFacility(VAFacility.CODE_ALBUQUERQUE);

		Person person = buildSimplePerson();

		IncomeTestSource testSource = getTestSource(IncomeTestSource.CODE_VAMC);
		IncomeTestType testType = getTestType(IncomeTestType.CODE_MEANS_TEST);
        MeansTestStatus.Code mtStatusCode = MeansTestStatus.MT_STATUS_MT_COPAY_EXEMPT;
        MeansTestStatus mtStatus = getMeansTestStatus(mtStatusCode);

		Date lastEditDate = new Date();
		IncomeTest tst = buildIncomeTestData(incomeYear, person, testType,
				testSource, mtStatus, facility);
		tst.setLastEditedDate(lastEditDate);
		tst.setDiscloseFinancialInformation(Boolean.FALSE);
		addTestStatus(tst, IncomeTestType.CODE_LTC_CO_PAY_TEST,
				MeansTestStatus.MT_STATUS_EXEMPT_LTC_COPAY_EXEMPTION_TEST);

		person = savePerson(person);

		// Clone the pristine to have a working version
		Person incoming = (Person) person.clone();
		Person onFile = getPersonService().getPerson(
				person.getPersonEntityKey());

		IncomeTest incomingTest = buildIncomeTestData(incomeYear, incoming,
				testType, getTestSource(IncomeTestSource.CODE_HEC), mtStatus,
				facility);
		incomingTest.setDiscloseFinancialInformation(Boolean.TRUE);
		incomingTest.removeAllStatuses();
		addTestStatus(incomingTest, IncomeTestType.CODE_MEANS_TEST,
				mtStatusCode);

		incomingTest.setCompletedDate(tst.getCompletedDate());
		incomingTest.setLastEditedDate(lastEditDate);

		try {
			Person result = invokeFinancialRuleService(incomeYear, facility,
					incoming, onFile);

			IncomeTest incomingIncomeTest = incoming.getIncomeTest(incomeYear);
			IncomeTest resultIncomeTest = result.getIncomeTest(incomeYear);
			assertNotNull("Result IncomeTest should not be null",
					resultIncomeTest);
			assertEquals(incomingIncomeTest.getCompletedDate(),
					resultIncomeTest.getCompletedDate());

			assertEquals(incomingIncomeTest.getStatus(), resultIncomeTest
					.getStatus());
			assertEquals(mtStatus, resultIncomeTest.getStatus());
			assertEquals(testType, resultIncomeTest.getType());
			assertNotNull(resultIncomeTest
					.getIncomeTestStatusOfType(IncomeTestType.CODE_MEANS_TEST));
			assertNull(resultIncomeTest
					.getIncomeTestStatusOfType(IncomeTestType.CODE_LTC_CO_PAY_TEST));
		} catch (Exception e) {
			this.fail("Failed due to exception ", e);
		}
	}

	public void testDuplicateTestLTC4Delete() throws Exception {
		Integer incomeYear = new Integer(2003);
		VAFacility facility = getVaFacility(VAFacility.CODE_ALBUQUERQUE);

		Person person = buildSimplePerson();
		IncomeTestSource src = getTestSource(IncomeTestSource.CODE_VAMC);
		IncomeTestType typ = getTestType(IncomeTestType.CODE_MEANS_TEST);
		MeansTestStatus status = getMeansTestStatus(MeansTestStatus.MT_STATUS_MT_COPAY_EXEMPT);

		Date lastEditDate = new Date();
		IncomeTest tst = buildIncomeTestData(incomeYear, person, typ, src,
				status, facility);
		tst.setLastEditedDate(lastEditDate);
		addTestStatus(tst, IncomeTestType.CODE_LTC_CO_PAY_TEST,
				MeansTestStatus.MT_STATUS_EXEMPT_LTC_COPAY_EXEMPTION_TEST);

		person = savePerson(person);

		// Clone the pristine to have a working version
		Person incoming = (Person) person.clone();
		Person onFile = getPersonService().getPerson(
				person.getPersonEntityKey());

		IncomeTest incomingTest = incoming.getIncomeTest(incomeYear);
		// Remove the LTC4 status
		incomingTest.removeStatus(incomingTest
				.getIncomeTestStatus(IncomeTestType.CODE_LTC_CO_PAY_TEST));

		try {
			Person result = invokeFinancialRuleService(incomeYear, facility,
					incoming, onFile);

			IncomeTest resultIncomeTest = result.getIncomeTest(incomeYear);
			assertNotNull("Result IncomeTest should not be null",
					resultIncomeTest);

			assertNull(resultIncomeTest
					.getIncomeTestStatusOfType(IncomeTestType.CODE_LTC_CO_PAY_TEST));
			assertNotNull(resultIncomeTest
					.getIncomeTestStatusOfType(IncomeTestType.CODE_MEANS_TEST));
		} catch (Exception e) {
			this.fail("Failed due to exception ", e);
		}
	}

	private void addTestStatus(IncomeTest tst, IncomeTestType.Code typeCode,
			MeansTestStatus.Code mtStatus) throws Exception {
		IncomeTestType type = getLookupService().getIncomeTestTypeByCode(
				typeCode.getName());
		IncomeTestStatus incomeTestStatus = tst
				.getIncomeTestStatusOfType(typeCode);
		if (incomeTestStatus == null) {
			incomeTestStatus = new IncomeTestStatus();
			incomeTestStatus.setType(type);
			tst.addStatus(incomeTestStatus);
		}
		MeansTestStatus status = getLookupService().getMeansTestStatusByCode(
				mtStatus.getName());
		incomeTestStatus.setStatus(status);
		incomeTestStatus.setDeterminedStatus(status);
	}

	public void testHardshipDelete() throws Exception {
		Integer incomeYear = new Integer(2003);
		VAFacility facility = getVaFacility(VAFacility.CODE_ALBUQUERQUE);

		Person p = buildSimplePerson();
		IncomeTestSource src = getTestSource(IncomeTestSource.CODE_VAMC);
		IncomeTestType typ = getTestType(IncomeTestType.CODE_MEANS_TEST);
		MeansTestStatus status = getMeansTestStatus(MeansTestStatus.MT_STATUS_GMT_COPAY_REQUIRED);

		IncomeTest tst = buildIncomeTestData(incomeYear, p, typ, src, status,
				facility);

		Person person = savePerson(p);

		// Clone the pristine to have a working version
		Person incoming = (Person) person.clone();
		Person onFile = getPersonService().getPerson(
				person.getPersonEntityKey());

		IncomeTestSource testSource = getTestSource(IncomeTestSource.CODE_VAMC);
		IncomeTestType testType = getTestType(IncomeTestType.CODE_MEANS_TEST);
		MeansTestStatus testStatus = getMeansTestStatus(MeansTestStatus.MT_STATUS_MT_COPAY_EXEMPT);

		IncomeTest tst1 = buildIncomeTestData(incomeYear, incoming, testType,
				testSource, testStatus, facility);
		tst.setDiscloseFinancialInformation(Boolean.FALSE);
		tst1.setHardship(null); // Delete hardship

		try {
			Person result = invokeFinancialRuleService(incomeYear, facility,
					incoming, onFile);

			IncomeTest incomingIncomeTest = incoming.getIncomeTest(incomeYear);
			IncomeTest resultIncomeTest = result.getIncomeTest(incomeYear);
			assertNotNull("Incoming IncomeTest should not be null",
					incomingIncomeTest);
			assertNotNull("Result IncomeTest should not be null",
					resultIncomeTest);
			Hardship hs = resultIncomeTest.getHardship();
			assertNull("Hardship should have been deleted.", hs);

		} catch (Exception e) {
			this.fail("Failed due to exception ", e);
		}
	}

	public void testHardshipAddWithValidEffectiveDates() throws Exception {
		Integer incomeYear = new Integer(2003);
		VAFacility facility = getVaFacility(VAFacility.CODE_ALBUQUERQUE);

		Person person = savePerson(buildSimplePerson());

		// Clone the pristine to have a working version
		Person incoming = (Person) person.clone();
		Person onFile = getPersonService().getPerson(
				person.getPersonEntityKey());

		IncomeTestSource testSource = getTestSource(IncomeTestSource.CODE_VAMC);
		IncomeTestType testType = getTestType(IncomeTestType.CODE_MEANS_TEST);
		MeansTestStatus testStatus = getMeansTestStatus(MeansTestStatus.MT_STATUS_MT_COPAY_EXEMPT);

		IncomeTest tst = buildIncomeTestData(incomeYear, incoming, testType,
				testSource, testStatus, facility);
		tst.setDiscloseFinancialInformation(Boolean.FALSE);

		Calendar cal = Calendar.getInstance();
		cal.setTime(tst.getEffectiveDate());
		cal.add(Calendar.MINUTE, -1);

		tst.getHardship().setEffectiveDate(cal.getTime());

		try {
			Person result = invokeFinancialRuleService(incomeYear, facility,
					incoming, onFile);

			IncomeTest incomingIncomeTest = incoming.getIncomeTest(incomeYear);
			IncomeTest resultIncomeTest = result.getIncomeTest(incomeYear);
			assertNotNull("Incoming IncomeTest should not be null",
					incomingIncomeTest);
			assertNotNull("Result IncomeTest should not be null",
					resultIncomeTest);
			Hardship hs = resultIncomeTest.getHardship();
			assertNull(
					"Hardship should be null as Hardship effective date is greater than MT effective date.",
					hs);

		} catch (Exception e) {
			this.fail("Failed due to exception ", e);
		}
	}

	public void testHardshipAdd() throws Exception {
		Integer incomeYear = new Integer(2003);
		VAFacility facility = getVaFacility(VAFacility.CODE_ALBUQUERQUE);

		Person person = savePerson(buildSimplePerson());

		// Clone the pristine to have a working version
		Person incoming = (Person) person.clone();
		Person onFile = getPersonService().getPerson(
				person.getPersonEntityKey());

		IncomeTestSource testSource = getTestSource(IncomeTestSource.CODE_VAMC);
		IncomeTestType testType = getTestType(IncomeTestType.CODE_MEANS_TEST);
		MeansTestStatus testStatus = getMeansTestStatus(MeansTestStatus.MT_STATUS_MT_COPAY_EXEMPT);

		IncomeTest tst = buildIncomeTestData(incomeYear, incoming, testType,
				testSource, testStatus, facility);
		tst.getHardship().setSiteGrantingHardship(
				getVaFacility(VAFacility.CODE_IRON_MOUNTAIN_VA_MEDICAL_CENTER));

		try {
			Person result = invokeFinancialRuleService(incomeYear, facility,
					incoming, onFile);

			IncomeTest incomingIncomeTest = incoming.getIncomeTest(incomeYear);
			IncomeTest resultIncomeTest = result.getIncomeTest(incomeYear);
			assertNotNull("Incoming IncomeTest should not be null",
					incomingIncomeTest);
			assertNotNull("Result IncomeTest should not be null",
					resultIncomeTest);
			Hardship hs = resultIncomeTest.getHardship();
			assertNotNull(hs);
			assertEquals(hs.getEffectiveDate(), tst.getHardship()
					.getEffectiveDate());
			assertEquals(hs.getHardshipGranted(), tst.getHardship()
					.getHardshipGranted());
			assertEquals(hs.getReason(), tst.getHardship().getReason());
			assertEquals(hs.getReviewDate(), tst.getHardship().getReviewDate());
			assertEquals(hs.getSiteGrantingHardship(), tst.getHardship()
					.getSiteGrantingHardship());

		} catch (Exception e) {
			this.fail("Failed due to exception ", e);
		}
	}

	public void testHardshipEditSameSite() throws Exception {
		Integer incomeYear = new Integer(2003);
		VAFacility facility = getVaFacility(VAFacility.CODE_ALBUQUERQUE);

		Person p = buildSimplePerson();
		IncomeTestSource src = getTestSource(IncomeTestSource.CODE_VAMC);
		IncomeTestType typ = getTestType(IncomeTestType.CODE_MEANS_TEST);
		MeansTestStatus status = getMeansTestStatus(MeansTestStatus.MT_STATUS_GMT_COPAY_REQUIRED);

		buildIncomeTestData(incomeYear, p, typ, src, status, facility);
		Person person = savePerson(p);

		// Clone the pristine to have a working version
		Person incoming = (Person) person.clone();
		Person onFile = getPersonService().getPerson(
				person.getPersonEntityKey());

		IncomeTestSource testSource = getTestSource(IncomeTestSource.CODE_VAMC);
		IncomeTestType testType = getTestType(IncomeTestType.CODE_MEANS_TEST);
		MeansTestStatus testStatus = getMeansTestStatus(MeansTestStatus.MT_STATUS_MT_COPAY_EXEMPT);

		IncomeTest tst1 = buildIncomeTestData(incomeYear, incoming, testType,
				testSource, testStatus, facility);
		tst1.setDiscloseFinancialInformation(Boolean.FALSE);
		Hardship newHs = tst1.getHardship();
		newHs.setReason("No new reason");
		newHs.setReviewDate(new Date());

		try {
			Person result = invokeFinancialRuleService(incomeYear, facility,
					incoming, onFile);

			IncomeTest incomingIncomeTest = incoming.getIncomeTest(incomeYear);
			IncomeTest resultIncomeTest = result.getIncomeTest(incomeYear);
			assertNotNull("Incoming IncomeTest should not be null",
					incomingIncomeTest);
			assertNotNull("Result IncomeTest should not be null",
					resultIncomeTest);
			Hardship hs = resultIncomeTest.getHardship();
			assertNotNull(hs);
			assertEquals(hs.getEffectiveDate(), newHs.getEffectiveDate());
			assertEquals(hs.getHardshipGranted(), newHs.getHardshipGranted());
			assertEquals(hs.getReason(), newHs.getReason());
			assertEquals(hs.getReviewDate(), newHs.getReviewDate());
			assertEquals(hs.getSiteGrantingHardship(), newHs
					.getSiteGrantingHardship());

		} catch (Exception e) {
			this.fail("Failed due to exception ", e);
		}
	}

	public void testHardshipEditFromDiffSites() throws Exception {
		Integer incomeYear = new Integer(2003);
		VAFacility facility = getVaFacility(VAFacility.CODE_ALBUQUERQUE);

		Person p = buildSimplePerson();
		IncomeTestSource src = getTestSource(IncomeTestSource.CODE_VAMC);
		IncomeTestType typ = getTestType(IncomeTestType.CODE_MEANS_TEST);
		MeansTestStatus status = getMeansTestStatus(MeansTestStatus.MT_STATUS_GMT_COPAY_REQUIRED);

		IncomeTest tst = buildIncomeTestData(incomeYear, p, typ, src, status,
				facility);
		tst.getHardship().setSiteGrantingHardship(
				getVaFacility(VAFacility.CODE_IRON_MOUNTAIN_VA_MEDICAL_CENTER));

		Person person = savePerson(p);

		// Clone the pristine to have a working version
		Person incoming = (Person) person.clone();
		Person onFile = getPersonService().getPerson(
				person.getPersonEntityKey());

		IncomeTestSource testSource = getTestSource(IncomeTestSource.CODE_VAMC);
		IncomeTestType testType = getTestType(IncomeTestType.CODE_MEANS_TEST);
		MeansTestStatus testStatus = getMeansTestStatus(MeansTestStatus.MT_STATUS_MT_COPAY_EXEMPT);

		IncomeTest tst1 = buildIncomeTestData(incomeYear, incoming, testType,
				testSource, testStatus, facility);
		tst1.setDiscloseFinancialInformation(Boolean.FALSE);
		Hardship newHs = tst1.getHardship();
		newHs.setReason("No new reason");
		newHs.setReviewDate(new Date());

		try {
			Person result = invokeFinancialRuleService(incomeYear, facility,
					incoming, onFile);

			IncomeTest incomingIncomeTest = incoming.getIncomeTest(incomeYear);
			IncomeTest resultIncomeTest = result.getIncomeTest(incomeYear);
			assertNotNull("Incoming IncomeTest should not be null",
					incomingIncomeTest);
			assertNotNull("Result IncomeTest should not be null",
					resultIncomeTest);
			Hardship hs = resultIncomeTest.getHardship();
			assertNotNull(hs);
			assertEquals(hs.getEffectiveDate(), tst.getHardship()
					.getEffectiveDate());
			assertEquals(hs.getHardshipGranted(), tst.getHardship()
					.getHardshipGranted());
			assertEquals(hs.getReason(), tst.getHardship().getReason());
			assertEquals(hs.getReviewDate(), tst.getHardship().getReviewDate());
			assertEquals(hs.getSiteGrantingHardship(), tst.getHardship()
					.getSiteGrantingHardship());

		} catch (Exception e) {
			this.fail("Failed due to exception ", e);
		}
	}

	private void deleteIncomeTest(Integer incomeYear, VAFacility facility,
			IncomeTestType testType, MeansTestStatus mtStatus,
			IncomeTestSource testSource) throws Exception {
		deleteIncomeTest(facility, incomeYear, facility, testType, mtStatus,
				testSource);
	}

	private void deleteIncomeTest(VAFacility sendingFacility,
			Integer incomeYear, VAFacility facility, IncomeTestType testType,
			MeansTestStatus mtStatus, IncomeTestSource testSource)
			throws Exception {
		Person p = buildSimplePerson();
		buildIncomeTestData(incomeYear, p, testType, testSource, mtStatus,
				facility);
		Person person = savePerson(p);

		// Clone the pristine to have a working version
		Person incoming = (Person) person.clone();
		Person onFile = getPersonService().getPerson(
				person.getPersonEntityKey());

		// remove the income test in the incoming person.
		incoming.setIncomeTest(incomeYear, null);

		try {
			Person result = invokeFinancialRuleService(incomeYear,
					sendingFacility, incoming, onFile);

			IncomeTest resultIncomeTest = result.getIncomeTest(incomeYear);
			if (sendingFacility.getCode().equals(facility.getCode())) {
				// both are same, so it test should be deleted
				assertNull(
						"Result IncomeTest should be null (should have been deleted)",
						resultIncomeTest);
			} else {
				assertNotNull("Result IncomeTest should not be null ",
						resultIncomeTest);
			}
		} catch (Exception e) {
			this.fail("Failed due to exception ", e);
		}
	}

	private void processInvalidIncomeTest(Integer incomeYear,
			VAFacility facility, IncomeTestType testType,
			IncomeTestSource testSource, MeansTestStatus testStatus)
			throws Exception {
		processInvalidIncomeTest(facility, incomeYear, facility, testType,
				testSource, testStatus, new Date(), new Date(), new Date());
	}

	private void processInvalidIncomeTest(VAFacility siteConductingTest,
			Integer incomeYear, VAFacility facility, IncomeTestType testType,
			IncomeTestSource testSource, MeansTestStatus testStatus)
			throws Exception {
		processInvalidIncomeTest(siteConductingTest, incomeYear, facility,
				testType, testSource, testStatus, new Date(), new Date(),
				new Date());
	}

	private void processInvalidIncomeTest(Integer incomeYear,
			VAFacility facility, IncomeTestType testType,
			IncomeTestSource testSource, MeansTestStatus testStatus,
			Date completedDate, Date lastEditTime, Date effectiveDate)
			throws Exception {
		processInvalidIncomeTest(facility, incomeYear, facility, testType,
				testSource, testStatus, completedDate, lastEditTime,
				effectiveDate);
	}

	private void processInvalidIncomeTest(VAFacility siteConductingTest,
			Integer incomeYear, VAFacility facility, IncomeTestType testType,
			IncomeTestSource testSource, MeansTestStatus testStatus,
			Date completedDate, Date lastEditTime, Date effectiveDate)
			throws Exception {

		Person person = savePerson(buildSimplePerson());

		// Clone the pristine to have a working version
		Person incoming = (Person) person.clone();
		Person onFile = getPersonService().getPerson(
				person.getPersonEntityKey());

		IncomeTest test = buildIncomeTestData(incomeYear, incoming, testType,
				testSource, testStatus, facility);
		test.setCompletedDate(completedDate);
		test.setEffectiveDate(effectiveDate);
		test.setLastEditedDate(lastEditTime);
		test.setSiteConductingTest(siteConductingTest);

		try {
			Person result = invokeFinancialRuleService(incomeYear, facility,
					incoming, onFile);

			IncomeTest incomingIncomeTest = incoming.getIncomeTest(incomeYear);
			IncomeTest resultIncomeTest = result.getIncomeTest(incomeYear);
			assertNotNull("Incoming IncomeTest should not be null",
					incomingIncomeTest);
			assertNull("Result IncomeTest should be null as processing should fail for this test",
					resultIncomeTest);
		} catch (Exception e) {
			this.fail("Failed due to exception ", e);
		}
	}

	private void processValidIncomeTest(VAFacility siteConductingTest,
			Integer incomeYear, VAFacility facility, IncomeTestType testType,
			IncomeTestSource testSource, MeansTestStatus testStatus,
			Date completedDate, Date lastEditTime, Date effectiveDate)
			throws Exception {

		Person person = savePerson(buildSimplePerson());

		// Clone the pristine to have a working version
		Person incoming = (Person) person.clone();
		Person onFile = getPersonService().getPerson(
				person.getPersonEntityKey());

		IncomeTest test = buildIncomeTestData(incomeYear, incoming, testType,
				testSource, testStatus, facility);
		test.setCompletedDate(completedDate);
		test.setEffectiveDate(effectiveDate);
		test.setLastEditedDate(lastEditTime);
		test.setSiteConductingTest(siteConductingTest);

		try {
			Person result = invokeFinancialRuleService(incomeYear, facility,
					incoming, onFile);

			IncomeTest incomingIncomeTest = incoming.getIncomeTest(incomeYear);
			IncomeTest resultIncomeTest = result.getIncomeTest(incomeYear);
			assertNotNull("Incoming IncomeTest should not be null",
					incomingIncomeTest);
			assertNull("Result IncomeTest should be null", resultIncomeTest);
		} catch (Exception e) {
			this.fail("Failed due to exception ", e);
		}
	}


    private FinancialInputParameter buildFinancialInputParameter(FinancialInputData data) throws ServiceException {
        FinancialInputParameter fip = (FinancialInputParameter) applicationContext.getBean("financialRuleParameter");
        fip.setRuleDataAware(data);
        fip.setLookupService(getLookupService());
        fip.setHelperService(getPersonHelperService());
        fip.setFinancialsHelperService(new FinancialsHelperServiceImpl());
        return fip;
    }
    
	protected Person invokeFinancialRuleService(Integer incomeYear,
			VAFacility facility, Person incoming, Person onFile)
			throws Exception {
		 EEResultInfo info = new EEResultInfo();
		return getFinancialInfoRuleService().processFinancialInfo(incomeYear,
				facility, incoming, onFile, info, true);
	}
	
	protected Date getDateOnly(Date date) {
	    Date modifiedDate = date;
	     if (date != null) {
	            Calendar calendar = DateUtils.createCalendar(date);
	            int year = calendar.get(Calendar.YEAR);
	            int month = calendar.get(Calendar.MONTH);
	            int day = calendar.get(Calendar.DATE);
	            int hour = calendar.get(Calendar.HOUR);	            
	            int minute = calendar.get(Calendar.MINUTE);
	            int second = calendar.get(Calendar.SECOND);	            
	            calendar.clear();
	            calendar.set(year,month,day);            
	            modifiedDate = calendar.getTime();        
	        }
	     return modifiedDate;
	}
}