/********************************************************************
 * Copyright  2004 EDS. All rights reserved
 ********************************************************************/

/*
 * Created on Jan 18, 2005
 *
 */
package gov.va.med.esr.common.rule.service;

// Java classes
import java.util.Set;

// Library classes

// Framework classes

// EDB Classes
import gov.va.med.esr.common.model.person.Person;
import gov.va.med.esr.common.model.ee.VerificationInfo;
import gov.va.med.esr.common.model.lookup.EligibilityType;
import gov.va.med.esr.common.model.lookup.AlliedCountry;
import gov.va.med.esr.common.util.RuleAbstractTestCase;;

/**
 * @author Carlos Ruiz
 * @version 1.0
 */
public class ProcessEligibilityCodeServiceTest extends RuleAbstractTestCase {
	
	
	/**
	 * Constructor for ProcessEligibilityCodeServiceTest.
	 * @param arg0
	 */
	public ProcessEligibilityCodeServiceTest(String arg0) throws Exception {
		super(arg0);		
	}
    
    
    protected void customSetUp() throws Exception {
		super.customSetUp();
    }    
	
	
	/**
	 * EligiblityCodeIsDeleted
	 * If  a messages list of eligibility codes has less items than a veterans list of eligibility codes 
	 * then remove all of the eligibility code that doesnt match those coming from a message
	 */
	public void testEligiblityCodeIsDeleted() throws Exception {

        Person onFile = buildSimplePerson();
        onFile.addReceivedSecondaryEligibility(this.createReceivedEligibility(EligibilityType.ALLIED_VETERAN));
        onFile.addReceivedSecondaryEligibility(this.createReceivedEligibility(EligibilityType.SHARING_AGREEMENT));        
        Person incoming = this.savePerson(onFile);
        incoming.removeAllReceivedSecondaryEligibilities();
        incoming.addReceivedSecondaryEligibility(this.createReceivedEligibility(EligibilityType.EMPLOYEE));
        this.getEnrollmentRuleService().manageReceivedEligibilities( incoming, onFile );
		
		Set results = onFile.getReceivedSecondaryEligibilities();
		
		assertNotNull( results );
		assertTrue ( results.size() == 1 );
	}
    
    public void testEligiblityCodeIsDeleteAll() throws Exception {

        Person onFile = buildSimplePerson();
        onFile.addReceivedSecondaryEligibility(this.createReceivedEligibility(EligibilityType.ALLIED_VETERAN));
        onFile.addReceivedSecondaryEligibility(this.createReceivedEligibility(EligibilityType.SHARING_AGREEMENT));        
        Person incoming = this.savePerson(onFile);
        incoming.removeAllReceivedSecondaryEligibilities();
        this.getEnrollmentRuleService().manageReceivedEligibilities( incoming, onFile );
        
        Set results = onFile.getReceivedSecondaryEligibilities();
        
        assertNotNull( results );
        assertTrue ( results.size() == 0 );
    }    
    
    public void testEligiblityCodeAdd() throws Exception {

        Person onFile = buildSimplePerson();
        Person incoming = this.savePerson(onFile);
        // add an invalid code (without country)
        incoming.addReceivedSecondaryEligibility(this.createReceivedEligibility(EligibilityType.ALLIED_VETERAN));
        // add a valid code
        incoming.addReceivedSecondaryEligibility(this.createReceivedEligibility(EligibilityType.SHARING_AGREEMENT));        

        this.getEnrollmentRuleService().manageReceivedEligibilities( incoming, onFile );
        
        Set results = onFile.getReceivedSecondaryEligibilities();
        
        assertNotNull( results );
        assertTrue ( results.size() == 1 ); // only one valid code stored 
    }    
	

	/**
	 * EligibilityCodeIsValid
	 *  
	 * If a messages eligibility code is one of the following: Mexican War Border, Collateral of vet,
	 * Employee, Reimbursable Insurance, Sharing Agreement, Tricare/Champus, Other federal Agency, Champva
	 * and a veterans eligiblity code is not one of those values coming from a message
	 * then update a veterans record with data coming from a message 
	 */
	public void testEligibilityCodeIsValid() throws Exception {
	   
		Person onFile = buildSimplePerson();
        Person incoming = this.savePerson(onFile);
		this.attachEmployee(incoming);
	
		this.getEnrollmentRuleService().manageReceivedEligibilities( incoming, onFile );
		
		Set results = onFile.getReceivedSecondaryEligibilities();
		
		assertNotNull( results );
		assertTrue ( results.size() == 1 );		
		
	}
    
    public void testPrimarySecondaryEligibilityCodeIsValid() throws Exception {
           
        Person onFile = buildSimplePerson();
        Person incoming = this.savePerson(onFile);
        this.attachEmployeeAsPrimary(incoming);
        this.attachOtherFederalAgency(incoming);
   
        this.getEnrollmentRuleService().manageReceivedEligibilities( incoming, onFile );
        
        assertNotNull(onFile.getReceivedPrimaryEligibility());
        assertTrue(onFile.getReceivedSecondaryEligibilities().size() == 1);
    }
    
    public void testPrimaryEligibilityCodeIsValid() throws Exception {
        
        Person onFile = buildSimplePerson();
        this.attachAlliedVeteranAsPrimary(onFile);
        Person incoming = this.savePerson(onFile);
        assertTrue(incoming.getReceivedPrimaryEligibility().getAlliedCountry().getCode().equals(AlliedCountry.CODE_CANADA.getCode()));
        incoming.getReceivedPrimaryEligibility().setAlliedCountry(this.getLookupService().getAlliedCountryByCode(AlliedCountry.CODE_AUSTRALIA.getCode()));
        this.getEnrollmentRuleService().processReceivedEligibilities( incoming, onFile, null,false );
        
        assertNotNull(onFile.getReceivedPrimaryEligibility());
        assertTrue(onFile.getReceivedSecondaryEligibilities().size() == 0);
        assertTrue(onFile.getReceivedPrimaryEligibility().getAlliedCountry().getCode().equals(AlliedCountry.CODE_AUSTRALIA.getCode()));
    }
    
    public void testPrimaryEligibilityCode() throws Exception {
        
        Person onFile = buildSimplePerson();
        Person incoming = this.savePerson(onFile);
        this.attachEmployee(incoming);
        this.getEnrollmentRuleService().manageReceivedEligibilities( incoming, onFile );
        
        assertNull(onFile.getReceivedPrimaryEligibility());
        assertTrue(onFile.getReceivedSecondaryEligibilities().size() == 1);
    }     
    
    public void testVoaHasAttachments() throws Exception {
        
        Person onFile = this.getPerson("4");
        Person incoming = (Person) onFile.clone();
        incoming.getReceivedPrimaryEligibility().setVoaHasAttachment(true);
        
        VerificationInfo verificationInfo = new VerificationInfo();
        verificationInfo.setMessageControlId("12345");

        this.getEnrollmentRuleService().processReceivedEligibilities(incoming, onFile, verificationInfo,false);
        
        this.setComplete();

    }     

}
