/********************************************************************
 * Copyright  2004 VHA. All rights reserved
 ********************************************************************/
package gov.va.med.esr.common.rule.service;

import gov.va.med.esr.common.model.financials.BeneficiaryTravel;
import gov.va.med.esr.common.model.lookup.VAFacility;
import gov.va.med.esr.common.model.person.Person;
import gov.va.med.esr.common.util.RuleAbstractTestCase;

import java.math.BigDecimal;
import java.util.Date;

/**
 * @author DNS   MANSOG
 * @version 1.0
 */
public class ProcessBeneficiaryTravelTest extends RuleAbstractTestCase {
	public ProcessBeneficiaryTravelTest(String testName) throws Exception {
		super(testName);
		// super.setDefaultRollback(false);
	}
    
	public void testAddTravelData() throws Exception {
		Integer incomeYear = new Integer(2003);

		VAFacility facility = getVaFacility(VAFacility.CODE_ALBUQUERQUE);
		Person person = savePerson(buildSimplePerson());

		// Clone the pristine to have a working version
		Person incoming = (Person) person.clone();
		Person onFile = getPersonService().getPerson(
				person.getPersonEntityKey());
		buildBeneficiaryTravelData(incomeYear, facility, incoming);
		try {
			Person result = invokeFinancialRuleService(incomeYear, facility,
					incoming, onFile);

			BeneficiaryTravel inBt = incoming.getBeneficiaryTravel(incomeYear,
					facility);
			BeneficiaryTravel rsBt = result.getBeneficiaryTravel(incomeYear,
					facility);
			assertNotNull("Result BeneficiaryTravel should not be null", rsBt);
			assertEquals(inBt.getAmountCertified(), rsBt.getAmountCertified());
			assertEquals(inBt.getClaimDate(), rsBt.getClaimDate());
			assertEquals(inBt.getDateCertified(), rsBt.getDateCertified());
			assertEquals(inBt.getEligible(), rsBt.getEligible());
			assertEquals(inBt.getFacilityVisited(), rsBt.getFacilityVisited());
			assertEquals(inBt.getYear(), rsBt.getYear());

		} catch (Exception e) {
			this.fail("Failed due to exception ", e);
		}
	}

	public void testDeleteExistingTravelData() throws Exception {
		Integer incomeYear = new Integer(2003);

		VAFacility facility = getVaFacility(VAFacility.CODE_ALBUQUERQUE);
		Person p = buildSimplePerson();
		buildBeneficiaryTravelData(incomeYear, facility, p);
		Person person = savePerson(p);

		// Clone the pristine to have a working version
		Person incoming = (Person) person.clone();
		Person onFile = getPersonService().getPerson(
				person.getPersonEntityKey());
		BeneficiaryTravel bt = incoming.getBeneficiaryTravel(incomeYear,
				facility);
		assertNotNull("BeneficiaryTravel should not be null", bt);
		assertEquals(facility, bt.getFacilityVisited());
		assertEquals(incomeYear, bt.getYear());

		incoming.setBeneficiaryTravel(incomeYear, facility, null);
		try {
			Person result = invokeFinancialRuleService(incomeYear, facility,
					incoming, onFile);

			BeneficiaryTravel rsBt = result.getBeneficiaryTravel(incomeYear,
					facility);
			assertNull("Result BeneficiaryTravel should be null", rsBt);

		} catch (Exception e) {
			this.fail("Failed due to exception ", e);
		}
	}

	public void testUpdateExistingTravelData() throws Exception {
		Integer incomeYear = new Integer(2003);

		VAFacility facility = getVaFacility(VAFacility.CODE_ALBUQUERQUE);
		Person p = buildSimplePerson();
		buildBeneficiaryTravelData(incomeYear, facility, p);
		Person person = savePerson(p);

		// Clone the pristine to have a working version
		Person incoming = (Person) person.clone();
		Person onFile = getPersonService().getPerson(
				person.getPersonEntityKey());
		BeneficiaryTravel bt = incoming.getBeneficiaryTravel(incomeYear,
				facility);
		assertNotNull("BeneficiaryTravel should not be null", bt);
		assertEquals(facility, bt.getFacilityVisited());
		assertEquals(incomeYear, bt.getYear());

		Date claimDate = new Date();
		BigDecimal amountCert = new BigDecimal(777);
		bt.setAmountCertified(amountCert);
		bt.setClaimDate(claimDate);
		bt.setEligible(Boolean.TRUE);
		try {
			Person result = invokeFinancialRuleService(incomeYear, facility,
					incoming, onFile);

			BeneficiaryTravel inBt = incoming.getBeneficiaryTravel(incomeYear,
					facility);
			BeneficiaryTravel rsBt = result.getBeneficiaryTravel(incomeYear,
					facility);
			assertNotNull("Result BeneficiaryTravel should not be null", rsBt);

			assertEquals(amountCert, rsBt.getAmountCertified());
			assertEquals(claimDate, rsBt.getClaimDate());
			assertEquals(Boolean.TRUE, rsBt.getEligible());

			assertEquals(inBt.getAmountCertified(), rsBt.getAmountCertified());
			assertEquals(inBt.getClaimDate(), rsBt.getClaimDate());
			assertEquals(inBt.getDateCertified(), rsBt.getDateCertified());
			assertEquals(inBt.getEligible(), rsBt.getEligible());
			assertEquals(inBt.getFacilityVisited(), rsBt.getFacilityVisited());
			assertEquals(inBt.getYear(), rsBt.getYear());

		} catch (Exception e) {
			this.fail("Failed due to exception ", e);
		}
	}

    protected BeneficiaryTravel buildBeneficiaryTravelData(Integer incomeYear,
            VAFacility facility, Person person) throws Exception {
        BeneficiaryTravel travel = person.getBeneficiaryTravel(incomeYear, facility);
        if (travel == null) {
            travel = createBeneficiaryTravel(incomeYear, facility);
            person.setBeneficiaryTravel(incomeYear, facility, travel);
        }
        travel.setAmountCertified(new BigDecimal(500.50));
        return travel;
    }
    
	protected Person invokeFinancialRuleService(Integer incomeYear,
			VAFacility facility, Person incoming, Person onFile)
			throws Exception {
		return getFinancialInfoRuleService().processFinancialInfo(incomeYear,
				facility, incoming, onFile);
	}
}