/*
 * Created on Jan 4, 2005
 *
 */
package gov.va.med.esr.common.rule.service;

// Java classes
import java.math.BigDecimal;
import java.util.Date;

import gov.va.med.esr.common.model.ee.AgentOrangeExposure;
import gov.va.med.esr.common.model.ee.ConflictExperience;
import gov.va.med.esr.common.model.ee.EnrollmentDetermination;
import gov.va.med.esr.common.model.ee.Eligibility;
import gov.va.med.esr.common.model.ee.EnrollmentOverride;
import gov.va.med.esr.common.model.ee.MilitaryService;
import gov.va.med.esr.common.model.ee.MilitaryServiceSiteRecord;
import gov.va.med.esr.common.model.financials.IncomeTest;
import gov.va.med.esr.common.model.financials.FinancialStatement;
import gov.va.med.esr.common.model.lookup.AssetType;
import gov.va.med.esr.common.model.lookup.AddressType;
import gov.va.med.esr.common.model.lookup.AgentOrangeExposureLocation;
import gov.va.med.esr.common.model.lookup.ConflictLocation;
import gov.va.med.esr.common.model.lookup.EligibilityStatus;
import gov.va.med.esr.common.model.lookup.EligibilityFactor;
import gov.va.med.esr.common.model.lookup.VAFacility;
import gov.va.med.esr.common.model.lookup.EligibilityType;
import gov.va.med.esr.common.model.lookup.EnrollmentCategory;
import gov.va.med.esr.common.model.lookup.EnrollmentPriorityGroup;
import gov.va.med.esr.common.model.lookup.EnrollmentPrioritySubGroup;
import gov.va.med.esr.common.model.lookup.EnrollmentSource;
import gov.va.med.esr.common.model.lookup.SSNType;
import gov.va.med.esr.common.model.lookup.EnrollmentStatus;
import gov.va.med.esr.common.model.lookup.IncomeTestType;
import gov.va.med.esr.common.model.lookup.SSAVerificationStatus;
import gov.va.med.esr.common.model.lookup.PseudoSSNReason;
import gov.va.med.esr.common.model.lookup.MeansTestStatus;
import gov.va.med.esr.common.model.financials.Hardship;
import gov.va.med.esr.common.model.financials.IncomeTestStatus;
import gov.va.med.esr.common.model.financials.SpouseFinancials;
import gov.va.med.esr.common.model.financials.DependentFinancials;
import gov.va.med.esr.common.model.person.Person;
import gov.va.med.esr.common.model.person.SSN;
import gov.va.med.esr.common.model.person.id.PersonIdEntityKeyImpl;
import gov.va.med.esr.common.rule.parameter.DemographicInputParameter;
import gov.va.med.esr.common.util.RuleAbstractTestCase;
import gov.va.med.fw.rule.RuleParameters;
import gov.va.med.fw.service.ServiceException;
import gov.va.med.fw.validation.ValidationMessages;

/**
 * @author Carlos Ruiz
 * 
 * 
 */
public class DetermineEnrollmentServiceTest extends RuleAbstractTestCase {

	protected static final String INVALID_APPLICATION_DATE_PRIOR_10011996="INVALID_APPLICATION_DATE_PRIOR_10011996";
	
    public DetermineEnrollmentServiceTest(String testName) throws Exception {
        super(testName);
    }

    public void testCCR8804_ContEnrollFailedCombatVet() throws Exception {
        
        String personId = "275792682";
        Person onFile = this.getPerson(personId);
        
        Person incoming = (Person)onFile.clone();

        Person updatedPerson = this.calculate(incoming);
        
        assertEquals(getEnrollmentDetermination(incoming).getEnrollmentStatus().getCode(), 
                getEnrollmentDetermination(updatedPerson).getEnrollmentStatus().getCode());

    }
    public void testEnrollmentApplicationDatePriorTo1001996() throws Exception {
        Person onFile = buildSimplePerson(); // bare-bones person
        this.setEligibilityStatusToVerified(onFile);
        this.attachVerifiedEnrollment(onFile);

        Person incoming = savePerson(onFile);
        this.attachApplication(incoming);
        // months start at 0
        incoming.getApplication().setApplicationDate(this.getDate(1996, 8, 30));
        ValidationMessages vm = this.getRuleValidationService().validateReceivedEnrollment(incoming, onFile, onFile, true);
        assertTrue(vm.getCount() == 1);
        assertTrue(vm.contains(INVALID_APPLICATION_DATE_PRIOR_10011996));
        
        incoming.getApplication().setApplicationDate(this.getDate(1996, 9, 1));
        vm = this.getRuleValidationService().validateReceivedEnrollment(incoming, onFile, onFile, true);
        assertTrue(vm.getCount() == 0);
    }    
    
    /**
     * EnrollmentStatusIsVerified- If Eligibility status is equal to verified
     * then set the enrollment status to verified and set the enrollment
     * effective date to the system date and set the source of enrollment to HEC
     * and set the enrollment priority equal to calculated enrollment priority
     * and set the enrollment category equal to enrolled Else set enrollment
     * status equal to pending Other and set the enrollment effective date
     * equal to the system date and set the source of enrollment to HEC and set
     * the enrollment priority equal to NULL and set the enrollment category
     * equal to in process
     * 
     * 
     */
    public void testEnrollmentStatusIsVerified() throws Exception {
        Person onFile = buildSimplePerson(); // bare-bones person
        this.setEligibilityStatusToVerified(onFile);
        this.attachVerifiedEnrollment(onFile);
        this.attachMeansTestWithMtCopayExempt(onFile);
        onFile.setFinancialStatement(new Integer(2005), this.createFinancialStatement());

        Person incoming = savePerson(onFile);
        Person updatedPerson = this.calculate(incoming);

        assessEnrollmentStatusResult(updatedPerson,
                EnrollmentStatus.CODE_VERIFIED);
    }
    
    public void testEnrollmentStatusForNonVeteranWithoutNONVetPEC() throws Exception {
        this.testEnrollmentStatusForNonVeteran(this.buildSimpleNonVeteran());
    }
    
    public void testEnrollmentStatusForNonVeteranWithNONVetPEC() throws Exception {
        Person onFile = this.buildSimpleNonVeteran(); // bare-bones person
        this.attachAlliedVeteranAsPrimary(onFile);
        this.testEnrollmentStatusForNonVeteran(onFile);
    }
    
    private void testEnrollmentStatusForNonVeteran(Person onFile) throws Exception {
        this.setEligibilityStatusToVerified(onFile);
        this.attachVerifiedEnrollment(onFile);

        Person incoming = savePerson(onFile);
        Person updatedPerson = this.calculate(incoming);

        assessEnrollmentStatusResult(updatedPerson, EnrollmentStatus.CODE_NOT_APPLICABLE);
    }        
    
    public void testEnrollmentStatusIsVerifiedWithNoFinancials() throws Exception {
        Person onFile = buildSimplePerson(); // bare-bones person
        this.setEligibilityStatusToVerified(onFile);
        this.attachVerifiedEnrollment(onFile);
        this.attachMeansTestWithMtCopayExempt(onFile);
   //     onFile.setFinancialStatement(new Integer(2005), this.createFinancialStatement());

        Person incoming = savePerson(onFile);
        Person updatedPerson = this.calculate(incoming);

        assessEnrollmentStatusResult(updatedPerson,
                EnrollmentStatus.CODE_VERIFIED);
    }    

    public void testPharmacy() throws Exception {
        Person onFile = buildSimplePerson(); // bare-bones person
        onFile.setUserEnrolleeValidThrough(new Integer(1999));
        this.attachApplication(onFile);
        onFile.getApplication().setApplicationDate(this.getDate(2000, 1, 1));

        this.attachVerifiedEnrollment(onFile);
        this.attachPharmacyCopayTestWithNoLongerApplicable(onFile);
        this.setEligibilityStatus(EligibilityStatus.CODE_VERIFIED, onFile);

        Person incoming = savePerson(onFile);

        this.setServiceConnectedPercentage(new Integer(75), incoming);
        Person updatedPerson = this.calculate(incoming);

        assessEnrollmentStatusResult(updatedPerson,
                EnrollmentStatus.CODE_VERIFIED);
    }

    /**
     * EnrollmentStatus = Pending Other
     * 
     * 
     */
    public void testEnrollmentStatusIsPendingOther() throws Exception {
        Person onFile = buildSimplePerson(); // bare-bones person
        this.attachAidAndAttendance(onFile);
        this.attachVerifiedEnrollment(onFile); // make existing person
        Person incoming = savePerson(onFile);
        this.getPersonHelperService().removeAABenefit(incoming);
        Person updatedPerson = this.calculate(incoming);

        assessEnrollmentStatusResult(updatedPerson,
                EnrollmentStatus.CODE_PENDING_OTHER);
    }

    public void testEnrollmentStatusRejectedBelowEGT() throws Exception {
        Person onFile = buildSimplePerson(); // bare-bones person
        this.setEligibilityStatusToVerified(onFile);
        this.attachVerifiedEnrollment(onFile);
        attachMeansTestWithMtCopayRequired(onFile, new BigDecimal(999),
                new BigDecimal(1000));
        onFile.setAppointmentRequestDate(this.getDate(2005, 2, 2));
        onFile.setAppointmentRequestResponse(Boolean.TRUE);        
        
        Person incoming = savePerson(onFile);
        Person result = calculate(incoming);

        assessEnrollmentStatusResult(result,
                EnrollmentStatus.CODE_REJECTED_BELOW_ENROLLMENT_GROUP_THRESHOLD);
        assertNotNull(result.getEnrollmentDetermination().getEndDate());
        assertEquals(result.getEnrollmentDetermination().getEndDate(), result
                .getEnrollmentDetermination().getEffectiveDate());
        assertNull(result.getAppointmentRequestDate());
        assertNull(result.getAppointmentRequestResponse());        
    }

    /**
     * EnrollmentStatusIsDeceased- If date of death is not null then set the
     * enrollment status equal to Deceased and set the enrollment effective
     * date to system date and set the enrollment priority equal to the
     * enrollment priority at time of last update and set the preferred
     * enrollment facility equal to NULL and set the enrollment date equal to
     * NULL and set the enrollment end date equal to the date of death and set
     * the enrollment category equal to not enrolled
     * 
     * @throws Exception
     */
    public void testEnrollmentStatusIsDeceased() throws Exception {
        Person onFile = buildSimplePerson(); // bare-bones person
        Person incoming = savePerson(onFile);

        this.attachDeathRecordFromSite(incoming);

        Person updatedPerson = this.calculate(incoming);
        assessEnrollmentStatusResult(updatedPerson,
                EnrollmentStatus.CODE_DECEASED);
    }

    /**
     * EnrollmentStatusIsUnverified- If enrollment status received from site is
     * equal to unverified then set the enrollment status equal to
     * unverified and set the enrollment effective date equal to enrollment
     * effective date received from VAMC and set the source of enrollment
     * received from VAMC and set the enrollment priority equal to the
     * enrollment priority received from VAMC and set the enrollment preferred
     * facility equal to preferred facility received from VAMC and set the
     * enrollment date equal to enrollment date received from VAMC and set
     * facility received from equal to VAMC received facility and set the
     * enrollment end date equal to enrollment end date received from VAMC and
     * set the enrollment category equal to enrollment category received from
     * VAMC
     * 
     */
    public void testEnrollmentStatusIsUnverified() throws Exception {
        Person onFile = buildSimplePerson(); // bare-bones person
        this.attachAidAndAttendance(onFile);
        this.attachVerifiedEnrollment(onFile); // make existing person
        Person incoming = savePerson(onFile);
        this.getPersonHelperService().removeAABenefit(incoming);
        this.attachReceivedEnrollment(incoming);
        Person updatedPerson = this.calculate(incoming);
        String category = this.getEnrollmentCategoryCode(updatedPerson);
        EnrollmentDetermination enrollmentDetermination = getEnrollmentDetermination(updatedPerson);
        assertNotNull(category);
        assertTrue(category.equals("P"));
        assessEnrollmentStatusResult(updatedPerson,
                EnrollmentStatus.CODE_UNVERIFIED);
        assertTrue(enrollmentDetermination.getCalculationSource().getCode()
                .equals(EnrollmentSource.CODE_VAMC.getName()));
    }

    /**
     * EnrollmentStatusIsRejectedFromVAMC- If enrollment status received from
     * site is equal to Rejected initial Application from VAMC then set the
     * enrollment status equal to Rejected initial Application from VAMC and
     * set the enrollment effective date equal to enrollment effective date
     * received from VAMC and set the source of enrollment to the preferred
     * facility received from VAMC and set the enrollment priority equal to the
     * enrollment priority received from VAMC and set the enrollment preferred
     * facility equal to the preferred facility received from VAMC and set the
     * enrollment date equal to the enrollment date received from VAMC and set
     * the facility from equal to received from VAMC facility and set the
     * enrollment end date equal to enrollment end received from VAMC and set
     * the enrollment category equal to the enrollment category received from
     * VAMC
     * 
     */
    public void testEnrollmentStatusIsRejectedFromVAMC() throws Exception {
        Person onFile = buildSimplePerson(); // bare-bones person
        this.attachAidAndAttendance(onFile);
        this.attachVerifiedEnrollment(onFile); // make existing person
        Person incoming = savePerson(onFile);
        this.getPersonHelperService().removeAABenefit(incoming);
        incoming
                .setReceivedEnrollment(createReceivedEnrollment(EnrollmentStatus.CODE_REJECTED_INITIAL_APPLICATION_BY_VAMC));
        incoming.setAppointmentRequestDate(this.getDate(2005, 2, 2));
        incoming.setAppointmentRequestResponse(Boolean.TRUE);
        
        Person updatedPerson = this.calculate(incoming);

        EnrollmentDetermination enrollmentDetermination = getEnrollmentDetermination(updatedPerson);
        assertTrue(enrollmentDetermination.getCalculationSource().getCode()
                .equals(EnrollmentSource.CODE_VAMC.getName()));
        assessEnrollmentStatusResult(updatedPerson,
                EnrollmentStatus.CODE_REJECTED_INITIAL_APPLICATION_BY_VAMC);
        assertNull(updatedPerson.getAppointmentRequestDate());
        assertNull(updatedPerson.getAppointmentRequestResponse());
        
    }

    /**
     * EnrollmentStatusIsPendingMTRequired - If eligibility status is equal to
     * verified and Means status is required then set the enrollment status
     * equal to Pending; Means Test Required and set the enrollment effective
     * date to the system date and set the source of enrollment to HEC and set
     * the enrollment priority equal to null and set the enrollment date equal
     * to null and set the facility received from to null and set the enrollment
     * category equal to in process
     * 
     */
    public void testEnrollmentStatusIsPendingMTRequired() throws Exception {
        Person onFile = buildSimplePerson(); // bare-bones person
        this.attachVerifiedEnrollment(onFile);
        Person incoming = savePerson(onFile);
        this.setEligibilityStatus(EligibilityStatus.CODE_VERIFIED, incoming);

        Person updatedPerson = this.calculate(incoming);

        assessEnrollmentStatusResult(updatedPerson,
                EnrollmentStatus.CODE_PENDING_MEANS_TEST_REQUIRED);
    }
    
    public void testEnrollmentStatusIsPendingMTRequired_WithRXTest() throws Exception {
        Person onFile = buildSimplePerson(); // bare-bones person
        this.attachVerifiedEnrollment(onFile);
        this.attachPharmacyCopayTestWithRXCopayExempt(onFile);
        Person incoming = savePerson(onFile);
        this.setEligibilityStatus(EligibilityStatus.CODE_VERIFIED, incoming);

        Person updatedPerson = this.calculate(incoming);

        assessEnrollmentStatusResult(updatedPerson,
                EnrollmentStatus.CODE_PENDING_MEANS_TEST_REQUIRED);
    }    

    /**
     * EnrollmentStatusIsPendingEligibilityUnverified If eligibility status is
     * equal to pending verification or pending reverification then set the
     * enrollment status equal to Pending; eligibility status is unverified
     * and set the enrollment effective date to the system date and set the
     * source of enrollment to HEC and set the enrollment priority equal to NULL
     * and set the enrollment date equal to null and set the facility received
     * from to null and set the enrollment category equal to in process
     * 
     */
    public void testEnrollmentStatusIsPendingEligibilityUnverified()
            throws Exception {
        Person onFile = buildSimplePerson(); // bare-bones person
        Person incoming = savePerson(onFile);
        this.setEligibilityStatus(EligibilityStatus.CODE_PENDING_VERIFICATION,
                incoming);

        Person updatedPerson = this.calculate(incoming);

        assessEnrollmentStatusResult(updatedPerson,
                EnrollmentStatus.CODE_PENDING_ELIGIBILITY_STATUS_IS_UNVERIFIED);

        onFile = buildSimplePerson(); // bare-bones person
        incoming = this.getPersonService().save(onFile);

        this.setEligibilityStatus(
                EligibilityStatus.CODE_PENDING_REVERIFICATION, incoming);

        updatedPerson = this.calculate(incoming);

        assessEnrollmentStatusResult(updatedPerson,
                EnrollmentStatus.CODE_PENDING_ELIGIBILITY_STATUS_IS_UNVERIFIED);
    }

    /**
     * EnrollmentStatusIsCancelledDeclined- If cancelled/declined indicator is
     * YES then set the enrollment status equal to Cancelled/Declined and
     * set the enrollment effective date to system date and set source of
     * enrollment to HEC and set enrollment priority equal to the enrollment at
     * time of last update and set the preferred enrollment facility equal to
     * NULL and set the enrollment date equal to NULL and set the facility
     * received from equal to HEC and set the enrollment end date equal to the
     * cancelled/declined effective date and set the enrollment category equal
     * to not enrolled
     * 
     * @throws Exception
     */
    public void testEnrollmentStatusIsCancelledDeclined() throws Exception {
        Person onFile = buildSimplePerson(); // bare-bones person
        onFile.setAppointmentRequestDate(this.getDate(2005, 2, 2));
        onFile.setAppointmentRequestResponse(Boolean.TRUE);
        Person incoming = savePerson(onFile);

        this.attachCancelDecline(incoming);
        
        Person updatedPerson = this.calculate(incoming);

        assessEnrollmentStatusResult(updatedPerson,
                EnrollmentStatus.CODE_CANCELLED_DECLINED);
        assertNull(updatedPerson.getAppointmentRequestDate());
        assertNull(updatedPerson.getAppointmentRequestResponse());
    }

    /**
     * EnrollmentStatusIsNotEligibleIneligibleDate- If ineligible date is not
     * null then set the enrollment status equal to Not Eligible: Ineligible
     * date and set the enrollment effective date to system date and set the
     * enrollment priority equal to null and set the preferred enrollment
     * facility equal to null and set the enrollment date equal to null and set
     * the enrollment end date equal to the ineligible date and set the
     * enrollment category equal to not enrolled
     * 
     */
    public void testEnrollmentStatusIsNotEligilibleIneligibleDate()
            throws Exception {
        Person onFile = buildSimplePerson(); // bare-bones person
        Person incoming = savePerson(onFile);
        this.attachIneligibleFactor(incoming);
        Person updatedPerson = this.calculate(incoming);

        assessEnrollmentStatusResult(updatedPerson,
                EnrollmentStatus.CODE_NOT_ELIGIBLE_INELIGIBLE_DATE);
    }

    /**
     * EnrollmentStatusIsNotEligibleRefusedToCopay- If current means test status
     * is MT copay required or current means test status is GMT copay
     * required or pending adjudication and agree to pay deductible indicator
     * is NO or IF -disclosure indicator is NO and agree to pay deductible
     * indicator is NO for the current means test then set enrollment status
     * equal to Not Eligible, refused to pay copay and set the enrollment
     * effective date equal to system date and set the source of enrollment to
     * HEC and set the enrollment priority equal to null and set the enrollment
     * date equal to null and set the enrollment end date equal to the effective
     * date of the primary means test when agree to pay deductible was set to
     * NO and set the enrollment category equal to not enrolled
     * 
     */
    public void testEnrollmentStatusIsNotEligilibleRefusedToCopay()
            throws Exception {
        Person onFile = buildSimplePerson(); // bare-bones person
        Person incoming = savePerson(onFile);
        IncomeTest it = createIncomeTest(IncomeTestType.CODE_MEANS_TEST,
                MeansTestStatus.MT_STATUS_MT_COPAY_REQUIRED, new Integer(2005));
        it.setAgreesToPayDeductible(Boolean.FALSE);
        it.setDiscloseFinancialInformation(Boolean.FALSE);

        incoming.setIncomeTest(new Integer(2005), it);

        Person updatedPerson = this.calculate(incoming);

        assessEnrollmentStatusResult(updatedPerson,
                EnrollmentStatus.CODE_NOT_ELIGIBLE_REFUSED_TO_PAY_COPAY);
    }

    /**
     * EnrollmentStatusIsPendingPHUnconfirmed- If purple heart indicator yes
     * and purple heart status is not confirmedthen set enrollment status
     * equal to Pending, purple heart unconfirmed and set the enrollment
     * effective date to system date and set the source of enrollment to HEC and
     * set the enrollment priority to calculated enrollment priority and set
     * the enrollment category equal to in process
     * 
     */
    public void testEnrollmentStatusIsPendingPHUnconfirmed() throws Exception {
        Person onFile = buildSimplePerson(); // bare-bones person
        Person incoming = savePerson(onFile);
        this.attachPurpleHeartWithPHStatusPending(incoming);

        Person updatedPerson = this.calculate(incoming);

        assessEnrollmentStatusResult(updatedPerson,
                EnrollmentStatus.CODE_PENDING_PURPLE_HEART_UNCONFIRMED);
    }

    public void testEnrollmentStatusIsPendingNoEligCode() throws Exception {
        Person onFile = buildSimplePerson(); // bare-bones person
        this.attachVerifiedEnrollment(onFile);

        this.setEligibilityStatus(EligibilityStatus.CODE_VERIFIED, onFile);
        Person incoming = savePerson(onFile);

        Person updatedPerson = this.calculate(incoming);

        assessEnrollmentStatusResult(updatedPerson,
                EnrollmentStatus.CODE_PENDING_NO_ELIGIBILITY_CODE_IN_VIVA);
    }

    public void testAssignRejectedSubpriorityE() throws Exception {

    }

    public void testAssignRejectedSubpriorityG() throws Exception {
        Person onFile = buildSimplePerson(); // bare-bones person
        this.attachMeansTestWithMtCopayRequired(onFile, new BigDecimal(1000),
                new BigDecimal(1000));
        this.setEligibilityStatusToVerified(onFile);
        this.attachApplication(onFile);
        Person incoming = savePerson(onFile);

        Person updated = calculate(incoming);
        assessPriorityResult(updated, EnrollmentPriorityGroup.GROUP_8);
        assessPrioritySubGroupResult(updated,
                EnrollmentPrioritySubGroup.SUBGROUP_G);
        assessEnrollmentStatusResult(updated,
                EnrollmentStatus.CODE_REJECTED_BELOW_ENROLLMENT_GROUP_THRESHOLD);
    }

    public void testIsCurrentRejectedAndOverridden() throws Exception {
        overrideTest(this.getLookupService().getEnrollmentStatusByCode(
                EnrollmentStatus.CODE_REJECTED_BELOW_ENROLLMENT_GROUP_THRESHOLD
                        .getName()));
    }

    public void testIsCurrentRejectedAndOverriddenVAMC() throws Exception {

        overrideTest(this.getLookupService().getEnrollmentStatusByCode(
                EnrollmentStatus.CODE_REJECTED_INITIAL_APPLICATION_BY_VAMC
                        .getName()));
    }

    /**
     * Test a person who was overridden already and has enrollment of verified.
     * Incoming is received with override equal to false. Keeps the verified.
     * 
     * @throws Exception
     */
    public void testOverriddenWithCurrentVerified() throws Exception {

        Person onFile = buildSimplePerson(); // bare-bones person

        attachMeansTestWithMtCopayRequired(onFile, new BigDecimal(999),
                new BigDecimal(1000));
        this.attachVerifiedEnrollment(onFile);
        this.turnOnEnrollmentOverride(onFile);
        onFile.getEnrollmentDetermination().setEnrollmentStatus(
                this.getLookupService().getEnrollmentStatusByCode(
                        EnrollmentStatus.CODE_VERIFIED.getName()));
        this.setEligibilityStatusToVerified(onFile);
        Person incoming = savePerson(onFile);

        Person result = calculate(incoming);

        assessEnrollmentStatusResult(result, EnrollmentStatus.CODE_VERIFIED);

        assessPriorityResult(result, EnrollmentPriorityGroup.GROUP_8);
        assessPrioritySubGroupResult(result,
                EnrollmentPrioritySubGroup.SUBGROUP_C);

        this.turnOffEnrollmentOverride(result);

        Person updated = calculate(result);

        assessEnrollmentStatusResult(updated, EnrollmentStatus.CODE_VERIFIED);

        assessPriorityResult(updated, EnrollmentPriorityGroup.GROUP_8);
        assessPrioritySubGroupResult(updated,
                EnrollmentPrioritySubGroup.SUBGROUP_C);
    }

    public void testIsUserOverridingRejectedEnrollment() throws Exception {

        Person onFile = buildSimplePerson(); // bare-bones person

        attachMeansTestWithMtCopayRequired(onFile, new BigDecimal(999),
                new BigDecimal(1000));
        this.attachVerifiedEnrollment(onFile);
        this.turnOffEnrollmentOverride(onFile);
        onFile
                .getEnrollmentDetermination()
                .setEnrollmentStatus(
                        this
                                .getLookupService()
                                .getEnrollmentStatusByCode(
                                        EnrollmentStatus.CODE_REJECTED_BELOW_ENROLLMENT_GROUP_THRESHOLD
                                                .getName()));
        this.setEligibilityStatusToVerified(onFile);
        Person incoming = savePerson(onFile);

        this.turnOnEnrollmentOverride(incoming);

        Person updated = calculate(incoming);

        assessEnrollmentStatusResult(updated, EnrollmentStatus.CODE_VERIFIED);

        assessPriorityResult(updated, EnrollmentPriorityGroup.GROUP_8);
        assessPrioritySubGroupResult(updated,
                EnrollmentPrioritySubGroup.SUBGROUP_C);
    }

    public void overrideTest(EnrollmentStatus status) throws Exception {
        Person onFile = buildSimplePerson(); // bare-bones person

        attachMeansTestWithMtCopayRequired(onFile, new BigDecimal(999),
                new BigDecimal(1000));
        this.attachVerifiedEnrollment(onFile);
        onFile.getEnrollmentDetermination().setEnrollmentStatus(status);
        onFile.getEnrollmentDetermination().setEffectiveDate(new Date());
        Person incoming = savePerson(onFile);

        Person result = calculate(incoming);

        assessEnrollmentStatusResult(result,
                EnrollmentStatus.CODE_REJECTED_BELOW_ENROLLMENT_GROUP_THRESHOLD);
        assessPriorityResult(result, EnrollmentPriorityGroup.GROUP_8);
        assessPrioritySubGroupResult(result,
                EnrollmentPrioritySubGroup.SUBGROUP_G);

        this.turnOnEnrollmentOverride(result);
        this.setEligibilityStatusToVerified(result);
        Person updated = calculate(result);

        assessEnrollmentStatusResult(updated, EnrollmentStatus.CODE_VERIFIED);

        assessPriorityResult(updated, EnrollmentPriorityGroup.GROUP_8);
        assessPrioritySubGroupResult(updated,
                EnrollmentPrioritySubGroup.SUBGROUP_C);
    }

    public void testVerifiedStatus() throws Exception {
        Person onFile = buildSimplePerson(); // bare-bones person
        this.attachMeansTestWithGmtCopayRequired(onFile, new BigDecimal(9),
                new BigDecimal(9));
        this.attachVerifiedEnrollment(onFile);
        this.setEligibilityStatusToVerified(onFile);
        Person incoming = savePerson(onFile);
        Person updated = this.calculate(incoming);
        assessEnrollmentStatusResult(updated, EnrollmentStatus.CODE_VERIFIED);
    }

    public void testUpdateFinancials() throws Exception {
        Person onFile = buildSimplePerson(); // bare-bones person
        this.attachMeansTestWithGmtCopayRequired(onFile, new BigDecimal(9),
                new BigDecimal(9));
        onFile.setFinancialStatement(new Integer(2005), this.createFinancialStatement());
        onFile.getFinancialStatement(new Integer(2005)).setAsset(
                this.getLookupService().getAssetTypeByCode(AssetType.CODE_OTHER.getCode()), this.createAsset((double)45458));
        onFile.getFinancialStatement(new Integer(2005)).setDebt(this.createDebt((double)5));
        this.attachVerifiedEnrollment(onFile);
        this.setEligibilityStatusToVerified(onFile);
        Person incoming = savePerson(onFile);
        incoming.getFinancialStatement(new Integer(2005)).setDebt(this.createDebt((double)45));
        Person updated = this.getFinancialsService().updateFinancialAssessment(
                new Integer(2005), (Person) incoming.clone());
        // Person updated = this.calculate(incoming);
        assessEnrollmentStatusResult(updated, EnrollmentStatus.CODE_VERIFIED);
    }

    /**
     * @return Returns the determineEnrollmentRuleParameters.
     */
    public RuleParameters getDetermineEnrollmentRuleParameters()
            throws ServiceException {
        return (RuleParameters) applicationContext.getBean("determineEnrollmentRuleParameters");
    }

    private void turnOnEnrollmentOverride(Person person) {
        EnrollmentOverride eo = new EnrollmentOverride();
        eo.setOverride(Boolean.TRUE);
        eo.setOverrideComment("override this person");
        eo.setOverrideDate(new Date());
        eo.setProcessOverride(Boolean.TRUE);
        person.setEnrollmentOverride(eo);
    }

    private void turnOffEnrollmentOverride(Person person) {
        EnrollmentOverride eo = person.getEnrollmentOverride();
        if (eo != null) {
            eo.setOverride(Boolean.FALSE);
            eo.setOverrideComment(null);
            eo.setOverrideDate(null);
            eo.setProcessOverride(Boolean.TRUE);
        }

    }

    /*
     * 6295 [UC13.10.3.1] RULE 2: If the enrollment record is in a Rejected
     * enrollment status due to a Manual Override (Enrollment Override =YES) the
     * beneficiary shall remain in a Rejected status unless the veteran is
     * assigned to an enrollment priority that is being enrolled, or until a new
     * EGT is set that could qualify the veteran for enrollment or if the record
     * is manually overridden to Enrolled. Exit Continuous Enrollment Processing
     * if Rule 2 has been met.
     */
    public void testRule2() throws Exception {
        Person onFile = this.getPerson("20009901");
        Person result = calculate(onFile);
        assertTrue(result.getEnrollmentDetermination().getEnrollmentStatus()
                .getEnrollmentCategory().getCode().equals(
                        EnrollmentCategory.CODE_NOT_ENROLLED.getCode()));
        System.out.println("Person id="
                + result.getPersonEntityKey().getKeyValueAsString());
    }

    public void testCreateRule2Data() throws Exception {
        Person onFile = buildSimplePerson(); // bare-bones person
        this.setEligibilityStatusToVerified(onFile);
        this.attachVerifiedEnrollment(onFile);
        onFile.getEnrollmentDetermination().setOverridden(Boolean.TRUE);
        onFile.getEnrollmentDetermination().setEffectiveDate(new Date());
        this.attachMeansTestWithMtCopayRequired(onFile, new BigDecimal(1),
                new BigDecimal(1));
        onFile
                .getEnrollmentDetermination()
                .setEnrollmentStatus(
                        this
                                .getLookupService()
                                .getEnrollmentStatusByCode(
                                        EnrollmentStatus.CODE_REJECTED_BELOW_ENROLLMENT_GROUP_THRESHOLD
                                                .getCode()));
        Person incoming = savePerson(onFile);
        this.setComplete();
        System.out.println("Person id="
                + incoming.getPersonEntityKey().getKeyValueAsString());
    }

    /**
     * 6296 [UC13.10.3.2] RULE 3: If the enrollment record is in a Rejected
     * enrollment status, it will stay Rejected as long as the veteran stays in
     * a priority group not being accepted for new enrollment. Exit Continuous
     * Enrollment Processing if Rule 3 has been met.
     */
    public void testRule3() throws Exception {
        Person onFile = this.getPerson("20003377");
        Person result = calculate(onFile);
        assertTrue(result.getEnrollmentDetermination().getEnrollmentStatus()
                .getEnrollmentCategory().getCode().equals(
                        EnrollmentCategory.CODE_NOT_ENROLLED.getCode()));
        System.out.println("Person id="
                + result.getPersonEntityKey().getKeyValueAsString());
    }

    public void testCreateRule3Data() throws Exception {
        Person onFile = buildSimplePerson(); // bare-bones person
        this.setEligibilityStatusToVerified(onFile);
        this.attachVerifiedEnrollment(onFile);

        onFile
                .getEnrollmentDetermination()
                .setEnrollmentStatus(
                        this
                                .getLookupService()
                                .getEnrollmentStatusByCode(
                                        EnrollmentStatus.CODE_REJECTED_BELOW_ENROLLMENT_GROUP_THRESHOLD
                                                .getCode()));

        this.attachMeansTestWithMtCopayRequired(onFile, new BigDecimal(1),
                new BigDecimal(1));
        Person incoming = savePerson(onFile);
        this.setComplete();
        System.out.println("Person id="
                + incoming.getPersonEntityKey().getKeyValueAsString());
    }

    /**
     * 6297 [UC13.10.3.3] RULE 4: If the enrollment record is in a Verified
     * enrollment status due to a Manual Override (Enrollment Override =YES) the
     * beneficiary shall remain in an Enrolled status until a new EGT is set
     * that could disqualify the veteran from enrollment or if the record is
     * manually overridden to Rejected. Exit Continuous Enrollment Processing if
     * Rule 4 has been met.
     */
    public void testRule4() throws Exception {
        Person onFile = this.getPerson("20006397");
        Person result = calculate(onFile);
        assertTrue(result.getEnrollmentDetermination().getEnrollmentStatus()
                .getEnrollmentCategory().getCode().equals(
                        EnrollmentCategory.CODE_ENROLLED.getCode()));
        System.out.println("Person id="
                + result.getPersonEntityKey().getKeyValueAsString());
    }

    public void testCreateRule4Data() throws Exception {
        Person onFile = buildSimplePerson(); // bare-bones person
        this.setEligibilityStatusToVerified(onFile);
        this.attachVerifiedEnrollment(onFile);
        onFile.getEnrollmentDetermination().setOverridden(Boolean.TRUE);
        onFile.getEnrollmentDetermination().setEffectiveDate(new Date());
        this.attachMeansTestWithMtCopayRequired(onFile, new BigDecimal(1),
                new BigDecimal(1));
        Person incoming = savePerson(onFile);
        this.setComplete();
        System.out.println("Person id="
                + incoming.getPersonEntityKey().getKeyValueAsString());
    }

    /**
     * 6298 [UC13.10.3.4] RULE 5: If the enrollment record is in a
     * Cancel/Declined enrollment status on or after the EGT Effective Date it
     * will be treated the same as a Rejected enrollment status and not be
     * continuously enrolled as long as a veteran stays in a priority not being
     * accepted for new enrollments. Exit Continuous Enrollment Processing if
     * Rule 5 has been met
     */
    public void testRule5() throws Exception {
        Person onFile = this.getPerson("91420");
        onFile.setCancelDecline(null);
        this.setEligibilityStatusToVerified(onFile);
        onFile.getEligibilityVerification().setVerificationMethod("test");
        
        Person result = this.getEligibilityEnrollmentService().updateEnrollmentData(onFile);
        assertTrue("expected:" + EnrollmentStatus.CODE_REJECTED_BELOW_ENROLLMENT_GROUP_THRESHOLD.getCode() + " was:" + result.getEnrollmentDetermination().getEnrollmentStatus().getCode(),
                result.getEnrollmentDetermination().getEnrollmentStatus().getCode().equals(
                EnrollmentStatus.CODE_REJECTED_BELOW_ENROLLMENT_GROUP_THRESHOLD.getCode()) );
        System.out.println("Person id="
                + result.getPersonEntityKey().getKeyValueAsString());
    }
    
    public void testRule5GettingCancelDecline() throws Exception {
        Person onFile = this.getPerson("91420");
        assertTrue("expected:" + EnrollmentStatus.CODE_CANCELLED_DECLINED.getCode() +
                " actual:" + onFile.getEnrollmentDetermination().getEnrollmentStatus().getCode(),
                onFile.getEnrollmentDetermination().getEnrollmentStatus().getCode().equals(
                EnrollmentStatus.CODE_CANCELLED_DECLINED.getCode()));
        System.out.println("Person id=" + onFile.getPersonEntityKey().getKeyValueAsString());
    }    

    public void testCreateRule5Data() throws Exception {
        Person onFile = buildSimplePerson(); // bare-bones person
        this.setEligibilityStatusToPendingVerification(onFile);
        attachVerifiedEnrollment(onFile);
        onFile.getEnrollmentDetermination().setEffectiveDate(this.getDate(2003, 6, 10));
        //this.attachMeansTestWithMtCopayRequired(onFile, new BigDecimal(1), new BigDecimal(1));
        //EnrollmentPriorityGroup;
        //import gov.va.med.esr.common.model.lookup.EnrollmentPrioritySubGroup;        
        this.attachCancelDecline(onFile);
        this.attachApplication(onFile);
        onFile.getApplication().setApplicationDate(getDate(1999,4,27));
        onFile.getCancelDecline().setEffectiveDate(this.getDate(1999, 4, 27));
        Eligibility e = new Eligibility();
        e.setPrimary();
        e.setType(this.getLookupService().getEligibilityTypeByCode(EligibilityType.NSC.getCode()));
        onFile.getEnrollmentDetermination().setEnrollmentDate(null);
        onFile.getEnrollmentDetermination().setPrimaryEligiblity(e);
        onFile.getEnrollmentDetermination().setEnrollmentStatus(
                this.getLookupService().getEnrollmentStatusByCode(EnrollmentStatus.CODE_CANCELLED_DECLINED.getCode()));
        
        onFile.getEnrollmentDetermination().setPriorityGroup(null);
        onFile.getEnrollmentDetermination().setPrioritySubGroup(null);
//        onFile.getEnrollmentDetermination().setPriorityGroup(this.getLookupService().getEnrollmentPriorityGroupByCode(
  //              EnrollmentPriorityGroup.GROUP_8.getCode()));
    //    onFile.getEnrollmentDetermination().setPrioritySubGroup(
      //          this.getLookupService().getEnrollmentPrioritySubGroupByCode(EnrollmentPrioritySubGroup.SUBGROUP_C.getCode()));        
        Person incoming = savePerson(onFile);
        this.setComplete();
        System.out.println("Person id="
                + incoming.getPersonEntityKey().getKeyValueAsString());
    }

    /**
     * 4384 [UC13.10.3.5] RULE 6 If the enrollment record is in an Enrollment
     * Status of Not Eligible; Refused to Pay Copay on or after the EGT
     * Effective Date. AND the veteran Agrees to Pay Deductible Indicator is
     * being changed from No to Yes AND The veteran has been sent a 601B OR 601C
     * letter had been triggered where the Communication Status is equal to any
     * of the following: 0-send to AAC 1-sent to AAC 2-mailed by HEC 3-reject at
     * HEC 4-reject by AAC 5- Error by AAC 6-Return by post office 7-mailed by
     * AAC 8-Sent to HEC printer 9-address changed and mailed by AAC
     * 
     * If the letter has been triggered and the Communication Status is one of
     * the above THEN it indicates that an attempt was made to notify the
     * Beneficiary that he/she is not eligible for enrollment, therefore the
     * system shall Reject Enrollment. Exit Continuous Enrollment Processing if
     * Rule 6 has been met. ELSE If the letters were NOT triggered and the
     * Communication Status is null or value 10-Cancel by HEC THEN it is
     * inferred that the veteran has not received notification of his ineligible
     * status. Continue executing the rules.
     */
    public void testRule6() throws Exception {
        Person onFile = this.getPerson("20003456");
        Person result = calculate(onFile);
        // assertTrue(result.getEnrollmentDetermination().getEnrollmentStatus().getEnrollmentCategory().getCode().equals(EnrollmentCategory.CODE_NOT_ENROLLED.getCode()));
        System.out.println("Person id="
                + result.getPersonEntityKey().getKeyValueAsString());
    }

    public void testCreateRule6Data() throws Exception {
        Person onFile = buildSimplePerson(); // bare-bones person
        this.setEligibilityStatusToVerified(onFile);
        attachVerifiedEnrollment(onFile);
        onFile.getEnrollmentDetermination().setEffectiveDate(new Date());
        this.attachMeansTestWithMtCopayRequired(onFile, new BigDecimal(1),
                new BigDecimal(1));
        onFile.getIncomeTest(new Integer(2005)).setAgreesToPayDeductible(
                Boolean.TRUE);
        onFile.getIncomeTest(new Integer(2005))
                .setDiscloseFinancialInformation(Boolean.TRUE);
        onFile.getEnrollmentDetermination().setEnrollmentStatus(
                this.getLookupService().getEnrollmentStatusByCode(
                        EnrollmentStatus.CODE_NOT_ELIGIBLE_REFUSED_TO_PAY_COPAY
                                .getCode()));
        Person incoming = savePerson(onFile);
        this.setComplete();
        System.out.println("Person id="
                + incoming.getPersonEntityKey().getKeyValueAsString());
    }

    /**
     * 1281 [UC13.10.4.1] RULE 7 If the earliest effective date of change is
     * prior to the EGT Effective Date, the system shall continuously enroll.
     * Exit Continuous Enrollment Processing if Rule 7 has been met.
     * 
     */
    public void testRule7() throws Exception {
        Person onFile = this.getPerson("20003519");
        Person result = calculate(onFile);
        assertTrue(result.getEnrollmentDetermination().getEnrollmentStatus()
                .getEnrollmentCategory().getCode().equals(
                        EnrollmentCategory.CODE_ENROLLED.getCode()));
        System.out.println("Person id="
                + result.getPersonEntityKey().getKeyValueAsString());
    }

    public void testCreateRule7Data() throws Exception {
        Person onFile = buildSimplePerson(); // bare-bones person
        this.setEligibilityStatusToVerified(onFile);
        this.attachVerifiedEnrollment(onFile);
        onFile.getEnrollmentDetermination().setEffectiveDate(
                getDate(2000, 1, 1));

        this.attachMeansTestWithMtCopayRequired(onFile, new BigDecimal(1),
                new BigDecimal(1));
        Person incoming = savePerson(onFile);
        this.setComplete();
        System.out.println("Person id="
                + incoming.getPersonEntityKey().getKeyValueAsString());
    }

    /**
     * 4382 [UC13.10.4.2] RULE 8 If there is any Application Date prior to the
     * EGT Effective Date, the system shall continuously enroll. Exit Continuous
     * Enrollment Processing if Rule 8 has been met.
     */
    public void testRule8() throws Exception {
        Person onFile = this.getPerson("20003528");
        Person result = calculate(onFile);
        assertTrue(result.getEnrollmentDetermination().getEnrollmentStatus()
                .getEnrollmentCategory().getCode().equals(
                        EnrollmentCategory.CODE_ENROLLED.getCode()));
        System.out.println("Person id="
                + result.getPersonEntityKey().getKeyValueAsString());
    }

    public void testCreateRule8Data() throws Exception {
        Person onFile = buildSimplePerson(); // bare-bones person
        this.setEligibilityStatusToVerified(onFile);
        this.attachVerifiedEnrollment(onFile);
        this.attachApplication(onFile);
        onFile.getApplication().setApplicationDate(getDate(2000, 1, 1));

        this.attachMeansTestWithMtCopayRequired(onFile, new BigDecimal(1),
                new BigDecimal(1));
        Person incoming = savePerson(onFile);
        this.setComplete();
        System.out.println("Person id="
                + incoming.getPersonEntityKey().getKeyValueAsString());
    }

    /**
     * RULE 9 SC 10% or greater and SC% is changed to SC 0% non Compensable
     * (total check amount $0 or null)
     */
    public void testRule9() throws Exception {
        Person onFile = this.getPerson("20461158");
        onFile.getServiceConnectionAward().setServiceConnectedPercentage(
                new Integer(0));
        Person result = calculate(onFile);
        assertTrue(result.getEnrollmentDetermination().getEnrollmentStatus()
                .getEnrollmentCategory().getCode().equals(
                        EnrollmentCategory.CODE_ENROLLED.getCode()));
        System.out.println("Person id="
                + result.getPersonEntityKey().getKeyValueAsString());
    }

    public void testCreateRule9Data() throws Exception {
        Person onFile = buildSimplePerson(); // bare-bones person
        this.setEligibilityStatusToVerified(onFile);
        this.attachVerifiedEnrollment(onFile);
        onFile.getEnrollmentDetermination().setEffectiveDate(new Date());
        this.attachServiceConnectionAward(onFile, new Integer(10));

        this.attachMeansTestWithMtCopayRequired(onFile, new BigDecimal(1),
                new BigDecimal(1));
        Person incoming = savePerson(onFile);
        this.setComplete();
        System.out.println("Person id="
                + incoming.getPersonEntityKey().getKeyValueAsString());
    }

    /**
     * RULE 10 Aid & Attendance = YES and A&A is now not YES
     */
    public void testRule10() throws Exception {
        Person onFile = this.getPerson("20461145");
        onFile.getMonetaryBenefitAward().removeAllMonetaryBenefits();
        Person result = calculate(onFile);
        assertTrue(result.getEnrollmentDetermination().getEnrollmentStatus()
                .getEnrollmentCategory().getCode().equals(
                        EnrollmentCategory.CODE_ENROLLED.getCode()));
        System.out.println("Person id="
                + result.getPersonEntityKey().getKeyValueAsString());
    }

    public void testCreateRule10Data() throws Exception {
        Person onFile = buildSimplePerson(); // bare-bones person
        this.setEligibilityStatusToVerified(onFile);
        this.attachVerifiedEnrollment(onFile);
        onFile.getEnrollmentDetermination().setEffectiveDate(new Date());
        this.attachAidAndAttendance(onFile);

        // I had to modify the test determined status to MT Copay Exempt in
        // order to prevent getting MT Required
        this.attachMeansTestWithMtCopayRequired(onFile, new BigDecimal(21),
                new BigDecimal(1));
        Person incoming = savePerson(onFile);
        this.getEligibilityEnrollmentService().updateEligibilityData(incoming);
        this.setComplete();
        System.out.println("Person id="
                + incoming.getPersonEntityKey().getKeyValueAsString());
    }

    /**
     * RULE 11 Housebound = YES and Housebound is now not YES
     */
    public void testRule11() throws Exception {
        Person onFile = this.getPerson("20003669");
        onFile.getMonetaryBenefitAward().removeAllMonetaryBenefits();
        Person result = calculate(onFile);
        assertTrue(result.getEnrollmentDetermination().getEnrollmentStatus()
                .getEnrollmentCategory().getCode().equals(
                        EnrollmentCategory.CODE_ENROLLED.getCode()));
        System.out.println("Person id="
                + result.getPersonEntityKey().getKeyValueAsString());
    }

    public void testCreateRule11Data() throws Exception {
        Person onFile = buildSimplePerson(); // bare-bones person
        this.setEligibilityStatusToVerified(onFile);
        this.attachVerifiedEnrollment(onFile);
        onFile.getEnrollmentDetermination().setEffectiveDate(new Date());
        this.attachHousebound(onFile);

        this.attachMeansTestWithMtCopayRequired(onFile, new BigDecimal(21),
                new BigDecimal(1));
        Person incoming = savePerson(onFile);
        this.getEligibilityEnrollmentService().updateEligibilityData(incoming);
        this.setComplete();
        System.out.println("Person id="
                + incoming.getPersonEntityKey().getKeyValueAsString());
    }
    
    /**
     * RULE 12 VA Pension = YES and VA Pension is now not YES
     */
    public void testRule12() throws Exception {
        Person onFile = this.getPerson("20009807");
        onFile.getMonetaryBenefitAward().removeAllMonetaryBenefits();
        Person result = calculate(onFile);
        assertTrue(result.getEnrollmentDetermination().getEnrollmentStatus()
                .getEnrollmentCategory().getCode().equals(
                        EnrollmentCategory.CODE_ENROLLED.getCode()));
        System.out.println("Person id="
                + result.getPersonEntityKey().getKeyValueAsString());
    }

    public void testCreateRule12Data() throws Exception {
        Person onFile = buildSimplePerson(); // bare-bones person
        this.setEligibilityStatusToVerified(onFile);
        this.attachVerifiedEnrollment(onFile);
        onFile.getEnrollmentDetermination().setEffectiveDate(new Date());
        this.attachVaPension(onFile);

        this.attachMeansTestWithMtCopayRequired(onFile, new BigDecimal(21),
                new BigDecimal(1));
        Person incoming = savePerson(onFile);
        this.getEligibilityEnrollmentService().updateEligibilityData(incoming);
        this.setComplete();
        System.out.println("Person id="
                + incoming.getPersonEntityKey().getKeyValueAsString());
    }
    
    /**
     * RULE 13 If AO indicator is currently = Yes and Location is currently =
     * DMZ and they were entered prior to ESR Implementation. (Date for this
     * software to be used will be 12/29/2040 until ESR is implemented. Date
     * will be modified with EVC)
     */
    public void testRule13() throws Exception {
        Person onFile = this.buildSimplePerson();
        this.attachVerifiedEnrollment(onFile);
        this.attachMeansTestWithMtCopayRequired(onFile, new BigDecimal(1),new BigDecimal(1));
        onFile.getEnrollmentDetermination().setPriorityGroup(this.getLookupService().getEnrollmentPriorityGroupByCode(EnrollmentPriorityGroup.GROUP_8.getCode()));
        onFile.getEnrollmentDetermination().setPrioritySubGroup(this.getLookupService().getEnrollmentPrioritySubGroupByCode(EnrollmentPrioritySubGroup.SUBGROUP_C.getCode()));        
        this.setEligibilityStatusToVerified(onFile);
        Person incoming = this.savePerson(onFile);
        this.attachAgentOrangeExposure(incoming);
        incoming.getAgentOrangeExposure().setLocation(this.getLookupService().getAgentOrangeExposureLocationByCode(AgentOrangeExposureLocation.CODE_KOREAN_DMZ.getCode()));
        Person result = this.getEligibilityEnrollmentService().updateEligibilityData(incoming);
        assertTrue(result.getEnrollmentDetermination().getEnrollmentStatus()
                .getEnrollmentCategory().getCode().equals(
                        EnrollmentCategory.CODE_ENROLLED.getCode()));
        System.out.println("Person id="
                + result.getPersonEntityKey().getKeyValueAsString());
    }
    //20751514
    public void testRule13_1() throws Exception {
        Person onFile = this.getPerson("20751781");
        Date date = new Date();
        onFile.getEligibilityVerification().setVerificationMethod(date.toString());
        Person result = this.getEligibilityEnrollmentService().updateEligibilityData(onFile);
        assertTrue(result.getEnrollmentDetermination().getEnrollmentStatus().getEnrollmentCategory().getCode().equals(
                        EnrollmentCategory.CODE_ENROLLED.getCode()));
    }    
    
    
    public void testRule13_Neg() throws Exception {
        Person onFile = this.buildSimplePerson();
        this.attachVerifiedEnrollment(onFile);
        this.attachMeansTestWithMtCopayRequired(onFile, new BigDecimal(1),new BigDecimal(1));
        onFile.getEnrollmentDetermination().setPriorityGroup(this.getLookupService().getEnrollmentPriorityGroupByCode(EnrollmentPriorityGroup.GROUP_8.getCode()));
        onFile.getEnrollmentDetermination().setPrioritySubGroup(this.getLookupService().getEnrollmentPrioritySubGroupByCode(EnrollmentPrioritySubGroup.SUBGROUP_C.getCode()));
        this.attachAgentOrangeExposure(onFile);        
        onFile.getAgentOrangeExposure().setLocation(this.getLookupService().getAgentOrangeExposureLocationByCode(AgentOrangeExposureLocation.CODE_KOREAN_DMZ.getCode()));        
        this.setEligibilityStatusToVerified(onFile);
        Person incoming = this.savePerson(onFile);
        incoming.getAgentOrangeExposure().setLocation(this.getLookupService().getAgentOrangeExposureLocationByCode(AgentOrangeExposureLocation.CODE_OTHER.getCode()));

        Person result = this.getEligibilityEnrollmentService().updateEligibilityData(incoming);
        assertTrue(result.getEnrollmentDetermination().getEnrollmentStatus()
                .getEnrollmentCategory().getCode().equals(EnrollmentCategory.CODE_NOT_ENROLLED.getCode()));
        System.out.println("Person id="
                + result.getPersonEntityKey().getKeyValueAsString());
    }    

   
        
    
    public void createRule13_Case2_Data() throws Exception {
        Person onFile = buildSimplePerson(); // bare-bones person
        this.attachVerifiedEnrollment(onFile);
        this.attachMeansTestWithMtCopayRequired(onFile, new BigDecimal(21),
                new BigDecimal(1));
        Person incoming = savePerson(onFile);
        this.setComplete();
        System.out.println("Person id="
                + incoming.getPersonEntityKey().getKeyValueAsString());
    }       
    
    public void testCreateRule13Data() throws Exception {
        Person onFile = buildSimplePerson(); // bare-bones person
        this.attachVerifiedEnrollment(onFile);
        onFile.getEnrollmentDetermination().setEnrollmentStatus(this.getLookupService().getEnrollmentStatusByCode(EnrollmentStatus.CODE_PENDING_OTHER.getCode()));
        this.attachMeansTestWithMtCopayRequired(onFile, new BigDecimal(21),
                new BigDecimal(1));
        onFile.getEnrollmentDetermination().setEffectiveDate(new Date());        
        Person incoming = savePerson(onFile);
        this.setEligibilityStatusToVerified(incoming);
        AgentOrangeExposure ao = this.createAgentOrange();
        ao.setLocation(this.getLookupService().getAgentOrangeExposureLocationByCode(AgentOrangeExposureLocation.CODE_KOREAN_DMZ.getName()));
        incoming.addSpecialFactor(ao);
        
        Person result = this.getEligibilityEnrollmentService().updateEligibilityData(incoming);
        this.setComplete();
        System.out.println("Person id="
                + result.getPersonEntityKey().getKeyValueAsString());
    }
    public void testCreateRuleCEData() throws Exception {
        Person onFile = buildSimplePerson(); // bare-bones person
        this.attachVerifiedEnrollment(onFile);
        onFile.getEnrollmentDetermination().setEnrollmentStatus(this.getLookupService().getEnrollmentStatusByCode(EnrollmentStatus.CODE_PENDING_OTHER.getCode()));
    
        onFile.getEnrollmentDetermination().setEffectiveDate(new Date());        
        Person incoming = savePerson(onFile);
        
        Person result = this.getEligibilityEnrollmentService().updateEligibilityData(incoming);
        this.setComplete();
        System.out.println("Person id="
                + result.getPersonEntityKey().getKeyValueAsString());
    }    
    
    /**
     * RULE 14 CV Eligibility End Date expires on or after Enrollment
     * Application Date (or in the absence of an Application Date, the earliest
     * Effective Date of Change) and the CV Eligibility End Date has not been
     * removed (set to NULL). 
     */
    public void testRule14() throws Exception {
       // Person onFile = this.getPerson("107121");
        PersonIdEntityKeyImpl personKey = new PersonIdEntityKeyImpl(new BigDecimal("107134"));        
        Person onFile = this.getPersonService().getPerson(personKey);
        
        onFile.getEligibilityVerification().setVerificationMethod("test2");
        //Person result = calculate(onFile);
        Person result = this.getEligibilityEnrollmentService().updateEligibilityData(onFile);
        assertTrue(result.getEnrollmentDetermination().getEnrollmentStatus()
                .getEnrollmentCategory().getCode().equals(
                        EnrollmentCategory.CODE_ENROLLED.getCode()));

        this.setComplete();
        System.out.println("Person id="
                + result.getPersonEntityKey().getKeyValueAsString());
    }
    protected ConflictExperience attachConflictExperience(Person person,
            ConflictLocation location) throws Exception {

        MilitaryService ms = this.attachMilitaryService(person);
        MilitaryServiceSiteRecord mssr = this.createMilitaryServiceSireRecord();
        mssr.setSite(this.getLookupService().getVaFacilityByCode(VAFacility.CODE_HEC.getCode()));
        // Create mse
        ConflictExperience ce = new ConflictExperience();
        ce.setConflictLocation(location);

        mssr.addConflictExperience(ce);
        ms.addMilitaryServiceSiteRecord(mssr);

        return ce;
    }
    
    public void testCreateRule14Data() throws Exception {
        Person onFile = buildSimplePerson(); // bare-bones person
        this.setEligibilityStatusToVerified(onFile);
        this.attachVerifiedEnrollment(onFile);
        onFile.getEnrollmentDetermination().setEffectiveDate(new Date());
        onFile.getEnrollmentDetermination().setPriorityGroup(
                this.getLookupService().getEnrollmentPriorityGroupByCode(EnrollmentPriorityGroup.GROUP_8.getCode()));
        onFile.getEnrollmentDetermination().setPrioritySubGroup(
                this.getLookupService().getEnrollmentPrioritySubGroupByCode(EnrollmentPrioritySubGroup.SUBGROUP_C.getCode()));        
        
        this.attachApplication(onFile);
        onFile.getApplication().setApplicationDate(this.getDate(2005, 4, 23));
        this.attachMeansTestWithMtCopayRequired(onFile, new BigDecimal(21),
                new BigDecimal(1));
        this.attachMilitaryServiceEpisode(onFile);
        this.attachConflictExperience(onFile, this.getDate(2005, 5, 23));

        /* disable since DMI does not insert Other codes
        Eligibility e = new Eligibility();
        e.setFactor(this.getLookupService().getEligibilityFactorByCode(EligibilityFactor.COMBAT_VETERAN.getCode()));
        onFile.getEnrollmentDetermination().addOtherEligibility(e);
        */
        Person incoming = savePerson(onFile);

        this.setComplete();
        System.out.println("Person id="
                + incoming.getPersonEntityKey().getKeyValueAsString());
    }            

    /**
     * RULE 15 Veteran is enrolled due to MT status, and a subsequent Income
     * Years Means Test was edited or added that would place the veteran in a
     * priority group not being enrolled, UNLESS RULE 15A The Means test on the
     * first verified enrollment record is edited to a Means Test Status that
     * places the record in a priority group not being enrolled and the veteran
     * has no subsequent record that would qualify for enrollment THEN the
     * veteran is NOT continuously enrolled and is placed in a Rejected
     * Enrollment Status. OR RULE 15 B The Means Test on the first verified
     * enrollment record is converted by IVM to a Means Test Status that places
     * the record in a priority group not being enrolled and the veteran has no
     * subsequent record that would qualify for enrollment THEN the veteran is
     * NOT continuously enrolled and is placed in a Rejected Enrollment Status.
     */     
    public void testRule15() throws Exception {
        Person onFile = this.getPerson("20459314");
        testRule15_Cases(onFile, true);        
    }
    
    public void testRule15_Neg() throws Exception {
        
        Person onFile = this.getPerson("20460708");
        this.attachMeansTestWithMtCopayRequired(onFile, new BigDecimal(21), new BigDecimal(1));        
        testRule15_Cases(onFile, false);
    }    
    


    private void testRule15_Cases(Person onFile, boolean posneg) throws Exception {
        onFile.getEligibilityVerification().setVerificationMethod((new Date()).toString());
        Person result = this.getEligibilityEnrollmentService().updateEligibilityData(onFile);
        if (posneg)
            assertTrue(result.getEnrollmentDetermination().getEnrollmentStatus()
                .getEnrollmentCategory().getCode().equals(
                        EnrollmentCategory.CODE_ENROLLED.getCode()));
        else
            assertTrue(result.getEnrollmentDetermination().getEnrollmentStatus()
                .getEnrollmentCategory().getCode().equals(
                        EnrollmentCategory.CODE_NOT_ENROLLED.getCode()));        
        System.out.println("Person id="
                + result.getPersonEntityKey().getKeyValueAsString());
    }
    
    
    public void testCreateRule15Data_2004Test_GMT_COPAY_REQ() throws Exception {
        createRule15TestData(new Integer(2004), this.getDate(2003,3,6), MeansTestStatus.MT_STATUS_GMT_COPAY_REQUIRED, EnrollmentPriorityGroup.GROUP_7, EnrollmentPrioritySubGroup.SUBGROUP_A);
    }
    public void testCreateRule21Data_GMT_COPAY_REQ() throws Exception {
        createRule21TestData(new Integer(2006), this.getDate(2006,3,6),
                MeansTestStatus.MT_STATUS_MT_COPAY_EXEMPT,
                MeansTestStatus.MT_STATUS_GMT_COPAY_REQUIRED,
                EnrollmentPriorityGroup.GROUP_7,
                EnrollmentPrioritySubGroup.SUBGROUP_A);
    }
    
    public void testCreateRule21Data_MT_COPAY_REQ() throws Exception {
        createRule21TestData(new Integer(2006), this.getDate(2006,3,6),
                MeansTestStatus.MT_STATUS_MT_COPAY_EXEMPT,
                MeansTestStatus.MT_STATUS_MT_COPAY_REQUIRED,
                EnrollmentPriorityGroup.GROUP_7,
                EnrollmentPrioritySubGroup.SUBGROUP_A);
    }        
    
    public void testCreateRule15Data_2004Test_PEND_ADJ() throws Exception {
        createRule15TestData(new Integer(2004), this.getDate(2003,3,6), MeansTestStatus.MT_STATUS_PENDING_ADJUDICATION, EnrollmentPriorityGroup.GROUP_7, EnrollmentPrioritySubGroup.SUBGROUP_C);
    }    
    
    public void testCreateRule15Data_2005Test_MT_COPAY_EX() throws Exception {
        createRule15TestData(new Integer(2005), this.getDate(2006,1,1), MeansTestStatus.MT_STATUS_MT_COPAY_EXEMPT, EnrollmentPriorityGroup.GROUP_5, null);
    }    
    
    private void createRule15TestData(Integer year, Date enEffective,
            MeansTestStatus.Code mtStatus, EnrollmentPriorityGroup.Code group,
            EnrollmentPrioritySubGroup.Code sub)  throws Exception {
        Person onFile = buildSimplePerson(); // bare-bones person
        onFile.addAddress(this.createAddress(this.getLookupService().getAddressTypeByCode(AddressType.CODE_PERMANENT_ADDRESS.getCode())));
        IncomeTest incomeTest = this.createIncomeTest(
                IncomeTestType.CODE_MEANS_TEST,
                mtStatus, year);
        incomeTest.setEffectiveDate(enEffective);
        IncomeTestStatus status = incomeTest.getIncomeTestStatus();
        status.setDeterminedStatus(this.getLookupService().getMeansTestStatusByCode(mtStatus.getCode()));        
        incomeTest.setCompletedDate(enEffective);
        incomeTest.setAgreesToPayDeductible(Boolean.TRUE);
        incomeTest.setDiscloseFinancialInformation(Boolean.TRUE);
        incomeTest.setModifiedOn(enEffective);

        onFile.setIncomeTest(year, incomeTest);
        this.attachApplication(onFile);
        this.attachVerifiedEnrollment(onFile);
        onFile.getEnrollmentDetermination().setPriorityGroup(this.getLookupService().getEnrollmentPriorityGroupByCode(group.getCode()));
        if (sub != null) {
            onFile.getEnrollmentDetermination().setPrioritySubGroup(this.getLookupService().getEnrollmentPrioritySubGroupByCode(sub.getCode()));
        }
        
        onFile.getEnrollmentDetermination().setEffectiveDate(enEffective);
        onFile.getEnrollmentDetermination().setModifiedOn(enEffective);
        this.setEligibilityStatusToVerified(onFile);        
        Person incoming = savePerson(onFile);
        

        this.setComplete();
        System.out.println("Person id="
                + incoming.getPersonEntityKey().getKeyValueAsString());        
    }
    
    private void createRule21TestData(Integer year, Date enEffective,
            MeansTestStatus.Code mtStatus, MeansTestStatus.Code testDeterminedStatus, EnrollmentPriorityGroup.Code group,
            EnrollmentPrioritySubGroup.Code sub)  throws Exception {
        Person onFile = buildSimplePerson(); // bare-bones person
        onFile.addAddress(this.createAddress(this.getLookupService().getAddressTypeByCode(AddressType.CODE_PERMANENT_ADDRESS.getCode())));
        IncomeTest incomeTest = this.createIncomeTest( IncomeTestType.CODE_MEANS_TEST, mtStatus, year);
        incomeTest.setEffectiveDate(enEffective);
        IncomeTestStatus status = incomeTest.getIncomeTestStatus();
        status.setDeterminedStatus(this.getLookupService().getMeansTestStatusByCode(testDeterminedStatus.getCode()));        
        incomeTest.setCompletedDate(enEffective);
        incomeTest.setAgreesToPayDeductible(Boolean.TRUE);
        incomeTest.setDiscloseFinancialInformation(Boolean.TRUE);
        incomeTest.setModifiedOn(enEffective);
        Hardship hardship = this.createHardship("test1");
        hardship.setEffectiveDate(this.getDate(2008, 0, 1));
        hardship.setHardshipGranted(Boolean.TRUE);
        incomeTest.setHardship(hardship);
        onFile.setIncomeTest(year, incomeTest);
        this.attachApplication(onFile);
        this.attachVerifiedEnrollment(onFile);
        onFile.getEnrollmentDetermination().setPriorityGroup(this.getLookupService().getEnrollmentPriorityGroupByCode(group.getCode()));
        if (sub != null) {
            onFile.getEnrollmentDetermination().setPrioritySubGroup(this.getLookupService().getEnrollmentPrioritySubGroupByCode(sub.getCode()));
        }
        
        onFile.getEnrollmentDetermination().setEffectiveDate(enEffective);
        onFile.getEnrollmentDetermination().setModifiedOn(enEffective);
        this.setEligibilityStatusToVerified(onFile);        
        Person incoming = savePerson(onFile);
        

        this.setComplete();
        System.out.println("Person id="
                + incoming.getPersonEntityKey().getKeyValueAsString());        
    }    

    public void testRule21Pos() throws Exception {
        rule21("100007995");
    }
    public void testRule21Neg() throws Exception {
        rule21("100007942");
        rule21("100008024");
    }    
    private void rule21(String id) throws Exception {
        PersonIdEntityKeyImpl personKey = new PersonIdEntityKeyImpl(new BigDecimal(id));        
        Person onFile = this.getPersonService().getPerson(personKey);              
        this.attachMTWithMtCopayRequired(onFile, new BigDecimal(1), new BigDecimal(1));
        Person incoming = savePerson(onFile);
        Person updated = this.getEligibilityEnrollmentService().updateEligibilityData(incoming);
       
        System.out.println("Person id="
                + updated.getPersonEntityKey().getKeyValueAsString());
    }    

    private void attachMTWithMtCopayRequired(Person person,
            BigDecimal gmtThresholdAmount, BigDecimal thresholdA)
            throws Exception {
        IncomeTest incomeTest = this.createIncomeTest(
                IncomeTestType.CODE_MEANS_TEST,
                MeansTestStatus.MT_STATUS_MT_COPAY_REQUIRED, new Integer("2007"));
        incomeTest.setGmtThresholdAmount(gmtThresholdAmount);
        incomeTest.setThresholdA(thresholdA);
        person.setIncomeTest(new Integer("2007"), incomeTest);
    }    
    
    public void updateRule15Data_2005Test() throws Exception {
        PersonIdEntityKeyImpl personKey = new PersonIdEntityKeyImpl(new BigDecimal("20459314"));        
        Person onFile = this.getPersonService().getPerson(personKey);              
        this.attachMeansTestWithMtCopayRequired(onFile, new BigDecimal(1), new BigDecimal(1));
        Person incoming = savePerson(onFile);
        this.getEligibilityEnrollmentService().updateEligibilityData(incoming);
        this.setComplete();
        System.out.println("Person id="
                + incoming.getPersonEntityKey().getKeyValueAsString());
    }
    
    public void updateRule15Data_2005TestStatus() throws Exception {
        PersonIdEntityKeyImpl personKey = new PersonIdEntityKeyImpl(new BigDecimal("20459660"));        
        Person onFile = this.getPersonService().getPerson(personKey);
        onFile.getIncomeTest(new Integer(2005)).setMeansTestStatus(this.getLookupService().getMeansTestStatusByCode(MeansTestStatus.MT_STATUS_MT_COPAY_REQUIRED.getCode()));
        Person incoming = savePerson(onFile);
        this.getEligibilityEnrollmentService().updateEligibilityData(incoming);
        this.setComplete();
        System.out.println("Person id="
                + incoming.getPersonEntityKey().getKeyValueAsString());
    }           
    
    /**
     * RULE 16 If Eligible for Medicaid is = Yes and it is changed to = No after
     * 365 days of the Enrollment Application Date, then the veteran is
     * continuously enrolled. (use Date Time Medicaid Last Updated in the
     * absence of the application date. Exit Continuous Enrollment Processing if
     * any of the Rules 9-16 have been met.
     */
    public void testRule16() throws Exception {
        Person onFile = this.getPerson("20005192");
        onFile.setMedicaidFactor(null);
        Person result = calculate(onFile);
        assertTrue(result.getEnrollmentDetermination().getEnrollmentStatus()
                .getEnrollmentCategory().getCode().equals(
                        EnrollmentCategory.CODE_ENROLLED.getCode()));
        System.out.println("Person id="
                + result.getPersonEntityKey().getKeyValueAsString());
    }

    public void testCreateRule16Data() throws Exception {
        Person onFile = buildSimplePerson(); // bare-bones person
        this.setEligibilityStatusToVerified(onFile);
        this.attachVerifiedEnrollment(onFile);
        onFile.getEnrollmentDetermination().setEffectiveDate(new Date());
        this.attachMedicaidFactor(onFile);
        this.attachMeansTestWithMtCopayRequired(onFile, new BigDecimal(21),
                new BigDecimal(1));
        Person incoming = savePerson(onFile);
        this.getEligibilityEnrollmentService().updateEligibilityData(incoming);
        this.setComplete();
        System.out.println("Person id="
                + incoming.getPersonEntityKey().getKeyValueAsString());
    }    

    public void testAgentOrangeEnteredDate() throws Exception {
        Person onFile = this.getPerson("20751514");
        AgentOrangeExposure aoe = onFile.getAgentOrangeExposure();
        if (aoe != null && aoe.getAgentOrangeExposureIndicator().toBoolean().booleanValue()) {
            System.out.println("modified date="  + aoe.getModifiedOn());
        }
        
    }
    
    public void testCreateAOData() throws Exception {
        Person onFile = buildSimplePerson(); // bare-bones person
        this.setEligibilityStatusToVerified(onFile);
        this.attachVerifiedEnrollment(onFile);
        this.attachMeansTestWithMtCopayRequired(onFile, new BigDecimal(21),
                new BigDecimal(1));
        Person incoming = savePerson(onFile);
        this.attachAgentOrangeExposure(incoming);
        incoming.getAgentOrangeExposure().setLocation(this.getLookupService().getAgentOrangeExposureLocationByCode(AgentOrangeExposureLocation.CODE_KOREAN_DMZ.getCode()));
        this.getEligibilityEnrollmentService().updateEligibilityData(incoming);
        this.setComplete();
        System.out.println("Person id="
                + incoming.getPersonEntityKey().getKeyValueAsString());
    }      
    
    
    public void testInputParameters() throws Exception {
        Person person = this.buildSimplePerson();
        FinancialStatement statement = this.createFinancialStatement();
        
        statement.addDependentFinancials(this.createDependentFinancials());
        statement.addSpouseFinancials(this.createSpouseFinancials());
        person.setFinancialStatement(new Integer(2005), statement);
        DemographicInputParameter param = new DemographicInputParameter();

        
        boolean result = param.hasInvalidSSNVerificationStatusForVetOrRelation(person);
        assertFalse(result);
        
        person.removeAllFinancialStatements();
        result = param.hasInvalidSSNVerificationStatusForVetOrRelation(person);
        assertFalse(result);
        
        SpouseFinancials spouse = this.createSpouseFinancials();
        DependentFinancials dependent = this.createDependentFinancials();
        
        spouse.getReportedOn().removeAllSsns();
        SSN ssn = this.createInvalidSSN();
        spouse.getReportedOn().addSsn(ssn);
        statement.removeSpouseFinancials();
        statement.removeAllDependentFinancials();
        statement.addSpouseFinancials(spouse);
        person.setFinancialStatement(new Integer(2005), statement);
        result = param.hasInvalidSSNVerificationStatusForVetOrRelation(person);
        assertTrue(result);
        
        statement.removeSpouseFinancials();
        dependent.getReportedOn().removeAllSsns();
        dependent.getReportedOn().addSsn(this.createInvalidSSN());
        statement.addDependentFinancials(dependent);
        result = param.hasInvalidSSNVerificationStatusForVetOrRelation(person);
        assertTrue(result);
        
        statement.removeSpouseFinancials();
        dependent.getReportedOn().removeAllSsns();
        dependent.getReportedOn().addSsn(this.createFollowUpSSN());
        statement.addDependentFinancials(dependent);
        param.setIncomingData(person);
        result = param.hasPseudoSSNReasonOfFollowup();
        assertTrue(result);
        
        statement.removeAllDependentFinancials();
        spouse = this.createSpouseFinancials();
        spouse.getReportedOn().removeAllSsns();
        spouse.getReportedOn().addSsn(this.createFollowUpSSN());
        statement.addSpouseFinancials(spouse);
        result = param.hasPseudoSSNReasonOfFollowup();
        assertTrue(result);                       
    }
    
    private SSN createInvalidSSN() throws Exception {
        SSN ssn = new SSN();
        String str = Long.toString(System.currentTimeMillis());
        String text = str.substring(str.length() - 9);
        ssn.setSsnText(text);
        ssn.setType(this.getLookupService().getSSNTypeByCode(SSNType.CODE_ACTIVE.getCode()));
        ssn.setSsaVerificationStatus(this.getLookupService().getSSAVerificationStatusByCode(SSAVerificationStatus.INVALID_PER_SSA.getCode()));
        return ssn;
    }
    
    private SSN createFollowUpSSN() throws Exception {
        SSN ssn = new SSN();
        String str = Long.toString(System.currentTimeMillis());
        String text = str.substring(str.length() - 9);
        ssn.setSsnText(text);
        ssn.setType(this.getLookupService().getSSNTypeByCode(SSNType.CODE_PSEUDO.getCode()));
        ssn.setPseudoSSNReason(this.getLookupService().getPseudoSSNReasonByCode(PseudoSSNReason.CODE_FOLLOWUP_REQD.getCode()));
        return ssn;
    }
    
    public void create600DPerson() throws Exception {

        Person onFile = buildSimplePerson(); // bare-bones person
        onFile.addAddress(this.createAddress(this.getLookupService().getAddressTypeByCode(AddressType.CODE_PERMANENT_ADDRESS.getCode())));
        this.attachVerifiedEnrollment(onFile);
        onFile.getEnrollmentDetermination().setEnrollmentStatus(
                this.getLookupService().getEnrollmentStatusByCode(
                        EnrollmentStatus.CODE_PENDING.getName()));
        this.setEligibilityStatusToPendingVerification(onFile);
        Person incoming = savePerson(onFile);
        this.setComplete();
        System.out.println("Person id="
                + incoming.getPersonEntityKey().getKeyValueAsString());        
    }    
    //21335500
    
    public void testGetcurrentTest() throws Exception {

        Person onFile = this.getPerson("21335500");
        assertNotNull(this.getPersonHelperService().getCurrentIncomeTest(onFile));
        assertTrue(this.getPersonHelperService().getCurrentIncomeTest(onFile).getType().getCode().equals(IncomeTestType.CODE_MEANS_TEST.getCode()));
    }        
}
