/********************************************************************
 * Copyright  2004 VHA. All rights reserved
 ********************************************************************/
package gov.va.med.esr.common.rule.service;

// Java classes
import java.math.BigDecimal;
import java.util.Date;

// ESR classes
import gov.va.med.esr.common.model.ee.Eligibility;
import gov.va.med.esr.common.model.ee.EnrollmentDetermination;
import gov.va.med.esr.common.model.ee.SHAD;
import gov.va.med.esr.common.model.lookup.EligibilityType;
import gov.va.med.esr.common.model.lookup.EligibilityFactor;
import gov.va.med.esr.common.model.person.Person;
import gov.va.med.esr.common.model.person.id.PersonIdEntityKeyImpl;
import gov.va.med.esr.common.util.RuleAbstractTestCase;
import gov.va.med.esr.common.persistent.history.RulesEnrollmentHistoryDAO;

/**
 * @author Carlos Ruiz
 * @version 1.0
 */
public class DetermineEligibilityParameterTest extends RuleAbstractTestCase {
    private RulesEnrollmentHistoryDAO rulesEnrollmentHistoryDAO = null;
    
    /**
     * @param testName
     */
    public DetermineEligibilityParameterTest(String testName) {
        super(testName);
    }
    public void testOtherIsAO() throws Exception {

        Person onFile = buildSimplePerson();
        this.attachAgentOrangeExposure(onFile);
        Person incoming = savePerson(onFile);
        
        Person result = calculate(incoming);
        this.assessIfHasOtherRanking(EligibilityFactor.AGENT_ORANGE_EXPOSURE.getName(),
                result);
    }
    public void testOthersMatch() throws Exception {
        EnrollmentDetermination ed = this.createEnrollmentDetermination();
        Eligibility e = this.getPersonHelperService().createOtherEligibility(EligibilityFactor.AGENT_ORANGE_EXPOSURE.getCode());
        ed.addOtherEligibility(e);
        Eligibility found = (Eligibility)this.getMergeRuleService().getMatchRuleService()
        .findMatchingElement(e, ed.getOtherEligibilities());
        assertNotNull(found);
    }
    
    public void testOtherAsia() throws Exception {

        Person onFile = buildSimplePerson();
        this.attachEnvironmentalContaminationExposure(onFile);
        Person incoming = savePerson(onFile);
        
        Person result = calculate(incoming);
        this.assessIfHasOtherRanking(EligibilityFactor.ENVIRONMENTAL_CONTAMINANTS.getName(),
                result);
    }
    
    public void testOtherRadiation() throws Exception {

        Person onFile = buildSimplePerson();
        this.attachRadiationExposure(onFile);
        Person incoming = savePerson(onFile);
        
        Person result = calculate(incoming);
        this.assessIfHasOtherRanking(EligibilityFactor.RADIATION_EXPOSURE.getName(),
                result);
    }    

    public void testOtherMedicaid() throws Exception {
        Person onFile = buildSimplePerson();
        
        this.attachMedicaidFactor(onFile);
        Person incoming = savePerson(onFile);        
        
        Person result = calculate(incoming);
        this.assessIfHasOtherRanking(EligibilityFactor.MEDICAID.getName(), result);        
    }
    
    public void testOtherCd() throws Exception {
        Person onFile = buildSimplePerson();
        
        this.attachCatastrophicDisability(onFile);
        Person incoming = savePerson(onFile);        
        
        Person result = calculate(incoming);
        this.assessIfHasOtherRanking(EligibilityFactor.CATASTROPHICALLY_DISABLED.getName(), result);        
    }
    
    public void testOtherCv() throws Exception {
        Person onFile = buildSimplePerson();
        
        this.attachConflictExperience(onFile, this.getDate(2006, 9, 23));
        Person incoming = savePerson(onFile);        
        
        Person result = calculate(incoming);
        this.assessIfHasOtherRanking(EligibilityFactor.COMBAT_VETERAN.getCode(), result);        
    }
    
    public void testOtherHardship() throws Exception {
        Person onFile = buildSimplePerson();
        
        this.attachMeansTestWithHardship(onFile);
        Person incoming = savePerson(onFile);        
        
        Person result = calculate(incoming);
        this.assessIfHasOtherRanking(EligibilityFactor.HARDSHIP_GRANTED.getCode(), result);        
    }
    
    public void testOtherNtr() throws Exception {
        Person onFile = buildSimplePerson();
        
        this.attachNoseThroatRadium(onFile);
        Person incoming = savePerson(onFile);        
        
        Person result = calculate(incoming);
        this.assessIfHasOtherRanking(EligibilityFactor.NOSE_THROAT_RADIUM.getCode(), result);        
    }            
    
    public void testOtherMst() throws Exception {
        Person onFile = buildSimplePerson();
        
        this.attachMilitarySexualTrauma(onFile);
        Person incoming = savePerson(onFile);        
        
        Person result = calculate(incoming);
        this.assessIfHasOtherRanking(EligibilityFactor.MILITARY_SEXUAL_TRAUMA.getCode(), result);        
    }                
    
    public void testOtherShad() throws Exception {
        Person onFile = buildSimplePerson();
        
        onFile.setShad(new SHAD());
        Person incoming = savePerson(onFile);        
        
        Person result = calculate(incoming);
        this.assessIfHasOtherRanking(EligibilityFactor.SHAD.getCode(), result);        
    }                
    public void TestEarliestVerifiedEnrollmentDate() throws Exception {
        PersonIdEntityKeyImpl personKey = new PersonIdEntityKeyImpl(new BigDecimal("72096"));        
        Date date = this.getEligibilityEnrollmentService().getEnrollmentDateForEarliestVerifiedUnlessCancelled(personKey);
        assertNotNull(date);
        System.out.println("date=" + date.toString());        
    }


    
    public void testPrimaryAidandAttendance() throws Exception {
        // PrimaryEligibilityIsA&A- if SC indicator is No or null and SC% is
        // Null
        // and the Aid and attendance indicator is equal to Yes then set the
        // primary eligibility equal to Aid and Attendance
        Person onFile = buildSimplePerson();
     
        this.attachAidAndAttendance(onFile);
        Person incoming = savePerson(onFile);        
        
        Person result = calculate(incoming);
        this.assessIfHasPrimaryRanking(
                EligibilityType.AID_AND_ATTENDANCE.getName(), result);
    }
    
    public void testPrimaryCodeIsAlliedVeteran() throws Exception {
        Person onFile = buildSimpleNonVeteran();
        
       // PrimaryCodeIsAlliedVeteran - If the SC indicator is no or null and
        // SC%
        // is null and has a eligibility code equal to Allied Veteran then set
        // the primary eligibility code equal to Allied Veteran
        this.attachAlliedVeteranAsPrimary(onFile);
        Person incoming = savePerson(onFile);
        
        Person result = calculate(incoming);
        this.assessIfHasPrimaryRanking(
                EligibilityType.ALLIED_VETERAN.getName(), result);
    }    

    public void testPrimaryCodeIsChampVA() throws Exception {
        // PrimaryCodeIsChampVA- If the SC is no or null and SC% is null and has
        // elgibility code received is equal to CHAMPVA then set the primary
        // eligibility code equal to ChampVA
        Person onFile = buildSimpleNonVeteran();
        this.attachChampVAAsPrimary(onFile);
        Person incoming = savePerson(onFile);
        
        Person result = calculate(incoming);
        this.assessIfHasPrimaryRanking(EligibilityType.CHAMPVA.getName(),
                result);
    }
    
    public void testPrimaryCodeIsCollateralOfVet() throws Exception {
        //  PrimaryCodeIsCollateralOfVet- If the SC is no or null and SC% is null
        // and eligibility code received is equal to Collateral Of Veteran
        // then
        // set the primary eligibility code equal to Collateral Of Vet
        Person onFile = buildSimpleNonVeteran();
        this.attachCollateralOfVeteran(onFile);
        Person incoming = savePerson(onFile);
        Person result = calculate(incoming);
        this.assessIfHasPrimaryRanking(
                EligibilityType.COLLATERAL_OF_VETERAN.getName(), result);
    }
    
    public void testPrimaryCodeIsEmployee() throws Exception {
        // PrimaryCodeIsEmployee- if the SC indicator is no or null and SC% is
        // null and has received eligibility code equal to employee then set
        // the
        // primary eligibility code equal to employee and set the patient
        // sensitive flag equal to yes
        Person onFile = this.buildSimpleNonVeteran();
        this.attachEmployee(onFile);
        this.attachCollateralOfVeteranAsPrimary(onFile);
        Person incoming = savePerson(onFile);
        
        Person result = calculate(incoming);
        this.assessIfHasPrimaryRanking(EligibilityType.EMPLOYEE.getName(),
                result);
    }
    
    public void testPrimaryCodeIsHE() throws Exception {
        // PrimaryCodeIsHE- if SC indicator is equal to no and SC% is null and
        // Ineligible date is not null then set the primary eligibility code
        // equal
        // to Humanitarian Emergency
        Person onFile = buildSimpleNonVeteran();
        this.attachIneligibleFactor(onFile);
        Person incoming = savePerson(onFile);
        
        Person result = calculate(incoming);
        this.assessIfHasPrimaryRanking(
                EligibilityType.HUMANTARIAN_EMERGENCY.getName(), result);
    }
    
    public void testPrimaryCodeIsHEWithNoIneligDate() throws Exception {
        // PrimaryCodeIsHE- if non-vet and inelig is null and received primary is humanitarian
        Person onFile = buildSimpleNonVeteran();
        onFile.setReceivedPrimaryEligiblity(this.createReceivedEligibility(EligibilityType.HUMANTARIAN_EMERGENCY));
        Person incoming = savePerson(onFile);
        
        Person result = calculate(incoming);
        this.assessIfHasPrimaryRanking(
                EligibilityType.HUMANTARIAN_EMERGENCY.getName(), result);
    }    
    
    public void testPrimaryCodeIsHE_SinceNoCode() throws Exception {
        // Last resort code by row 21 equal to Humanitarian Emergency
        Person onFile = buildSimpleNonVeteran();
        Person incoming = savePerson(onFile);
        
        Person result = calculate(incoming);
        this.assessIfHasPrimaryRanking(
                EligibilityType.HUMANTARIAN_EMERGENCY.getName(), result);
    }    
    
    public void testPrimaryCodeIsSC50And100() throws Exception {
        // PrimaryCodeIsSC50And!00 - If SC% is between 50% and 100% then set the
        // primary eligibility code equal to service connected 50% to 100%
        Person onFile = buildSimplePerson();
        this.attachServiceConnectionAward(onFile, new Integer(50));
        Person incoming = savePerson(onFile);
        
        Person result = calculate(incoming);
        this.assessIfHasPrimaryRanking(EligibilityType.SERVICE_CONNECTED_50_TO_100_PERCENT.getName(),
                result);
    }
    
    public void testPrimaryCodeIsSCLessThan50() throws Exception {
        // PrimaryEligibilityCodeIsSCLessThan50- if the SC% is between 1 and 49
        // then set the primary eligibility code equal to Service connected
        // less
        // than 50%
        Person onFile = buildSimplePerson();
        this.attachServiceConnectionAward(onFile, new Integer(49));
        Person incoming = savePerson(onFile);
        
        Person result = calculate(incoming);
        this.assessIfHasPrimaryRanking(
                EligibilityType.SC_LESS_THAN_50_PERCENT.getName(), result);
    }
    
    public void testPrimaryEligibilityIsPOW() throws Exception {
        // PrimaryEligibilityIsPOW- If the SC is no or null and SC% is null and
        // POW is Yes then set the primary eligibility code equal to prisoner
        // of war
        Person onFile = buildSimplePerson();
        this.attachPOW(onFile);
        Person incoming = savePerson(onFile);
        
        Person result = calculate(incoming);
        this.assessIfHasPrimaryRanking(
                EligibilityType.PRISONER_OF_WAR.getName(), result);
    }
    
    public void testPrimaryEligibilityIsPH() throws Exception {
        //    PrimaryEligibilityIsPH- if SC is no or null and SC% is null and PH
        // indicator is YES and purple heart status is either Pending or In
        // process or Confirmed or initial Letter sent then set the primary
        // eligibility code equal to Purple Heart Recipent
        Person onFile = buildSimplePerson();
        this.attachPurpleHeartWithPHStatusPending(onFile);
        Person incoming = savePerson(onFile);
        
        Person result = calculate(incoming);
        this.assessIfHasPrimaryRanking(EligibilityType.PURPLE_HEART_RECIPIENT.getName(), result);
    }
    
    public void testPrimaryEligibilityIsHouseBound() throws Exception {

        //    PrimaryEligibilityIsHouseBound- if SC indicator is No or is null and
        // SC% is null and HouseBound is equal to YES then set the primary
        // eligibility code equal to HouseBound
        Person onFile = buildSimplePerson();
        this.attachHousebound(onFile);
        Person incoming = savePerson(onFile);
        
        Person result = calculate(incoming);
        this.assessIfHasPrimaryRanking(EligibilityType.HOUSEBOUND.getName(),
                result);
    }
    
    public void testPrimaryEligibilityIsNSCAndVAPension() throws Exception {
        //    PrimaryEligibilityIsNSCAndVAPension- if SC indicator is No or is null
        // and SC% is null and VA Pension is equal to YES then set the primary
        // eligibility code equal to NSC, VA Pension
        Person onFile = buildSimplePerson();
        this.attachVaPension(onFile);
        Person incoming = savePerson(onFile);
        
        Person result = calculate(incoming);
        this.assessIfHasPrimaryRanking(
                EligibilityType.NSC_VA_PENSION.getName(), result);
    }
    
    public void testPrimaryElgibilityIsWWI() throws Exception {
        // PrimaryElgibilityIsWWI- if the SC is No or null and SC% is null and
        // World war I eligibility code received from the site is true and date
        // of
        // birth is on or before 12/31/1907 or if SC indicator is No or null and
        // SC% is Null and period of service is equal to world war I and date
        // of
        // birth is on or before 12/31/1907 then set the primary eligibility
        // code
        // equal to World War I
        Person onFile = buildSimplePerson();
        Person incoming = savePerson(onFile);
        this.attachWWIAsPrimary(incoming);
        
        Person result = calculate(incoming);
        this.assessIfHasPrimaryRanking(EligibilityType.WORLD_WAR_I.getName(),
                result);
    }
    
    public void testPrimaryEligibilityIsMexicanWar() throws Exception {
        // PrimaryEligibilityIsMexicanWar- if the SC indicator is No or null and
        // SC% is null and has a Mexican Border War Eligibility code and date of
        // birth is on or before 12/31/1906 then the primary eligibility code is
        // equal to  Mexican Border War
        Person onFile = buildSimplePerson();
        Person incoming = savePerson(onFile);
        this.attachMexicanBorderWarAsPrimary(incoming);
        
        Person result = calculate(incoming);
        this.assessIfHasPrimaryRanking(
                EligibilityType.MEXICAN_BORDER_WAR.getName(), result);
    }
    public void testPrimaryEligibilityTriCareAndSharing() throws Exception {
        // PrimaryEligibilityIsTriCare- if the SC indicator is No or null and
        // SC%
        // is null and has a Tricare Eligibility code then set the primary
        // eligibility code equal to TRICARE
        Person onFile = buildSimpleNonVeteran();
        this.attachSharingAgreement(onFile);
        this.attachTriCare(onFile);
        Person incoming = savePerson(onFile);
        
        Person result = calculate(incoming);
        this.assessIfHasPrimaryRanking(
                EligibilityType.SHARING_AGREEMENT.getName(), result);
        this.assessIfHasSecondaryRanking(
                EligibilityType.TRICARE_CHAMPUS.getName(), result);
    }    
    public void testPrimaryEligibilityIsTriCare() throws Exception {
        // PrimaryEligibilityIsTriCare- if the SC indicator is No or null and
        // SC%
        // is null and has a Tricare Eligibility code then set the primary
        // eligibility code equal to TRICARE
        Person onFile = buildSimpleNonVeteran();
        this.attachTriCareAsPrimary(onFile);
        Person incoming = savePerson(onFile);
        
        Person result = calculate(incoming);
        this.assessIfHasPrimaryRanking(
                EligibilityType.TRICARE_CHAMPUS.getName(), result);
    }
    
    public void PrimaryCodeIsSharingAgreement() throws Exception {
        // PrimaryCodeIsSharingAgreement- If the SC indicator is No or null and
        // SC% is null and has a eligibility code equal to sharing agreement
        // eligibility then set the primary eligibility code equal to sharing
        // agreement
        Person onFile = buildSimplePerson();
        this.attachSharingAgreementAsPrimary(onFile);
        Person incoming = savePerson(onFile);
        
        Person result = calculate(incoming);
        this.assessIfHasPrimaryRanking(
                EligibilityType.SHARING_AGREEMENT.getName(), result);
    }
    
    public void testPrimaryCodeIsOtherFederalAgency() throws Exception {
        // PrimaryCodeIsOtherFederalAgency- If the SC is no or null and SC% is
        // null and eligibility code received is equal to Other Federal Agency
        // then set the primary eligibility code equal to Other Federal Agency
        // Else the primary care is equal to NSC
        Person onFile = buildSimpleNonVeteran();
        this.attachOtherFederalAgencyAsPrimary(onFile);
        Person incoming = savePerson(onFile);
        
        Person result = calculate(incoming);
        this.assessIfHasPrimaryRanking(
                EligibilityType.OTHER_FEDERAL_AGENCY.getName(), result);
    }    
    
    

    public void testAddSecondaryPURPLE_HEART_RECIPIENT() {

        try {
            Person onFile = buildSimplePerson(); // bare-bones person
            // SecondaryEligibilityCodeIsPH  If Purple heart indicator is YES
            // and
            // Purple heart status is equal to Pending  or in process or
            // confirmed or
            // initial Letter Sent and primary eligibility is not equal to
            // purple heart
            // then set the secondary eligibility code equal to purple heart
            // recipient
            this.attachServiceConnectionAward(onFile, new Integer(75));            
            this.attachPurpleHeartWithPHStatusPending(onFile);
            Person incoming = savePerson(onFile);            
            Person result = calculate(incoming);
            
            this.assessIfHasSecondaryRanking(
                    EligibilityType.PURPLE_HEART_RECIPIENT.getName(),
                    result);

            logger.info("Completed adding Secondary Test");
        } catch (Exception e) {
            fail("test failed due to exception: " + e);
        }
    }        

    // SecondaryEligibilityCodeIsPOW- If POW indicator is YES and
    // primary
    // eligibility is not
    // equal to POW then set the secondary eligibility code equal to
    // POW
    public void testAddSecondaryPRISONER_OF_WAR() {

        try {
            Person onFile = buildSimplePerson(); // bare-bones person
            this.attachServiceConnectionAward(onFile, new Integer(75));
            this.attachPOW(onFile);
            
            Person incoming = savePerson(onFile);            
            Person result = calculate(incoming);            
            
            this.assessIfHasSecondaryRanking(
                    EligibilityType.PRISONER_OF_WAR.getName(), result);
            
            logger.info("Completed adding Secondary Test");
        } catch (Exception e) {
            fail("test failed due to exception: " + e);
        }
    }        

    // SecondaryEligibilityCodeIsHouseBound- If HouseBound indicator is
    // YES
    // and
    // primary eligibility is not equal to House Bound then set the
    // secondary eligibility code equal
    // to HouseBound    
    public void testAddSecondaryHOUSEBOUND() {

        try {
            Person onFile = buildSimplePerson(); // bare-bones person
            
            this.attachServiceConnectionAward(onFile, new Integer(75));
            this.attachHousebound(onFile);
            
            Person incoming = savePerson(onFile);            
            Person result = calculate(incoming);
            
            this.assessIfHasSecondaryRanking(
                    EligibilityType.HOUSEBOUND.getName(), result);
            
            logger.info("Completed adding Secondary Test");
        } catch (Exception e) {
            fail("test failed due to exception: " + e);
        }
    }        
    
    // SecondaryEligibilityCodeIsA&A- If Aid and attendance indicator is
    // YES and
    // primary eligibility is not equal to Aide and attendance then
    // set
    // the secondary
    // eligibility code equal to Aid And Attendance    
    public void testAddSecondaryAID_AND_ATTENDANCE() {

        try {
            Person onFile = buildSimplePerson(); // bare-bones person
            this.attachServiceConnectionAward(onFile, new Integer(75));
            this.attachAidAndAttendance(onFile);
            onFile.getMonetaryBenefitAward().setCheckAmount(
                    new BigDecimal(20));
            
            Person incoming = savePerson(onFile);            
            Person result = calculate(incoming);
            
            this.assessIfHasSecondaryRanking(
                    EligibilityType.AID_AND_ATTENDANCE.getName(), result);
            
            logger.info("Completed adding Secondary Test");
        } catch (Exception e) {
            fail("test failed due to exception: " + e);
        }
    }        
    // SecondaryEligibilityCodeIsEmployee- If eligibility code received
    // from the site is equal
    // to Employee and primary eligibility code is not equal to
    // Employee then set the
    // secondary eligibility code equal to Employee and Patient
    // Sensitive flag to yes
    public void testAddSecondaryEMPLOYEE() {

        try {
            Person onFile = buildSimplePerson(); // bare-bones person
            this.attachEmployee(onFile);        
            Person incoming = savePerson(onFile);            
            Person result = calculate(incoming);
            
            this.assessIfHasSecondaryRanking(
                    EligibilityType.EMPLOYEE.getName(), result);

            logger.info("Completed adding Secondary Test");
        } catch (Exception e) {
            fail("test failed due to exception: " + e);
        }
    }        

    // SecondaryEligibilityCodeIsAlliedVeteran- If eligibility code
    // received
    // from the site is equal to Allied Veteran and primary
    // eligibility
    // code
    // is not equal to Allied Veteran then set the secondary
    // eligibility
    // code equal to Allied Veteran
    public void testAddSecondaryALLIED_VETERAN() {

        try {
            Person onFile = buildSimplePerson(); // bare-bones person
            this.attachAlliedVeteran(onFile);
            
            Person incoming = savePerson(onFile);            
            Person result = calculate(incoming);            
            
            this.assessIfHasSecondaryRanking(
                    EligibilityType.ALLIED_VETERAN.getName(), result);

            logger.info("Completed adding Secondary Test");
        } catch (Exception e) {
            fail("test failed due to exception: " + e);
        }
    }        

    // SecondaryEligibilityCodeIsWWI- If (Eligibility code received from
    // the site is equal to World War I or
    // period of service is world war I) and Date of Birth is on or
    // before 12/31/1907 and primary
    // eligibility is not equal to World War I then set the secondary
    // eligibility equal to World war I    
    public void testAddSecondaryWORLD_WAR_I() {

        try {
            Person onFile = buildSimplePerson(); // bare-bones person
            Person incoming = savePerson(onFile);
            this.attachServiceConnectionAward(incoming, new Integer(75));            
            this.attachWWI(incoming);            
            Person result = calculate(incoming);
            
            this.assessIfHasSecondaryRanking(
                    EligibilityType.WORLD_WAR_I.getName(), result);

            logger.info("Completed adding Secondary Test");
        } catch (Exception e) {
            fail("test failed due to exception: " + e);
        }
    }        

    // SecondaryEligibilityCodeIsTriCare- - If eligibility code received
    // from the site is equal to TRICARE and primary eligibility
    // code is not equal to TRICARE then set the secondary
    // eligibility code equal to TRICARE
    public void testAddSecondaryTRICARE_CHAMPUS() {

        try {
            Person onFile = buildSimplePerson();
            this.attachTriCare(onFile);
            
            Person incoming = savePerson(onFile);            
            Person result = calculate(incoming);
            
            this.assessIfHasSecondaryRanking(
                    EligibilityType.TRICARE_CHAMPUS.getName(), result);

            logger.info("Completed adding Secondary Test");
        } catch (Exception e) {
            fail("test failed due to exception: " + e);
        }
    }            
    
    // SecondaryEligibilityCodeIsSharingAgreement- If eligibility code
    // received from the site is equal
    // to Sharing Agreement and primary eligibility code is not equal
    // to
    // Sharing Agreement then set
    // the secondary eligibility code equal to Sharing Agreement    
    public void testAddSecondarySHARING_AGREEMENT() {

        try {
            Person onFile = buildSimplePerson();
            this.attachSharingAgreement(onFile);            
            Person incoming = savePerson(onFile);            
            Person result = calculate(incoming);
            
            this.assessIfHasSecondaryRanking(
                    EligibilityType.SHARING_AGREEMENT.getName(), result);
            
            logger.info("Completed adding Secondary Test");
        } catch (Exception e) {
            fail("test failed due to exception: " + e);
        }
    }   

    // SecondaryEligibilityCodeIsChampVA- If eligibility code received
    // from
    // the site is equal to
    // ChampVA and primary eligibility code is not equal to ChampVA
    // then set the secondary
    // eligibility code equal to ChampVA
    public void testAddSecondaryCHAMPVA() {

        try {
            Person onFile = buildSimplePerson();
           
            this.attachChampVA(onFile);            
            Person incoming = savePerson(onFile);            
            Person result = calculate(incoming);
            
            this.assessIfHasSecondaryRanking(EligibilityType.CHAMPVA.getName(),
                    result);
            
            logger.info("Completed adding Secondary Test");
        } catch (Exception e) {
            fail("test failed due to exception: " + e);
        }
    }
    
    // SecondaryEligibilityIsCollateralOfVet- If eligibility code
    // received
    // from the site is equal to
    // Collateral Of Veteran and primary eligibility code is not equal
    // to
    // Collateral Of Veteran then
    // set the secondary eligibility code equal to Collateral of
    // Veteran
    
    public void testAddSecondaryCOLLATERAL_OF_VETERAN() {

        try {
            Person onFile = buildSimplePerson();
            
            this.attachCollateralOfVeteran(onFile);
            
            Person incoming = savePerson(onFile);            
            Person result = calculate(incoming);
            
            this.assessIfHasSecondaryRanking(
                    EligibilityType.COLLATERAL_OF_VETERAN.getName(),
                    result);
            
            logger.info("Completed adding Secondary Test");
        } catch (Exception e) {
            fail("test failed due to exception: " + e);
        }
    }          

    // SecondaryEligibilityIsOtherFederal- If eligibility code received
    // from the site is equal
    // to Other Federal Agency and primary eligibility code is not
    // equal
    // to Other Federal Agency then
    // set the secondary eligibility code equal to Other Federal
    // Agency
    public void testAddSecondaryOTHER_FEDERAL_AGENCY() {

        try {
            Person onFile = buildSimplePerson(); // bare-bones person
           
            this.attachOtherFederalAgency(onFile);
            
            Person incoming = savePerson(onFile);            
            Person result = calculate(incoming);
            
            this.assessIfHasSecondaryRanking(
                    EligibilityType.OTHER_FEDERAL_AGENCY.getName(),
                    result);
            
            logger.info("Completed adding Secondary Test");
        } catch (Exception e) {
            fail("test failed due to exception: " + e);
        }
    }          

    // If eligibility code received from the site is equal to Mexican
    // Border War and
    // date of birth is on or before 12/31/1906 and primary eligibility
    // code
    // is not equal to Mexican Border War then set the secondary
    // eligibility
    // code equal to Mexican Border War
    public void testAddSecondaryMEXICAN_BORDER_WAR() {

        try {
            Person onFile = buildSimplePerson(); // bare-bones person
            Person incoming = savePerson(onFile);            
            this.attachServiceConnectionAward(incoming, new Integer(75));
            
            this.attachMexicanBorderWar(incoming);
            
            Person result = calculate(incoming);
            this.assessIfHasSecondaryRanking(
                    EligibilityType.MEXICAN_BORDER_WAR.getName(), result);
            
            logger.info("Completed adding Secondary Test");
        } catch (Exception e) {
            fail("test failed due to exception: " + e);
        }
    }          
    
    public void testAddSecondaryHUMANTARIAN_EMERGENCY() {

        try {
            Person onFile = buildSimplePerson(); // bare-bones person
            this.attachServiceConnectionAward(onFile, new Integer(75));

            
            // Setup for Humanitarian
            this.attachServiceConnectionAward(onFile, new Integer(0));
            this.attachIneligibleFactor(onFile);
            Person incoming = savePerson(onFile);            
            Person result = calculate(incoming);
            // SecondaryEligibilityCodeIsHumantarian- If the SC indicator is
            // yes
            // and SC% is 0 and ineligible
            // date is not null and primary eligibility is not equal to
            // humanitarian emergency then set the
            // secondary eligibility code equal to Humanitarian Emergency
            this.assessIfHasSecondaryRanking(
                    EligibilityType.HUMANTARIAN_EMERGENCY.getName(),
                    result);
            
            
            logger.info("Completed adding Secondary Test");
        } catch (Exception e) {
            fail("test failed due to exception: " + e);
        }
    }              
   
    /**
     * @return Returns the rulesEnrollmentHistoryDAO.
     */
    public RulesEnrollmentHistoryDAO getRulesEnrollmentHistoryDAO() {
        return rulesEnrollmentHistoryDAO;
    }



    /**
     * @param rulesEnrollmentHistoryDAO The rulesEnrollmentHistoryDAO to set.
     */
    public void setRulesEnrollmentHistoryDAO(
            RulesEnrollmentHistoryDAO rulesEnrollmentHistoryDAO) {
        this.rulesEnrollmentHistoryDAO = rulesEnrollmentHistoryDAO;
    }
    
    
}