/********************************************************************
 * Copyright  2004 VHA. All rights reserved
 ********************************************************************/
package gov.va.med.esr.common.rule.service;

// Java classes

// Library classes
import org.apache.commons.lang.Validate;

// Framework classes
import gov.va.med.fw.validation.ValidationFieldMessage;
import gov.va.med.fw.validation.ValidationMessages;

// ESR classes
import gov.va.med.esr.common.infra.ImpreciseDateUtils;
import gov.va.med.esr.common.model.ee.MilitaryService;
import gov.va.med.esr.common.model.ee.MilitaryServiceEpisode;
import gov.va.med.esr.common.model.ee.MilitaryServiceSiteRecord;
import gov.va.med.esr.common.model.ee.ConflictExperience;
import gov.va.med.esr.common.model.person.Person;
import gov.va.med.esr.common.model.person.DeathRecord;
import gov.va.med.esr.common.model.lookup.ConflictLocation;
import gov.va.med.esr.common.model.lookup.VAFacility;
import gov.va.med.esr.common.rule.service.impl.RuleValidationMessage;
import gov.va.med.esr.common.util.RuleAbstractTestCase;


/**
 * Provides methods to test BAL rules in ConflictExperienceValidation rule flow
 *
 * Project: Common</br>
 * Created on: 4:45:47 PM </br>
 *
 * @author DNS   LEV
 */
public class ConflictExperienceValidationTest extends RuleAbstractTestCase {

	/**
	 * An instance of onFile
	 */
	private Person onFile = null;

	/**
	 * constructor
	 */
	public ConflictExperienceValidationTest(String testName) throws Exception {
		super(testName);
	}

	/** Tests doesConflictOverlap BAL rule in ConflictExperienceValidation.irl
	 * @throws Exception Thrown in case of errors
	 */
	public void testdoesConflictOverlap() throws Exception {
		
		/** 3323[UC23.22] Multiple Conflict Locations
		 * The actor is allowed to enter more than one Conflict during which he/she served.  
		 * However, the dates cannot overlap his/her service in the same or another Conflict Location.
		 * 
		 * 3324[UC23.22.1] If the actor enters overlapping dates the system displays this message: 
		 * "Conflict Location Dates cannot overlap." 
		 * 
		 * IF The received conflict record overlaps with another conflict record 
		 * THEN Generate Error Message "Conflict Location Dates cannot overlap."
		 **/
		// Persist a person so we will have a record on file
		Person pristine = this.getPersonService().save( onFile );
		
		// Create a military service record
		Person working = (Person)onFile.clone();
		this.attachMilitaryServiceEpisode( working );

		// Create 2 mes for a period from 02/20/1968 to 04/30/1990
		ConflictExperience ce = this.attachConflictExperience( working );
		ce.setStartDate( ImpreciseDateUtils.createImpreciseDate( 1968,2,20) );
		ce.setEndDate( ImpreciseDateUtils.createImpreciseDate( 1975,4,30) );
	
		MilitaryServiceSiteRecord mssr = ce.getMilitaryServiceSiteRecord();
		ce = new ConflictExperience();
		ce.setStartDate( ImpreciseDateUtils.createImpreciseDate( 1978,2,20) );
		ce.setEndDate( ImpreciseDateUtils.createImpreciseDate( 1990,4,30) );
		mssr.addConflictExperience( ce );
		
		// Create an overlapping mse here 02/20/1970 to 04/30/1980
		ce = new ConflictExperience();
		ce.setStartDate( ImpreciseDateUtils.createImpreciseDate( 1974,2,20) );
		ce.setEndDate( ImpreciseDateUtils.createImpreciseDate( 1980,4,30) );
		mssr.addConflictExperience( ce );
		
		try {
			ValidationMessages messages = 
				this.getRuleValidationService().validateConflictExperience( ce, working, pristine, false );

			assertTrue( !messages.isEmpty() );
			assertEquals( ((ValidationFieldMessage)messages.get().next()).getKey(), 
							  RuleValidationMessage.INVALID_CONFLICT_EXPERIENCES_CANNOT_OVERLAP.getName() );
		}
		catch( Exception e ) {
			if( logger.isDebugEnabled() ) {
				logger.debug( "Got an un-expected validation exception ", e );
			}
			this.fail( "Got an un-expected validation exception", e );
		}
	}
	
	/** Tests isConflictExperienceNotAllowed BAL rule in ConflictExperienceValidation.irl
	 * @throws Exception Thrown in case of errors
	 */
	public void testisConflictExperienceNotAllowed() throws Exception {
		
		/** 3253[UC23.19] Conflict From Date 
		 * The actor is not allowed to enter the Conflict From Date unless 
		 * there is at least one military service episode. 
		 * 
		 * IF ( The received conflict from date for the veteran is provided OR
		 * The received conflict to date for the veteran is provided OR 
		 * The received conflict location for the veteran is provided ) 
		 * AND There are no received military service episode for the veteran 
		 * THEN Generate Error Message "Entry of conflict information requires at 
		 * least one military service episode."
		 **/		
		// Create a military service record
		Person pristine = this.getPersonService().save( onFile );

		// Create a working version
		Person working = (Person)pristine.clone();

		ConflictExperience ce = this.attachConflictExperience( working, this.getConflictLocation( ConflictLocation.CODE_VIETNAM ) );
		ce.setStartDate( ImpreciseDateUtils.createImpreciseDate( 2002, 1, 1) );

		try {
			ValidationMessages messages = 
				this.getRuleValidationService().validateConflictExperience( ce, working, pristine, false );
			
			assertTrue( !messages.isEmpty() );
			assertEquals( ((ValidationFieldMessage)messages.get().next()).getKey(), 
							  RuleValidationMessage.MSE_REQUIRED_FOR_CONFLICT_ENTRY.getName() );
		}
		catch( Exception e ) {
			if( logger.isDebugEnabled() ) {
				logger.debug( "Got an un-expected validation exception ", e );
			}
			this.fail( "Got an un-expected validation exception", e );
		}
	}
	
	/** Tests isConflictNotforOneMSE BAL rule in ConflictExperienceValidation.irl
	 * @throws Exception Thrown in case of errors
	 */
	public void testisConflictNotforOneMSE() throws Exception {
		
		/** 3260[UC23.19.3] 
		 * The actor must enter a Conflict From Date and To Date that must fall within a single MSE. 
		 * 
		 * 3261[UC23.19.3.1] 
		 * If the actor enters a Conflict From Date or To Date that does not fall within the 
		 * prescribed range, the system displays this message: 
		 * "The Conflict From Date or To Date must fall within a single MSE."
		 * 
		 * IF the received conflict from date for the veteran is provided AND 
		 * the received conflict to date for the veteran is provided AND 
		 * ( The received conflict from date for the veteran is less than 
		 * the service entry date for a military service episode OR 
		 * The received conflict end date for the veteran is greater than 
		 * the service separation date for the same military service episode ) 
		 * THEN Generate Error Message "The Conflict From Date or To Date must fall within a single MSE."
		 **/
		
		/** Veteran has two mse's
		 * MSE 1 = Army from 1/1/1940 until 1/1/1960
		 * MSE 2 = Navy from 2/5/1962 until 12/1/1980
		 * The conflict from/to date must fall within 
		 * one of these two date ranges to be considered valid.
		 * So a conflict from 11/11/1941 until 1/1/1945 is good
		 * But a conflict from 1/1/1950 until 1/1/1961 is bad
		 * AND A conflict from 1/1/1950 until 1/1/1970 is bad as well.
		 */
		// Persist a person so we will have a record on file
		Person pristine = this.getPersonService().save( onFile );
		
		// Create a military service record
		Person working = (Person)onFile.clone();

		// Create a military service
		MilitaryService ms = new MilitaryService();
		
		MilitaryServiceSiteRecord mssr = new MilitaryServiceSiteRecord();
		mssr.setSite( this.getVaFacility( VAFacility.CODE_HEC ) );
		
		// Create 2 mes for a period from 02/20/1968 to 04/30/1990
		MilitaryServiceEpisode mse = new MilitaryServiceEpisode();
		mse.setStartDate( ImpreciseDateUtils.createImpreciseDate( 1940,1,1) );
		mse.setEndDate( ImpreciseDateUtils.createImpreciseDate( 1960,1,1) );
		mse.setDischargeType( this.getRandomDischargeType() );
		mse.setServiceBranch( this.getRandomServiceBranch() );
		mssr.addMilitaryServiceEpisode( mse );
		
		mse = new MilitaryServiceEpisode();
		mse.setStartDate( ImpreciseDateUtils.createImpreciseDate( 1962,2,5) );
		mse.setEndDate( ImpreciseDateUtils.createImpreciseDate( 1980,12,1) );
		mse.setDischargeType( this.getRandomDischargeType() );
		mse.setServiceBranch( this.getRandomServiceBranch() );
		mssr.addMilitaryServiceEpisode( mse );

		// Create an overlapping conflict
		ConflictExperience ce = new ConflictExperience();
		ce.setStartDate( ImpreciseDateUtils.createImpreciseDate( 1950,1,1) );
		ce.setEndDate( ImpreciseDateUtils.createImpreciseDate( 1961,1,1) );
		mssr.addConflictExperience( ce );

		ms.addMilitaryServiceSiteRecord( mssr );
		working.setMilitaryService( ms );
		
		try {
			ValidationMessages messages = 
				this.getRuleValidationService().validateConflictExperience( ce, working, pristine, false );

			assertTrue( !messages.isEmpty() );
			assertEquals( ((ValidationFieldMessage)messages.get().next()).getKey(), 
							  RuleValidationMessage.CONFLICT_EXPERIENCE_NOT_FOR_ONE_MSE.getName() );
		}
		catch( Exception e ) {
			if( logger.isDebugEnabled() ) {
				logger.debug( "Got an un-expected validation exception ", e );
			}
			this.fail( "Got an un-expected validation exception", e );
		}
	}
	
	/** Tests isFromDateFutureDate BAL rule in ConflictExperienceValidation.irl
	 * @throws Exception Thrown in case of errors
	 */
	public void testisFromDateFutureDate() throws Exception {
		
		/** 3266[UC23.19.6]
		 * The Conflict From Date cannot be a future date.
		 * 
		 * 3267[UC23.19.6.1] 
		 * If the actor enters a future Conflict From Date the system displays this message: 
		 * "Conflict From Date cannot be a future date." 
		 * 
		 * =====================
		 * IF The received conflict from date is provided 
		 * AND The received conflict from date for the veteran is in the future 
		 * THEN Generate Error Message "Conflict From Date cannot be a future date."
		 */
		Person pristine = this.getPersonService().save( onFile );

		// Create a working version
		Person working = (Person)pristine.clone();
		this.attachMilitaryServiceEpisode( working );

		ConflictExperience ce = this.attachConflictExperience( working );
		ce.setStartDate( ImpreciseDateUtils.createImpreciseDate( 2008, 1, 1) );
		
		try {
			ValidationMessages messages = 
				this.getRuleValidationService().validateConflictExperience( ce, working, pristine, false );

			assertTrue( !messages.isEmpty() );
			assertEquals( ((ValidationFieldMessage)messages.get().next()).getKey(), 
							  RuleValidationMessage.INVALID_FUTURE_CONFLICT_FROM_DATE.getName() );
		}
		catch( Exception e ) {
			if( logger.isDebugEnabled() ) {
				logger.debug( "Got an un-expected validation exception ", e );
			}
			this.fail( "Got an un-expected validation exception", e );
		}
	}
	
	/** Tests isFromDateGreaterThanDOD BAL rule in ConflictExperienceValidation.irl
	 * @throws Exception Thrown in case of errors
	 */
	public void testisFromDateGreaterThanDOD() throws Exception {
		
		/** 
		 * The Conflict From Date must be prior to or equal to the Date of Death
		 *  
		 * 3265[UC23.19.5.1]   
		 * If the actor enters a Conflict From Date that is out of this range, 
		 * the system displays this message: 
		 * "Conflict From Date must be prior to or equal to the Date of Death." 
		 *
		 *	=====================
		 * IF The received conflict from date for the veteran is provided
		 * AND The veteran has date of death information is TRUE
		 * AND The received conflict date for the veteran is greater than the veterans date of death
		 * THEN Generate Error Message "Conflict From Date must be prior to or equal to the Date of Death."
		 **/
		// Create a military service record
		Person pristine = this.getPersonService().save( onFile );

		// Create a working version
		Person working = (Person)pristine.clone();
		this.attachMilitaryServiceEpisode( working );

		ConflictExperience ce = this.attachConflictExperience( working, this.getConflictLocation( ConflictLocation.CODE_VIETNAM ) );
		ce.setStartDate( ImpreciseDateUtils.createImpreciseDate( 2002, 1, 1) );

		DeathRecord record = this.createDeathRecord();
		record.setDeathDate( ImpreciseDateUtils.createImpreciseDate( 2000, 1, 1) );
		working.setDeathRecord( record );
		
		try {
			ValidationMessages messages = 
				this.getRuleValidationService().validateConflictExperience( ce, working, pristine, false );
			
			assertTrue( !messages.isEmpty() );
			assertEquals( ((ValidationFieldMessage)messages.get().next()).getKey(), 
							  RuleValidationMessage.INVALID_CONFLICT_FROM_DATE_AFTER_DOD.getName() );
		}
		catch( Exception e ) {
			if( logger.isDebugEnabled() ) {
				logger.debug( "Got an un-expected validation exception ", e );
			}
			this.fail( "Got an un-expected validation exception", e );
		}
	}
	
	/** Tests isFromDateLessThanOfficialStartDate BAL rule in ConflictExperienceValidation.irl
	 * @throws Exception Thrown in case of errors
	 */
	public void testisFromDateLessThanOfficialStartDate() throws Exception {
		
		/** 
		 * 3262[UC23.19.4] 
		 * The Conflict From Date must be greater than or 
		 * equal to the Conflict Locations Start Date. 
		 * 
		 * 3263[UC23.19.4.1]
		 * If the actor enters a Conflict From Date that is prior to the Conflict 
		 * Locations Start Date, the system displays this message:
		 * "Conflict From Date must be greater than or equal to [date]", 
		 * where [date] is the official start date of the selected Conflict location. 
		 * 
		 * =======================
		 * IF The received conflict from date for the veteran is provided
		 * AND The received conflict location for the veteran is provided
		 * AND The received conflict from date is less than the associated 
		 * conflict locations official start date 
		 * THEN Generate Error Message "Conflict From Date must be greater than or 
		 * equal to [date], where [date] is the official start date of the selected Conflict location.
		 **/
		Person pristine = this.getPersonService().save( onFile );

		// Create a working version
		Person working = (Person)pristine.clone();
		this.attachMilitaryServiceEpisode( working );

		ConflictExperience ce = this.attachConflictExperience( working, this.getConflictLocation( ConflictLocation.CODE_VIETNAM ) );
		ce.setStartDate( ImpreciseDateUtils.createImpreciseDate( 2002, 1, 1) );
		
		try {
			ValidationMessages messages = 
				this.getRuleValidationService().validateConflictExperience( ce, working, pristine, false );
			
			assertTrue( !messages.isEmpty() );
			assertEquals( ((ValidationFieldMessage)messages.get().next()).getKey(), 
							  RuleValidationMessage.INVALID_CONFLICT_FROM_DATE_BEFORE_OFFICIAL_START_DATE.getName() );
		}
		catch( Exception e ) {
			if( logger.isDebugEnabled() ) {
				logger.debug( "Got an un-expected validation exception ", e );
			}
			this.fail( "Got an un-expected validation exception", e );
		}
	}
	
	/** Tests isFromDateNotPrecise BAL rule in ConflictExperienceValidation.irl
	 * @throws Exception Thrown in case of errors
	 */
	public void testisFromDateNotPrecise() throws Exception {
		
		/** 
		 * 3256[UC23.19.1]  
		 * Conflict From Date must be precise. 
		 * 
		 * 3259[UC23.19.2]  
		 * If the Conflict From Date entered does not meet the date requirements 
		 * the system displays this message: "Conflict From Date must be a precise date." 
		 *
		 *	=====================
		 * IF The received conflict from date is provided
		 * AND It is not true that The received conflict from date for the veteran is precise
		 * THEN Generate Error Message "Conflict From Date must be a precise date."
		 **/
		// Persist a person so we will have a record on file
		Person pristine = this.getPersonService().save( onFile );

		// Create a military service record
		Person working = (Person)pristine.clone();
		this.attachMilitaryServiceEpisode( working );

		// Create a military service record
		ConflictExperience ce = this.attachConflictExperience( working );
		ce.setStartDate( ImpreciseDateUtils.createImpreciseDate( 2001, 10, 10 ) );
		ce.setEndDate( ImpreciseDateUtils.createImpreciseDate( 2003, 10, 10 ) );
		
		try {
			ValidationMessages messages = 
				this.getRuleValidationService().validateConflictExperience( ce, working, pristine, false );
			
			assertTrue( messages.isEmpty() );
		}
		catch( Exception e ) {
			if( logger.isDebugEnabled() ) {
				logger.debug( "Got an un-expected validation exception ", e );
			}
			this.fail( "Got an un-expected validation exception", e );
		}
	}
	
	/** Tests isToDateFutureDate BAL rule in ConflictExperienceValidation.irl
	 * @throws Exception Thrown in case of errors
	 */
	public void testisToDateFutureDate() throws Exception {
		
		/** 
		 * 3283[UC23.20.6]  
		 * The Conflict To Date cannot be a future date. 
		 * 
		 * 3284[UC23.20.6.1]  
		 * If the actor enters a future Conflict To Date, the system displays this message:
		 * "Conflict To Date cannot be a future date." 
		 *
		 *	==========================
		 * IF The received conflict to date for the veteran is provided
		 * AND The received conflict to date for the veteran is future date
		 * THEN Generate Error "Conflict To Date cannot be a future date."
		 **/
		Person pristine = this.getPersonService().save( onFile );

		// Create a working version
		Person working = (Person)pristine.clone();
		this.attachMilitaryServiceEpisode( working );

		ConflictExperience ce = this.attachConflictExperience( working );
		ce.setEndDate( ImpreciseDateUtils.createImpreciseDate( 2008, 1, 1) );
		
		try {
			ValidationMessages messages = 
				this.getRuleValidationService().validateConflictExperience( ce, working, pristine, false );

			assertTrue( !messages.isEmpty() );
			assertEquals( ((ValidationFieldMessage)messages.get().next()).getKey(), 
							  RuleValidationMessage.INVALID_FUTURE_CONFLICT_TO_DATE.getName() );
		}
		catch( Exception e ) {
			if( logger.isDebugEnabled() ) {
				logger.debug( "Got an un-expected validation exception ", e );
			}
			this.fail( "Got an un-expected validation exception", e );
		}
	}
	
	/** Tests isToDateGreaterThanDOD BAL rule in ConflictExperienceValidation.irl
	 * @throws Exception Thrown in case of errors
	 */
	public void testisToDateGreaterThanDOD() throws Exception {
		
		/** 
		 * 3281[UC23.20.5]  
		 * The Conflict To Date must be prior to or equal to the Date of Death 
		 * 
		 * 3282[UC23.20.5.1]  
		 * If the actor enters a Conflict To Date that does not fall within the 
		 * specified range,  the system displays this message:
		 * "Conflict To Date must be prior to or equal to the Date of Death."
		 * 
		 * =================================
		 * IF The received conflict to date for the veteran is provided
		 * AND The veteran has date of death information is TRUE
		 * AND The received conflict to date for the veteran is greater than 
		 * the veterans date of death
		 * THEN Generate Error "Conflict To Date must be prior to or equal to 
		 * the Date of Death."
		 **/
		// Create a military service record
		Person pristine = this.getPersonService().save( onFile );

		// Create a working version
		Person working = (Person)pristine.clone();
		this.attachMilitaryServiceEpisode( working );

		ConflictExperience ce = this.attachConflictExperience( working );
		ce.setEndDate( ImpreciseDateUtils.createImpreciseDate( 2002, 1, 1) );
		
		DeathRecord record = this.createDeathRecord();
		record.setDeathDate( ImpreciseDateUtils.createImpreciseDate( 2000, 1, 1) );
		working.setDeathRecord( record );
		
		try {
			ValidationMessages messages = 
				this.getRuleValidationService().validateConflictExperience( ce, working, pristine, false );

			assertTrue( !messages.isEmpty() );
			assertEquals( ((ValidationFieldMessage)messages.get().next()).getKey(), 
							  RuleValidationMessage.INVALID_CONFLICT_TO_DATE_AFTER_DOD.getName() );
		}
		catch( Exception e ) {
			if( logger.isDebugEnabled() ) {
				logger.debug( "Got an un-expected validation exception ", e );
			}
			this.fail( "Got an un-expected validation exception", e );
		}
	}
	
	/** Tests isToDateGreaterThanOfficialEndDate BAL rule in ConflictExperienceValidation.irl
	 * @throws Exception Thrown in case of errors
	 */
	public void testisToDateGreaterThanOfficialEndDate() throws Exception {
		
		/** 
		 * 3279[UC23.20.4]  
		 * The Conflict To Date must be prior to or equal to the Conflict 
		 * Locations End Date. 
		 *
		 * 3280[UC23.20.4.1]  
		 * If the actor enters Conflict To Date that does not fall within 
		 * the specified range, the system displays this message: "Conflict 
		 * To Date must be prior to or equal to [date]", where [date] is the 
		 * official end date of the Conflict selected. 
		 *
		 * =====================
		 * IF The received conflict to date for the veteran is provided
		 * AND The received conflict location for the veteran is provided
		 * AND The received conflict to date is greater than the associated 
		 * conflict locations official end date 
		 * THEN Generate Error Message "Conflict To Date must be prior to or 
		 * equal to [date]", where [date] is the official end date of the Conflict selected
		 **/
		Person pristine = this.getPersonService().save( onFile );

		// Create a working version
		Person working = (Person)pristine.clone();
		this.attachMilitaryServiceEpisode( working );

		ConflictExperience ce = this.attachConflictExperience( working, this.getConflictLocation( ConflictLocation.CODE_VIETNAM ) );
		ce.setEndDate( ImpreciseDateUtils.createImpreciseDate( 2002, 1, 1) );
		
		try {
			ValidationMessages messages = 
				this.getRuleValidationService().validateConflictExperience( ce, working, pristine, false );
			
			assertTrue( !messages.isEmpty() );
			assertEquals( ((ValidationFieldMessage)messages.get().next()).getKey(), 
							  RuleValidationMessage.INVALID_CONFLICT_TO_DATE_AFTER_OFFICIAL_END_DATE.getName() );
		}
		catch( Exception e ) {
			if( logger.isDebugEnabled() ) {
				logger.debug( "Got an un-expected validation exception ", e );
			}
			this.fail( "Got an un-expected validation exception", e );
		}
	}
	
	/** Tests isToDateLessThanFromDate BAL rule in ConflictExperienceValidation.irl
	 * @throws Exception Thrown in case of errors
	 */
	public void testisToDateLessThanFromDate() throws Exception {
		
		/** 
		 * 3277[UC23.20.3]  
		 * The Conflict To Date must be later than or equal to the Conflict From Date. 
		 * 
		 * 3278[UC23.20.3.2]  
		 * If the actor enters a Conflict To Date that does not fall within the specified 
		 * range, the system displays this message:
		 * "The Conflict To Date must be later than or equal to the Conflict From Date" 
		 *
		 * =================================
		 * IF The received conflict to date for the veteran is provided
		 * AND The received conflict from date for the veteran is provided
		 * AND The received conflict to date for the veteran is less than the received 
		 * conflict from date
		 * THEN Generate Error Message "The Conflict To Date must be later than or equal 
		 * to the Conflict From Date."
		 **/
		// Create a military service record
		Person pristine = this.getPersonService().save( onFile );

		// Create a working version
		Person working = (Person)pristine.clone();
		this.attachMilitaryServiceEpisode( working );

		ConflictExperience ce = this.attachConflictExperience( working );
		ce.setStartDate( ImpreciseDateUtils.createImpreciseDate( 2002, 1, 1) );
		ce.setEndDate( ImpreciseDateUtils.createImpreciseDate( 2000, 1, 1) );
		
		try {
			ValidationMessages messages = 
				this.getRuleValidationService().validateConflictExperience( ce, working, pristine, false );
			
			assertTrue( !messages.isEmpty() );
			assertEquals( ((ValidationFieldMessage)messages.get().next()).getKey(), 
							  RuleValidationMessage.INVALID_CONFLICT_TO_DATE_BEFORE_FROM_DATE.getName() );
		}
		catch( Exception e ) {
			if( logger.isDebugEnabled() ) {
				logger.debug( "Got an un-expected validation exception ", e );
			}
			this.fail( "Got an un-expected validation exception", e );
		}
	}
	
	/** Tests isToDateNotPrecise BAL rule in ConflictExperienceValidation.irl
	 * @throws Exception Thrown in case of errors
	 */
	public void testisToDateNotPrecise() throws Exception {
		
		/** 
		 * 3268[UC23.20] Conflict To Date 
		 * The actor is not allowed to enter the Conflict To Date unless 
		 * there is at least one military service episode. 
		 *
		 *	3271[UC23.20.1]  
		 * Conflict To Date must be precise. 
		 *
		 * 3274[UC23.20.1.1]  
		 * If the Conflict To Date entered does not meet the date 
		 * requirement above, the system displays this message:
		 * "Conflict To Date must be a precise date." 
		 *
		 * IF the received conflict to date for the veteran is provided
		 * AND It is not true that the received conflict to date for the 
		 * veteran is precise 
		 * THEN Generate Error Message "Conflict To Date must be a precise date."
		 **/
		// Persist a person so we will have a record on file
		Person pristine = this.getPersonService().save( onFile );

		// Create a military service record
		Person working = (Person)pristine.clone();
		this.attachMilitaryServiceEpisode( working );

		// Create a military service record
		ConflictExperience ce = this.attachConflictExperience( working );
		ce.setStartDate( ImpreciseDateUtils.createImpreciseDate( 2001, 10, 10 ) );
		ce.setEndDate( ImpreciseDateUtils.createImpreciseDate( 2003, 10, 10 ) );
		
		try {
			ValidationMessages messages = 
				this.getRuleValidationService().validateConflictExperience( ce, working, pristine, false );
			
			assertTrue( messages.isEmpty() );
		}
		catch( Exception e ) {
			if( logger.isDebugEnabled() ) {
				logger.debug( "Got an un-expected validation exception ", e );
			}
			this.fail( "Got an un-expected validation exception", e );
		}
	}
	
	/** Tests to get multiple validaion messages at a time for the UI in ConflictExperienceValidation.irl
	 * @throws Exception Thrown in case of errors
	 */
	public void testConflictServiceValidation() throws Exception {
		
		/** 
		 * Validate an empty Conflict experience object to obtain more than one validation messages.
		 */
		// Persist a person so we will have a record on file
		Person pristine = this.getPersonService().save( onFile );
		
		// Create a military service record
		Person working = (Person)pristine.clone();
		this.attachMilitaryServiceEpisode( working );
		
		//Create an empty conflict service (no fields populated)
		ConflictExperience ce = attachConflictExperience(working);
		//ce.setStartDate(ImpreciseDateUtils.createImpreciseDate( 2001, 10, 10 ));
		//ce.setEndDate(ImpreciseDateUtils.createImpreciseDate( 2002, 10, 10 ));
		
		try {
			ValidationMessages messages = 
				this.getRuleValidationService().validateConflictExperience( ce, working, pristine, true );
			
			System.out.println(messages.getCount());
			//Currently I am not very sure how many validation messages I am getting, just checking for more than 1.
			//assertTrue(messages.getCount() == 0);
		}
		catch( Exception e ) {
			if( logger.isDebugEnabled() ) {
				logger.debug( "Got an un-expected validation exception ", e );
			}
			this.fail( "Got an un-expected validation exception", e );
		}
	}
	
	/**
	 * @see gov.va.med.fw.util.AbstractTestCase#customSetUp()
	 */
	protected void customSetUp() throws Exception {
		super.customSetUp();
		onFile = this.buildSimplePerson();
	}

	/**
	 * @see gov.va.med.fw.util.AbstractTestCase#customTearDown()
	 */
	protected void customTearDown() throws Exception {
		super.customTearDown();
		onFile = null;
	}

	/**
	 * @see org.springframework.beans.factory.InitializingBean#afterPropertiesSet()
	 */
	public void afterPropertiesSet() throws Exception {
		super.afterPropertiesSet();
		Validate.notNull(this.getRuleValidationService(), "A rule validation service is required");
	}
}