package gov.va.med.esr.common.rule.service;

// Java classes
import java.util.Date;
import java.util.List;
import java.util.ArrayList;
import java.math.BigDecimal;

// Library classes

// Framework classes

// ESR classes
import gov.va.med.esr.common.model.comms.CommsLogEntry;

import gov.va.med.esr.common.model.ee.PurpleHeart;
import gov.va.med.esr.common.model.ee.Eligibility;
import gov.va.med.esr.common.model.ee.PurpleHeartDocument;
import gov.va.med.esr.common.model.financials.Hardship;

import gov.va.med.esr.common.rule.data.CommsInputData;
import gov.va.med.esr.common.rule.parameter.CommunicationsInputParameter;
import gov.va.med.esr.common.model.party.Address;
import gov.va.med.esr.common.model.person.Person;
import gov.va.med.esr.common.model.person.SSN;
import gov.va.med.esr.common.model.person.id.PersonIdEntityKeyImpl;
import gov.va.med.esr.common.model.lookup.AddressType;
import gov.va.med.esr.common.model.lookup.EligibilityType;
import gov.va.med.esr.common.model.lookup.ComLetterTemplateType;
import gov.va.med.esr.common.model.lookup.ComMailingStatusType;
import gov.va.med.esr.common.model.lookup.ComLetterTemplateType.Code;
import gov.va.med.esr.common.model.lookup.DecorationStatus;
import gov.va.med.esr.common.model.lookup.EnrollmentPriorityGroup;
import gov.va.med.esr.common.model.lookup.EnrollmentStatus;
import gov.va.med.esr.common.model.lookup.SSAVerificationStatus;
import gov.va.med.esr.common.model.lookup.PseudoSSNReason;
import gov.va.med.esr.common.model.lookup.EligibilityVerificationSource;
import gov.va.med.esr.common.model.lookup.SSNType;
import gov.va.med.esr.common.model.lookup.PHDocumentType;
import gov.va.med.esr.common.util.RuleAbstractTestCase;
import gov.va.med.esr.common.infra.ImpreciseDateUtils;
import gov.va.med.esr.common.persistent.comms.CommsLogEntryDAO;
import gov.va.med.esr.service.trigger.LetterTriggerEvent;
import gov.va.med.esr.service.CommsLetterRequestService;
import gov.va.med.esr.service.IVMFinancialInfo;
import gov.va.med.fw.model.EntityKey;
import gov.va.med.fw.model.EntityKeyFactory;
import gov.va.med.fw.rule.RuleException;
import gov.va.med.fw.validation.ValidationMessages;

/**
 * @author DNS   RUIZC
 * @version 1.0
 */
public class CommunicationRuleServiceTest extends RuleAbstractTestCase {

	private CommsLogEntryDAO commsLogEntryDAO = null;

	public CommunicationRuleServiceTest(String testName) {
		super(testName);
	
	}
	/**
	 * FAILING with this error. Need to add exception for the Compassion letter
[DEBUG] 23 May 03:22:45.013 PM main [gov.va.med.fw.rule.RuleFlowManager]
Invoked rule flow: ProcessLetterSendRequest
(+) activate rule : IsUserOverridingLetterRequest
(-) deactivate rule : IsUserOverridingLetterRequest
(+) activate rule : IsAddressBad
(-) deactivate rule : IsAddressBad
(+) activate rule : IsVeteranDeceased
(+) add rule instance : IsVeteranDeceased@1b17d11   (priority = 0)   [gov.va.med.esr.common.rule.parameter.DemographicInputParameter@1d64969,  gov.va.med.fw.rule.RuleState@c7cee9]
(->) fire rule instance : IsVeteranDeceased@1b17d11
(-) deactivate rule : IsVeteranDeceased 
	 * @throws Exception
	 */
	public void testCCR12858CompassionLetter() throws Exception {
	   	String personId = "8859561";
    	//8859561
		PersonIdEntityKeyImpl personKey = new PersonIdEntityKeyImpl(new BigDecimal(personId));
		Person person = this.getPersonService().getPerson(personKey);    	
		
    	CommsLogEntryDAO dao =   (CommsLogEntryDAO)applicationContext.getBean("commsLogEntryDAO");
		EntityKey key = EntityKeyFactory.createEntityKey(new BigDecimal(378180276),CommsLogEntry.class);
		CommsLogEntry entry = (CommsLogEntry)dao.getByKey(key);
		
		ValidationMessages errMsgs1 = this.getRuleValidationService().validateLetterMailing(entry, person);
		List logs = new ArrayList();
		logs.add(entry);
		ComLetterTemplateType form = this.getComLetterTemplateType(ComLetterTemplateType.FORM_NUMBER_683A);
		errMsgs1 = this.getRuleValidationService().validateLetterSendRequest(person, form,
				null,  LetterTriggerEvent.AUTO_MAIL, false, logs, LetterTriggerEvent.AUTO_MAIL.getName(), CommsLetterRequestService.CONTEXT_REQUEST_LETTER);
		assertTrue(errMsgs1.isEmpty());
	}
	
	public void testPreviouslyMailedFinalLetters() throws Exception {
		String personId = "8859561";
		PersonIdEntityKeyImpl personKey = new PersonIdEntityKeyImpl(new BigDecimal(personId));
		Person person = this.getPersonService().getPerson(personKey);    		
		CommunicationsInputParameter param = (CommunicationsInputParameter)applicationContext.getBean("communicationsRuleParameter");
		CommsInputData ruleData = new CommsInputData(person, person);
		param.setRuleDataAware(ruleData);		
		boolean mailed = param.hasLetterBeenMailedIncludingMailedStatus(ComLetterTemplateType.FORM_NUMBER_683A.getCode());
		System.out.println("mailed="+mailed);
	}
	
	public void testIVMFinalLetterIndicator() throws Exception {
		String personId = "375200524";
		PersonIdEntityKeyImpl personKey = new PersonIdEntityKeyImpl(new BigDecimal(personId));
		Person person = this.getPersonService().getPerson(personKey);    		
		CommunicationsInputParameter param = (CommunicationsInputParameter)applicationContext.getBean("communicationsRuleParameter");
		CommsInputData ruleData = new CommsInputData(person, person);
		param.setRuleDataAware(ruleData);		
		boolean active = param.isIVMFinalLetterActive();

	}
		
	
	public void testFinalLetterValidationsSecondaryEligibilities() throws Exception {
		Person person = this.buildSimplePerson();
		this.attachVerifiedEnrollment(person);
		person.getEnrollmentDetermination().setPrimaryEligiblity(null);
		CommunicationsInputParameter param = (CommunicationsInputParameter)applicationContext.getBean("communicationsRuleParameter");
		CommsInputData ruleData = new CommsInputData(person, person);
		CommsLogEntry entry = new CommsLogEntry();
		entry.setIvmCaseNumber("1234IVM88");
		entry.setIvmCaseClosureDate(new Date());
		ruleData.setCommsLogEntry(entry);
		param.setRuleDataAware(ruleData);		
		assertTrue(param.meetsIvmEligibilityCriteria(person));
		assertNotNull(param.getIvmCaseNumber());
		assertNotNull(param.getIvmCaseClosureDate());
		
		//this.attachPrisonerOfWar(person, ImpreciseDateUtils.createImpreciseDate(2005, 1, 1),ImpreciseDateUtils.createImpreciseDate(2006, 1, 1));
		person.getEnrollmentDetermination().addSecondaryEligibility(this.createEligibility(EligibilityType.PRISONER_OF_WAR.getCode()));		
		assertFalse(param.meetsIvmEligibilityCriteria(person));
		
		person.setEnrollmentDetermination(this.createEnrollmentDetermination());		
		assertTrue(param.meetsIvmEligibilityCriteria(person));	
		
		person.getEnrollmentDetermination().addSecondaryEligibility(this.createEligibility(EligibilityType.PURPLE_HEART_RECIPIENT.getCode()));		
		assertFalse(param.meetsIvmEligibilityCriteria(person));
		
		person.setEnrollmentDetermination(this.createEnrollmentDetermination());		
		assertTrue(param.meetsIvmEligibilityCriteria(person));
		
		person.getEnrollmentDetermination().addSecondaryEligibility(this.createEligibility(EligibilityType.AID_AND_ATTENDANCE.getCode()));		
		assertFalse(param.meetsIvmEligibilityCriteria(person));
		
		person.setEnrollmentDetermination(this.createEnrollmentDetermination());		
		assertTrue(param.meetsIvmEligibilityCriteria(person));
		
		person.getEnrollmentDetermination().addSecondaryEligibility(this.createEligibility(EligibilityType.HOUSEBOUND.getCode()));		
		assertFalse(param.meetsIvmEligibilityCriteria(person));
		
		person.setEnrollmentDetermination(this.createEnrollmentDetermination());		
		assertTrue(param.meetsIvmEligibilityCriteria(person));					
	}
	private Eligibility createEligibility (String type) throws Exception {
		Eligibility e = new Eligibility();
		e.setType(this.getLookupService().getEligibilityTypeByCode(type));
		return e;
	}
	
	public void testCommunicationInputParamter() throws Exception {
		Person person = this.buildSimplePerson();
		CommunicationsInputParameter param = new CommunicationsInputParameter();
		IVMFinancialInfo info = new IVMFinancialInfo();
		info.setIvmActionCode("C");
		info.setIvmCaseClosureDate(new Date());
		info.setIvmCaseNumber("1234IVM88");
		
		CommsInputData ruleData = new CommsInputData(person, person);
		ruleData.setIvmFinancialInfo(info);		
		param.setRuleDataAware(ruleData);
		
		param.setLookupService(this.getLookupService());
		param.triggerLetterMailing(ComLetterTemplateType.FORM_NUMBER_400.getCode());
	}
	
	
	public void testValidateSendRequest_601B_Neg() throws Exception {
		Person onFile = this.createTestPerson();
		this.attachMeansTestWithMtCopayRequired(onFile, new BigDecimal(100),
				new BigDecimal(100));
		onFile.getIncomeTest(new Integer(2005)).setAgreesToPayDeductible(
				Boolean.FALSE);
		onFile.getIncomeTest(new Integer(2005))
				.setDiscloseFinancialInformation(Boolean.TRUE);
		Person incoming = this.savePerson(onFile);
		this.setEligibilityStatusToVerified(incoming);
		incoming.getEligibilityVerification().setVerificationMethod("gdfggd");
		this.calculate(incoming);

		ValidationMessages vm = this.doValidateLetterSendRequest(incoming,
				Boolean.FALSE, ComLetterTemplateType.FORM_NUMBER_601B,
				ComMailingStatusType.SEND_TO_AAC, false);
		assertTrue(vm.getCount() == 1);
	}

	public void testValidateSendRequest_600B_INVALID() throws Exception {
		Person onFile = this.createTestPerson();
		Person incoming = this.savePerson(onFile);

		ValidationMessages vm = this.doValidateLetterSendRequest(incoming,
				Boolean.FALSE, ComLetterTemplateType.FORM_NUMBER_600B,
				ComMailingStatusType.SEND_TO_AAC);
		assertTrue(vm.getCount() == 1);
	}

	public void testValidateSendRequest_601B_Pos() throws Exception {
		Person onFile = this.createTestPerson();
		this.attachMeansTestWithMtCopayRequired(onFile, new BigDecimal(100),
				new BigDecimal(100));
		onFile.getIncomeTest(new Integer(2005)).setAgreesToPayDeductible(
				Boolean.FALSE);
		onFile.getIncomeTest(new Integer(2005))
				.setDiscloseFinancialInformation(Boolean.TRUE);
		Person incoming = this.savePerson(onFile);
		this.setEligibilityStatusToVerified(incoming);
		incoming.getEligibilityVerification().setVerificationMethod("gdfggd");
		this.calculate(incoming);

		ValidationMessages vm = this.doValidateLetterSendRequest(incoming,
				Boolean.FALSE, ComLetterTemplateType.FORM_NUMBER_601B,
				ComMailingStatusType.SEND_TO_AAC, true);
		assertTrue(vm.getCount() == 0);
	}

	public void testValidateSendRequest_601B_REMAIL() throws Exception {
		Person onFile = this.createTestPerson();
		this.attachMeansTestWithMtCopayRequired(onFile, new BigDecimal(100),
				new BigDecimal(100));
		onFile.getIncomeTest(new Integer(2005)).setAgreesToPayDeductible(
				Boolean.FALSE);
		onFile.getIncomeTest(new Integer(2005))
				.setDiscloseFinancialInformation(Boolean.TRUE);
		Person incoming = this.savePerson(onFile);
		this.setEligibilityStatusToVerified(incoming);
		incoming.getEligibilityVerification().setVerificationMethod("gdfggd");
		this.calculate(incoming);

		ValidationMessages vm = this.doValidateLetterSendRequest(incoming,
				Boolean.FALSE, ComLetterTemplateType.FORM_NUMBER_601B,
				ComMailingStatusType.SEND_TO_AAC, true,
				LetterTriggerEvent.REMAIL_MAIL);
		assertTrue(vm.getCount() == 0);
	}

	public void testValidateSendRequest_601B_Pos2() throws Exception {
		Person onFile = this.createTestPerson();
		this.attachMeansTestWithMtCopayRequired(onFile, new BigDecimal(100),
				new BigDecimal(100));
		onFile.getIncomeTest(new Integer(2005)).setAgreesToPayDeductible(
				Boolean.FALSE);
		onFile.getIncomeTest(new Integer(2005))
				.setDiscloseFinancialInformation(Boolean.TRUE);
		Person incoming = this.savePerson(onFile);
		this.setEligibilityStatusToVerified(incoming);
		incoming.getEligibilityVerification().setVerificationMethod("gdfggd");
		this.calculate(incoming);

		ValidationMessages vm = this.doValidateLetterSendRequest(incoming,
				Boolean.FALSE, ComLetterTemplateType.FORM_NUMBER_601B,
				ComMailingStatusType.SEND_TO_AAC, true);
		assertTrue(vm.getCount() == 0);
	}

	public void testValidateSendRequest_640D_and_remail() throws Exception {
		Person onFile = this.createTestPerson();
		this.attachMeansTestWithMtStatusPendingAdjudication(onFile,
				new BigDecimal(100), new BigDecimal(100));
		Person incoming = this.savePerson(onFile);
		Person calculated = calculate(incoming);
		calculated.getEnrollmentDetermination().setEnrollmentStatus(
				this.getLookupService().getEnrollmentStatusByCode(
						EnrollmentStatus.CODE_VERIFIED.getName()));
		ValidationMessages vm = this.doValidateLetterSendRequest(calculated,
				Boolean.FALSE, ComLetterTemplateType.FORM_NUMBER_640D,
				ComMailingStatusType.SEND_TO_AAC);
		assertTrue(vm.getCount() == 0);
		vm = this.doValidateLetterSendRequest(calculated, Boolean.FALSE,
				ComLetterTemplateType.FORM_NUMBER_640D,
				ComMailingStatusType.SEND_TO_AAC, false);
		assertTrue(vm.getCount() == 1);

		vm = this.doValidateLetterSendRequest(calculated, Boolean.FALSE,
				ComLetterTemplateType.FORM_NUMBER_640D,
				ComMailingStatusType.SEND_TO_AAC, false,
				LetterTriggerEvent.REMAIL_MAIL);
		assertTrue(vm.getCount() == 2);

		vm = this.doValidateLetterSendRequest(calculated, Boolean.FALSE,
				ComLetterTemplateType.FORM_NUMBER_640D,
				ComMailingStatusType.SEND_TO_AAC, true,
				LetterTriggerEvent.REMAIL_MAIL);
		assertTrue(vm.getCount() == 0);
	}

	public void testValidateSendRequest_400_ph() throws Exception {

		Person onFile = this.createTestPerson();
		this.attachVerifiedEnrollment(onFile);
		Eligibility e = new Eligibility();
		this.setEligibilityStatusToVerified(onFile);		
		e.setPrimary();
		e.setType(getLookupService().getEligibilityTypeByCode(EligibilityType.PURPLE_HEART_RECIPIENT.getCode()));
		onFile.getEnrollmentDetermination().setPrimaryEligiblity(e);    
		Person incoming = savePerson(onFile);

		ValidationMessages vm = this.doValidateLetterSendRequest(incoming,
				Boolean.FALSE, ComLetterTemplateType.FORM_NUMBER_400,
				ComMailingStatusType.SEND_TO_CMS);

		assertTrue(vm.getCount() == 1);

	}
	public void testValidateSendRequest_400_hardship() throws Exception {

		Person onFile = this.createTestPerson();
		this.attachVerifiedEnrollment(onFile);
		Eligibility e = new Eligibility();
		this.setEligibilityStatusToVerified(onFile);		
		e.setPrimary();
		e.setType(getLookupService().getEligibilityTypeByCode(EligibilityType.NSC.getCode()));
		onFile.getEnrollmentDetermination().setPrimaryEligiblity(e);
		this.attachMeansTestWithMtCopayRequired(onFile, new BigDecimal(3), new BigDecimal(3));
		this.getPersonHelperService().getCurrentIncomeTest(onFile).setHardship(this.createHardship("test"));
		Person incoming = savePerson(onFile);

		ValidationMessages vm = this.doValidateLetterSendRequest(incoming,
				Boolean.FALSE, ComLetterTemplateType.FORM_NUMBER_400,
				ComMailingStatusType.SEND_TO_CMS);

		assertTrue(vm.getCount() == 1);

	}		
	protected Hardship createHardship(String reason) {
		Hardship hardship = new Hardship();
		hardship.setReason(reason);
		hardship.setEffectiveDate(new Date());
		hardship.setHardshipGranted(Boolean.TRUE);
		return hardship;
	}	
	public void testValidateSendRequest_600C() throws Exception {

		Person onFile = this.createTestPerson();
		this.attachMeansTestWithMtCopayExempt(onFile);
		Person incoming = savePerson(onFile);
		this.setEligibilityStatusToPendingVerification(incoming);
		calculate(incoming);
		ValidationMessages vm = this.doValidateLetterSendRequest(incoming,
				Boolean.FALSE, ComLetterTemplateType.FORM_NUMBER_600C,
				ComMailingStatusType.SEND_TO_AAC);

		assertTrue(vm.getCount() == 1);
		this.setEligibilityStatusToVerified(incoming);

		calculate(incoming);
		vm = this.doValidateLetterSendRequest(incoming, Boolean.FALSE,
				ComLetterTemplateType.FORM_NUMBER_600C,
				ComMailingStatusType.SEND_TO_AAC);

		assertTrue(vm.getCount() == 0);
	}

	public void testValidateSendRequest_400H() throws Exception {
		Person onFile = createTestPerson();
		attachMeansTestWithMtCopayExempt(onFile);
		Person incoming = savePerson(onFile);
		setEligibilityStatusToPendingVerification(incoming);
		calculate(incoming);
		ValidationMessages vm = doValidateLetterSendRequest(incoming,
				Boolean.FALSE, ComLetterTemplateType.FORM_NUMBER_400H,
				ComMailingStatusType.SEND_TO_CMS);
		assertTrue(vm.getCount() == 1);
		setEligibilityStatusToVerified(incoming);
		calculate(incoming);
		vm = doValidateLetterSendRequest(incoming, Boolean.FALSE,
				ComLetterTemplateType.FORM_NUMBER_400H,
				ComMailingStatusType.SEND_TO_CMS);
		assertTrue(vm.getCount() == 0);
	}

	public void testValidateSendRequest_620A() throws Exception {
		Person onFile = this.createTestPerson();
		this.attachCancelDecline(onFile);
		Person incoming = this.savePerson(onFile);
		this.setEligibilityStatusToVerified(incoming);
		calculate(incoming);

		ValidationMessages vm = this.doValidateLetterSendRequest(incoming,
				Boolean.FALSE, ComLetterTemplateType.FORM_NUMBER_620A,
				ComMailingStatusType.SEND_TO_AAC);
		assertTrue(vm.getCount() == 0);
		this.attachIneligibleFactor(incoming);
		vm = this.doValidateLetterSendRequest(incoming, Boolean.FALSE,
				ComLetterTemplateType.FORM_NUMBER_620A,
				ComMailingStatusType.SEND_TO_AAC);
		assertTrue(vm.getCount() == 1);
	}

	public void testValidateSendRequest_623A() throws Exception {
		// use testCreateHistoricalPersonEGT to create person
		PersonIdEntityKeyImpl personKey = new PersonIdEntityKeyImpl(
				new BigDecimal("441417"));
		Person incoming = this.getPersonService().getPerson(personKey);
		this.addPermanentAddress(incoming, getDate(2006, 1, 1), null);
		this.attachVerifiedEnrollment(incoming);
		incoming.getEnrollmentDetermination()
				.setEnrollmentStatus(
						this.getLookupService()
								.getEnrollmentStatusByCode(
										EnrollmentStatus.CODE_REJECTED_BELOW_ENROLLMENT_GROUP_THRESHOLD
												.getCode()));
		ValidationMessages vm = this.doValidateLetterSendRequest(incoming,
				Boolean.FALSE, ComLetterTemplateType.FORM_NUMBER_623A,
				ComMailingStatusType.SEND_TO_AAC);
		assertTrue(vm.getCount() == 0);
	}

	public void testValidateSendRequest_623D() throws Exception {
		Person incoming = this.createHistoricalPersonEGT();

		ValidationMessages vm = this.doValidateLetterSendRequest(incoming,
				Boolean.FALSE, ComLetterTemplateType.FORM_NUMBER_623D,
				ComMailingStatusType.SEND_TO_AAC);
		assertTrue(vm.getCount() == 0);
	}

	public void updatedPersonFor_623A() throws Exception {
		// use testCreateHistoricalPersonEGT to create person
		PersonIdEntityKeyImpl personKey = new PersonIdEntityKeyImpl(
				new BigDecimal("441417"));
		Person updated = this.getPersonService().getPerson(personKey);
		updated.getEnrollmentDetermination()
				.setEnrollmentStatus(
						this.getLookupService()
								.getEnrollmentStatusByCode(
										EnrollmentStatus.CODE_REJECTED_BELOW_ENROLLMENT_GROUP_THRESHOLD
												.getCode()));
		this.getPersonService().save(updated);
		this.setComplete();
	}

	public void testValidateSendRequest_630A() throws Exception {
		Person onFile = this.createTestPerson();
		Person incoming = this.savePerson(onFile);
		this.attachPurpleHeartWithPHStatusPending(incoming);
		calculate(incoming);

		ValidationMessages vm = this.doValidateLetterSendRequest(incoming,
				Boolean.FALSE, ComLetterTemplateType.FORM_NUMBER_630A,
				ComMailingStatusType.SEND_TO_AAC);
		assertTrue(vm.getCount() == 0);
	}

	public void testValidateSendRequest_630B() throws Exception {
		Person onFile = this.createTestPerson();
		Person incoming = this.savePerson(onFile);
		this.setEligibilityStatusToVerified(incoming);
		calculate(incoming);

		ValidationMessages vm = this.doValidateLetterSendRequest(incoming,
				Boolean.FALSE, ComLetterTemplateType.FORM_NUMBER_630B,
				ComMailingStatusType.SEND_TO_AAC);
		assertTrue(vm.getCount() == 0);
	}

	public void testValidateSendRequest_630C() throws Exception {
		Person onFile = this.createTestPerson();
		Person incoming = this.savePerson(onFile);
		this.setEligibilityStatusToVerified(incoming);
		calculate(incoming);

		ValidationMessages vm = this.doValidateLetterSendRequest(incoming,
				Boolean.FALSE, ComLetterTemplateType.FORM_NUMBER_630C,
				ComMailingStatusType.SEND_TO_AAC);
		assertTrue(vm.getCount() == 0);
	}

	public void testValidateSendRequest_630D() throws Exception {
		doTestValidateSendRequest_630D(LetterTriggerEvent.AUTO_MAIL);
	}

	public void testValidateSendRequest_630D_Remail() throws Exception {
		doTestValidateSendRequest_630D(LetterTriggerEvent.REMAIL_MAIL);
	}

	private void doTestValidateSendRequest_630D(
			LetterTriggerEvent.MailType mailType) throws Exception {
		Person onFile = this.createTestPerson();
		this.attachVerifiedEnrollment(onFile);
		this.setEligibilityStatusToVerified(onFile);
		this.attachPurpleHeartWithPHStatusPending(onFile);
		onFile.getPurpleHeart().setStatus(
				this.getLookupService().getDecorationStatusByCode(
						DecorationStatus.CODE_CONFIRMED.getCode()));
		// Defect 5147 says person might have had ee data change so that prior
		// enrollment is P3. This test simulates
		// that scenario to make sure it works. The letter request is still
		// valid provided that person has P3 and
		// PH status is Confirmed, etc.
		onFile.getEnrollmentDetermination().setPriorityGroup(
				this.getLookupService().getEnrollmentPriorityGroupByCode(
						EnrollmentPriorityGroup.GROUP_3.getCode()));
		Person incoming = this.getPersonService().save(onFile);

		ValidationMessages vm = this.doValidateLetterSendRequest(incoming,
				Boolean.FALSE, ComLetterTemplateType.FORM_NUMBER_630D,
				ComMailingStatusType.MAILED_BY_AAC, true, mailType);
		assertTrue(vm.getCount() == 0);
	}

	public void testValidateSendRequest_640C() throws Exception {
		doTestValidateSendRequest_640C(LetterTriggerEvent.AUTO_MAIL);
	}

	public void testValidateSendRequest_640C_Remail() throws Exception {
		doTestValidateSendRequest_640C(LetterTriggerEvent.REMAIL_MAIL);
	}

	private void doTestValidateSendRequest_640C(
			LetterTriggerEvent.MailType mailType) throws Exception {
		Person onFile = this.createTestPerson();
		this.attachVerifiedEnrollment(onFile);
		this.setEligibilityStatusToVerified(onFile);
		this.attachPrisonerOfWar(onFile,
				ImpreciseDateUtils.createImpreciseDate(2005, 1, 1),
				ImpreciseDateUtils.createImpreciseDate(2005, 2, 1));
		Eligibility e = new Eligibility();
		e.setPrimary();
		e.setType(this.getLookupService().getEligibilityTypeByCode(
				EligibilityType.PRISONER_OF_WAR.getCode()));
		onFile.getEnrollmentDetermination().addSecondaryEligibility(e);
		// Defect 5244 says person might have had ee data change so that prior
		// enrollment is P3. This test simulates
		// that scenario to make sure it works. The letter request is still
		// valid provided that person has P3 and
		// POW eligibility, etc.
		onFile.getEnrollmentDetermination().setPriorityGroup(
				this.getLookupService().getEnrollmentPriorityGroupByCode(
						EnrollmentPriorityGroup.GROUP_3.getCode()));
		Person incoming = this.getPersonService().save(onFile);

		ValidationMessages vm = this.doValidateLetterSendRequest(incoming,
				Boolean.FALSE, ComLetterTemplateType.FORM_NUMBER_640C,
				ComMailingStatusType.MAILED_BY_AAC, true, mailType);
		assertTrue(vm.getCount() == 0);
	}

	public void testValidateSendRequest_630G() throws Exception {
		Person onFile = this.createTestPerson();
		onFile.getEnrollmentDetermination().setPriorityGroup(
				this.getLookupService().getEnrollmentPriorityGroupByCode(
						EnrollmentPriorityGroup.GROUP_3.getCode()));
		Person incoming = this.savePerson(onFile);
		this.attachPurpleHeartWithPHStatusPending(incoming);
		DecorationStatus ds = this.getLookupService()
				.getDecorationStatusByCode(
						DecorationStatus.CODE_CONFIRMED.getName());
		((PurpleHeart) incoming.getDecorations().iterator().next())
				.setStatus(ds);

		ValidationMessages vm = this.doValidateLetterSendRequest(incoming,
				Boolean.FALSE, ComLetterTemplateType.FORM_NUMBER_630G,
				ComMailingStatusType.MAILED_BY_AAC);
		assertTrue(vm.getCount() == 0);
	}

	public void testValidateSendRequest_630H_Neg() throws Exception {
		Person onFile = this.createTestPerson();
		onFile.getEnrollmentDetermination().setPriorityGroup(
				this.getLookupService().getEnrollmentPriorityGroupByCode(
						EnrollmentPriorityGroup.GROUP_3.getCode()));
		Person incoming = this.savePerson(onFile);
		this.attachPurpleHeartWithPHStatusPending(incoming);
		DecorationStatus ds = this.getLookupService()
				.getDecorationStatusByCode(
						DecorationStatus.CODE_CONFIRMED.getName());
		((PurpleHeart) incoming.getDecorations().iterator().next())
				.setStatus(ds);
		this.setServiceConnectedPercentage(new Integer(100), incoming);

		ValidationMessages vm = this.doValidateLetterSendRequest(incoming,
				Boolean.FALSE, ComLetterTemplateType.FORM_NUMBER_630H,
				ComMailingStatusType.MAILED_BY_AAC);
		assertTrue(vm.getCount() == 2);
	}

	public void testValidateSendRequest_640BWithPriorCD() throws Exception {
		//
		PersonIdEntityKeyImpl personKey = new PersonIdEntityKeyImpl(
				new BigDecimal("20722948"));
		Person incoming = this.getPersonService().getPerson(personKey);
		incoming.addAddress(this.createAddress(this.getLookupService()
				.getAddressTypeByCode(
						AddressType.CODE_PERMANENT_ADDRESS.getCode())));
		ValidationMessages vm = this.doValidateLetterSendRequest(incoming,
				Boolean.FALSE, ComLetterTemplateType.FORM_NUMBER_640B,
				ComMailingStatusType.SEND_TO_AAC);
		// Expect no change in P and No CD related change
		assertTrue(vm.getCount() == 0);
	}

	// 20722788

	public void testValidateSendRequest_640B() throws Exception {
		// no Prior CD
		Person onFile = this.createHistPerson();
		Person incoming = this.getPersonService().save(onFile);
		this.attachVerifiedEnrollment(incoming);
		this.attachCatastrophicDisability(incoming);
		setEnrollmentPriority(incoming, EnrollmentPriorityGroup.GROUP_4);
		ValidationMessages vm = this.doValidateLetterSendRequest(incoming,
				Boolean.FALSE, ComLetterTemplateType.FORM_NUMBER_640B,
				ComMailingStatusType.SEND_TO_AAC);
		assertTrue(vm.getCount() == 0);
	}

	/*
	 * public void testValidateSendRequest_640C() throws Exception{ //264243
	 * PersonIdEntityKeyImpl personKey = new PersonIdEntityKeyImpl(new
	 * BigDecimal("264243")); Person incoming =
	 * this.getPersonService().getPerson(personKey); this.attachPOW(incoming);
	 * setEnrollmentPriority(incoming, EnrollmentPriorityGroup.GROUP_3);
	 * ValidationMessages vm = this.doValidateLetterSendRequest(incoming,
	 * Boolean.FALSE, ComLetterTemplateType.FORM_NUMBER_640C,
	 * ComMailingStatusType.SEND_TO_AAC); assertTrue(vm.getCount() == 0); }
	 */

	public void testValidateSendRequest_604A() throws Exception {
		Person onFile = this.createHistPerson();
		Person incoming = this.getPersonService().save(onFile);
		this.attachVerifiedEnrollment(incoming);
		setEnrollmentPriority(incoming, EnrollmentPriorityGroup.GROUP_8);
		ValidationMessages vm = this.doValidateLetterSendRequest(incoming,
				Boolean.FALSE, ComLetterTemplateType.FORM_NUMBER_604A,
				ComMailingStatusType.SEND_TO_AAC);
		assertTrue(vm.getCount() == 0);
	}

	public void testValidateSendRequest_604B() throws Exception {
		Person onFile = this.createHistPerson();
		Person incoming = this.getPersonService().save(onFile);
		setEnrollmentPriority(incoming, EnrollmentPriorityGroup.GROUP_4);
		ValidationMessages vm = this.doValidateLetterSendRequest(incoming,
				Boolean.FALSE, ComLetterTemplateType.FORM_NUMBER_604B,
				ComMailingStatusType.SEND_TO_AAC);
		assertTrue(vm.getCount() == 0);
	}

	public void testValidateSendRequest_640F() throws Exception {
		Person onFile = this.createHistPerson();
		Person incoming = this.getPersonService().save(onFile);
		setEnrollmentPriority(incoming, EnrollmentPriorityGroup.GROUP_6);

		ValidationMessages vm = this.doValidateLetterSendRequest(incoming,
				Boolean.FALSE, ComLetterTemplateType.FORM_NUMBER_640F,
				ComMailingStatusType.SEND_TO_AAC);
		assertTrue(vm.getCount() == 0);
	}

	public void testValidateSendRequest_290_All() throws Exception {
		Person onFile = this
				.createSSAPerson(SSAVerificationStatus.INVALID_PER_SSA);
		Person incoming = this.getPersonService().save(onFile);

		ValidationMessages vm = this.doValidateLetterSendRequest(incoming,
				Boolean.FALSE, ComLetterTemplateType.FORM_NUMBER_290,
				ComMailingStatusType.SEND_TO_AAC);
		assertTrue(vm.getCount() == 0);
		vm = this.doValidateLetterSendRequest(incoming, Boolean.FALSE,
				ComLetterTemplateType.FORM_NUMBER_290,
				ComMailingStatusType.SEND_TO_AAC, true,
				LetterTriggerEvent.REMAIL_MAIL);
		assertTrue(vm.getCount() == 0);

		vm = this.doValidateLetterSendRequest(incoming, Boolean.FALSE,
				ComLetterTemplateType.FORM_NUMBER_290,
				ComMailingStatusType.SEND_TO_AAC, true,
				LetterTriggerEvent.MANUAL_MAIL);
		assertTrue(vm.getCount() == 0);
		incoming.getOfficialSsn().setSsaVerificationStatus(null);
		vm = this.doValidateLetterSendRequest(incoming, Boolean.FALSE,
				ComLetterTemplateType.FORM_NUMBER_290,
				ComMailingStatusType.SEND_TO_AAC, true,
				LetterTriggerEvent.MANUAL_MAIL);
		assertTrue(vm.getCount() == 1);
	}

	public void testValidateSendRequest_291() throws Exception {
		Person onFile = this
				.createSSAPerson(SSAVerificationStatus.INVALID_PER_SSA);
		Person incoming = this.getPersonService().save(onFile);

		ValidationMessages vm = this.doValidateLetterSendRequest(incoming,
				Boolean.FALSE, ComLetterTemplateType.FORM_NUMBER_291,
				ComMailingStatusType.SEND_TO_AAC);
		assertTrue(vm.getCount() == 0);
	}

	public void testValidateSendRequest_291_Remail() throws Exception {
		Person onFile = this
				.createSSAPerson(SSAVerificationStatus.INVALID_PER_SSA);
		Person incoming = this.getPersonService().save(onFile);

		ValidationMessages vm = doValidateLetterSendRequest(incoming,
				Boolean.FALSE, ComLetterTemplateType.FORM_NUMBER_291,
				ComMailingStatusType.SEND_TO_AAC, true,
				LetterTriggerEvent.REMAIL_MAIL);

		assertTrue(vm.getCount() == 0);
	}

	public void testValidateSendRequest_292_All() throws Exception {
		Person onFile = this
				.createSSAPerson(PseudoSSNReason.CODE_FOLLOWUP_REQD);
		Person incoming = this.getPersonService().save(onFile);

		ValidationMessages vm = this.doValidateLetterSendRequest(incoming,
				Boolean.FALSE, ComLetterTemplateType.FORM_NUMBER_292,
				ComMailingStatusType.SEND_TO_AAC);
		assertTrue(vm.getCount() == 0);

		vm = this.doValidateLetterSendRequest(incoming, Boolean.FALSE,
				ComLetterTemplateType.FORM_NUMBER_292,
				ComMailingStatusType.SEND_TO_AAC, true,
				LetterTriggerEvent.REMAIL_MAIL);
		assertTrue(vm.getCount() == 0);

		vm = this.doValidateLetterSendRequest(incoming, Boolean.FALSE,
				ComLetterTemplateType.FORM_NUMBER_292,
				ComMailingStatusType.SEND_TO_AAC, true,
				LetterTriggerEvent.MANUAL_MAIL);
		assertTrue(vm.getCount() == 0);
		SSN ssn = SSN.getSSNOfType(incoming.getSsns(),
				SSNType.CODE_PSEUDO.getCode());
		ssn.setPseudoSSNReason(null);
		vm = this.doValidateLetterSendRequest(incoming, Boolean.FALSE,
				ComLetterTemplateType.FORM_NUMBER_292,
				ComMailingStatusType.SEND_TO_AAC, true,
				LetterTriggerEvent.MANUAL_MAIL);
		assertTrue(vm.getCount() == 1);
	}

	public void testValidateSendRequest_293_All() throws Exception {
		Person onFile = this
				.createSSAPerson(PseudoSSNReason.CODE_FOLLOWUP_REQD);
		Person incoming = this.getPersonService().save(onFile);

		ValidationMessages vm = this.doValidateLetterSendRequest(incoming,
				Boolean.FALSE, ComLetterTemplateType.FORM_NUMBER_293,
				ComMailingStatusType.SEND_TO_AAC);
		assertTrue(vm.getCount() == 0);

		vm = this.doValidateLetterSendRequest(incoming, Boolean.FALSE,
				ComLetterTemplateType.FORM_NUMBER_293,
				ComMailingStatusType.SEND_TO_AAC, true,
				LetterTriggerEvent.REMAIL_MAIL);
		assertTrue(vm.getCount() == 0);

		vm = this.doValidateLetterSendRequest(incoming, Boolean.FALSE,
				ComLetterTemplateType.FORM_NUMBER_293,
				ComMailingStatusType.SEND_TO_AAC, true,
				LetterTriggerEvent.MANUAL_MAIL);
		assertTrue(vm.getCount() == 0);
		SSN ssn = SSN.getSSNOfType(incoming.getSsns(),
				SSNType.CODE_PSEUDO.getCode());
		ssn.setPseudoSSNReason(null);
		vm = this.doValidateLetterSendRequest(incoming, Boolean.FALSE,
				ComLetterTemplateType.FORM_NUMBER_293,
				ComMailingStatusType.SEND_TO_AAC, true,
				LetterTriggerEvent.MANUAL_MAIL);
		assertTrue(vm.getCount() == 1);
	}

	public void createHistoricalPerson() throws Exception {
		Person onFile = this.createHistPerson();
		onFile.getEnrollmentDetermination().setPriorityGroup(
				this.getLookupService().getEnrollmentPriorityGroupByCode(
						EnrollmentPriorityGroup.GROUP_5.getName()));
		Person incoming = this.getPersonService().save(onFile);
		this.setComplete();
		System.out.println("Person id="
				+ incoming.getPersonEntityKey().getKeyValueAsString());
	}

	// For creating data
	public void createHistoricalPersonPH() throws Exception {
		Person onFile = this.createPHPerson(EnrollmentPriorityGroup.GROUP_3);// this.createHistoricalPerson();
		// onFile.getEnrollmentDetermination().setPriorityGroup(this.getLookupService().getEnrollmentPriorityGroupByCode(EnrollmentPriorityGroup.GROUP_3.getName()));
		Person incoming = this.getPersonService().save(onFile);
		this.setComplete();
		System.out.println("Person id="
				+ incoming.getPersonEntityKey().getKeyValueAsString());
	}

	// For creating data
	public void createHistoricalPersonCd() throws Exception {
		Person onFile = this.createHistPerson();
		attachCatastrophicDisability(onFile);
		this.attachVerifiedEnrollment(onFile);
		setEnrollmentPriority(onFile, EnrollmentPriorityGroup.GROUP_4);
		Person incoming = this.getPersonService().save(onFile);
		this.setComplete();
		System.out.println("Person id="
				+ incoming.getPersonEntityKey().getKeyValueAsString());
	}

	// For creating data
	public void createHistoricalPersonP1() throws Exception {
		Person onFile = this.createHistPerson();
		onFile.getEnrollmentDetermination().setPriorityGroup(
				this.getLookupService().getEnrollmentPriorityGroupByCode(
						EnrollmentPriorityGroup.GROUP_1.getName()));
		Person incoming = this.getPersonService().save(onFile);
		this.setComplete();
		System.out.println("Person id="
				+ incoming.getPersonEntityKey().getKeyValueAsString());
	}

	// For creating data
	public Person createHistoricalPersonEGT() throws Exception {
		Person onFile = this.buildSimplePerson();
		this.attachVerifiedEnrollment(onFile);
		onFile.getEnrollmentDetermination()
				.setEnrollmentStatus(
						this.getLookupService()
								.getEnrollmentStatusByCode(
										EnrollmentStatus.CODE_REJECTED_BELOW_ENROLLMENT_GROUP_THRESHOLD
												.getCode()));
		this.setEligibilityStatusToVerified(onFile);
		onFile.addAddress(this.createAddress(this.getLookupService()
				.getAddressTypeByCode(
						AddressType.CODE_PERMANENT_ADDRESS.getName())));
		attachMeansTestWithMtCopayRequired(onFile, new BigDecimal(999),
				new BigDecimal(1000));

		return this.savePerson(onFile);

	}

	private Person createSSAPerson(SSAVerificationStatus.Code status)
			throws Exception {
		Person onFile = this.createHistPerson();
		SSN ssn = this.createSSN();
		onFile.addSsn(ssn);
		onFile.getOfficialSsn().setSsaVerificationStatus(
				this.getLookupService().getSSAVerificationStatusByCode(
						status.getName()));
		return onFile;
	}

	private Person createSSAPerson(PseudoSSNReason.Code reason)
			throws Exception {
		Person onFile = this.createHistPerson();
		SSN ssn = this.createSSN();
		ssn.setType(this.getLookupService().getSSNTypeByCode(
				SSNType.CODE_PSEUDO.getName()));
		ssn.setPseudoSSNReason(this.getLookupService()
				.getPseudoSSNReasonByCode(reason.getName()));
		onFile.addSsn(ssn);
		return onFile;
	}

	protected Person createHistPerson() throws Exception {
		Person onFile = this.createTestPerson();
		this.attachEligibilityVerification(onFile,
				EligibilityVerificationSource.CODE_CEV.getName());
		this.attachMeansTestWithMtCopayExempt(onFile);
		this.attachVerifiedEnrollment(onFile);
		return onFile;
	}

	public void testTriggerAutoMail() throws Exception {
		Person onFile = this.buildSimplePerson();
		Address address = this.createAddress();
		onFile.addAddress(address);
		Person incoming = this.getPersonService().save(onFile);

		this.getCommunicationRuleService().triggerAutomaticLetters(incoming,
				null, false, null);
	}

	private ValidationMessages doValidateLetterSendRequest(Person incoming,
			Boolean override, Code type, ComMailingStatusType.Code statusType)
			throws Exception {
		return doValidateLetterSendRequest(incoming, override, type,
				statusType, true);
	}

	private ValidationMessages doValidateLetterSendRequest(Person incoming,
			Boolean override, Code type, ComMailingStatusType.Code statusType,
			boolean prereqs) throws Exception {
		return doValidateLetterSendRequest(incoming, override, type,
				statusType, prereqs, LetterTriggerEvent.AUTO_MAIL);
	}

	private ValidationMessages doValidateLetterSendRequest(Person incoming,
			Boolean override, Code type, ComMailingStatusType.Code statusType,
			boolean prereqs, LetterTriggerEvent.MailType mailType)
			throws Exception {
		// Prerequisite letters
		CommsLogEntry _600c = this.createCommsLogEntry(incoming,
				ComLetterTemplateType.FORM_NUMBER_600C,
				ComMailingStatusType.MAILED_BY_AAC);
		this.getCommsLogEntryDAO().insert(_600c);
		CommsLogEntry _600d = this.createCommsLogEntry(incoming,
				ComLetterTemplateType.FORM_NUMBER_600D,
				ComMailingStatusType.MAILED_BY_AAC);
		this.getCommsLogEntryDAO().insert(_600d);
		CommsLogEntry _630a = this.createCommsLogEntry(incoming,
				ComLetterTemplateType.FORM_NUMBER_630A,
				ComMailingStatusType.MAILED_BY_AAC);
		this.getCommsLogEntryDAO().insert(_630a);
		CommsLogEntry _630d = this.createCommsLogEntry(incoming,
				ComLetterTemplateType.FORM_NUMBER_630D,
				ComMailingStatusType.MAILED_BY_AAC);
		this.getCommsLogEntryDAO().insert(_630d);
		CommsLogEntry _601b = this.createCommsLogEntry(incoming,
				ComLetterTemplateType.FORM_NUMBER_601B,
				ComMailingStatusType.MAILED_BY_AAC);
		this.getCommsLogEntryDAO().insert(_601b);
		CommsLogEntry _640c = this.createCommsLogEntry(incoming,
				ComLetterTemplateType.FORM_NUMBER_640C,
				ComMailingStatusType.MAILED_BY_AAC);
		this.getCommsLogEntryDAO().insert(_640c);
		CommsLogEntry _640d = this.createCommsLogEntry(incoming,
				ComLetterTemplateType.FORM_NUMBER_640D,
				ComMailingStatusType.MAILED_BY_AAC);
		this.getCommsLogEntryDAO().insert(_640d);
		CommsLogEntry _290 = this.createCommsLogEntry(incoming,
				ComLetterTemplateType.FORM_NUMBER_290,
				ComMailingStatusType.MAILED_BY_AAC);
		_290.setMailingDate(this.getDate(2005, 12, 1));
		this.getCommsLogEntryDAO().insert(_290);
		CommsLogEntry _291 = this.createCommsLogEntry(incoming,
				ComLetterTemplateType.FORM_NUMBER_291,
				ComMailingStatusType.MAILED_BY_AAC);
		_291.setMailingDate(this.getDate(2005, 12, 1));
		this.getCommsLogEntryDAO().insert(_291);

		CommsLogEntry _292 = this.createCommsLogEntry(incoming,
				ComLetterTemplateType.FORM_NUMBER_292,
				ComMailingStatusType.MAILED_BY_AAC);
		_292.setMailingDate(this.getDate(2005, 12, 1));
		this.getCommsLogEntryDAO().insert(_292);

		CommsLogEntry _293 = this.createCommsLogEntry(incoming,
				ComLetterTemplateType.FORM_NUMBER_293,
				ComMailingStatusType.MAILED_BY_AAC);
		_293.setMailingDate(this.getDate(2005, 12, 1));
		this.getCommsLogEntryDAO().insert(_293);

		// The historical log entries
		List entries = new ArrayList();
		if (prereqs) {
			if (type.equals(ComLetterTemplateType.FORM_NUMBER_601B)) {
				_601b.setMailingDate(this.getDate(2003, 1, 1));
				entries.add((CommsLogEntry) _601b.clone());
			} else {
				entries.add((CommsLogEntry) _600c.clone());
				entries.add((CommsLogEntry) _600d.clone());
				entries.add((CommsLogEntry) _630a.clone());
				entries.add((CommsLogEntry) _630d.clone());
				entries.add((CommsLogEntry) _640c.clone());
				entries.add((CommsLogEntry) _640d.clone());
				entries.add((CommsLogEntry) _290.clone());
				entries.add((CommsLogEntry) _291.clone());
				entries.add((CommsLogEntry) _292.clone());
				entries.add((CommsLogEntry) _293.clone());
			}
		} else {
			if (type.equals(ComLetterTemplateType.FORM_NUMBER_601B)) {
				_601b.setMailingDate(this.getDate(2006, 1, 1));

				entries.add((CommsLogEntry) _600c.clone());
				entries.add((CommsLogEntry) _600d.clone());
				entries.add((CommsLogEntry) _601b.clone());
			}

		}
		ComLetterTemplateType form = this.getComLetterTemplateType(type);
		return this.getRuleValidationService().validateLetterSendRequest(
				incoming, form, "", mailType, override, entries, null,
				CommsLetterRequestService.CONTEXT_REQUEST_LETTER);
	}

	/**
	 * @return Returns the commsLogEntryDAO.
	 */
	public CommsLogEntryDAO getCommsLogEntryDAO() {
		return commsLogEntryDAO;
	}

	/**
	 * @param commsLogEntryDAO
	 *            The commsLogEntryDAO to set.
	 */
	public void setCommsLogEntryDAO(CommsLogEntryDAO commsLogEntryDAO) {
		this.commsLogEntryDAO = commsLogEntryDAO;
	}

	protected Person createTestPerson() throws Exception {
		Person onFile = this.buildSimplePerson();
		Address address = this.createAddress();
		address.setType(this.getLookupService().getAddressTypeByCode(
				AddressType.CODE_PERMANENT_ADDRESS.getCode()));
		onFile.addAddress(address);
		this.attachVerifiedEnrollment(onFile);
		return onFile;
	}

	protected ComLetterTemplateType getComLetterTemplateType(Code type)
			throws Exception {
		return this.getLookupService().getComLetterTemplateTypeByCode(
				type.getName());
	}

	protected void setEnrollmentPriority(Person person,
			EnrollmentPriorityGroup.Code type) throws Exception {
		person.getEnrollmentDetermination().setPriorityGroup(
				this.getLookupService().getEnrollmentPriorityGroupByCode(
						type.getName()));
	}

	protected CommsLogEntry createCommsLogEntry(Person person,
			ComLetterTemplateType.Code type, ComMailingStatusType.Code status)
			throws Exception {
		CommsLogEntry entry = this.createCommsLogEntry(person, type.getName());
		ComMailingStatusType statusCode = this.getLookupService()
				.getComMailingStatusTypeByCode(status.getName());
		entry.getLatestMailingStatus().setMailingStatus(statusCode);
		return entry;
	}

	protected Person createPHPerson(EnrollmentPriorityGroup.Code code)
			throws Exception {
		Person onFile = this.createHistPerson();
		this.setEnrollmentPriority(onFile, code);
		this.attachPurpleHeartWithPHStatusPending(onFile);
		onFile.getPurpleHeart().setStatus(
				this.getLookupService().getDecorationStatusByCode(
						DecorationStatus.CODE_CONFIRMED.getCode()));
		this.setEligibilityStatusToVerified(onFile);
		PurpleHeartDocument purpleHeartDocument = new PurpleHeartDocument();
		purpleHeartDocument.setAcceptable(Boolean.TRUE);
		purpleHeartDocument.setDocumentType((PHDocumentType) this
				.getLookupService().getByCode(PHDocumentType.class, "1"));

		onFile.getPurpleHeart().addDocument(purpleHeartDocument);

		return onFile;
	}

}
