/********************************************************************
 * Copyright  2004 VHA. All rights reserved
 ********************************************************************/
package gov.va.med.esr.common.rule.service;

// Java classes
import java.util.Date;

// Library classes
import org.apache.commons.lang.Validate;

// Framework classes

// ESR classes
import gov.va.med.esr.common.infra.ImpreciseDateUtils;
import gov.va.med.esr.common.model.person.Person;
import gov.va.med.esr.common.model.ee.EligibilityVerification;
import gov.va.med.esr.common.model.ee.IneligibilityFactor;
import gov.va.med.esr.common.model.ee.MilitaryService;
import gov.va.med.esr.common.model.ee.MilitaryServiceEpisode;
import gov.va.med.esr.common.model.ee.MilitaryServiceSiteRecord;
import gov.va.med.esr.common.model.ee.ServiceConnectionAward;
import gov.va.med.esr.common.model.lookup.ServiceBranch;
import gov.va.med.esr.common.model.lookup.VAFacility;
import gov.va.med.esr.common.util.RuleAbstractTestCase;
import gov.va.med.esr.common.model.lookup.ServicePeriod;

/**
 * Provides methods to test BAL rules in ConflictExperienceValidation rule flow
 *
 * Project: Common</br>
 * Created on: 4:45:47 PM </br>
 *
 * @author DNS   LEV
 */
public class CalculatePOSRuleServiceTest extends RuleAbstractTestCase {

	/**
	 * An instance of onFile
	 */
	private Person onFile = null;

	/**
	 * constructor
	 */
	public CalculatePOSRuleServiceTest(String testName) throws Exception {
		super(testName);
	}
	
	public void testRegularExpr() throws Exception {
		String ACTION_COMMENT_SEPARATOR = System.getProperty("line.separator") + System.getProperty("line.separator");
		String test = escapeRegExprChars(ACTION_COMMENT_SEPARATOR);
	}
	
    /**
     * Escape certain regular expression characters
     * @param name
     * @return filtered name
     */    
    private static String escapeRegExprChars(String name) {
 	   if (name == null) {
 		   return null;
 	   }
        StringBuffer sb = new StringBuffer();
        for (int i = 0; i < name.length(); i++) {
            char curChar = name.charAt(i);
            switch (curChar) {
	            case '\\': 
	            case '/':
	            case ',':
	            case '.':
	            case '+':
	            case '-':
	            case '"':
	            case '@':
	            case '!':
	            case '?':
	            case '<':
	            case '>':
	            case '(':
	            case ')':  
	            case '[':
	            case ']': 
                case '{':
                case '}': 	            	
	            case ';':
	            case ':':
	            case '^':
	            case '$':
	            case '*':
             	   break;
                default:
                    sb.append(curChar);
            }
        }       
        return sb.toString();
    }	
	public void testisPOSPostKorean() throws Exception {
		/**
		 * Post Korean  2/1/1955  2/27/1961
		 */
		this.getPersonService().save( onFile );
		
		// create an incoming record 
		Person incoming = (Person)onFile.clone();
		incoming.setBirthRecord( this.createBirthRecord( ImpreciseDateUtils.createImpreciseDate(1925,1,1)));
		
		// Create a site record for HEC with one MSE
		MilitaryService ms = new MilitaryService();
		MilitaryServiceSiteRecord record = this.createMilitaryServiceSireRecord();
		record.setSite( this.getVaFacility( VAFacility.CODE_HEC ) );
		
		MilitaryServiceEpisode mse = new MilitaryServiceEpisode();
		mse.setServiceBranch( this.getServiceBranch( ServiceBranch.CODE_ARMY ) );
		mse.setStartDate( ImpreciseDateUtils.createImpreciseDate( 1957, 12, 1 ) );
		mse.setEndDate( ImpreciseDateUtils.createImpreciseDate( 1961, 1, 1 ) );
		mse.setDischargeType( this.getRandomDischargeType() );
		
		record.addMilitaryServiceEpisode( mse );
		ms.addMilitaryServiceSiteRecord( record );
		incoming.setMilitaryService( ms );
		
		try {
			Person updated = this.getMilitaryInfoService().updateCalculatePeriodOfService( incoming );
			ms = updated.getMilitaryService();
			record = ms.getHECMilitaryServiceSiteRecord();
			
			// Get a service period to check if the newly calculated period of service is correct
			ServicePeriod period = record.getServicePeriod();
			String code = period != null ? period.getCode() : null;
			assertEquals( code, ServicePeriod.CODE_POST_KOREAN.getName() );
		}
		catch( Exception e ) {
			if( logger.isDebugEnabled() ) {
				logger.debug( "Got an un-expected validation exception ", e );
			}
			fail( "Got an un-expected validation exception", e );
		}
	}
	
	public void testisPOSPostVietnamEra() throws Exception {
		/**
		 * Post-Vietnam Era  5/8/75 to 08/01/1990
		 */
		this.getPersonService().save( onFile );
		
		// create an incoming record 
		Person incoming = (Person)onFile.clone();
		incoming.setBirthRecord( this.createBirthRecord( ImpreciseDateUtils.createImpreciseDate(1960,1,1)));
		
		// Create a site record for HEC with one MSE
		MilitaryService ms = new MilitaryService();
		MilitaryServiceSiteRecord record = this.createMilitaryServiceSireRecord();
		record.setSite( this.getVaFacility( VAFacility.CODE_HEC ) );
		
		MilitaryServiceEpisode mse = new MilitaryServiceEpisode();
		mse.setServiceBranch( this.getServiceBranch( ServiceBranch.CODE_ARMY ) );
		mse.setStartDate( ImpreciseDateUtils.createImpreciseDate( 1988, 12, 1 ) );
		mse.setEndDate( ImpreciseDateUtils.createImpreciseDate( 1989, 1, 31 ) );
		mse.setDischargeType( this.getRandomDischargeType() );
		
		record.addMilitaryServiceEpisode( mse );
		ms.addMilitaryServiceSiteRecord( record );
		incoming.setMilitaryService( ms );
		
		try {
			Person updated = this.getMilitaryInfoService().updateCalculatePeriodOfService( incoming );
			ms = updated.getMilitaryService();
			record = ms.getHECMilitaryServiceSiteRecord();

			// Get a service period to check if the newly calculated period of service is correct
			ServicePeriod period = record.getServicePeriod();
			String code = period != null ? period.getCode() : null;
			assertEquals( code, ServicePeriod.CODE_POST_VIETNAM.getName() );
		}
		catch( Exception e ) {
			if( logger.isDebugEnabled() ) {
				logger.debug( "Got an un-expected validation exception ", e );
			}
			fail( "Got an un-expected validation exception", e );
		}
	}
	
	public void testisPOSSpanishAmerican() throws Exception {
		/**
		 * Spanish American 4/21/1898  7/4/1902                           
		 */
		this.getPersonService().save( onFile );
		
		// create an incoming record 
		Person incoming = (Person)onFile.clone();
		incoming.setBirthRecord( this.createBirthRecord( ImpreciseDateUtils.createImpreciseDate(1880,1,1)));
		
		// Create a site record for HEC with one MSE
		MilitaryService ms = new MilitaryService();
		MilitaryServiceSiteRecord record = this.createMilitaryServiceSireRecord();
		record.setSite( this.getVaFacility( VAFacility.CODE_HEC ) );
		
		MilitaryServiceEpisode mse = new MilitaryServiceEpisode();
		mse.setServiceBranch( this.getServiceBranch( ServiceBranch.CODE_ARMY ) );
		mse.setStartDate( ImpreciseDateUtils.createImpreciseDate( 1899, 12, 1 ) );
		mse.setEndDate( ImpreciseDateUtils.createImpreciseDate( 1904, 1, 31 ) );
		mse.setDischargeType( this.getRandomDischargeType() );
		
		record.addMilitaryServiceEpisode( mse );
		ms.addMilitaryServiceSiteRecord( record );
		incoming.setMilitaryService( ms );
		
		try {
			Person updated = this.getMilitaryInfoService().updateCalculatePeriodOfService( incoming );
			ms = updated.getMilitaryService();
			record = ms.getHECMilitaryServiceSiteRecord();

			// Get a service period to check if the newly calculated period of service is correct
			ServicePeriod period = record.getServicePeriod();
			String code = period != null ? period.getCode() : null;
			assertEquals( code, ServicePeriod.CODE_SPANISH_AMERICAN.getName() );
		}
		catch( Exception e ) {
			if( logger.isDebugEnabled() ) {
				logger.debug( "Got an un-expected validation exception ", e );
			}
			fail( "Got an un-expected validation exception", e );
		}
	}
	
	public void testisPOSWorldWarI() throws Exception {
		/**
		 * World War I  4/6/1917  11/11/1918
		 */
		this.getPersonService().save( onFile );
		
		// create an incoming record 
		Person incoming = (Person)onFile.clone();
		incoming.setBirthRecord( this.createBirthRecord( ImpreciseDateUtils.createImpreciseDate(1900,1,1)));
		
		// Create a site record for HEC with one MSE
		MilitaryService ms = new MilitaryService();
		MilitaryServiceSiteRecord record = this.createMilitaryServiceSireRecord();
		record.setSite( this.getVaFacility( VAFacility.CODE_HEC ) );
		
		MilitaryServiceEpisode mse = new MilitaryServiceEpisode();
		mse.setServiceBranch( this.getServiceBranch( ServiceBranch.CODE_ARMY ) );
		mse.setStartDate( ImpreciseDateUtils.createImpreciseDate( 1917, 12, 1 ) );
		mse.setEndDate( ImpreciseDateUtils.createImpreciseDate( 1920, 1, 31 ) );
		mse.setDischargeType( this.getRandomDischargeType() );
		
		record.addMilitaryServiceEpisode( mse );
		ms.addMilitaryServiceSiteRecord( record );
		incoming.setMilitaryService( ms );
		
		try {
			Person updated = this.getMilitaryInfoService().updateCalculatePeriodOfService( incoming );
			ms = updated.getMilitaryService();
			record = ms.getHECMilitaryServiceSiteRecord();

			// Get a service period to check if the newly calculated period of service is correct
			ServicePeriod period = record.getServicePeriod();
			String code = period != null ? period.getCode() : null;
			assertEquals( code, ServicePeriod.CODE_WORLD_WAR_I.getName() );
		}
		catch( Exception e ) {
			if( logger.isDebugEnabled() ) {
				logger.debug( "Got an un-expected validation exception ", e );
			}
			fail( "Got an un-expected validation exception", e );
		}
	}
	
	public void testisPOSMerchantMarine() throws Exception {
		/**
		 * Merchant Marine: 12/01/1941 - 08/15/1945 
		 * World War II: 12/07/1941 - 12/31/1946
		 */
		this.getPersonService().save( onFile );
		
		// create an incoming record 
		Person incoming = (Person)onFile.clone();
		incoming.setBirthRecord( this.createBirthRecord( ImpreciseDateUtils.createImpreciseDate(1920,1,1)));
		
		// Create a site record for HEC with one MSE
		MilitaryService ms = new MilitaryService();
		MilitaryServiceSiteRecord record = this.createMilitaryServiceSireRecord();
		record.setSite( this.getVaFacility( VAFacility.CODE_HEC ) );
		
		MilitaryServiceEpisode mse = new MilitaryServiceEpisode();
		mse.setServiceBranch( this.getServiceBranch( ServiceBranch.CODE_ARMY ) );
		mse.setStartDate( ImpreciseDateUtils.createImpreciseDate( 1941, 12, 2 ) );
		mse.setEndDate( ImpreciseDateUtils.createImpreciseDate( 1941, 12, 6 ) );
		mse.setDischargeType( this.getRandomDischargeType() );
		
		record.addMilitaryServiceEpisode( mse );
		ms.addMilitaryServiceSiteRecord( record );
		incoming.setMilitaryService( ms );
		
		try {
			Person updated = this.getMilitaryInfoService().updateCalculatePeriodOfService( incoming );
			ms = updated.getMilitaryService();
			record = ms.getHECMilitaryServiceSiteRecord();

			// Get a service period to check if the newly calculated period of service is correct
			ServicePeriod period = record.getServicePeriod();
			String code = period != null ? period.getCode() : null;
			assertEquals( code, ServicePeriod.CODE_MERCHANT_MARINE.getName() );
		}
		catch( Exception e ) {
			if( logger.isDebugEnabled() ) {
				logger.debug( "Got an un-expected validation exception ", e );
			}
			fail( "Got an un-expected validation exception", e );
		}
	}
	
	public void testisPOSWorldWarII() throws Exception {
		/**
		 * World War II  12/7/1941- 12/31/1946
		 */
		this.getPersonService().save( onFile );
		
		// create an incoming record 
		Person incoming = (Person)onFile.clone();
		incoming.setBirthRecord( this.createBirthRecord( ImpreciseDateUtils.createImpreciseDate(1920,1,1)));
		
		// Create a site record for HEC with one MSE
		MilitaryService ms = new MilitaryService();
		MilitaryServiceSiteRecord record = this.createMilitaryServiceSireRecord();
		record.setSite( this.getVaFacility( VAFacility.CODE_HEC ) );
		
		MilitaryServiceEpisode mse = new MilitaryServiceEpisode();
		mse.setServiceBranch( this.getServiceBranch( ServiceBranch.CODE_ARMY ) );
		mse.setStartDate( ImpreciseDateUtils.createImpreciseDate( 1942, 6, 27 ) );
		mse.setEndDate( ImpreciseDateUtils.createImpreciseDate( 1943, 1, 31 ) );
		mse.setDischargeType( this.getRandomDischargeType() );
		
		record.addMilitaryServiceEpisode( mse );
		ms.addMilitaryServiceSiteRecord( record );
		incoming.setMilitaryService( ms );
		
		try {
			Person updated = this.getMilitaryInfoService().updateCalculatePeriodOfService( incoming );
			ms = updated.getMilitaryService();
			record = ms.getHECMilitaryServiceSiteRecord();

			// Get a service period to check if the newly calculated period of service is correct
			ServicePeriod period = record.getServicePeriod();
			String code = period != null ? period.getCode() : null;
			assertEquals( code, ServicePeriod.CODE_WORLD_WAR_II.getName() );
		}
		catch( Exception e ) {
			if( logger.isDebugEnabled() ) {
				logger.debug( "Got an un-expected validation exception ", e );
			}
			fail( "Got an un-expected validation exception", e );
		}
	}
	
	/** Test rule isPOSKorean in CalculatePeriodOfService.irl
	 * @throws Exception In case of errors
	 */
	public void testisPOSKorean() throws Exception {
		
		/**
		 * Korean- 6/27/1950  1/31/1955
		 */
		this.getPersonService().save( onFile );
		
		// create an incoming record 
		Person incoming = (Person)onFile.clone();
		incoming.setBirthRecord( this.createBirthRecord( ImpreciseDateUtils.createImpreciseDate(1920,1,1)));
		
		// Create a site record for HEC with one MSE
		MilitaryService ms = new MilitaryService();
		MilitaryServiceSiteRecord record = this.createMilitaryServiceSireRecord();
		record.setSite( this.getVaFacility( VAFacility.CODE_HEC ) );
		
		MilitaryServiceEpisode mse = new MilitaryServiceEpisode();
		mse.setServiceBranch( this.getServiceBranch( ServiceBranch.CODE_ARMY ) );
		mse.setStartDate( ImpreciseDateUtils.createImpreciseDate( 1945, 6, 27 ) );
		mse.setEndDate( ImpreciseDateUtils.createImpreciseDate( 1960, 1, 31 ) );
		mse.setDischargeType( this.getRandomDischargeType() );
		
		record.addMilitaryServiceEpisode( mse );
		ms.addMilitaryServiceSiteRecord( record );
		incoming.setMilitaryService( ms );
		
		try {
			Person updated = this.getMilitaryInfoService().updateCalculatePeriodOfService( incoming );
			ms = updated.getMilitaryService();
			record = ms.getHECMilitaryServiceSiteRecord();

			// Get a service period to check if the newly calculated period of service is correct
			ServicePeriod period = record.getServicePeriod();
			String code = period != null ? period.getCode() : null;
			assertEquals( code, ServicePeriod.CODE_KOREAN.getName() );
		}
		catch( Exception e ) {
			if( logger.isDebugEnabled() ) {
				logger.debug( "Got an un-expected validation exception ", e );
			}
			fail( "Got an un-expected validation exception", e );
		}
		
	}
	
	public void testisPOSVietnamEra() throws Exception {
		
		/**
		 * Vietnam Era 2/28/1961  5/7/1975
		 */
		onFile.setBirthRecord( this.createBirthRecord( ImpreciseDateUtils.createImpreciseDate(1920,1,1)));
		this.getPersonService().save( onFile );
		
		// create an incoming record 
		Person incoming = (Person)onFile.clone();
		
		// Create a site record for HEC with one MSE
		MilitaryService ms = new MilitaryService();
		MilitaryServiceSiteRecord record = this.createMilitaryServiceSireRecord();
		record.setSite( this.getVaFacility( VAFacility.CODE_HEC ) );
		
		MilitaryServiceEpisode mse = new MilitaryServiceEpisode();
		mse.setServiceBranch( this.getServiceBranch( ServiceBranch.CODE_ARMY ) );
		mse.setStartDate( ImpreciseDateUtils.createImpreciseDate( 1970, 6, 27 ) );
		mse.setEndDate( ImpreciseDateUtils.createImpreciseDate( 1980, 1, 31 ) );
		mse.setDischargeType( this.getRandomDischargeType() );
		
		record.addMilitaryServiceEpisode( mse );
		ms.addMilitaryServiceSiteRecord( record );
		incoming.setMilitaryService( ms );
		
		try {
			Person updated = this.getMilitaryInfoService().updateCalculatePeriodOfService( incoming );
			ms = updated.getMilitaryService();
			record = ms.getHECMilitaryServiceSiteRecord();

			// Get a service period to check if the newly calculated period of service is correct
			ServicePeriod period = record.getServicePeriod();
			String code = period != null ? period.getCode() : null;
			assertEquals( code, ServicePeriod.CODE_VIETNAM_ERA.getName() );
		}
		catch( Exception e ) {
			if( logger.isDebugEnabled() ) {
				logger.debug( "Got an un-expected validation exception ", e );
			}
			fail( "Got an un-expected validation exception", e );
		}
	}
	
	public void testisPOSPersianGulfWar() throws Exception {
		/**
		 * Persian Gulf War  8/2/1990
		 */
		onFile.setBirthRecord( this.createBirthRecord( ImpreciseDateUtils.createImpreciseDate(1920,1,1)));
		this.getPersonService().save( onFile );
		
		// create an incoming record 
		Person incoming = (Person)onFile.clone();
		
		// Create a site record for HEC with one MSE
		MilitaryService ms = new MilitaryService();
		MilitaryServiceSiteRecord record = this.createMilitaryServiceSireRecord();
		record.setSite( this.getVaFacility( VAFacility.CODE_HEC ) );
		
		MilitaryServiceEpisode mse = new MilitaryServiceEpisode();
		mse.setServiceBranch( this.getServiceBranch( ServiceBranch.CODE_ARMY ) );
		mse.setStartDate( ImpreciseDateUtils.createImpreciseDate( 1985, 6, 27 ) );
		mse.setEndDate( ImpreciseDateUtils.createImpreciseDate( 1995, 1, 31 ) );
		mse.setDischargeType( this.getRandomDischargeType() );
		
		record.addMilitaryServiceEpisode( mse );
		ms.addMilitaryServiceSiteRecord( record );
		incoming.setMilitaryService( ms );
		
		try {
			Person updated = this.getMilitaryInfoService().updateCalculatePeriodOfService( incoming );
			ms = updated.getMilitaryService();
			record = ms.getHECMilitaryServiceSiteRecord();

			// Get a service period to check if the newly calculated period of service is correct
			ServicePeriod period = record.getServicePeriod();
			String code = period != null ? period.getCode() : null;
			assertEquals( code, ServicePeriod.CODE_PERSIAN_GULF_WAR.getName() );
		}
		catch( Exception e ) {
			if( logger.isDebugEnabled() ) {
				logger.debug( "Got an un-expected validation exception ", e );
			}
			fail( "Got an un-expected validation exception", e );
		}
	}
	
	/** Test rule assignPOSDefault in CalculatePeriodOfService.irl
	 * @throws Exception In case of errors
	 */
	public void testassignPOSDefault() throws Exception {

		/** 3337[UC23.26] Period of Service (POS) 
		 * The system will automatically assign the HEC (Veteran) Period of Service based on the following criteria:                                                                                       The system assigns the POS using the veterans Service Entry Dates and Service Separation Dates and assigns the POS in the following order                                                   
		 * Persian Gulf War  8/2/1990
		 * Vietnam Era 2/28/1961  5/7/1975
		 * Korean- 6/27/1950  1/31/1955
		 * World War II  12/7/1941- 12/31/1946
		 * Merchant Marine- 12/1941- 8/15/1945 
		 * World War I  4/6/1917  11/11/1918
		 * Spanish American 4/21/1898  7/4/1902                           
		 * Post-Vietnam Era  On or after 5/8/75
		 * Post Korean  2/1/1955  2/27/1961
		 * Pre-Korean  Peacetime before 6/27/1950                                                                                         
		 * 
		 * If there are NO Service Entry and Service Separation Dates 
		 * THEN the system will use Site Service Entry and Separation Dates on file 
		 * and assign in accordance with the order of assignment above.
		 * 
		 * If there is NO Service Data on file THEN the system assigns Other or None. 
		 */

		/**
		 * IF it is not true that
		 * ( there are HEC Entered military service episodes for the veteran OR
		 *   there are site entered military service episodes on file )
		 *	Then update a veterans period of service to OTHER OR NONE
		 **/
		this.getPersonService().save( onFile );
		
		// create an incoming record 
		Person incoming = (Person)onFile.clone();
		
		// Create an empty site record for HEC
		MilitaryService ms = new MilitaryService();
		MilitaryServiceSiteRecord record = this.createMilitaryServiceSireRecord();
		record.setSite( this.getVaFacility( VAFacility.CODE_HEC ) );
		ms.addMilitaryServiceSiteRecord( record );
		incoming.setMilitaryService( ms );
		
		try {
			Person updated = this.getMilitaryInfoService().updateCalculatePeriodOfService( incoming );
			ms = updated.getMilitaryService();
			record = ms.getHECMilitaryServiceSiteRecord();

			// Get a service period to check if the newly calculated period of service is correct
			ServicePeriod period = record.getServicePeriod();
			String code = period != null ? period.getCode() : null;
			assertEquals( code, ServicePeriod.CODE_OTHER_OR_NONE.getName() );
		}
		catch( Exception e ) {
			if( logger.isDebugEnabled() ) {
				logger.debug( "Got an un-expected validation exception ", e );
			}
			this.fail( "Got an un-expected validation exception", e );
		}
	}
	
	public void testisPOSPreKorean() throws Exception {
		/**
		 * Pre-Korean  01/01/1947 to 06/26/1961                          
		 * 
		 * If there are NO Service Entry and Service Separation Dates 
		 * THEN the system will use Site Service Entry and Separation Dates on file 
		 * and assign in accordance with the order of assignment above.
		 * 
		 * If there is NO Service Data on file THEN the system assigns Other or None. 
		 */
		this.getPersonService().save( onFile );
		
		// create an incoming record 
		Person incoming = (Person)onFile.clone();
		incoming.setBirthRecord( this.createBirthRecord( ImpreciseDateUtils.createImpreciseDate(1920,1,1)));
		
		// Create a site record for HEC with one MSE
		MilitaryService ms = new MilitaryService();
		MilitaryServiceSiteRecord record = this.createMilitaryServiceSireRecord();
		record.setSite( this.getVaFacility( VAFacility.CODE_HEC ) );
		
		MilitaryServiceEpisode mse = new MilitaryServiceEpisode();
		mse.setServiceBranch( this.getServiceBranch( ServiceBranch.CODE_ARMY ) );
		mse.setStartDate( ImpreciseDateUtils.createImpreciseDate( 1948, 6, 27 ) );
		mse.setEndDate( ImpreciseDateUtils.createImpreciseDate( 1949, 1, 31 ) );
		mse.setDischargeType( this.getRandomDischargeType() );
		
		record.addMilitaryServiceEpisode( mse );
		ms.addMilitaryServiceSiteRecord( record );
		incoming.setMilitaryService( ms );
		
		try {
			Person updated = this.getMilitaryInfoService().updateCalculatePeriodOfService( incoming );
			ms = updated.getMilitaryService();
			record = ms.getHECMilitaryServiceSiteRecord();

			// Get a service period to check if the newly calculated period of service is correct
			ServicePeriod period = record.getServicePeriod();
			String code = period != null ? period.getCode() : null;
			assertEquals( code, ServicePeriod.CODE_PRE_KOREAN.getName() );
		}
		catch( Exception e ) {
			if( logger.isDebugEnabled() ) {
				logger.debug( "Got an un-expected validation exception ", e );
			}
			fail( "Got an un-expected validation exception", e );
		}
	}
	
	/** Test rule AssignPOSOtherNonVeteran in CalculatePeriodOfService.irl
	 * @throws Exception In case of errors
	 */
	public void testAssignPOSOtherNonVeteran() throws Exception {
		
		
	   	// on file person has no ineligible date
	   	onFile.setIneligibilityFactor(null);	 
	   	// result person has no SC
	   	onFile.setServiceConnectionAward(null);
		this.getPersonService().save( onFile );
		
		// create an incoming record 
		Person incoming = (Person)onFile.clone();
		// result person has inelgibile date
		IneligibilityFactor ineligFactor = new IneligibilityFactor();
		ineligFactor.setIneligibleDate(new Date());
		ineligFactor.setReason("some reason");
		incoming.setIneligibilityFactor(ineligFactor);
		
		// Create a site record for HEC
		MilitaryService ms = new MilitaryService();
		MilitaryServiceSiteRecord record = this.createMilitaryServiceSireRecord();
		record.setSite( this.getVaFacility( VAFacility.CODE_HEC ) );
		ms.addMilitaryServiceSiteRecord( record );		
		incoming.setMilitaryService( ms );
				
		try {
	        this.getEnrollmentRuleService().manageIneligibility(incoming, onFile);						
			
			Person updated = this.getMilitaryInfoService().updateCalculatePeriodOfService( incoming );
			MilitaryService resultMs = updated.getMilitaryService();
			MilitaryServiceSiteRecord hecRecord = resultMs.getHECMilitaryServiceSiteRecord();			
			assertEquals( hecRecord.getServicePeriod().getCode(), ServicePeriod.CODE_OTHER_NON_VET.getName() );
		}
		catch( Exception e ) {
			if( logger.isDebugEnabled() ) {
				logger.debug( "Got an un-expected validation exception ", e );
			}
			fail( "Got an un-expected validation exception", e );
		}
		
	}	
	
	/** Test rule testisIneligibleAddedSCEqualYesUI() in CalculatePeriodOfService.irl
	 * @throws Exception In case of errors
	 */
	public void testisIneligibleAddedSCEqualYesUI() throws Exception {
		
		
	   	// on file person has no ineligible date
	   	onFile.setIneligibilityFactor(null);	 
	   	// result person has SC
	   	ServiceConnectionAward award = new ServiceConnectionAward();
	   	award.setServiceConnectedPercentage(new Integer(0));
	   	onFile.setServiceConnectionAward(award);
		MilitaryService ms = new MilitaryService();
		MilitaryServiceSiteRecord record = this.createMilitaryServiceSireRecord();
		record.setSite( this.getVaFacility( VAFacility.CODE_HEC ) );
		record.setServicePeriod(getLookupService().getServicePeriodByCode("0"));
		ms.addMilitaryServiceSiteRecord( record );		
		onFile.setMilitaryService( ms );

		this.getPersonService().save( onFile );
		
		// create an incoming record 
		Person incoming = (Person)onFile.clone();
		// result person has inelgibile date
		IneligibilityFactor ineligFactor = new IneligibilityFactor();
		ineligFactor.setIneligibleDate(new Date());
		ineligFactor.setReason("some reason");
		incoming.setIneligibilityFactor(ineligFactor);
		
		// Create a site record for HEC
				
		try {
	        this.getEnrollmentRuleService().manageIneligibility(incoming, onFile);						
			
		    this.getMilitaryRuleService().calculatePeriodOfService(incoming,onFile, "FIXME", null);	
			MilitaryService resultMs = onFile.getMilitaryService();
			MilitaryServiceSiteRecord hecRecord = resultMs.getHECMilitaryServiceSiteRecord();			
			assertEquals( hecRecord.getServicePeriod().getCode(), record.getServicePeriod().getCode() );
		}
		catch( Exception e ) {
			if( logger.isDebugEnabled() ) {
				logger.debug( "Got an un-expected validation exception ", e );
			}
			fail( "Got an un-expected validation exception", e );
		}
		
	}		

	/** Test rule testisIneligibleAddedSCEqualYesMessaging() in CalculatePeriodOfService.irl
	 * @throws Exception In case of errors
	 */
	public void testisIneligibleAddedSCEqualYesMessaging() throws Exception {
		
		
	   	// on file person has no ineligible date
	   	onFile.setIneligibilityFactor(null);	 
	   	// result person has SC
	   	ServiceConnectionAward award = new ServiceConnectionAward();
	   	award.setServiceConnectedPercentage(new Integer(0));
	   	onFile.setServiceConnectionAward(award);
		MilitaryService ms = new MilitaryService();
		MilitaryServiceSiteRecord record = this.createMilitaryServiceSireRecord();
		record.setSite( this.getVaFacility( VAFacility.CODE_HEC ) );
		record.setServicePeriod(getLookupService().getServicePeriodByCode("0"));
		ms.addMilitaryServiceSiteRecord( record );		
		onFile.setMilitaryService( ms );
		EligibilityVerification ev = new EligibilityVerification();
	   	ev.setEligibilityStatus(getLookupService().getEligibilityStatusByCode("P")); //Pending
	   	onFile.setEligibilityVerification(ev);

		this.getPersonService().save( onFile );
		
		// create an incoming record 
		Person incoming = (Person)onFile.clone();
		// result person has inelgibile date
		IneligibilityFactor ineligFactor = new IneligibilityFactor();
		ineligFactor.setIneligibleDate(new Date());
		ineligFactor.setReason("some reason");
		incoming.setIneligibilityFactor(ineligFactor);
		
		// Create a site record for HEC
				
		try {			
			
			this.getEnrollmentRuleService().processIneligibility(incoming,onFile,null,null,false);
		    this.getMilitaryRuleService().calculatePeriodOfService(incoming,onFile,"FIXME", null);	
			MilitaryService resultMs = onFile.getMilitaryService();
			MilitaryServiceSiteRecord hecRecord = resultMs.getHECMilitaryServiceSiteRecord();			
			assertEquals( hecRecord.getServicePeriod().getCode(), ServicePeriod.CODE_KOREAN.getName() );
		}
		catch( Exception e ) {
			if( logger.isDebugEnabled() ) {
				logger.debug( "Got an un-expected validation exception ", e );
			}
			fail( "Got an un-expected validation exception", e );
		}
		
	}			
	/** Test rule testIsIneligibleRemovedSCEqualYes() in CalculatePeriodOfService.irl
	 * @throws Exception In case of errors
	 */
	public void testIsIneligibleRemovedSCEqualYes() throws Exception {
		
		
	   	// on file person has no ineligible date
		IneligibilityFactor ineligFactor = new IneligibilityFactor();
		ineligFactor.setIneligibleDate(new Date());
		ineligFactor.setReason("some reason");
	   	onFile.setIneligibilityFactor(ineligFactor);	 
	   	// result person has SC
	   	ServiceConnectionAward award = new ServiceConnectionAward();
	   	award.setServiceConnectedPercentage(new Integer(0));
	   	onFile.setServiceConnectionAward(award);
		MilitaryService ms = new MilitaryService();
		MilitaryServiceSiteRecord record = this.createMilitaryServiceSireRecord();
		record.setSite( this.getVaFacility( VAFacility.CODE_HEC ) );
		record.setServicePeriod(getLookupService().getServicePeriodByCode("0")); //Koeran
		ms.addMilitaryServiceSiteRecord( record );		
		onFile.setMilitaryService( ms );

		this.getPersonService().save( onFile );
		
		// create an incoming record 
		Person incoming = (Person)onFile.clone();
		incoming.setIneligibilityFactor(null);
		
		// Create a site record for HEC
				
		try {
	        this.getEnrollmentRuleService().manageIneligibility(incoming, onFile);						
		    this.getMilitaryRuleService().calculatePeriodOfService(incoming,onFile,"FIXME", null);	
			MilitaryService resultMs = onFile.getMilitaryService();
			MilitaryServiceSiteRecord hecRecord = resultMs.getHECMilitaryServiceSiteRecord();			
			assertEquals( hecRecord.getServicePeriod().getCode(), record.getServicePeriod().getCode() );
		}
		catch( Exception e ) {
			if( logger.isDebugEnabled() ) {
				logger.debug( "Got an un-expected validation exception ", e );
			}
			fail( "Got an un-expected validation exception", e );
		}
		
	}		
	
	/** Test rule testIsIneligibleRemovedSCEqualYes() in CalculatePeriodOfService.irl
	 * @throws Exception In case of errors
	 */
	public void testIsIneligibleRemovedNonVet() throws Exception {
		
		
	   	// on file person has no ineligible date
		IneligibilityFactor ineligFactor = new IneligibilityFactor();
		ineligFactor.setIneligibleDate(new Date());
		ineligFactor.setReason("some reason");
	   	onFile.setIneligibilityFactor(ineligFactor);	 
		MilitaryService ms = new MilitaryService();
		MilitaryServiceSiteRecord record = this.createMilitaryServiceSireRecord();
		record.setSite( this.getVaFacility( VAFacility.CODE_HEC ) );
		record.setServicePeriod(getLookupService().getServicePeriodByCode("0"));//Koeran
		ms.addMilitaryServiceSiteRecord( record );		
		onFile.setMilitaryService( ms );

		this.getPersonService().save( onFile );
		
		// create an incoming record 
		Person incoming = (Person)onFile.clone();
		incoming.setIneligibilityFactor(null);
		this.attachEmployeeAsPrimary(incoming);
		incoming.setVeteran(Boolean.FALSE);
		
		// Create a site record for HEC
				
		try {
			
	        this.getEnrollmentRuleService().manageIneligibility(incoming, onFile);
	        this.getEnrollmentRuleService().manageReceivedEligibilities(incoming,onFile);
	        this.getEnrollmentRuleService().calculateEE(incoming,true,false);
		    this.getMilitaryRuleService().calculatePeriodOfService(incoming,onFile,"FIXME", null);	
			MilitaryService resultMs = onFile.getMilitaryService();
			MilitaryServiceSiteRecord hecRecord = resultMs.getHECMilitaryServiceSiteRecord();			
			assertEquals( hecRecord.getServicePeriod().getCode(), ServicePeriod.CODE_OTHER_NON_VET.getName() );
		}
		catch( Exception e ) {
			if( logger.isDebugEnabled() ) {
				logger.debug( "Got an un-expected validation exception ", e );
			}
			fail( "Got an un-expected validation exception", e );
		}
		
	}			
	/**
	 * @see gov.va.med.fw.util.AbstractTestCase#customSetUp()
	 */
	protected void customSetUp() throws Exception {
		super.customSetUp();
		onFile = this.buildSimplePerson();
	}

	/**
	 * @see gov.va.med.fw.util.AbstractTestCase#customTearDown()
	 */
	protected void customTearDown() throws Exception {
		super.customTearDown();
		onFile = null;
	}

	/**
	 * @see org.springframework.beans.factory.InitializingBean#afterPropertiesSet()
	 */
	public void afterPropertiesSet() throws Exception {
		super.afterPropertiesSet();
		Validate.notNull(this.getRuleValidationService(), "A rule validation service is required");
	}
}