/*******************************************************************************
 * Copyright  2004-2005 VHA. All rights reserved
 ******************************************************************************/
//Package
package gov.va.med.esr.service.impl;

//Java classes
import gov.va.med.esr.common.model.financials.GMTThreshold;
import gov.va.med.esr.common.model.financials.IncomeThreshold;
import gov.va.med.esr.common.model.lookup.*;
import gov.va.med.esr.common.model.party.Address;
import gov.va.med.esr.common.model.system.SystemParameter;
import gov.va.med.esr.common.persistent.demographic.ConfidentialAddressCategoryTypeDAO;
import gov.va.med.esr.common.persistent.lookup.CountyDAO;
import gov.va.med.esr.common.persistent.lookup.LookupsDAO;
import gov.va.med.esr.common.persistent.lookup.StateDAO;
import gov.va.med.esr.common.persistent.lookup.VAFacilityDAO;
import gov.va.med.esr.common.persistent.lookup.ZipCodeDAO;
import gov.va.med.esr.service.LookupService;
import gov.va.med.esr.service.SystemParameterService;
import gov.va.med.esr.service.UnknownLookupCodeException;
import gov.va.med.esr.service.UnknownLookupTypeException;
import gov.va.med.fw.model.lookup.AbstractCode;
import gov.va.med.fw.model.lookup.AbstractLookup;
import gov.va.med.fw.persistent.DAOException;
import gov.va.med.fw.persistent.DAOOperations;
import gov.va.med.fw.report.ReportExportedType;
import gov.va.med.fw.service.AbstractComponent;
import gov.va.med.fw.service.ServiceException;

import java.math.BigDecimal;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;

import org.apache.commons.lang.Validate;

/**
 * A service preconfigured with DAOs to query lookup (read-only) data. A caller
 * can either query look-up data using generic getByCode and findAll method or
 * use specific methods to query look-up data based on its business domain.
 *
 * @author Alex Yoon, Martin Francisco, Vu Le
 * @version 1.0
 */
public class LookupServiceImpl extends AbstractComponent implements LookupService {

	private Map           daoMap;
	private DAOOperations genericDAO;

	/**
	 * A default constructor
	 */
	public LookupServiceImpl() {
		super();
	}

	/**
	 * Returns a map containing lookup entities and DAO classes
	 *
	 * @return Map a map containing lookup entities and DAO classes
	 */
	public Map getDaoMap() {
		return this.daoMap;
	}

	/**
	 * Sets a map containing lookup entities and DAO classes
	 *
	 * @param daoMap the dao map.
	 */
	public void setDaoMap(Map daoMap) {
		this.daoMap = daoMap;
	}

	public DAOOperations getGenericDAO() {
		return genericDAO;
	}

	public void setGenericDAO(DAOOperations genericDAO) {
		this.genericDAO = genericDAO;
	}

	/**
	 * @see org.springframework.beans.factory.InitializingBean#afterPropertiesSet()
	 */
	public void afterPropertiesSet() throws Exception {
		super.afterPropertiesSet();
		Validate.notNull(daoMap, "Missing required genericDAO map");
	}

	/**
	 * @see gov.va.med.esr.service.LookupService#findAll(java.lang.Class)
	 */
	public List findAll(Class lookupType) throws UnknownLookupTypeException {
		try {
			return this.getDAO(lookupType).findAll(lookupType);
		} catch (DAOException e) {
			throw new RuntimeException(e);
		}
	}

	/**
	 * @see gov.va.med.esr.service.LookupService#getByCode(java.lang.Class,
	 *      java.lang.String)
	 */
	public AbstractLookup getByCode(Class lookupType, String code)
	throws UnknownLookupTypeException, UnknownLookupCodeException {
		try {
			if (code == null || code.trim().length() == 0)
				return null;

			AbstractLookup value = this.getDAO(lookupType).getByCode(lookupType, code);

			if (value == null) {
				throw new UnknownLookupCodeException(
						"Could not find the lookup by code", code, lookupType);
			} else {
				return value;
			}
		} catch (DAOException e) {
			throw new RuntimeException(e);
		}
	}

	public AbstractLookup getByDescription(Class lookupType, String description)
	throws UnknownLookupTypeException, UnknownLookupCodeException {
		try {
			if (description == null || description.trim().length() == 0)
				return null;

			AbstractLookup value = this.getDAO(lookupType).getByDescription(lookupType, description);

			if (value == null) {
				throw new UnknownLookupCodeException(
						"Could not find the unique lookup by description", description, lookupType);
			} else {
				return value;
			}
		} catch (DAOException e) {
			throw new RuntimeException(e);
		}
	}

	public AbstractLookup getByName(Class lookupType, String name)
	throws UnknownLookupTypeException, UnknownLookupCodeException {
		try {
			if (name == null || name.trim().length() == 0)
				return null;

			AbstractLookup value = this.getDAO(lookupType).getByName(lookupType, name);

			if (value == null) {
				throw new UnknownLookupCodeException(
						"Could not find the unique lookup by name", name, lookupType);
			} else {
				return value;
			}
		} catch (DAOException e) {
			throw new RuntimeException(e);
		}
	}

	protected LookupsDAO getDAO(Class lookupType)
	throws UnknownLookupTypeException {

		LookupsDAO dao = null;

		if (this.daoMap != null && lookupType != null) {
			dao = (LookupsDAO) this.daoMap.get(lookupType.getName());
		}

		if (dao == null) {
			throw new UnknownLookupTypeException(
					"Could not find a genericDAO for the lookup type",
					lookupType);
		}
		return dao;
	}

	/*
	 * (non-Javadoc)
	 *
	 * @see gov.va.med.esr.service.LookupService#getByCode(java.lang.Class,
	 *      gov.va.med.fw.model.lookup.AbstractCode)
	 */
	public AbstractLookup getByCode(Class lookupType, AbstractCode code)
	throws UnknownLookupTypeException, UnknownLookupCodeException {
		return getByCode(lookupType, code.getName());
	}

	/**
	 * Find all AacIndicator
	 *
	 * @return List a list of all AacIndicator
	 */
	public List getAllAACIndicators() throws UnknownLookupTypeException,
	UnknownLookupCodeException {
		return this.findAll(AACIndicator.class);
	}

	/**
	 * Get AacIndicator by code
	 *
	 * @return AacIndicator A corresponding AacIndicator by code
	 */
	public AACIndicator getAACIndicatorByCode(String code)
	throws UnknownLookupTypeException, UnknownLookupCodeException {
		return (AACIndicator) this.getByCode(AACIndicator.class, code);
	}

	/**
	 * Find all AddressChangeSource
	 *
	 * @return List a list of all AddressChangeSource
	 */
	public List getAllAddressChangeSources() throws UnknownLookupTypeException,
	UnknownLookupCodeException {
		return this.findAll(AddressChangeSource.class);
	}
	

	/**
	 * Get AddressChangeSource by code
	 *
	 * @return AddressChangeSource A corresponding AddressChangeSource by code
	 */
	public AddressChangeSource getAddressChangeSourceByCode(String code)
	throws UnknownLookupTypeException, UnknownLookupCodeException {
		return (AddressChangeSource) this.getByCode(AddressChangeSource.class,
				code);
	}

	/**
	 * Find all PhoneSourceOfChange
	 *
	 * @return List a list of all PhoneSourceOfChange
	 */
	public List getAllPhoneSourcesOfChange() throws UnknownLookupTypeException,
	UnknownLookupCodeException {
		return this.findAll(PhoneSourceOfChange.class);
	}

	/**
	 * Get PhoneSourceOfChange by code
	 *
	 * @return PhoneSourceOfChange A corresponding PhoneSourceOfChange by code
	 */
	public PhoneSourceOfChange getPhoneSourceOfChangeByCode(String code)
	throws UnknownLookupTypeException, UnknownLookupCodeException {
		return (PhoneSourceOfChange) this.getByCode(PhoneSourceOfChange.class,
				code);
	}

	/**
	 * Find all EmailSourceOfChange
	 *
	 * @return List a list of all EmailSourceOfChange
	 */
	public List getAllEmailSourcesOfChange() throws UnknownLookupTypeException,
	UnknownLookupCodeException {
		return this.findAll(EmailSourceOfChange.class);
	}

	/**
	 * Get EmailSourceOfChange by code
	 *
	 * @return EmailSourceOfChange A corresponding EmailSourceOfChange by code
	 */
	public EmailSourceOfChange getEmailSourceOfChangeByCode(String code)
	throws UnknownLookupTypeException, UnknownLookupCodeException {
		return (EmailSourceOfChange) this.getByCode(EmailSourceOfChange.class,
				code);
	}

	/**
	 * Find all AddressType
	 *
	 * @return List a list of all AddressType
	 */
	public List getAllAddressTypes() throws UnknownLookupTypeException,
	UnknownLookupCodeException {
		return this.findAll(AddressType.class);
	}

	/**
	 * Get AddressType by code
	 *
	 * @return AddressType A corresponding AddressType by code
	 */
	public AddressType getAddressTypeByCode(String code)
	throws UnknownLookupTypeException, UnknownLookupCodeException {
		return (AddressType) this.getByCode(AddressType.class, code);
	}

	/**
	 * Find all AffectedExtremity
	 *
	 * @return List a list of all AffectedExtremity
	 */
	public List getAllAffectedExtremitys() throws UnknownLookupTypeException,
	UnknownLookupCodeException {
		return this.findAll(AffectedExtremity.class);
	}

	/**
	 * Get AffectedExtremity by code
	 *
	 * @return AffectedExtremity A corresponding AffectedExtremity by code
	 */
	public AffectedExtremity getAffectedExtremityByCode(String code)
	throws UnknownLookupTypeException, UnknownLookupCodeException {
		return (AffectedExtremity) this.getByCode(AffectedExtremity.class, code);
	}

	/**
	 * Find all AlliedCountry
	 *
	 * @return List a list of all AlliedCountry
	 */
	public  List getAllAlliedCountries()
	throws UnknownLookupTypeException, UnknownLookupCodeException {
		return this.findAll(AlliedCountry.class);
	}

	/**
	 * Get AlliedCountry by code
	 *
	 * @return AlliedCountry A corresponding AlliedCountry by code
	 */
	public  AlliedCountry getAlliedCountryByCode(String code)
	throws UnknownLookupTypeException, UnknownLookupCodeException {
		return (AlliedCountry) this.getByCode(AlliedCountry.class, code);
	}


	/**
	 * Find all FederalAgency
	 *
	 * @return List a list of all FederalAgency
	 */
	public  List getAllFederalAgencies()
	throws UnknownLookupTypeException, UnknownLookupCodeException {
		return this.findAll(FederalAgency.class);
	}

	/**
	 * Get FederalAgency by code
	 *
	 * @return FederalAgency A corresponding FederalAgency by code
	 */
	public  FederalAgency getFederalAgencyByCode(String code)
	throws UnknownLookupTypeException, UnknownLookupCodeException {
		return (FederalAgency) this.getByCode(FederalAgency.class, code);
	}

	/**
	 * Find all FeeBasisProgrm
	 *
	 * @return List a list of all FeeBasisProgram
	 */
	public List getAllFeeBasisPrograms() throws UnknownLookupTypeException,
	UnknownLookupCodeException {
		return this.findAll(FeeBasisProgram.class);
	}

	/**
	 * Get FeeBasisProgram by code
	 *
	 * @return FeeBasisProgram A corresponding
	 *         FeeBasisProgram by code
	 */
	public FeeBasisProgram getFeeBasisProgramByCode(String code)
	throws UnknownLookupTypeException, UnknownLookupCodeException {
		return (FeeBasisProgram) this.getByCode(FeeBasisProgram.class, code);
	}

	/**
	 * Find all FeeBasisTreatmentType
	 *
	 * @return List a list of all FeeBasisTreatmentType
	 */
	public List getAllFeeBasisTreatmentTypes()
	throws UnknownLookupTypeException, UnknownLookupCodeException {
		return this.findAll(FeeBasisTreatmentType.class);
	}

	/**
	 * Get FeeBasisTreatmentType by code
	 *
	 * @return FeeBasisTreatmentType A corresponding
	 *         FeeBasisTreatmentType by code
	 */
	public FeeBasisTreatmentType getFeeBasisTreatmentTypeByCode(String code)
	throws UnknownLookupTypeException, UnknownLookupCodeException {
		return (FeeBasisTreatmentType) this.getByCode(
				FeeBasisTreatmentType.class, code);
	}

	/**
	 * Find all AgentOrangeExposureLocation
	 *
	 * @return List a list of all AgentOrangeExposureLocation
	 */
	public List getAllAgentOrangeExposureLocations()
	throws UnknownLookupTypeException, UnknownLookupCodeException {
		return this.findAll(AgentOrangeExposureLocation.class);
	}

	/**
	 * Get AgentOrangeExposureLocation by code
	 *
	 * @return AgentOrangeExposureLocation A corresponding
	 *         AgentOrangeExposureLocation by code
	 */
	public AgentOrangeExposureLocation getAgentOrangeExposureLocationByCode(
			String code) throws UnknownLookupTypeException,
			UnknownLookupCodeException {
		return (AgentOrangeExposureLocation) this.getByCode(
				AgentOrangeExposureLocation.class, code);
	}

	/**
	 * Find all AssetType
	 *
	 * @return List a list of all AssetType
	 */
	public List getAllAssetTypes() throws UnknownLookupTypeException,
	UnknownLookupCodeException {
		return this.findAll(AssetType.class);
	}

	/**
	 * Get AssetType by code
	 *
	 * @return AssetType A corresponding AssetType by code
	 */
	public AssetType getAssetTypeByCode(String code)
	throws UnknownLookupTypeException, UnknownLookupCodeException {
		return (AssetType) this.getByCode(AssetType.class, code);
	}

	/**
	 * Find all AssociationType
	 *
	 * @return List a list of all AssociationType
	 */
	public List getAllAssociationTypes() throws UnknownLookupTypeException,
	UnknownLookupCodeException {
		return this.findAll(AssociationType.class);
	}

	/**
	 * Get AssociationType by code
	 *
	 * @return AssociationType A corresponding AssociationType by code
	 */
	public AssociationType getAssociationTypeByCode(String code)
	throws UnknownLookupTypeException, UnknownLookupCodeException {
		return (AssociationType) this.getByCode(AssociationType.class, code);
	}

	/**
	 * Find all CdDeterminationMethod
	 *
	 * @return List a list of all CdDeterminationMethod
	 */
	public List getAllCDDeterminationMethods()
	throws UnknownLookupTypeException, UnknownLookupCodeException {
		return this.findAll(CDDeterminationMethod.class);
	}

	/**
	 * Get CdDeterminationMethod by code
	 *
	 * @return CdDeterminationMethod A corresponding CdDeterminationMethod by
	 *         code
	 */
	public CDDeterminationMethod getCdDeterminationMethodByCode(String code)
	throws UnknownLookupTypeException, UnknownLookupCodeException {
		return (CDDeterminationMethod) this.getByCode(
				CDDeterminationMethod.class, code);
	}

	/**
	 * Find all CancelDeclineNotProcessedReason
	 *
	 * @return List a list of all CancelDeclineNotProcessedReason
	 */
	public List getAllCancelDeclineNotProcessedReasons()
	throws UnknownLookupTypeException, UnknownLookupCodeException {
		return this.findAll(CancelDeclineNotProcessedReason.class);
	}

	/**
	 * Get CancelDeclineNotProcessedReason by code
	 *
	 * @return CancelDeclineNotProcessedReason A corresponding
	 *         CancelDeclineNotProcessedReason by code
	 */
	public CancelDeclineNotProcessedReason getCancelDeclineNotProcessedReasonByCode(
			String code) throws UnknownLookupTypeException,
			UnknownLookupCodeException {
		return (CancelDeclineNotProcessedReason) this.getByCode(
				CancelDeclineNotProcessedReason.class, code);
	}

	/**
	 * Find all CancelDeclineReason
	 *
	 * @return List a list of all CancelDeclineReason
	 */
	public List getAllCancelDeclineReasons() throws UnknownLookupTypeException,
	UnknownLookupCodeException {
		return this.findAll(CancelDeclineReason.class);
	}

	/**
	 * Get CancelDeclineReason by code
	 *
	 * @return CancelDeclineReason A corresponding CancelDeclineReason by code
	 */
	public CancelDeclineReason getCancelDeclineReasonByCode(String code)
	throws UnknownLookupTypeException, UnknownLookupCodeException {
		return (CancelDeclineReason) this.getByCode(CancelDeclineReason.class,
				code);
	}

	/**
	 * Find all ComLetterTemplateType
	 *
	 * @return List a list of all ComLetterTemplateType
	 */
	public List getAllComLetterTemplateTypes() throws UnknownLookupTypeException,
	UnknownLookupCodeException {
		return this.findAll(ComLetterTemplateType.class);
	}

	/**
	 * Get ComLetterTemplateType by code
	 *
	 * @return ComLetterTemplateType A corresponding ComLetterTemplateType by code
	 */
	public ComLetterTemplateType getComLetterTemplateTypeByCode(String code)
	throws UnknownLookupTypeException, UnknownLookupCodeException {
		return (ComLetterTemplateType) this.getByCode(ComLetterTemplateType.class, code);
	}

	/**
	 * Find all Condition
	 *
	 * @return List a list of all Condition
	 */
	public List getAllConditions() throws UnknownLookupTypeException,
	UnknownLookupCodeException {
		return this.findAll(Condition.class);
	}

	/**
	 * Get Condition by code
	 *
	 * @return Condition A corresponding Condition by code
	 */
	public Condition getConditionByCode(String code)
	throws UnknownLookupTypeException, UnknownLookupCodeException {
		return (Condition) this.getByCode(Condition.class, code);
	}

	/**
	 * Find all Conflict
	 *
	 * @return List a list of all Conflict
	 */
	public List getAllConflicts() throws UnknownLookupTypeException,
	UnknownLookupCodeException {
		return this.findAll(ConflictLocation.class);
	}

	/**
	 * Get Conflict by code
	 *
	 * @return Conflict A corresponding Conflict by code
	 */
	public ConflictLocation getConflictByCode(String code)
	throws UnknownLookupTypeException, UnknownLookupCodeException {
		return (ConflictLocation) this.getByCode(ConflictLocation.class, code);
	}

	/**
	 * Find all Country
	 *
	 * @return List a list of all Country
	 */
	public List getAllCountrys() throws UnknownLookupTypeException,
	UnknownLookupCodeException {
		return this.findAll(Country.class);
	}

	/**
	 * Get Country by code
	 *
	 * @return Country A corresponding Country by code
	 */
	public Country getCountryByCode(String code)
	throws UnknownLookupTypeException, UnknownLookupCodeException {
		return (Country) this.getByCode(Country.class, code);
	}

	/**
	 * Get MsdsCountry by code
	 *
	 * @return Country A corresponding Country by code
	 */
	public MsdsCountry getMsdsCountryByCode(String code)
	throws UnknownLookupTypeException, UnknownLookupCodeException {
		return (MsdsCountry) this.getByCode(MsdsCountry.class, code);
	}


	/**
	 * Find all County
	 *
	 * @return List a list of all County
	 */
	public List getAllCountys() throws UnknownLookupTypeException,
	UnknownLookupCodeException {
		return this.findAll(County.class);
	}

	/**
	 * Get County by code
	 *
	 * @return County A corresponding County by code
	 */
	public County getCountyByCode(String code)
	throws UnknownLookupTypeException, UnknownLookupCodeException {
		return (County) this.getByCode(County.class, code);
	}

	/**
	 * Get County by County name and State Postal Code Returns null if there is no
	 * match.
	 * e.g County = "Montgomery", State Postal Code = "OH"
	 *
	 * @return County A corresponding County by County name and state code
	 */
	public County getCountyByCountyNameAndStateCode(String countyName,
			String stateCode) throws UnknownLookupTypeException,
			UnknownLookupCodeException
			{
		try
		{
			CountyDAO countyDAO = (CountyDAO) this.getDAO(County.class);
			return countyDAO.getCountyByCountyNameAndStateCode(countyName,
					stateCode);
		} catch (DAOException e)
		{
			throw new RuntimeException(e);
		}
	}

	public County getCountyByAddress(Address address) throws UnknownLookupTypeException, UnknownLookupCodeException {
		if(address == null)
			return null;

		String stateCode = address.getState();
		String countyName = address.getCounty();
		String zipCode = address.getZipCode();

		// Common ESR logic in one place

		// first try to use stateCode and countyName to get the County code...
		County county = (stateCode == null || countyName == null) ? null
				: getCountyByCountyNameAndStateCode(countyName, stateCode);

		// ..if county is still null, try to look up by zip code
		if (county == null && zipCode != null)
		{
			ZipCode zipCodeObj = getZipCodeByCode(zipCode);
			county = zipCodeObj == null ? null : zipCodeObj.getCounty();
		}

		return county;
	}

	/**
	 * Find all DataSource
	 *
	 * @return List a list of all DataSource
	 */
	public List getAllDataSources() throws UnknownLookupTypeException,
	UnknownLookupCodeException {
		return this.findAll(DataSource.class);
	}

	/**
	 * Get DataSource by code
	 *
	 * @return DataSource A corresponding DataSource by code
	 */
	public DataSource getDataSourceByCode(String code)
	throws UnknownLookupTypeException, UnknownLookupCodeException {
		return (DataSource) this.getByCode(DataSource.class, code);
	}

	/**
	 * Find all DeathDataSource
	 *
	 * @return List a list of all DeathDataSource
	 */
	public List getAllDeathDataSources() throws UnknownLookupTypeException,
	UnknownLookupCodeException {
		return this.findAll(DeathDataSource.class);
	}

	/**
	 * Get DeathDataSource by code
	 *
	 * @return DeathDataSource A corresponding DeathDataSource by code
	 */
	public DeathDataSource getDeathDataSourceByCode(String code)
	throws UnknownLookupTypeException, UnknownLookupCodeException {
		return (DeathDataSource) this.getByCode(DeathDataSource.class, code);
	}

	/**
	 * Find all DecorationStatus
	 *
	 * @return List a list of all DecorationStatus
	 */
	public List getAllDecorationStatuss() throws UnknownLookupTypeException,
	UnknownLookupCodeException {
		return this.findAll(DecorationStatus.class);
	}

	/**
	 * Get DecorationStatus by code
	 *
	 * @return DecorationStatus A corresponding DecorationStatus by code
	 */
	public DecorationStatus getDecorationStatusByCode(String code)
	throws UnknownLookupTypeException, UnknownLookupCodeException {
		return (DecorationStatus) this.getByCode(DecorationStatus.class, code);
	}

	/**
	 * Find all Diagnosis
	 *
	 * @return List a list of all Diagnosis
	 */
	public List getAllDiagnosiss() throws UnknownLookupTypeException,
	UnknownLookupCodeException {
		return this.findAll(Diagnosis.class);
	}

	/**
	 * Get Diagnosis by code
	 *
	 * @return Diagnosis A corresponding Diagnosis by code
	 */
	public Diagnosis getDiagnosisByCode(String code)
	throws UnknownLookupTypeException, UnknownLookupCodeException {
		return (Diagnosis) this.getByCode(Diagnosis.class, code);
	}

	/**
	 * Find all Disability
	 *
	 * @return List a list of all Disability
	 */
	public List getAllDisabilitys() throws UnknownLookupTypeException,
	UnknownLookupCodeException {
		return this.findAll(Disability.class);
	}

	/**
	 * Get Disability by code
	 *
	 * @return Disability A corresponding Disability by code
	 */
	public Disability getDisabilityByCode(String code)
	throws UnknownLookupTypeException, UnknownLookupCodeException {
		return (Disability) this.getByCode(Disability.class, code);
	}

	/**
	 * Find all DischargeType
	 *
	 * @return List a list of all DischargeType
	 */
	public List getAllDischargeTypes() throws UnknownLookupTypeException,
	UnknownLookupCodeException {
		return this.findAll(DischargeType.class);
	}

	/**
	 * Get DischargeType by code
	 *
	 * @return DischargeType A corresponding DischargeType by code
	 */
	public DischargeType getDischargeTypeByCode(String code)
	throws UnknownLookupTypeException, UnknownLookupCodeException {
		return (DischargeType) this.getByCode(DischargeType.class, code);
	}

	public DocumentType getDocumentTypeByCode(String code)
	throws UnknownLookupTypeException, UnknownLookupCodeException {
		return (DocumentType) this.getByCode(DocumentType.class, code);
	}


	/**
	 * Find all EgtSettingType
	 *
	 * @return List a list of all EgtSettingType
	 */
	public List getAllEgtSettingTypes() throws UnknownLookupTypeException,
	UnknownLookupCodeException {
		return this.findAll(EGTSettingType.class);
	}

	/**
	 * Get EgtSettingType by code
	 *
	 * @return EgtSettingType A corresponding EgtSettingType by code
	 */
	public EGTSettingType getEgtSettingTypeByCode(String code)
	throws UnknownLookupTypeException, UnknownLookupCodeException {
		return (EGTSettingType) this.getByCode(EGTSettingType.class, code);
	}

	/**
	 * Find all EmailType
	 *
	 * @return List a list of all EmailType
	 */
	public List getAllEmailTypes() throws UnknownLookupTypeException,
	UnknownLookupCodeException {
		return this.findAll(EmailType.class);
	}

	/**
	 * Get EmailType by code
	 *
	 * @return EmailType A corresponding EmailType by code
	 */
	public EmailType getEmailTypeByCode(String code)
	throws UnknownLookupTypeException, UnknownLookupCodeException {
		return (EmailType) this.getByCode(EmailType.class, code);
	}

	/**
	 * Find all EligibilityStatus
	 *
	 * @return List a list of all EligibilityStatus
	 */
	public List getAllEligibilityStatuss() throws UnknownLookupTypeException,
	UnknownLookupCodeException {
		return this.findAll(EligibilityStatus.class);
	}

	/**
	 * Get EligibilityStatus by code
	 *
	 * @return EligibilityStatus A corresponding EligibilityStatus by code
	 */
	public EligibilityStatus getEligibilityStatusByCode(String code)
	throws UnknownLookupTypeException, UnknownLookupCodeException {
		return (EligibilityStatus) this.getByCode(EligibilityStatus.class, code);
	}
	

	/**
	 * Find all EligibilityStatusPendVerfiReason
	 *
	 * @return List a list of all EligibilityStatusPendVerfiReason
	 */
	public List getAllEligibilityStatusReasons() throws UnknownLookupTypeException,
	UnknownLookupCodeException {
		return this.findAll(EligibilityStatusPendVerfiReason.class);
	}

	/**
	 * Get EligibilityStatusPendVerfiReason by code
	 *
	 * @return EligibilityStatusPendVerfiReason A corresponding EligibilityStatusPendVerfiReason by code
	 */
	public EligibilityStatusPendVerfiReason getEligibilityStatusReasonByCode(String code)
	throws UnknownLookupTypeException, UnknownLookupCodeException {
		return (EligibilityStatusPendVerfiReason) this.getByCode(EligibilityStatusPendVerfiReason.class, code);
	}

	/**
	 * Find all EligibilityType
	 *
	 * @return List a list of all EligibilityType
	 */
	public List getAllEligibilityTypes() throws UnknownLookupTypeException,
	UnknownLookupCodeException {
		return this.findAll(EligibilityType.class);
	}

	/**
	 * Get EligibilityType by code
	 *
	 * @return EligibilityType A corresponding EligibilityType by code
	 */
	public EligibilityType getEligibilityTypeByCode(String code)
	throws UnknownLookupTypeException, UnknownLookupCodeException {
		return (EligibilityType) this.getByCode(EligibilityType.class, code);
	}

	/**
	 * Get EligibilityFactor by code
	 *
	 * @return EligibilityFactor A corresponding EligibilityFactor by code
	 */
	public  EligibilityFactor getEligibilityFactorByCode(String code)
	throws UnknownLookupTypeException, UnknownLookupCodeException {
		return (EligibilityFactor) this.getByCode(EligibilityFactor.class, code);
	}


	/**
	 * Find all EligibilityVerificationSource
	 *
	 * @return List a list of all EligibilityVerificationSource
	 */
	public List getAllEligibilityVerificationSources()
	throws UnknownLookupTypeException, UnknownLookupCodeException {
		return this.findAll(EligibilityVerificationSource.class);
	}

	/**
	 * Get EligibilityVerificationSource by code
	 *
	 * @return EligibilityVerificationSource A corresponding
	 *         EligibilityVerificationSource by code
	 */
	public EligibilityVerificationSource getEligibilityVerificationSourceByCode(
			String code) throws UnknownLookupTypeException,
			UnknownLookupCodeException {
		return (EligibilityVerificationSource) this.getByCode(
				EligibilityVerificationSource.class, code);
	}

	/**
	 * Find all EmploymentStatus
	 *
	 * @return List a list of all EmploymentStatus
	 */
	public List getAllEmploymentStatuss() throws UnknownLookupTypeException,
	UnknownLookupCodeException {
		return this.findAll(EmploymentStatus.class);
	}

	/**
	 * Get EmploymentStatus by code
	 *
	 * @return EmploymentStatus A corresponding EmploymentStatus by code
	 */
	public EmploymentStatus getEmploymentStatusByCode(String code)
	throws UnknownLookupTypeException, UnknownLookupCodeException {
		return (EmploymentStatus) this.getByCode(EmploymentStatus.class, code);
	}

	/**
	 * Find all EnrollmentCategory
	 *
	 * @return List a list of all EnrollmentCategory
	 */
	public List getAllEnrollmentCategorys() throws UnknownLookupTypeException,
	UnknownLookupCodeException {
		return this.findAll(EnrollmentCategory.class);
	}

	/**
	 * Get EnrollmentCategory by code
	 *
	 * @return EnrollmentCategory A corresponding EnrollmentCategory by code
	 */
	public EnrollmentCategory getEnrollmentCategoryByCode(String code)
	throws UnknownLookupTypeException, UnknownLookupCodeException {
		return (EnrollmentCategory) this.getByCode(EnrollmentCategory.class,
				code);
	}

	/**
	 * Find all EnrollmentOverrideReason
	 *
	 * @return List a list of all EnrollmentOverrideReason
	 */
	public List getAllEnrollmentOverrideReasons() throws UnknownLookupTypeException,
	UnknownLookupCodeException {
		return this.findAll(EnrollmentOverrideReason.class);
	}

	/**
	 * Get EnrollmentOverrideReason by code
	 *
	 * @return EnrollmentOverrideReason A corresponding EnrollmentOverrideReason by code
	 */
	public EnrollmentOverrideReason getEnrollmentOverrideReasonByCode(String code)
	throws UnknownLookupTypeException, UnknownLookupCodeException {
		return (EnrollmentOverrideReason) this.getByCode(EnrollmentOverrideReason.class,
				code);
	}

	/**
	 * Find all EnrollmentPriorityGroup
	 *
	 * @return List a list of all EnrollmentPriorityGroup
	 */
	public List getAllEnrollmentPriorityGroups()
	throws UnknownLookupTypeException, UnknownLookupCodeException {
		return this.findAll(EnrollmentPriorityGroup.class);
	}

	/**
	 * Get EnrollmentPriorityGroup by code
	 *
	 * @return EnrollmentPriorityGroup A corresponding EnrollmentPriorityGroup
	 *         by code
	 */
	public EnrollmentPriorityGroup getEnrollmentPriorityGroupByCode(String code)
	throws UnknownLookupTypeException, UnknownLookupCodeException {
		return (EnrollmentPriorityGroup) this.getByCode(
				EnrollmentPriorityGroup.class, code);
	}

	/**
	 * Find all EnrollmentPrioritySubGroup
	 *
	 * @return List a list of all EnrollmentPrioritySubGroup
	 */
	public List getAllEnrollmentPrioritySubGroups()
	throws UnknownLookupTypeException, UnknownLookupCodeException {
		return this.findAll(EnrollmentPrioritySubGroup.class);
	}

	/**
	 * Get EnrollmentPrioritySubGroup by code
	 *
	 * @return EnrollmentPrioritySubGroup A corresponding
	 *         EnrollmentPrioritySubGroup by code
	 */
	public EnrollmentPrioritySubGroup getEnrollmentPrioritySubGroupByCode(
			String code) throws UnknownLookupTypeException,
			UnknownLookupCodeException {
		return (EnrollmentPrioritySubGroup) this.getByCode(
				EnrollmentPrioritySubGroup.class, code);
	}

	/**
	 * Find all EnrollmentSource
	 *
	 * @return List a list of all EnrollmentSource
	 */
	public List getAllEnrollmentSources() throws UnknownLookupTypeException,
	UnknownLookupCodeException {
		return this.findAll(EnrollmentSource.class);
	}

	/**
	 * Get EnrollmentSource by code
	 *
	 * @return EnrollmentSource A corresponding EnrollmentSource by code
	 */
	public EnrollmentSource getEnrollmentSourceByCode(String code)
	throws UnknownLookupTypeException, UnknownLookupCodeException {
		return (EnrollmentSource) this.getByCode(EnrollmentSource.class, code);
	}

	/**
	 * Find all EnrollmentStatus
	 *
	 * @return List a list of all EnrollmentStatus
	 */
	public List getAllEnrollmentStatuss() throws UnknownLookupTypeException,
	UnknownLookupCodeException {
		return this.findAll(EnrollmentStatus.class);
	}

	/**
	 * Get EnrollmentStatus by code
	 *
	 * @return EnrollmentStatus A corresponding EnrollmentStatus by code
	 */
	public EnrollmentStatus getEnrollmentStatusByCode(String code)
	throws UnknownLookupTypeException, UnknownLookupCodeException {
		return (EnrollmentStatus) this.getByCode(EnrollmentStatus.class, code);
	}

	/**
	 * Find all ExpenseType
	 *
	 * @return List a list of all ExpenseType
	 */
	public List getAllExpenseTypes() throws UnknownLookupTypeException,
	UnknownLookupCodeException {
		return this.findAll(ExpenseType.class);
	}

	/**
	 * Get ExpenseType by code
	 *
	 * @return ExpenseType A corresponding ExpenseType by code
	 */
	public ExpenseType getExpenseTypeByCode(String code)
	throws UnknownLookupTypeException, UnknownLookupCodeException {
		return (ExpenseType) this.getByCode(ExpenseType.class, code);
	}

	/**
	 * Find all FilipinoVeteranProof
	 *
	 * @return List a list of all FilipinoVeteranProof
	 */
	public List getAllFilipinoVeteranProofs()
	throws UnknownLookupTypeException, UnknownLookupCodeException {
		return this.findAll(FilipinoVeteranProof.class);
	}

	/**
	 * Get FilipinoVeteranProof by code
	 *
	 * @return FilipinoVeteranProof A corresponding FilipinoVeteranProof by code
	 */
	public FilipinoVeteranProof getFilipinoVeteranProofByCode(String code)
	throws UnknownLookupTypeException, UnknownLookupCodeException {
		return (FilipinoVeteranProof) this.getByCode(
				FilipinoVeteranProof.class, code);
	}

	/**
	 * Find all Gender
	 *
	 * @return List a list of all Gender
	 */
	public List getAllGenders() throws UnknownLookupTypeException,
	UnknownLookupCodeException {
		return this.findAll(Gender.class);
	}

	/**
	 * Get Gender by code
	 *
	 * @return Gender A corresponding Gender by code
	 */
	public Gender getGenderByCode(String code)
	throws UnknownLookupTypeException, UnknownLookupCodeException {
		return (Gender) this.getByCode(Gender.class, code);
	}

	/**
	 * Find all IncomeTestSource
	 *
	 * @return List a list of all IncomeTestSource
	 */
	public List getAllIncomeTestSources() throws UnknownLookupTypeException,
	UnknownLookupCodeException {
		return this.findAll(IncomeTestSource.class);
	}

	public Indicator getIndicatorByCode(Indicator.Code code)
	throws UnknownLookupTypeException, UnknownLookupCodeException {
		return (Indicator) getByCode(Indicator.class, code);
	}

	public  CombatPayType getCombatPayTypeByCode(CombatPayType.Code code)
	throws UnknownLookupTypeException, UnknownLookupCodeException {
		return (CombatPayType) getByCode(CombatPayType.class, code);
	}


	/**
	 * Get IncomeTestSource by code
	 *
	 * @return IncomeTestSource A corresponding IncomeTestSource by code
	 */
	public IncomeTestSource getIncomeTestSourceByCode(String code)
	throws UnknownLookupTypeException, UnknownLookupCodeException {
		return (IncomeTestSource) this.getByCode(IncomeTestSource.class, code);
	}

	/**
	 * Find all IncomeTestType
	 *
	 * @return List a list of all IncomeTestType
	 */
	public List getAllIncomeTestTypes() throws UnknownLookupTypeException,
	UnknownLookupCodeException {
		return this.findAll(IncomeTestType.class);
	}

	/**
	 * Get IncomeTestType by code
	 *
	 * @return IncomeTestType A corresponding IncomeTestType by code
	 */
	public IncomeTestType getIncomeTestTypeByCode(String code)
	throws UnknownLookupTypeException, UnknownLookupCodeException {
		return (IncomeTestType) this.getByCode(IncomeTestType.class, code);
	}

	/**
	 * Find all IncomeType
	 *
	 * @return List a list of all IncomeType
	 */
	public List getAllIncomeTypes() throws UnknownLookupTypeException,
	UnknownLookupCodeException {
		return this.findAll(IncomeType.class);
	}

	/**
	 * Get IncomeType by code
	 *
	 * @return IncomeType A corresponding IncomeType by code
	 */
	public IncomeType getIncomeTypeByCode(String code)
	throws UnknownLookupTypeException, UnknownLookupCodeException {
		return (IncomeType) this.getByCode(IncomeType.class, code);
	}

	/**
	 * Find all InsurancePlanType
	 *
	 * @return List a list of all InsurancePlanType
	 */
	public List getAllInsurancePlanTypes() throws UnknownLookupTypeException,
	UnknownLookupCodeException {
		return this.findAll(InsurancePlanType.class);
	}

	/**
	 * Get InsurancePlanType by code
	 *
	 * @return InsurancePlanType A corresponding InsurancePlanType by code
	 */
	public InsurancePlanType getInsurancePlanTypeByCode(String code)
	throws UnknownLookupTypeException, UnknownLookupCodeException {
		return (InsurancePlanType) this.getByCode(InsurancePlanType.class, code);
	}

	/**
	 * Find all InsuredRelationship
	 *
	 * @return List a list of all InsuredRelationship
	 */
	public List getAllInsuredRelationships() throws UnknownLookupTypeException,
	UnknownLookupCodeException {
		return this.findAll(InsuredRelationship.class);
	}

	/**
	 * Get InsuredRelationship by code
	 *
	 * @return InsuredRelationship A corresponding InsuredRelationship by code
	 */
	public InsuredRelationship getInsuredRelationshipByCode(String code)
	throws UnknownLookupTypeException, UnknownLookupCodeException {
		return (InsuredRelationship) this.getByCode(InsuredRelationship.class,
				code);
	}

	/**
	 * Find all MaritalStatus
	 *
	 * @return List a list of all MaritalStatus
	 */
	public List getAllMaritalStatuss() throws UnknownLookupTypeException,
	UnknownLookupCodeException {
		return this.findAll(MaritalStatus.class);
	}

	/**
	 * Get MaritalStatus by code
	 *
	 * @return MaritalStatus A corresponding MaritalStatus by code
	 */
	public MaritalStatus getMaritalStatusByCode(String code)
	throws UnknownLookupTypeException, UnknownLookupCodeException {
		return (MaritalStatus) this.getByCode(MaritalStatus.class, code);
	}

	/**
	 * Find all MeansTestStatus
	 *
	 * @return List a list of all MeansTestStatus
	 */
	public List getAllMeansTestStatuss() throws UnknownLookupTypeException,
	UnknownLookupCodeException {
		return this.findAll(MeansTestStatus.class);
	}

	/**
	 * Get MeansTestStatus by code
	 *
	 * @return MeansTestStatus A corresponding MeansTestStatus by code
	 */
	public MeansTestStatus getMeansTestStatusByCode(String code)
	throws UnknownLookupTypeException, UnknownLookupCodeException {
		return (MeansTestStatus) this.getByCode(MeansTestStatus.class, code);
	}

	/**
	 * Find all MessageStatus
	 *
	 * @return List a list of all MessageStatus
	 */
	public List getAllMessageStatuss() throws UnknownLookupTypeException,
	UnknownLookupCodeException {
		return this.findAll(MessageStatus.class);
	}

	/**
	 * Get MessageStatus by code
	 *
	 * @return MessageStatus A corresponding MessageStatus by code
	 */
	public MessageStatus getMessageStatusByCode(String code)
	throws UnknownLookupTypeException, UnknownLookupCodeException {
		return (MessageStatus) this.getByCode(MessageStatus.class, code);
	}

	/**
	 * Find all MessageType
	 *
	 * @return List a list of all MessageType
	 */
	public List getAllMessageTypes() throws UnknownLookupTypeException,
	UnknownLookupCodeException {
		return this.findAll(MessageType.class);
	}

	/**
	 * Get MessageType by code
	 *
	 * @return MessageType A corresponding MessageType by code
	 */
	public MessageType getMessageTypeByCode(String code)
	throws UnknownLookupTypeException, UnknownLookupCodeException {
		return (MessageType) this.getByCode(MessageType.class, code);
	}

	/**
	 * Find all MilitarySexualTraumaStatus
	 *
	 * @return List a list of all MilitarySexualTraumaStatus
	 */
	public List getAllMilitarySexualTraumaStatuss()
	throws UnknownLookupTypeException, UnknownLookupCodeException {
		return this.findAll(MilitarySexualTraumaStatus.class);
	}

	/**
	 * Get MilitarySexualTraumaStatus by code
	 *
	 * @return MilitarySexualTraumaStatus A corresponding
	 *         MilitarySexualTraumaStatus by code
	 */
	public MilitarySexualTraumaStatus getMilitarySexualTraumaStatusByCode(
			String code) throws UnknownLookupTypeException,
			UnknownLookupCodeException {
		return (MilitarySexualTraumaStatus) this.getByCode(
				MilitarySexualTraumaStatus.class, code);
	}

	/**
	 * Find all MonetaryBenefitType
	 *
	 * @return List a list of all MonetaryBenefitType
	 */
	public List getAllMonetaryBenefitTypes() throws UnknownLookupTypeException,
	UnknownLookupCodeException {
		return this.findAll(MonetaryBenefitType.class);
	}

	/**
	 * Get MonetaryBenefitType by code
	 *
	 * @return MonetaryBenefitType A corresponding MonetaryBenefitType by code
	 */
	public MonetaryBenefitType getMonetaryBenefitTypeByCode(String code)
	throws UnknownLookupTypeException, UnknownLookupCodeException {
		return (MonetaryBenefitType) this.getByCode(MonetaryBenefitType.class,
				code);
	}

	/**
	 * Find all NameType
	 *
	 * @return List a list of all NameType
	 */
	public List getAllNameTypes() throws UnknownLookupTypeException,
	UnknownLookupCodeException {
		return this.findAll(NameType.class);
	}

	/**
	 * Get NameType by code
	 *
	 * @return NameType A corresponding NameType by code
	 */
	public NameType getNameTypeByCode(String code)
	throws UnknownLookupTypeException, UnknownLookupCodeException {
		return (NameType) this.getByCode(NameType.class, code);
	}

	/**
	 * Find all PhoneType
	 *
	 * @return List a list of all PhoneType
	 */
	public List getAllPhoneTypes() throws UnknownLookupTypeException,
	UnknownLookupCodeException {
		return this.findAll(PhoneType.class);
	}

	/**
	 * Get PhoneType by code
	 *
	 * @return PhoneType A corresponding PhoneType by code
	 */
	public PhoneType getPhoneTypeByCode(String code)
	throws UnknownLookupTypeException, UnknownLookupCodeException {
		return (PhoneType) this.getByCode(PhoneType.class, code);
	}

	/**
	 * Find all PrisonerOfWarLocation
	 *
	 * @return List a list of all PrisonerOfWarLocation
	 */
	public List getAllPrisonerOfWarLocations()
	throws UnknownLookupTypeException, UnknownLookupCodeException {
		return this.findAll(PrisonerOfWarLocation.class);
	}

	/**
	 * Get PrisonerOfWarLocation by code
	 *
	 * @return PrisonerOfWarLocation A corresponding PrisonerOfWarLocation by
	 *         code
	 */
	public PrisonerOfWarLocation getPrisonerOfWarLocationByCode(String code)
	throws UnknownLookupTypeException, UnknownLookupCodeException {
		return (PrisonerOfWarLocation) this.getByCode(
				PrisonerOfWarLocation.class, code);
	}

	/**
	 * Find all Procedure
	 *
	 * @return List a list of all Procedure
	 */
	public List getAllProcedures() throws UnknownLookupTypeException,
	UnknownLookupCodeException {
		return this.findAll(Procedure.class);
	}

	/**
	 * Get Procedure by code
	 *
	 * @return Procedure A corresponding Procedure by code
	 */
	public Procedure getProcedureByCode(String code)
	throws UnknownLookupTypeException, UnknownLookupCodeException {
		return (Procedure) this.getByCode(Procedure.class, code);
	}

	/**
	 * Find all EGTStatus
	 *
	 * @return List a list of all EGTStatus
	 */
	public List getAllEGTStatuss() throws UnknownLookupTypeException,
	UnknownLookupCodeException {
		return this.findAll(EGTStatus.class);
	}

	/**
	 * Get EGTStatus by code
	 *
	 * @return EGTStatus A corresponding EGTStatus by code
	 */
	public EGTStatus getEGTStatusByCode(String code)
	throws UnknownLookupTypeException, UnknownLookupCodeException {
		return (EGTStatus) this.getByCode(EGTStatus.class, code);
	}

	/**
	 * Find all RadiationExposureMethod
	 *
	 * @return List a list of all RadiationExposureMethod
	 */
	public List getAllRadiationExposureMethods()
	throws UnknownLookupTypeException, UnknownLookupCodeException {
		return this.findAll(RadiationExposureMethod.class);
	}

	/**
	 * Get RadiationExposureMethod by code
	 *
	 * @return RadiationExposureMethod A corresponding RadiationExposureMethod
	 *         by code
	 */
	public RadiationExposureMethod getRadiationExposureMethodByCode(String code)
	throws UnknownLookupTypeException, UnknownLookupCodeException {
		return (RadiationExposureMethod) this.getByCode(
				RadiationExposureMethod.class, code);
	}

	/**
	 * Find all RejectionRemark
	 *
	 * @return List a list of all RejectionRemark
	 */
	public List getAllRejectionRemarks() throws UnknownLookupTypeException,
	UnknownLookupCodeException {
		return this.findAll(RejectionRemark.class);
	}

	/**
	 * Get RejectionRemark by code
	 *
	 * @return RejectionRemark A corresponding RejectionRemark by code
	 */
	public RejectionRemark getRejectionRemarkByCode(String code)
	throws UnknownLookupTypeException, UnknownLookupCodeException {
		return (RejectionRemark) this.getByCode(RejectionRemark.class, code);
	}

	/**
	 * Find all Relationship
	 *
	 * @return List a list of all Relationship
	 */
	public List getAllRelationships() throws UnknownLookupTypeException,
	UnknownLookupCodeException {
		return this.findAll(Relationship.class);
	}

	/**
	 * Get Relationship by code
	 *
	 * @return Relationship A corresponding Relationship by code
	 */
	public Relationship getRelationshipByCode(String code)
	throws UnknownLookupTypeException, UnknownLookupCodeException {
		return (Relationship) this.getByCode(Relationship.class, code);
	}

	/**
	 * Get RegistryType by code
	 *
	 * @return RegistryType
	 */
	public  RegistryType getRegistryTypeByCode(String code)
	throws UnknownLookupTypeException, UnknownLookupCodeException
	{
		return (RegistryType) this.getByCode(RegistryType.class, code);
	}
	/**
	 * Find all Religion
	 *
	 * @return List a list of all Religion
	 */
	public List getAllReligions() throws UnknownLookupTypeException,
	UnknownLookupCodeException {
		return this.findAll(Religion.class);
	}

	/**
	 * Get Religion by code
	 *
	 * @return Religion A corresponding Religion by code
	 */
	public Religion getReligionByCode(String code)
	throws UnknownLookupTypeException, UnknownLookupCodeException {
		return (Religion) this.getByCode(Religion.class, code);
	}

	/**
	 * Find all ServiceBranch
	 *
	 * @return List a list of all ServiceBranch
	 */
	public List getAllServiceBranchs() throws UnknownLookupTypeException,
	UnknownLookupCodeException {
		return this.findAll(ServiceBranch.class);
	}

	/**
	 * Get ServiceBranch by code
	 *
	 * @return ServiceBranch A corresponding ServiceBranch by code
	 */
	public ServiceBranch getServiceBranchByCode(String code)
	throws UnknownLookupTypeException, UnknownLookupCodeException {
		return (ServiceBranch) this.getByCode(ServiceBranch.class, code);
	}

	/**
	 * Find all ServicePeriod
	 *
	 * @return List a list of all ServicePeriod
	 */
	public List getAllServicePeriods() throws UnknownLookupTypeException,
	UnknownLookupCodeException {
		return this.findAll(ServicePeriod.class);
	}

	/**
	 * Get ServicePeriod by code
	 *
	 * @return ServicePeriod A corresponding ServicePeriod by code
	 */
	public ServicePeriod getServicePeriodByCode(String code)
	throws UnknownLookupTypeException, UnknownLookupCodeException {
		return (ServicePeriod) this.getByCode(ServicePeriod.class, code);
	}

	/**
	 * Find all State
	 *
	 * @return List a list of all State
	 */
	public List getAllStates() throws UnknownLookupTypeException,
	UnknownLookupCodeException {
		return this.findAll(State.class);
	}

	/**
	 * Get State by code
	 *
	 * @return State A corresponding State by code
	 */
	public State getStateByCode(String code) throws UnknownLookupTypeException,
	UnknownLookupCodeException {
		return (State) this.getByCode(State.class, code);
	}

	/**
	 * Get State by State name
	 *
	 * @return State A corresponding State by name
	 */
	public State getStateByName(String name) throws UnknownLookupTypeException,
	UnknownLookupCodeException
	{
		try
		{
			StateDAO stateDAO = (StateDAO) this.getDAO(State.class);
			return stateDAO.getStateByName(name);
		} catch (DAOException e)
		{
			throw new RuntimeException(e);
		}
	}

	
	/**
	 * Find all VaFacilitys
	 *
	 * @return List a list of all VaFacility
	 */
	public List getAllVaFacilitys() throws UnknownLookupTypeException,
	UnknownLookupCodeException {
		return this.findAll(VAFacility.class);
	}

	/**
	 * @see gov.va.med.esr.service.LookupService#getAllMfnzegRecipients()
	 */
	public List getAllMfnzegRecipients() throws ServiceException
	{
		try
		{
			List result = getGenericDAO().find(VAFacility.QUERY_FIND_ALL_MFNZEG_RECIPIENTS);
			return result == null ? new ArrayList() : result;
		}
		catch (DAOException ex)
		{
			throw new ServiceException("Error loading VAFacility objects from the database.", ex);
		}
	}

	//CCR 13856
	public boolean getDODSharingInd() throws ServiceException {

		SystemParameterService systemParameterService = (SystemParameterService) this.getComponent("systemParameterService");

		return systemParameterService.getDODSharing();
	}

	/**
	 * @see gov.va.med.esr.service.LookupService#getAllDeathNotificationSites()
	 */
	public List getAllDeathNotificationSites() throws ServiceException
	{
		List result = null;
		try
		{
			SystemParameterService systemParameterService = (SystemParameterService) this.getComponent("systemParameterService");

			if (systemParameterService.getDODIndicator())
			{
			   result = getGenericDAO().find(VAFacility.QUERY_FIND_ALL_DEATH_NOTIFICATION_SITES_NCA);
			}
			else
			{
				result = getGenericDAO().find(VAFacility.QUERY_FIND_ALL_DEATH_NOTIFICATION_SITES);
			}

			return result == null ? new ArrayList() : result;
		}
		catch (DAOException ex)
		{
			throw new ServiceException("Error loading VAFacility objects from the database.", ex);
		}
	}
	/**
	 * Get VaFacility by code
	 *
	 * @return VaFacility A corresponding VaFacility by code
	 */
	public VAFacility getVaFacilityByCode(String code)
	throws UnknownLookupTypeException, UnknownLookupCodeException {
		return (VAFacility) this.getByCode(VAFacility.class, code);
	}

	/**
	 * Get VaFacility by station number. This is just a helper method, It gets
	 * the VAFacility by code as code and station number are same.
	 *
	 * @return VaFacility A corresponding VaFacility by station number
	 */
	public VAFacility getVaFacilityByStationNumber(String stationNum)
	throws UnknownLookupTypeException, UnknownLookupCodeException {
		return getVaFacilityByCode(stationNum);
	}

	/**
	 * Get VaFacility by id
	 *
	 * @return VaFacility A corresponding VaFacility by id
	 */
	public  VAFacility getVAFacilityById(BigDecimal identifier)
	throws UnknownLookupTypeException, UnknownLookupCodeException {
		try {
			VAFacilityDAO vaFacilityDAO = (VAFacilityDAO) this.getDAO(VAFacility.class);
			return vaFacilityDAO.getById(identifier);
		} catch (DAOException e) {
			throw new RuntimeException(e);
		}
	}


	/**
	 * Find all VaFacilityType
	 *
	 * @return List a list of all VaFacilityType
	 */
	public List getAllVaFacilityTypes() throws UnknownLookupTypeException,
	UnknownLookupCodeException {
		return this.findAll(VAFacilityType.class);
	}

	/**
	 * Get VaFacilityType by code
	 *
	 * @return VaFacilityType A corresponding VaFacilityType by code
	 */
	public VAFacilityType getVaFacilityTypeByCode(String code)
	throws UnknownLookupTypeException, UnknownLookupCodeException {
		return (VAFacilityType) this.getByCode(VAFacilityType.class, code);
	}

	public Map getVISNSiteMap() throws UnknownLookupTypeException, ClassNotFoundException {
		List facilities = this.findAll(VAFacility.class);

		return getVISNSiteMapForFacilities(facilities);

	}
	public Map getParentVISNSiteMap()throws UnknownLookupTypeException, ClassNotFoundException,ServiceException {
		List facilities = this.getAllMfnzegRecipients();
		return getVISNSiteMapForFacilities(facilities);

	}
	public Map getVISNSiteMapForFacilities(List facilities) throws UnknownLookupTypeException, ClassNotFoundException {

		Map facilityByVisn = new LinkedHashMap();
		if (facilities != null) {
			for (Iterator iter = facilities.iterator(); iter.hasNext();) {

				VAFacility facility = (VAFacility)iter.next();
				BigDecimal visnId = facility.getVisnId();

				//If the VISN Id is null, then create a dummy No Visn Id
				if(visnId == null)
					visnId = VISN.CODE_NO_VISN_IDENTIFIER;

				List visnFacility = (List)facilityByVisn.get(visnId);
				if(visnFacility == null) {
					visnFacility = new ArrayList();

					facilityByVisn.put(visnId,visnFacility);
				}
				if(visnFacility != null) {
					visnFacility.add(facility);
				}
			}
		}

		List visns = this.findAll(VISN.class);
		//Add a dummy VISN to the list
		visns.add(VISN.getNoVISN());

		Map facilitySiteMap = new LinkedHashMap();
		for(Iterator iter = visns.iterator(); iter.hasNext();) {
			VISN visn = (VISN)iter.next();
			List visnFacilities = (List)facilityByVisn.get(visn.getIdentifier());
			facilitySiteMap.put(visn,visnFacilities);
		}
		return facilitySiteMap;
	}


	/**
	 * Find all NTRVerificationMethod
	 *
	 * @return List a list of all NTRVerificationMethod
	 */
	public List getAllNTRVerificationMethods()
	throws UnknownLookupTypeException, UnknownLookupCodeException {
		return this.findAll(NTRVerificationMethod.class);
	}

	/**
	 * Get NTRVerificationMethod by code
	 *
	 * @return NTRVerificationMethod A corresponding NTRVerificationMethod by
	 *         code
	 */
	public NTRVerificationMethod getNTRVerificationMethodByCode(String code)
	throws UnknownLookupTypeException, UnknownLookupCodeException {
		return (NTRVerificationMethod) this.getByCode(
				NTRVerificationMethod.class, code);
	}

	/**
	 * Find all ZipCode
	 *
	 * @return List a list of all ZipCode
	 */
	public List getAllZipCodes() throws UnknownLookupTypeException,
	UnknownLookupCodeException {
		return this.findAll(ZipCode.class);
	}

	/**
	 * Get ZipCode by code
	 *
	 * @return ZipCode A corresponding ZipCode by code
	 */
	public ZipCode getZipCodeByCode(String code)
	throws UnknownLookupTypeException, UnknownLookupCodeException {
		return (ZipCode) this.getByCode(ZipCode.class, code);
	}

	/**
	 * Gets a partially filled in zip code object by code with the following
	 * attributes filled in:
	 *
	 * ZipCode.id, ZipCode.zipCode, ZipCode.state.name, ZipCode.cities.name (abbreviation when present or full name when not),
	 * ZipCode.county.name
	 *
	 * @return The partially filled in zip code object.
	 */
	public ZipCode getPartialZipCodeByCode(String code)
	throws UnknownLookupTypeException {
		try {
			ZipCodeDAO zipCodeDAO = (ZipCodeDAO) this.getDAO(ZipCode.class);
			return zipCodeDAO.getPartialZipCodeByCode(code);
		} catch (DAOException e) {
			throw new RuntimeException(e);
		}
	}

	/**
	 * Gets a list of zip code objects that has only the following attributes filled in:
	 * <p>
	 * ZipCode.id
	 * ZipCode.code
	 * <p>
	 * If no records are found, an empty list is returned.
	 *
	 * @param cityName The name of the city to query on
	 * @param stateCode The state code to query on
	 *
	 * @return The zip code object
	 */
	public List getPartialZipCodeByCityState(String cityName, String stateCode) throws UnknownLookupTypeException
	{
		try
		{
			ZipCodeDAO zipCodeDAO = (ZipCodeDAO) this.getDAO(ZipCode.class);
			return zipCodeDAO.getPartialZipCodeByCityState(cityName, stateCode);
		}
		catch (DAOException e)
		{
			throw new RuntimeException(e);
		}
	}

	/**
	 * Gets a list of zip code objects that has only the following attributes filled in:
	 * <p>
	 * ZipCode.id
	 * ZipCode.code
	 * <p>
	 * If no records are found, an empty list is returned.
	 *
	 * @param cityName The name of the city to query on
	 *
	 * @return The zip code object
	 */
	public List getPartialZipCodeByCity(String cityName) throws UnknownLookupTypeException
	{
		try
		{
			ZipCodeDAO zipCodeDAO = (ZipCodeDAO) this.getDAO(ZipCode.class);
			return zipCodeDAO.getPartialZipCodeByCity(cityName);
		}
		catch (DAOException e)
		{
			throw new RuntimeException(e);
		}
	}

	/**
	 * Find all BadAddressReason
	 *
	 * @return List a list of all BadAddressReason
	 */
	public List getAllBadAddressReasons() throws UnknownLookupTypeException,
	UnknownLookupCodeException {
		return this.findAll(BadAddressReason.class);
	}

	/**
	 * Get BadAddressReason by code
	 *
	 * @return ZipCode A corresponding BadAddressReason by code
	 */
	public BadAddressReason getBadAddressReasonByCode(String code)
	throws UnknownLookupTypeException, UnknownLookupCodeException {
		return (BadAddressReason) this.getByCode(BadAddressReason.class, code);
	}

	/**
	 * Find all MeansTestNotRequiredReason
	 *
	 * @return List a list of all MeansTestNotRequiredReason
	 */
	public List getAllMeansTestNotRequiredReasons()
	throws UnknownLookupTypeException, UnknownLookupCodeException {
		return this.findAll(MeansTestNotRequiredReason.class);
	}

	/**
	 * Get MeansTestNotRequiredReason by code
	 *
	 * @return ZipCode A corresponding MeansTestNotRequiredReason by code
	 */
	public MeansTestNotRequiredReason getMeansTestNotRequiredReasonByCode(
			String code) throws UnknownLookupTypeException,
			UnknownLookupCodeException {
		return (MeansTestNotRequiredReason) this.getByCode(
				MeansTestNotRequiredReason.class, code);
	}

	/**
	 * Find all InsuranceReportSource
	 *
	 * @return List a list of all InsuranceReportSource
	 */
	public List getAllInsuranceReportSources()
	throws UnknownLookupTypeException, UnknownLookupCodeException {
		return this.findAll(InsuranceReportSource.class);
	}

	/**
	 * Get InsuranceReportSource by code
	 *
	 * @return ZipCode A corresponding InsuranceReportSource by code
	 */
	public InsuranceReportSource getInsuranceReportSourceByCode(String code)
	throws UnknownLookupTypeException, UnknownLookupCodeException {
		return (InsuranceReportSource) this.getByCode(
				InsuranceReportSource.class, code);
	}

	/**
	 * Find all SensitivityChangeSource
	 *
	 * @return List a list of all SensitivityChangeSource
	 */
	public List getAllSensitivityChangeSources() throws UnknownLookupTypeException,
	UnknownLookupCodeException {
		return this.findAll(SensitivityChangeSource.class);
	}

	/**
	 * Get SensitivityChangeSource by code
	 *
	 * @return SensitivityChangeSource A corresponding SensitivityChangeSource by code
	 */
	public SensitivityChangeSource getSensitivityChangeSourceByCode(String code)
	throws UnknownLookupTypeException, UnknownLookupCodeException {
		return (SensitivityChangeSource) this.getByCode(SensitivityChangeSource.class,
				code);
	}

	/**
	 * Find all SSNType
	 *
	 * @return List a list of all SSNType
	 */
	public List getAllSSNTypes() throws UnknownLookupTypeException,
	UnknownLookupCodeException {
		return this.findAll(SSNType.class);
	}

	/**
	 * Get SSNType by code
	 *
	 * @return SSNType A corresponding SSNType by code
	 */
	public SSNType getSSNTypeByCode(String code)
	throws UnknownLookupTypeException, UnknownLookupCodeException {
		return (SSNType) this.getByCode(SSNType.class, code);
	}

	/**
	 * Find all SSNChangeSource
	 *
	 * @return List a list of all SSNChangeSource
	 */
	public List getAllSSNChangeSources() throws UnknownLookupTypeException,
	UnknownLookupCodeException {
		return this.findAll(SSNChangeSource.class);
	}

	/**
	 * Get SSNChangeSource by code
	 *
	 * @return SSNChangeSource A corresponding SSNChangeSource by code
	 */
	public SSNChangeSource getSSNChangeSourceByCode(String code)
	throws UnknownLookupTypeException, UnknownLookupCodeException {
		return (SSNChangeSource) this.getByCode(SSNChangeSource.class, code);
	}

	/**
	 * Find all SSAVerificationStatus
	 *
	 * @return List a list of all SSAVerificationStatus
	 */
	public List getAllSSAVerificationStatuss()
	throws UnknownLookupTypeException, UnknownLookupCodeException {
		return this.findAll(SSAVerificationStatus.class);
	}

	/**
	 * Get SSAVerificationStatus by code
	 *
	 * @return SSAVerificationStatus A corresponding SSAVerificationStatus by
	 *         code
	 */
	public SSAVerificationStatus getSSAVerificationStatusByCode(String code)
	throws UnknownLookupTypeException, UnknownLookupCodeException {
		return (SSAVerificationStatus) this.getByCode(
				SSAVerificationStatus.class, code);
	}

	/**
	 * Find all SSAMessage
	 *
	 * @return List a list of all SSAMessage
	 */
	public List getAllSSAMessages() throws UnknownLookupTypeException,
	UnknownLookupCodeException {
		return this.findAll(SSAMessage.class);
	}

	/**
	 * Get SSAMessage by code
	 *
	 * @return SSAMessage A corresponding SSAMessage by code
	 */
	public SSAMessage getSSAMessageByCode(String code)
	throws UnknownLookupTypeException, UnknownLookupCodeException {
		return (SSAMessage) this.getByCode(SSAMessage.class, code);

	}

	/**
	 * Find all PseudoSSNReason
	 *
	 * @return List a list of all PseudoSSNReason
	 */
	public List getAllPseudoSSNReasons() throws UnknownLookupTypeException,
	UnknownLookupCodeException {
		return this.findAll(PseudoSSNReason.class);
	}

	/**
	 * Get PseudoSSNReason by code
	 *
	 * @return PseudoSSNReason A corresponding PseudoSSNReason by code
	 */
	public PseudoSSNReason getPseudoSSNReasonByCode(String code)
	throws UnknownLookupTypeException, UnknownLookupCodeException {
		return (PseudoSSNReason) this.getByCode(PseudoSSNReason.class, code);
	}

	/**
	 * Find all NcoaDeliveryCode
	 *
	 * @return List a list of all NcoaDeliveryCode
	 */
	public List getAllNcoaDeliveryCodes() throws UnknownLookupTypeException,
	UnknownLookupCodeException {
		return this.findAll(NcoaDeliveryCode.class);
	}

	/**
	 * Get NcoaDeliveryCode by code
	 *
	 * @return NcoaDeliveryCode A corresponding NcoaDeliveryCode by code
	 */
	public NcoaDeliveryCode getNcoaDeliveryCodeByCode(String code)
	throws UnknownLookupTypeException, UnknownLookupCodeException {
		return (NcoaDeliveryCode) this.getByCode(NcoaDeliveryCode.class, code);
	}

	/**
	 * Find all NcoaStatus
	 *
	 * @return List a list of all NcoaStatus
	 */
	public List getAllNcoaStatuses() throws UnknownLookupTypeException,
	UnknownLookupCodeException {
		return this.findAll(NcoaStatus.class);
	}

	/**
	 * Get NcoaStatus by code
	 *
	 * @return NcoaStatus A corresponding NcoaStatus by code
	 */
	public NcoaStatus getNcoaStatusByCode(String code)
	throws UnknownLookupTypeException, UnknownLookupCodeException {
		return (NcoaStatus) this.getByCode(NcoaStatus.class, code);
	}

	/**
	 * Find all NcoaZipMatch
	 *
	 * @return List a list of all NcoaZipMatch
	 */
	public List getAllNcoaZipMatches() throws UnknownLookupTypeException,
	UnknownLookupCodeException {
		return this.findAll(NcoaZipMatch.class);
	}

	/**
	 * Get NcoaZipMatch by code
	 *
	 * @return NcoaZipMatch A corresponding NcoaZipMatch by code
	 */
	public NcoaZipMatch getNcoaZipMatchByCode(String code)
	throws UnknownLookupTypeException, UnknownLookupCodeException {
		return (NcoaZipMatch) this.getByCode(NcoaZipMatch.class, code);
	}

	/**
	 * Find all ZipPlus4NoMatchReason
	 *
	 * @return List a list of all ZipPlus4NoMatchReason
	 */
	public List getAllZipPlus4NoMatchReasons()
	throws UnknownLookupTypeException, UnknownLookupCodeException {
		return this.findAll(ZipPlus4NoMatchReason.class);
	}

	/**
	 * Get ZipPlus4NoMatchReason by code
	 *
	 * @return ZipPlus4NoMatchReason A corresponding ZipPlus4NoMatchReason by code
	 */
	public ZipPlus4NoMatchReason getZipPlus4NoMatchReasonByCode(String code)
	throws UnknownLookupTypeException, UnknownLookupCodeException {
		return (ZipPlus4NoMatchReason) this.getByCode(
				ZipPlus4NoMatchReason.class, code);
	}

	/**
	 * Find all ComAACErrorType
	 *
	 * @return List a list of all ComAACErrorType
	 */
	public List getAllComAACErrorTypes() throws UnknownLookupTypeException,
	UnknownLookupCodeException {
		return this.findAll(ComAACErrorType.class);
	}

	/**
	 * Get ComAACErrorType by code
	 *
	 * @return ComAACErrorType A corresponding ComAACErrorType by code
	 */
	public ComAACErrorType getComAACErrorTypeByCode(String code)
	throws UnknownLookupTypeException, UnknownLookupCodeException {
		return (ComAACErrorType) this.getByCode(ComAACErrorType.class, code);
	}

	/**
	 * Find all ComAACRejectReasonType
	 *
	 * @return List a list of all ComAACRejectReasonType
	 */
	public List getAllComAACRejectReasonTypes()
	throws UnknownLookupTypeException, UnknownLookupCodeException {
		return this.findAll(ComAACRejectReasonType.class);
	}

	/**
	 * Get ComAACRejectReasonType by code
	 *
	 * @return ComAACRejectReasonType A corresponding ComAACRejectReasonType by
	 *         code
	 */
	public ComAACRejectReasonType getComAACRejectReasonTypeByCode(String code)
	throws UnknownLookupTypeException, UnknownLookupCodeException {
		return (ComAACRejectReasonType) this.getByCode(
				ComAACRejectReasonType.class, code);
	}

	/**
	 * Find all ComLetterGeneratedType
	 *
	 * @return List a list of all ComLetterGeneratedType
	 */
	public List getAllComLetterGeneratedTypes()
	throws UnknownLookupTypeException, UnknownLookupCodeException {
		return this.findAll(ComLetterGeneratedType.class);
	}

	/**
	 * Get ComLetterGeneratedType by code
	 *
	 * @return ComLetterGeneratedType A corresponding ComLetterGeneratedType by
	 *         code
	 */
	public ComLetterGeneratedType getComLetterGeneratedTypeByCode(String code)
	throws UnknownLookupTypeException, UnknownLookupCodeException {
		return (ComLetterGeneratedType) this.getByCode(
				ComLetterGeneratedType.class, code);
	}

	/**
	 * Find all ComLetterType
	 *
	 * @return List a list of all ComLetterType
	 */
	public List getAllComLetterTypes() throws UnknownLookupTypeException,
	UnknownLookupCodeException {
		return this.findAll(ComLetterType.class);
	}

	/**
	 * Get ComLetterType by code
	 *
	 * @return ComLetterType A corresponding ComLetterType by code
	 */
	public ComLetterType getComLetterTypeByCode(String code)
	throws UnknownLookupTypeException, UnknownLookupCodeException {
		return (ComLetterType) this.getByCode(ComLetterType.class, code);
	}

	/**
	 * Find all ComMailingStatusType
	 *
	 * @return List a list of all ComMailingStatusType
	 */
	public List getAllComMailingStatusTypes()
	throws UnknownLookupTypeException, UnknownLookupCodeException {
		return this.findAll(ComMailingStatusType.class);
	}

	/**
	 * Find all ComMailingTriggerType
	 *
	 * @return List a list of all ComMailingStatusType
	 */
	public List getAllComMailingTriggerTypes()
	throws UnknownLookupTypeException, UnknownLookupCodeException {
		return this.findAll(ComMailingTriggerType.class);
	}

	/**
	 * Get ComMailingStatusType by code
	 *
	 * @return ComMailingStatusType A corresponding ComMailingStatusType by code
	 */
	public ComMailingStatusType getComMailingStatusTypeByCode(String code)
	throws UnknownLookupTypeException, UnknownLookupCodeException {
		return (ComMailingStatusType) this.getByCode(
				ComMailingStatusType.class, code);
	}

	/**
	 * Get ComMailingTriggerType by code
	 *
	 * @return ComMailingStatusType A corresponding ComMailingStatusType by code
	 */
	public ComMailingTriggerType getComMailingTriggerTypeByCode(String code)
	throws UnknownLookupTypeException, UnknownLookupCodeException {
		return (ComMailingTriggerType) this.getByCode(
				ComMailingTriggerType.class, code);
	}

	/**
	 * Find all WkfCaseActivityType
	 *
	 * @return List a list of all WkfCaseActivityType
	 */
	public List getAllWkfCaseActivityTypes() throws UnknownLookupTypeException,
	UnknownLookupCodeException {
		return this.findAll(WkfCaseActivityType.class);
	}

	/**
	 * Get WkfCaseActivityType by code
	 *
	 * @return WkfCaseActivityType A corresponding WkfCaseActivityType by code
	 */
	public WkfCaseActivityType getWkfCaseActivityTypeByCode(String code)
	throws UnknownLookupTypeException, UnknownLookupCodeException {
		return (WkfCaseActivityType) this.getByCode(WkfCaseActivityType.class,
				code);
	}

	/**
	 * Find all WkfCaseRequestType
	 *
	 * @return List a list of all WkfCaseRequestType
	 */
	public List getAllWkfCaseRequestTypes() throws UnknownLookupTypeException,
	UnknownLookupCodeException {
		return this.findAll(WkfCaseRequestType.class);
	}

	/**
	 * Get WkfCaseRequestType by code
	 *
	 * @return WkfCaseRequestType A corresponding WkfCaseRequestType by code
	 */
	public WkfCaseRequestType getWkfCaseRequestTypeByCode(String code)
	throws UnknownLookupTypeException, UnknownLookupCodeException {
		return (WkfCaseRequestType) this.getByCode(WkfCaseRequestType.class,
				code);
	}

	/**
	 * Find all WkfCaseStatusType
	 *
	 * @return List a list of all WkfCaseStatusType
	 */
	public List getAllWkfCaseStatusTypes() throws UnknownLookupTypeException,
	UnknownLookupCodeException {
		return this.findAll(WkfCaseStatusType.class);
	}

	/**
	 * Get WkfCaseStatusType by code
	 *
	 * @return WkfCaseStatusType A corresponding WkfCaseStatusType by code
	 */
	public WkfCaseStatusType getWkfCaseStatusTypeByCode(String code)
	throws UnknownLookupTypeException, UnknownLookupCodeException {
		return (WkfCaseStatusType) this.getByCode(WkfCaseStatusType.class, code);
	}

	/**
	 * Find all WkfCaseType
	 *
	 * @return List a list of all WkfCaseType
	 */
	public List getAllWkfCaseTypes() throws UnknownLookupTypeException,
	UnknownLookupCodeException {
		return this.findAll(WkfCaseType.class);
	}

	/**
	 * Get WkfCaseType by code
	 *
	 * @return WkfCaseType A corresponding WkfCaseType by code
	 */
	public WkfCaseType getWkfCaseTypeByCode(String code)
	throws UnknownLookupTypeException, UnknownLookupCodeException {
		return (WkfCaseType) this.getByCode(WkfCaseType.class, code);
	}

	/**
	 * Find all WkfIssueType
	 *
	 * @return List a list of all WkfIssueType
	 */
	public List getAllWkfIssueTypes() throws UnknownLookupTypeException,
	UnknownLookupCodeException {
		return this.findAll(WkfIssueType.class);
	}

	/**
	 * Get WkfIssueType by code
	 *
	 * @return WkfIssueType A corresponding WkfIssueType by code
	 */
	public WkfIssueType getWkfIssueTypeByCode(String code)
	throws UnknownLookupTypeException, UnknownLookupCodeException {
		return (WkfIssueType) this.getByCode(WkfIssueType.class, code);
	}

	/**
	 * (non-Javadoc)
	 * @see gov.va.med.esr.service.LookupService#getMessageStatusByCode(gov.va.med.esr.common.model.lookup.MessageStatus.Code)
	 */
	public MessageStatus getMessageStatusByCode(MessageStatus.Code code)
	throws UnknownLookupTypeException, UnknownLookupCodeException {
		return (MessageStatus) getByCode(MessageStatus.class, code);
	}

	/**
	 *  (non-Javadoc)
	 * @see gov.va.med.esr.service.LookupService#getAllAckTypes()
	 */
	public List getAllAckTypes() throws UnknownLookupTypeException {
		return this.findAll(AckType.class);
	}

	/**
	 *  (non-Javadoc)
	 * @see gov.va.med.esr.service.LookupService#getAckTypeByCode(AckType.Code)
	 */
	public AckType getAckTypeByCode(AckType.Code code)
	throws UnknownLookupTypeException, UnknownLookupCodeException {
		return (AckType) this.getByCode(AckType.class, code);
	}

	/**
	 * Get the GMTThreshold by income year and fips code.
	 * @param year the year.
	 * @param countyCode the county code.
	 * @return List of GMTThreshold Objects.
	 * @throws ServiceException
	 */
	public List getGMTThreshold(Integer year, String countyCode)
	throws ServiceException {
		Validate.notNull(year, "GMT Year can not be null.");
		Validate.notNull(countyCode, "County Code can not be null.");
		List result = null;
		try {
			String[] paramNames = { GMTThreshold.QUERY_PARAM_YEAR,
					GMTThreshold.QUERY_PARAM_FIPS_CODE };
			Object[] paramValues = { year, countyCode };
			result = getGenericDAO().findByNamedQueryAndNamedParam(
					GMTThreshold.QUERY_FIND_BY_YEAR_FIPS, paramNames,
					paramValues);
		} catch (DAOException ex) {
			throw new ServiceException(
					"Error finding GMT Threshold information from the database.",
					ex);
		}
		return result;
	}

	/**
	 * Get the GMTThreshold by income year, fips code and msa code.
	 *
	 * @param year the year.
	 * @param countyCode the county code.
	 * @param msaCode the msa code.
	 * @return List of GMTThreshold Objects.
	 * @throws ServiceException
	 */
	public List getGMTThreshold(Integer year, String countyCode, String msaCode)
	throws ServiceException {
		Validate.notNull(year, "GMT Year can not be null.");
		Validate.notNull(countyCode, "County Code can not be null.");
		Validate.notNull(msaCode, "MSA Code can not be null.");
		List result = null;
		try {
			String[] paramNames = { GMTThreshold.QUERY_PARAM_YEAR,
					GMTThreshold.QUERY_PARAM_FIPS_CODE,
					GMTThreshold.QUERY_PARAM_MSA_CODE };
			Object[] paramValues = { year, countyCode, msaCode };
			result = getGenericDAO().findByNamedQueryAndNamedParam(
					GMTThreshold.QUERY_FIND_BY_YEAR_FIPS_MSA, paramNames,
					paramValues);
			//ES 4.1_CodeCR13795
			//If No GMT Thresholds found for given IY, county and MSA code
			//then lookup again with default MSA code of 9999
			if (result == null || result.size() == 0) {
				Object[] defParamValues = { year, countyCode, GMTThreshold.DEFAULT_MSA_CODE };
				result = getGenericDAO().findByNamedQueryAndNamedParam(
						GMTThreshold.QUERY_FIND_BY_YEAR_FIPS_MSA, paramNames,
						defParamValues);
				//ES 4.1_CodeCR13795
				//If No GMT Thresholds found for given IY, county and default MSA code of 9999
				//then lookup again with given IY and county
				if (result == null || result.size() == 0) {
					result = getGMTThreshold(year, countyCode);
				}
			}
		} catch (DAOException ex) {
			throw new ServiceException(
					"Error finding GMT Threshold information from the database.",
					ex);
		}
		return result == null ? new ArrayList() : result;
	}

	public String getFipsCode(String zipCode)
	throws ServiceException {
		Validate.notNull(zipCode, "ZipCode can not be null.");
		List result = null;
		try {
			String[] paramNames = { "code"};
			Object[] paramValues = { zipCode };
			result = getGenericDAO().findByNamedQueryAndNamedParam(
					"zipCodeQuery_getFipsCode", paramNames,
					paramValues);
		} catch (DAOException ex) {
			throw new ServiceException(
					"Error finding ZipCode information from the database.",
					ex);
		}
		return result == null || result.size() ==0 ? null : (String)result.get(0);
	}


	public String getMsaCode(String zipCode) throws ServiceException {
		Validate.notNull(zipCode, "ZipCode can not be null.");
		List result = null;
		try {
			String[] paramNames = { "code" };
			Object[] paramValues = { zipCode };
			result = getGenericDAO().findByNamedQueryAndNamedParam(
					"msaCodeQuery_GetMsaCodeByZipCode", paramNames, paramValues);
		} catch (DAOException ex) {
			throw new ServiceException(
					"Error finding MsaCode information from the database.", ex);
		}
		return result == null || result.size() == 0 ? null : (String) result
				.get(0);
	}

	/**
	 * Get the IncomeThreshold by income year.
	 * @param year the year.
	 * @throws ServiceException
	 */
	public IncomeThreshold getIncomeThreshold(Integer year) throws ServiceException {
		Validate.notNull(year, "Year can not be null.");

		List result = null;
		try {
			result = getGenericDAO().findByNamedQueryAndNamedParam(
					IncomeThreshold.QUERY_FIND_BY_YEAR,
					IncomeThreshold.QUERY_PARAM_YEAR,
					year);
		} catch (DAOException ex) {
			throw new ServiceException(
					"Error finding Income Threshold information from the database.",
					ex);
		}
		return result == null || result.size() ==0 ? null : (IncomeThreshold)result.get(0);
	}

	/**
	 * @see gov.va.med.esr.service.LookupService#getAllIncomeThresholds()
	 */
	public List getAllIncomeThresholds() throws ServiceException {
		List result = null;
		try {
			result = getGenericDAO().find(IncomeThreshold.QUERY_FIND_ALL);
		} catch (DAOException ex) {
			throw new ServiceException(
					"Error loading Income Threshold information from the database.",
					ex);
		}
		return result == null ? new ArrayList() : result;
	}
	/**
	 * Find all SystemParameterType
	 *
	 * @return List a list of all SystemParameterType
	 */
	public List getAllSystemParameterTypes() throws UnknownLookupTypeException,
	UnknownLookupCodeException {
		return this.findAll(SystemParameterType.class);
	}

	/**
	 * Get SystemParameterType by code
	 *
	 * @return SystemParameterType A corresponding SystemParameterType by code
	 */
	public SystemParameterType getSystemParameterTypeByCode(String code)
	throws UnknownLookupTypeException, UnknownLookupCodeException {
		return (SystemParameterType) this.getByCode(SystemParameterType.class, code);
	}
	/**
	 * Find all SystemParameterUnit
	 *
	 * @return List a list of all SystemParameterUnit
	 */
	public List getAllSystemParameterUnits() throws UnknownLookupTypeException,
	UnknownLookupCodeException {
		return this.findAll(SystemParameterUnit.class);
	}

	/**
	 * Get SystemParameterUnit by code
	 *
	 * @return SystemParameterUnit A corresponding SystemParameterUnit by code
	 */
	public SystemParameterUnit getSystemParameterUnitByCode(String code)
	throws UnknownLookupTypeException, UnknownLookupCodeException {
		return (SystemParameterUnit) this.getByCode(SystemParameterUnit.class, code);
	}

	/**
	 * Find all Capability
	 *
	 * @return List a list of all Capability
	 */
	public List getAllCapabilities() throws UnknownLookupTypeException,
	UnknownLookupCodeException {
		return this.findAll(Capability.class);
	}

	/**
	 * Get Capability by code
	 *
	 * @return Capability A corresponding Capability by code
	 */
	public Capability getCapabilityByCode(String code)
	throws UnknownLookupTypeException, UnknownLookupCodeException {
		return (Capability) this.getByCode(Capability.class, code);
	}
	/**
	 * Get All Functional Groups
	 */
	public List getAllFunctionalGroups() throws UnknownLookupTypeException, UnknownLookupCodeException {
		return this.findAll(FunctionalGroup.class);
	}

	/**
	 * Get Functional Group By Code
	 */
	public FunctionalGroup getFunctionalGroupByCode(String code)
	throws UnknownLookupTypeException, UnknownLookupCodeException {
		return (FunctionalGroup) this.getByCode(FunctionalGroup.class, code);
	}

	/**
	 * @see gov.va.med.esr.service.LookupService#getAllReportDivisions()
	 */
	public List getAllReportDivisions() throws UnknownLookupTypeException, UnknownLookupCodeException {
		return this.findAll(ReportDivision.class);
	}

	/**
	 * @see gov.va.med.esr.service.LookupService#getAllReportExportedTypes()
	 */
	public List getAllReportExportedTypes() throws UnknownLookupTypeException, UnknownLookupCodeException {
		return this.findAll(ReportExportedType.class);
	}

	/**
	 * @see gov.va.med.esr.service.LookupService#getAllReportFormats()
	 */
	public List getAllReportFormats() throws UnknownLookupTypeException, UnknownLookupCodeException {
		return this.findAll(ReportFormat.class);
	}

	/**
	 * @see gov.va.med.esr.service.LookupService#getAllStandardReports()
	 */
	public List getAllStandardReports() throws UnknownLookupTypeException, UnknownLookupCodeException {
		return this.findAll(StandardReport.class);
	}
	/**
	 * @see gov.va.med.esr.service.LookupService#getReportDivisionByCode(java.lang.String)
	 */
	public ReportDivision getReportDivisionByCode(String code) throws UnknownLookupTypeException, UnknownLookupCodeException {
		return (ReportDivision) this.getByCode(ReportDivision.class, code);
	}

	/**
	 * @see gov.va.med.esr.service.LookupService#getReportExportedTypeByCode(java.lang.String)
	 */
	public ReportExportedType getReportExportedTypeByCode(String code) throws UnknownLookupTypeException, UnknownLookupCodeException {
		return (ReportExportedType) this.getByCode(ReportExportedType.class, code);
	}

	/**
	 * @see gov.va.med.esr.service.LookupService#getReportFormatByCode(java.lang.String)
	 */
	public ReportFormat getReportFormatByCode(String code) throws UnknownLookupTypeException, UnknownLookupCodeException {
		return (ReportFormat) this.getByCode(ReportFormat.class, code);
	}

	/**
	 * @see gov.va.med.esr.service.LookupService#getStandardReportByCode(java.lang.String)
	 */
	public StandardReport getStandardReportByCode(String code) throws UnknownLookupTypeException, UnknownLookupCodeException {
		return (StandardReport) this.getByCode(StandardReport.class, code);
	}

	/**
	 * @see gov.va.med.esr.service.LookupService#getReportTypeByCode(java.lang.String)
	 */
	public ReportType getReportTypeByCode(String code)
	throws UnknownLookupTypeException, UnknownLookupCodeException
	{
		return (ReportType) this.getByCode(ReportType.class, code);
	}

	/**
	 * @see gov.va.med.esr.service.LookupService#getReportPeriodTypeByCode(java.lang.String)
	 */
	public ReportPeriodType getReportPeriodTypeByCode(String code) throws UnknownLookupTypeException, UnknownLookupCodeException {
		return (ReportPeriodType) this.getByCode(ReportPeriodType.class, code);
	}

	/**
	 * @see gov.va.med.esr.service.LookupService#getAllReportPeriodTypes()
	 */
	public List getAllReportPeriodTypes() throws UnknownLookupTypeException, UnknownLookupCodeException {
		return this.findAll(ReportPeriodType.class);
	}

	/**
	 * @see gov.va.med.esr.service.LookupService#getReportPeriodTypeByCode(java.lang.String)
	 */
	public ReportEEDWeeklyType getReportEEDWeeklyTypeByCode(String code) throws UnknownLookupTypeException, UnknownLookupCodeException {
		return (ReportEEDWeeklyType) this.getByCode(ReportEEDWeeklyType.class, code);
	}

	/**
	 * @see gov.va.med.esr.service.LookupService#getAllReportEEDWeeklyTypes()
	 */
	public List getAllReportEEDWeeklyTypes() throws UnknownLookupTypeException, UnknownLookupCodeException {
		return this.findAll(ReportEEDWeeklyType.class);
	}

	/**
	 * @see gov.va.med.esr.service.LookupService#getQuarterByCode(java.lang.String)
	 */
	public Quarter getQuarterByCode(String code) throws UnknownLookupTypeException, UnknownLookupCodeException {
		return (Quarter) this.getByCode(Quarter.class, code);
	}

	/**
	 * @see gov.va.med.esr.service.LookupService#getAllQuarters()
	 */
	public List getAllQuarters() throws UnknownLookupTypeException, UnknownLookupCodeException {
		return this.findAll(Quarter.class);
	}

	/**
	 * @see gov.va.med.esr.service.LookupService#getEmergencyResponseByCode(java.lang.String)
	 */
	public EmergencyResponse getEmergencyResponseByCode(String code) throws UnknownLookupTypeException, UnknownLookupCodeException {
		return (EmergencyResponse) this.getByCode(EmergencyResponse.class, code);
	}

	/**
	 * @see gov.va.med.esr.service.LookupService#getAllEmergencyResponses()
	 */
	public List getAllEmergencyResponses() throws UnknownLookupTypeException, UnknownLookupCodeException {
		return this.findAll(EmergencyResponse.class);
	}

	public BenefitType getBenefitTypeByCode(String code) throws UnknownLookupTypeException, UnknownLookupCodeException {
		return (BenefitType) this.getByCode(BenefitType.class, code);
	}

	public List getAllBenefitTypes() throws UnknownLookupTypeException, UnknownLookupCodeException {
		return this.findAll(BenefitType.class);
	}

	public EthnicityType getEthnicityTypeByCode(String code) throws UnknownLookupTypeException, UnknownLookupCodeException {
		return (EthnicityType) this.getByCode(EthnicityType.class, code);
	}

	public List getAllEthnicityTypes() throws UnknownLookupTypeException, UnknownLookupCodeException {
		return this.findAll(EthnicityType.class);
	}

	public RaceType getRaceTypeByCode(String code) throws UnknownLookupTypeException, UnknownLookupCodeException {
		return (RaceType) this.getByCode(RaceType.class, code);
	}

	public List getAllRaceTypes() throws UnknownLookupTypeException, UnknownLookupCodeException {
		return this.findAll(RaceType.class);
	}

	public SpinalCordInjuryType getSpinalCordInjuryTypeByCode(String code) throws UnknownLookupTypeException, UnknownLookupCodeException {
		return (SpinalCordInjuryType) this.getByCode(SpinalCordInjuryType.class, code);
	}

	public List getAllSpinalCordInjuryTypes() throws UnknownLookupTypeException, UnknownLookupCodeException {
		return this.findAll(SpinalCordInjuryType.class);
	}

	public List getAllMHDataSources() throws UnknownLookupTypeException, UnknownLookupCodeException {
		return this.findAll(MHDataSource.class);
	}

	public List getAllMHDocumentTypes() throws UnknownLookupTypeException, UnknownLookupCodeException {
		return this.findAll(MHDocumentType.class);
	}

	public List getAllMilitaryServiceQueryStatuses() throws UnknownLookupTypeException, UnknownLookupCodeException {
		return this.findAll(MilitaryServiceQueryStatus.class);
	}

	public List getAllNamedContingencys() throws UnknownLookupTypeException, UnknownLookupCodeException {
		return this.findAll(NamedContingency.class);
	}

	public List getAllSpecialOperations() throws UnknownLookupTypeException, UnknownLookupCodeException {
		return this.findAll(SpecialOperation.class);
	}

	public List getAllTerminationReasons() throws UnknownLookupTypeException, UnknownLookupCodeException {
		return this.findAll(TerminationReason.class);
	}
	public List getAllCombatServiceSources() throws UnknownLookupTypeException, UnknownLookupCodeException {
		return this.findAll(CombatServiceSource.class);
	}
	public CombatServiceSource getCombatServiceSourceByCode(String code) throws UnknownLookupTypeException, UnknownLookupCodeException {
		return (CombatServiceSource)this.getByCode(CombatServiceSource.class, code);
	}

	public MHDataSource getMHDataSourceByCode(String code) throws UnknownLookupTypeException, UnknownLookupCodeException {
		return (MHDataSource)this.getByCode(MHDataSource.class, code);
	}

	public MHDocumentType getMHDocumentTypeByCode(String code) throws UnknownLookupTypeException, UnknownLookupCodeException {
		return (MHDocumentType)this.getByCode(MHDocumentType.class, code);
	}

	public MilitaryServiceQueryStatus getMilitaryServiceQueryStatusByCode(String code) throws UnknownLookupTypeException, UnknownLookupCodeException {
		return (MilitaryServiceQueryStatus)this.getByCode(MilitaryServiceQueryStatus.class, code);
	}

	public NamedContingency getNamedContingencyByCode(String code) throws UnknownLookupTypeException, UnknownLookupCodeException {
		return (NamedContingency)this.getByCode(NamedContingency.class, code);
	}

	public SpecialOperation getSpecialOperationByCode(String code) throws UnknownLookupTypeException, UnknownLookupCodeException {
		return (SpecialOperation)this.getByCode(SpecialOperation.class, code);
	}

	public TerminationReason getTerminationReasonByCode(String code) throws UnknownLookupTypeException, UnknownLookupCodeException {
		return (TerminationReason)this.getByCode(TerminationReason.class, code);
	}

	/**
	 * Get CollectionMethod by code
	 *
	 * @return CollectionMethod
	 */
	 public CollectionMethod getCollectionMethodByCode(String code)
	 throws UnknownLookupTypeException, UnknownLookupCodeException {
		 return (CollectionMethod) this.getByCode(CollectionMethod.class, code);
	 }

	 /**
	  * Find all collectionMethods
	  *
	  * @return List a list of all CollectionMethod
	  */
	 public List getAllCollectionMethods()
	 throws UnknownLookupTypeException, UnknownLookupCodeException {
		 return this.findAll(CollectionMethod.class);
	 }

	 /**
	  * Get EEServiceGroup by code
	  *
	  * @return Gender A corresponding Gender by code
	  */
	 public EEServiceGroup getEEServiceGroupByCode(String code)
	 throws UnknownLookupTypeException, UnknownLookupCodeException {
		 return (EEServiceGroup) this.getByCode(EEServiceGroup.class, code);
	 }

	 /**
	  * Get EEServiceGroup by code
	  *
	  * @return Gender A corresponding Gender by code
	  */
	 public EEServiceField getEEServiceFieldByCode(String code)
	 throws UnknownLookupTypeException, UnknownLookupCodeException {
		 return (EEServiceField) this.getByCode(EEServiceField.class, code);
	 }


	 public PensionReasonCode getPensionReasonByCode(String code)
	 throws UnknownLookupTypeException, UnknownLookupCodeException {
		 return (PensionReasonCode) this.getByCode(PensionReasonCode.class, code);
	 }

	 /**
	  * Get SeedStatus by code
	  *
	  * @return SeedStatus A corresponding seeding status by code
	  */
	 public SeedStatus getSeedStatusByCode(String code)
	 throws UnknownLookupTypeException, UnknownLookupCodeException {
		 return (SeedStatus) this.getByCode(SeedStatus.class, code);
	 }

	 /**
	  * Find all Source Designations
	  *
	  * @return List a list of all SourceDesignations
	  */
	 public List getAllSourceDesignations() throws UnknownLookupTypeException,
	 UnknownLookupCodeException {
		 return this.findAll(SourceDesignation.class);
	 }


	 /**
	  * Get SourceDesignation by code
	  *
	  * @return SourceDesignation
	  */
	 public SourceDesignation getSourceDesignationByCode(String code)
	 throws UnknownLookupTypeException, UnknownLookupCodeException {
		 return (SourceDesignation) this.getByCode(SourceDesignation.class, code);

	 }

	 /**
	  * Get TeamPurpose by code
	  *
	  * @return TeamPurpose
	  */
	 public TeamPurpose getTeamPurposeByCode(String code)
	 throws UnknownLookupTypeException, UnknownLookupCodeException {
		 return (TeamPurpose) this.getByCode(TeamPurpose.class, code);
	 }

	 public ApplicationMethod getApplicationMethodByCode(String code)
	 throws UnknownLookupTypeException, UnknownLookupCodeException {
		 return (ApplicationMethod) this.getByCode(ApplicationMethod.class, code);
	 }

	 public ApplicationStatus getApplicationStatusByCode(String code)
	 throws UnknownLookupTypeException, 	UnknownLookupCodeException {
		 return (ApplicationStatus) this.getByCode(ApplicationStatus.class, code);
	 }
	 /**
	  * Find all HandBookRequestStatusType
	  *
	  * @return List a list of all HandBookRequestStatusType
	  */
	 public List getAllHandBookBatchRequestStatus() throws UnknownLookupTypeException,
	 UnknownLookupCodeException {
		 return this.findAll(HandBookRequestStatusType.class);
	 }

	 /**
	  * Get HandBookRequestStatusType by code
	  *
	  * @return HandBookRequestStatusType A corresponding HandBookRequestStatusType by code
	  */
	 public HandBookRequestStatusType getHandBookRequestStatusTypeByCode(String code)
	 throws UnknownLookupTypeException, UnknownLookupCodeException {
		 return (HandBookRequestStatusType) this.getByCode(HandBookRequestStatusType.class, code);
	 }

	 /**
	  * Get HandBookMailStatusType by code
	  *
	  * @return HandBookMailStatusType A corresponding HandBookMailStatusType by code
	  */
	 public HandBookMailStatusType getHandBookMailingStatusTypeByCode(String code)
	 throws UnknownLookupTypeException, UnknownLookupCodeException {
		 return (HandBookMailStatusType) this.getByCode(
				 HandBookMailStatusType.class, code);
	 }

	/**
	 * Find all DeliveryPreferenceType
	 *
	 * @return a list of all DeliveryPreferenceType
	 */
	@SuppressWarnings("rawtypes")
	public List getAllDeliveryPreferenceTypes()
			throws UnknownLookupTypeException, UnknownLookupCodeException {
		return this.findAll(DeliveryPreferenceType.class);
	}

	/**
	 * Find all DeliveryPreferenceSourceOfChange
	 *
	 * @return a list of all DeliveryPreferenceSourceOfChange
	 */
	@SuppressWarnings("rawtypes")
	public List getAllDeliveryPreferenceSourceOfChangeTypes()
			throws UnknownLookupTypeException, UnknownLookupCodeException {
		return this.findAll(DeliveryPreferenceSourceOfChange.class);
	}

	/**
	 * Get DeliveryPreferenceType by code
	 *
	 * @return DeliveryPreferenceType A corresponding DeliveryPreferenceType by
	 *         code
	 */
	public DeliveryPreferenceType getDeliveryPreferenceTypeByCode(String code)
			throws UnknownLookupTypeException, UnknownLookupCodeException {
		return (DeliveryPreferenceType) getByCode(DeliveryPreferenceType.class,
				code);
	}

	/**
	 * Get DeliveryPreferenceSourceOfChange by code
	 *
	 * @return DeliveryPreferenceSourceOfChange A corresponding
	 *         DeliveryPreferenceSourceOfChange by code
	 */
	public DeliveryPreferenceSourceOfChange getDeliveryPreferenceSourceOfChangeByCode(
			String code) throws UnknownLookupTypeException,
			UnknownLookupCodeException {
		return (DeliveryPreferenceSourceOfChange) getByCode(
				DeliveryPreferenceSourceOfChange.class, code);
	}

	public List getAllAuthenticationLevels() throws UnknownLookupTypeException, UnknownLookupCodeException {
		return this.findAll(AuthenticationLevel.class);
	}

	public List getAllVOAApplicationStatuses() throws UnknownLookupTypeException, UnknownLookupCodeException {
		return this.findAll(VOAApplicationStatus.class);
	}

	public List getAllVOAFormTypes() throws UnknownLookupTypeException, UnknownLookupCodeException {
		return this.findAll(VOAFormType.class);
	}

	public AuthenticationLevel getAuthenticationLevelByCode(String code) throws UnknownLookupTypeException, UnknownLookupCodeException {
		return (AuthenticationLevel) getByCode(AuthenticationLevel.class, code);
	}

	public VOAApplicationStatus getVOAApplicationStatusByCode(String code) throws UnknownLookupTypeException, UnknownLookupCodeException {
		return (VOAApplicationStatus) getByCode(VOAApplicationStatus.class, code);
	}

	public VOAFormType getVOAFormTypeByCode(String code) throws UnknownLookupTypeException, UnknownLookupCodeException {
		return (VOAFormType) getByCode(VOAFormType.class, code);
	}

	public List getAllEligibilityDocTypes() throws UnknownLookupTypeException, UnknownLookupCodeException {
		return this.findAll(EligibilityDocType.class);
	}

	public EligibilityDocType getEligibilityDocTypeByCode(String code) throws UnknownLookupTypeException, UnknownLookupCodeException {
		return (EligibilityDocType) getByCode(EligibilityDocType.class, code);
	}

	public List getAllVeteranIdentifierTypes() throws UnknownLookupTypeException, UnknownLookupCodeException {
		return this.findAll(VeteranIdentifierType.class);
	}

	public VeteranIdentifierType getVeteranIdentifierTypeByCode(String code) throws UnknownLookupTypeException, UnknownLookupCodeException {
		return (VeteranIdentifierType) getByCode(VeteranIdentifierType.class, code);
	}

	public List getAllDataChangeSources() throws UnknownLookupTypeException, UnknownLookupCodeException {
		return this.findAll(DataChangeSource.class);
	}

	public DataChangeSource getDataChangeSourceByCode(String code) throws UnknownLookupTypeException, UnknownLookupCodeException {
		return (DataChangeSource) getByCode(DataChangeSource.class, code);
	}

	public List getAllHealthBenefitPlanTypes() throws UnknownLookupTypeException, UnknownLookupCodeException {
		return this.findAll(HealthBenefitPlanType.class);
	}

	public HealthBenefitPlanType getHealthBenefitPlanTypeByCode(String code) throws UnknownLookupTypeException, UnknownLookupCodeException {
		return (HealthBenefitPlanType) getByCode(HealthBenefitPlanType.class, code);
	}

	public List getAllDescriptorTypes() throws UnknownLookupTypeException, UnknownLookupCodeException {
		return this.findAll(DescriptorType.class);
	}

	public DescriptorType getDescriptorTypeByCode(String code) throws UnknownLookupTypeException, UnknownLookupCodeException {
		return (DescriptorType) this.getByCode(DescriptorType.class, code);
	}

    /**
     * Find all Languages.
     *
     * @return List a list of all Language
     */
    public List getAllLanguages() throws UnknownLookupTypeException,
            UnknownLookupCodeException {
        return this.findAll(Language.class);
    }

    /**
     * Get Language by code.
     *
     * @param code
     * @return Language A corresponding Language by code
     * @throws UnknownLookupTypeException Thrown if the type of lookup cannot be found.
     * @throws UnknownLookupCodeException Thrown if the specified code cannot be found.
     */
    public Language getLanguageByCode(String code)
            throws UnknownLookupTypeException, UnknownLookupCodeException {
        return (Language) this.getByCode(Language.class, code);
    }

    /**
     * Find all HealthBenefitPlanCategory.
     *
     * @return List a list of all HealthBenefitPlanCategory
     */
    public List getAllHealthBenefitPlanCategories() throws UnknownLookupTypeException,
            UnknownLookupCodeException {
        return this.findAll(HealthBenefitPlanCategory.class);
    }

    /**
     * Get HealthBenefitPlanCategory by code.
     *
     * @param code
     * @return HealthBenefitPlanCategory A corresponding HealthBenefitPlanCategory by code
     * @throws UnknownLookupTypeException Thrown if the type of lookup cannot be found.
     * @throws UnknownLookupCodeException Thrown if the specified code cannot be found.
     */
    public HealthBenefitPlanCategory getHealthBenefitPlanCategoryByCode(String code)
            throws UnknownLookupTypeException, UnknownLookupCodeException {
        return (HealthBenefitPlanCategory) this.getByCode(HealthBenefitPlanCategory.class, code);
    }

    public CampLejeuneVerificationType getCampLejeuneVerificationTypeByCode(String code) throws UnknownLookupTypeException, UnknownLookupCodeException {
		return (CampLejeuneVerificationType) getByCode(CampLejeuneVerificationType.class, code);
	}

    public CampLejeuneChangeSource getCampLejeuneChangeSourceByCode(String code) throws UnknownLookupTypeException, UnknownLookupCodeException {
		return (CampLejeuneChangeSource) getByCode(CampLejeuneChangeSource.class, code);
	}
    

	public List getAllConfAddressCategories()throws UnknownLookupTypeException,
	UnknownLookupCodeException {
		try
		{
			ConfidentialAddressCategoryTypeDAO confidentialAddressCategoryTypeDAO = (ConfidentialAddressCategoryTypeDAO) this.getDAO(ConfidentialAddressCategoryTypeDAO.class);
			return confidentialAddressCategoryTypeDAO.getAllConfidentialAddressCategoryTypes();
		} catch (DAOException e)
		{
			throw new RuntimeException(e);
		}

	}
}