/*******************************************************************************
 * Copyright  2005 VHA. All rights reserved
 ******************************************************************************/
package gov.va.med.esr.service.impl;

// Java Classes

// Framework Classes
import gov.va.med.fw.service.AbstractComponent;
import gov.va.med.fw.service.ServiceException;
import gov.va.med.fw.service.trigger.TriggerEvent;
import gov.va.med.fw.service.trigger.TriggerableService;
import gov.va.med.fw.util.InvalidConfigurationException;

import gov.va.med.esr.common.model.person.Person;
import gov.va.med.esr.common.model.person.id.PersonEntityKey;
import gov.va.med.esr.service.CommsEmailBulletinService;
import gov.va.med.esr.service.PersonService;
import gov.va.med.esr.service.trigger.BulletinTriggerEvent;
import gov.va.med.esr.service.trigger.BulletinTriggerIdentity;

/**
 * This class processes a bulletin trigger event message by preparing any needed
 * data and calling the appropriate bulletin service API.
 * 
 * @author Andrew Pach
 * @version 3.0
 */
public class CommsBulletinMessageProcessor extends AbstractComponent implements TriggerableService {

    private PersonService personService = null;

    private CommsEmailBulletinService commsEmailBulletinService = null;

    /**
     * Process the BulletinTriggerEvent.
     *
     * @param triggerEvent The trigger event
     * @throws gov.va.med.fw.service.ServiceException If there were any problems encountered.
     */
    public void trigger(TriggerEvent triggerEvent) throws ServiceException {

        if( this.logger.isDebugEnabled() ) {
            this.logger.debug( "Received a trigger event " + triggerEvent );
        }

        BulletinTriggerEvent bulletinTriggerEvent = (BulletinTriggerEvent) triggerEvent;

        // Get the person based on the person Id in the trigger event
        Person person = getPerson(bulletinTriggerEvent);
        
        // Get the bulletin trigger identity
        BulletinTriggerIdentity bulletinTriggerIdentity = (BulletinTriggerIdentity) bulletinTriggerEvent
                .getTriggerIdentity();

        // Call the bulletin service to send the email.
        commsEmailBulletinService.sendEmailBulletin( bulletinTriggerIdentity
                .getDataType(), bulletinTriggerEvent.getFieldMap(), person);
    }

    private Person getPerson(BulletinTriggerEvent bulletinTriggerEvent) throws ServiceException {
        try {
        	Person person = null;
			if(bulletinTriggerEvent.getIdentityTraits() != null)
			{
				person = personService.find(bulletinTriggerEvent.getIdentityTraits());
			}else 
			{
				PersonEntityKey key = bulletinTriggerEvent.getPersonId();
				if(key != null)
					person = personService.getPerson(key);
			}			
        	
            return person;
        }
        catch (ServiceException e) {
            throw new ServiceException(
                    "Error retrieving a person from the BulletinTriggerEvent",
                    e);
        }
    }

    /**
     * Verify that the required properties are set.
     */
    public void afterPropertiesSet() {
        if (personService == null) {
            throw new InvalidConfigurationException(
                    "personService must be set on " + getClass().getName());
        }
        if (commsEmailBulletinService == null) {
            throw new InvalidConfigurationException(
                    "commsEmailBulletinService must be set on "
                            + getClass().getName());
        }
    }

    public PersonService getPersonService() {
        return personService;
    }

    public void setPersonService(PersonService personService) {
        this.personService = personService;
    }

    public CommsEmailBulletinService getCommsEmailBulletinService() {
        return commsEmailBulletinService;
    }

    public void setCommsEmailBulletinService(
            CommsEmailBulletinService commsEmailBulletinService) {
        this.commsEmailBulletinService = commsEmailBulletinService;
    }
}