/********************************************************************
 * Copyright  2004 VHA. All rights reserved
 ********************************************************************/
// Package
package gov.va.med.esr.service;

// Java classes
import java.io.Serializable;
import java.util.List;
import java.util.Map;

import gov.va.med.esr.common.model.report.CompletedReport;
import gov.va.med.esr.common.model.report.ReportSetup;
import gov.va.med.esr.common.model.report.SimpleCompletedReport;
import gov.va.med.esr.common.model.system.SystemParameter;
import gov.va.med.esr.common.report.data.CommonExtractFileCriteria;
import gov.va.med.fw.model.EntityKey;
import gov.va.med.fw.report.ReportException;
import gov.va.med.fw.security.UserPrincipal;
import gov.va.med.fw.service.ServiceException;
import gov.va.med.fw.persistent.NoRecordFoundException;

/** Provides methods for scheduling, generating, and viewing reports,
 * report schedules, and report setups.  These methods are mainly
 * invoked by the ESR UI layer to display reports, report setups 
 * on the screen.
 *
 * Project: Common</br>
 * Created on: 11:25:09 AM </br>
 *
 * @author DNS   LEV
 */
public interface StandardReportService extends Serializable {
	public static final String REPORT_USER_ID = "reportUser.id";
	public static final String REPORT_USER_LOGIN = "reportUser.login";
	public static final String REPORT_USER_PASSWORD = "reportUser.password";
	public static final String REPORT_SETUP_ID = "reportSetup.id";
	
    /** Gets a list of standard reports.  
     *  This method is intended for a Scheduled Reports screen.
     */
    public List getStandardReports() throws ServiceException;
    
    /** Gets a list of standard reports accessible for the user.  
     *  This method is intended for a Scheduled Reports screen.
     *  @param user The user who setup the reports.
     */
    public List getStandardReports(UserPrincipal user) throws ServiceException ;
    
	
	/**
	 * 
	 * @param identifier
	 * @return ReportSetup with the specified id or null if not found
	 * @throws ServiceException
	 */
	public ReportSetup getReportSetup(EntityKey identifier) throws ServiceException ; 

	/**
	 * Add or update a CompletedReport
	 * @param reportSetup
	 * @throws ServiceException
	 */
    public void saveReportSetup(ReportSetup reportSetup) throws ServiceException;

	
	/**
	 * 
	 * @param identifier
	 * @return CompletedReport with the specified id or null if not found.
	 * @throws ServiceException
	 */
	public CompletedReport getCompletedReport(EntityKey identifier) throws ServiceException;

	/**
	 * Add or update a CompletedReport
	 * @param completedReport
	 * @throws ServiceException
	 */
    public void saveCompletedReport(CompletedReport completedReport) throws ServiceException;	
		
		
	/** Generates a report asycnhronously with the specific report setup.
	 * This method is intended for a Report setup screen when a user presses
	 * a button to generate a report asynchronously.
	 * 
	 * @param user A current logged-in user
	 * @param setup A report setup to generate a report
	 */
	public void generateReport( UserPrincipal user, ReportSetup setup ) throws ServiceException ;

	/** Generates a report asycnhronously with the specific report data in the Map.
	 * Useful for persistence of generateReport arguments into a database.
	 * 
	 * @param reportData ReportData abiding by keys in StandardReportService
	 */
	public void generateReport( Map reportData ) throws ServiceException ;
	
	/** Generates a report asycnhronously with the specific report setup.
	 * This method is intended for a Report setup screen when a user presses
	 * a button to generate a report asynchronously.
	 * 
	 * @param user A current logged-in user
	 * @param criteria An OPP report criteria to generate an OPP report
	 */
	public void generateOPPReport( UserPrincipal user, CommonExtractFileCriteria criteria ) throws ServiceException;
	
	/** Schedules a report generation.  This method is intended for a Report 
	 * schedule screen when a user presses a button to schedule a report generation.
	 * 
	 * @param user A current logged-in user
	 * @param setup A report schedule to schedule for a report generation
	 */
	public void scheduleReport( UserPrincipal user, ReportSetup setup ) throws ServiceException ;
	
	/**
	 * Cancels a previously scheduled report.  The ReportSetup will be deleted, and any
	 * scheduled triggers will be unscheduled.
	 * @param identifier
	 * @throws ServiceException
	 */
	public void cancelScheduledReport(EntityKey identifier, UserPrincipal user) throws NoRecordFoundException, ServiceException;
    
    /**
     * Returns the SimpleCompletedReport for the given identifier.
     * @param identifier
     * @return
     * @throws ServiceException
     */
    public SimpleCompletedReport getSimpleCompletedReport(EntityKey identifier) throws ServiceException;
    
    /**
     * This method deleted completed report.
     * @param identifier
     * @param user
     * @throws NoRecordFoundException
     * @throws ServiceException
     */
    
    public void deleteCompletedReport(EntityKey identifier, UserPrincipal user) throws NoRecordFoundException, ServiceException;  
    
    /**
     * This method inactivates a completed report.
     * 
     * @param identifier
     * @param user
     * @throws NoRecordFoundException
     * @throws ServiceException
     */
    public void inactivateCompletedReport(EntityKey identifier, UserPrincipal user) throws NoRecordFoundException, ServiceException, ReportException;
    
    /**
     * This method inactivates all those reports which are older than specified system parameter.
     * At present, only batch process uses this method.
     * @param updatedQueryString This is update query statement
     * @param auditInfo This info is used auditing.
     * @param systemParameter Archive Reports Parameter info.
     * @return
     * @throws ServiceException
     */
    public int inactivateCompletedReports(String updatedQueryString, String auditInfo, SystemParameter systemParameter) throws ServiceException;
    
    /**
     * This method purges all those inactive reports which are older than specified system parameter.
     * At present, only batch process uses this method.
     * @param purgeQueryString This is delete query statement
     * @param auditInfo This info is used auditing.
     * @param systemParameter Purge Archived Reports Parameter info.
     * @return
     * @throws ServiceException
     */
    public int purgeCompletedReports(String purgeQueryString,String auditInfo, SystemParameter systemParameter) throws ServiceException;
    
    /**
     * This method returns all the information required to disply filter dropdown menus on 
     * completed reports screen.
     * @return
     * @throws ServiceException
     */
    public ReportFilter getCompletedReportFilter() throws ServiceException;
   
}