/********************************************************************
 * Copyright   2004-2005 EDS. All rights reserved
 ********************************************************************/
package gov.va.med.esr.service;

// Java Classes

// Framework classes
import java.util.List;
import java.util.Set;

import gov.va.med.esr.common.model.lookup.VAFacility;
import gov.va.med.esr.common.model.person.FullyQualifiedIdentity;
import gov.va.med.esr.common.model.person.Person;
import gov.va.med.esr.common.model.person.id.PersonEntityKey;
import gov.va.med.esr.common.model.person.id.PersonIdEntityKey;
import gov.va.med.esr.common.model.person.id.VPIDEntityKey;
import gov.va.med.esr.service.external.person.PersonLinkInfo;
import gov.va.med.esr.service.external.person.PersonMoveInfo;
import gov.va.med.esr.service.external.person.PersonPrimaryViewInfo;
import gov.va.med.esr.service.impl.IdMSearchInfo;
import gov.va.med.fw.service.ServiceException;
import gov.va.med.fw.service.pagination.PaginatedSearchService;
import gov.va.med.ps.model.PatientIdentifier;
import gov.va.med.esr.common.batchprocess.AppointmentConversionResult;

/**
 * @author Carlos Ruiz & Andrew Pach
 * @version 3.0
 */
public interface PersonService extends PaginatedSearchService {
   /**
    * Finds the person that best fits the criteria.  Intended for unattended search requests, where only
    * a singular Person is requested.
    *
    * @param traits A subset of identity traits
    * @return The person for the traits
    */
   public Person find(PersonIdentityTraits traits) throws ServiceException;

   /**
    * Find method that only returns the Person's VPID.
    *
    * @param traits A subset of identity traits
    * @return The VPID for the person
    */
   public VPIDEntityKey findVPIDOnly(PersonIdentityTraits traits) throws ServiceException;

   /**
    * Gets refreshed information about the person.
    * @param personID The unique business key of the person.
    * @return Person with refreshed information.
    */
   public Person getPerson(PersonEntityKey personID) throws ServiceException;

   /**
    * CCR11593 If person traits are passed in, no need to call 1305 search again
    * Gets refreshed information about the person.
    * @param personID The unique business key of the person.
    * @return Person with refreshed information.
    */
   public Person getPerson(PersonEntityKey personID, PersonIdentityTraits traits) throws ServiceException;
   /**
     * Get the person without calling PISM for identity trait information.
     *
     * @param personID
     * @return
     * @throws ServiceException
     */
   public Person getPersonWithoutIdentityTraits(PersonEntityKey personID) throws ServiceException;

   /** Receives a person created from site data stored in a HL7 message log table </br>
    * based on a received eligiblity status. </br>
    *
    * @param personID A person id
    * @param status A received eligibility status
    * @return A person built from a site data
    * @throws ServiceException In case of errors retrieving a person
    */
   public Person getPersonFromSiteData( PersonEntityKey personID, String status ) throws ServiceException;

   /**
    * Persist a person into a database
    * @param person A person to persist
    * @retrun Updated Person
    * @throws ServiceException Thrown if failed to persist a person
    */
   public Person save(Person person) throws ServiceException;

   /**
    * Persist a person into a database
    * @param person A person to persist
    * @param overrideLocking
    * @retrun Updated Person
    * @throws ServiceException Thrown if failed to persist a person
    */
   public Person save(Person person, boolean overrideLocking) throws ServiceException;

   /**
    * Entry point for non-ESR initiated person link
    *
    * @param linkInfo
    * @throws ServiceException
    */
   public void processExternalPersonLink(PersonLinkInfo linkInfo) throws ServiceException;

   /**
    * Entry point for non-ESR initiated person move
    *
    * @param moveInfo
    * @throws ServiceException
    */
   public void processExternalPersonMove(PersonMoveInfo moveInfo) throws ServiceException;

   /**
    * Entry point for non-ESR initiated person move
    *
    * @param moveInfo
    * @throws ServiceException
    */
   public void processExternalPersonPrimaryview(PersonPrimaryViewInfo primaryviewInfo) throws ServiceException;

   /**
    * Performs a lookup to get VPID using a PersonId
    * @param personId
    * @return
    * @throws ServiceException
    */
   public VPIDEntityKey getVPIDByPersonId(PersonIdEntityKey personId) throws ServiceException;

   /**
    * Performs a lookup to get PersonId using a VPID
    * @param personId
    * @return
    * @throws ServiceException
    */
   public PersonIdEntityKey getPersonIdByVPID(VPIDEntityKey vpid) throws ServiceException;

   /**
    * Sync the person data with data from HEC legacy.
    *
    * @param person
    * @return
    * @throws ServiceException
    */
   public Person processDataSyncFromHECLegacy(Person person)  throws ServiceException;

   /**
    * Execute bulk ee on single person
    *
    * @param key
    * @return
    * @throws ServiceException
    */
   public EEResult processEEForMigratedPerson(PersonEntityKey key)  throws ServiceException;


   /**
    * Execute the apointment data conversion on person.
    *
    * @param key
    * @return
    * @throws ServiceException
    */
   public AppointmentConversionResult processAppointmentConversion(PersonEntityKey key) throws  ServiceException;

   /**
    * Persist the PersonChangeLogEntry objects
    *
    * @param personChangeLogEntries Set<PersonChangeLogEntry>
    * @throws ServiceException
    */
   public void logChanges(Set personChangeLogEntries) throws ServiceException;

   /**
    * Overloaded search for special case handling.  Enforces ESR Security for attended searches.
    *
    * @param personSearchQueryInfo
    * @param isAttendedMode
    * @param shouldRetrieveFullPerson
    * @param maxResults
    * @return
    * @throws ServiceException
    */
   public List search(PersonSearchQueryInfo personSearchQueryInfo,  boolean isAttendedMode, boolean shouldRetrieveFullPerson, int maxResults) throws ServiceException;

   /**
    * Overloaded search for special case handling.
    *
    * @param personSearchQueryInfo
    * @param isAttendedMode
    * @param shouldRetrieveFullPerson
    * @param enforceSecurity
    * @param maxResults
    * @return
    * @throws ServiceException
    */
   public List search(PersonSearchQueryInfo personSearchQueryInfo,  boolean isAttendedMode, boolean shouldRetrieveFullPerson, boolean enforceSecurity, int maxResults) throws ServiceException;

   /**
    * Search for ADD A PERSON.
    *
    * @param personSearchQueryInfo
    * @param isAttendedMode
    * @param shouldRetrieveFullPerson
    * @param enforceSecurity
    * @param maxResults
    * @return
    * @throws ServiceException
    */
   // Removed for Jean
 //  public List searchForAddAPerson(PersonSearchQueryInfo personSearchQueryInfo,  boolean isAttendedMode, boolean shouldRetrieveFullPerson, boolean enforceSecurity, int maxResults) throws ServiceException;

	/**
	 * ADD A PERSON: send traits information to MVI to add a person
	 *
	 * @param incoming  incoming Person to be added
	 * @return vpid
	 * @throws ServiceException
	 */
	//CCR 12789 - added for convenience so it can be handled in a standard way
   public VPIDEntityKey addPersonExplicit(Person incoming, IdMSearchInfo searchInfo) throws ServiceException;

   /**
    * Add Person for ADD A PERSON.
    *
    * @param traits  Person Identity Traits data
    * @return vpid
    * @throws ServiceException
    */
   public VPIDEntityKey addPerson(IdmServiceVO idmServiceVO) throws ServiceException;

   public PersonIdentityTraits getESRCorrelation(VPIDEntityKey vpid) throws ServiceException;

	/**
	 * Adds ESR correlation for person, updating the profile with the person identity trait information
	 * @param person
	 * @return VPIDEntityKey that is created
	 */
  public VPIDEntityKey addESRCorrelation(Person person) throws ServiceException;

	/**
	 * Check to see if ESR200 Correlation exists for the person and Adds ESR correlation if necessary
	 * @param person
	 * @return
	 */
  public void checkAndAddESRCorrelation(Person person) throws ServiceException;
  /**
   * Update the profile for ESR 200 Correlation with idm.
   * @param incomingPerson
   * @throws ServiceException
   */
  public void updateProfileForESRCorrelation(Person incomingPerson) throws ServiceException;

  /**
   * Update the profile for ESR 200 Correlation with idm
   *  @param incomingPerson
   * @throws ServiceException
   */
  //CCR13856
  public void updateProfileForDeathEvent(Person incomingPerson) throws ServiceException;

  /**
   * Update the profile for ESR 200 Correlation with idm
   *  @param incomingPerson
   * @throws ServiceException
   */
  //CCR 13856
  public void updateProfileForAAP(Person incomingPerson) throws ServiceException;

  /**
   * Update the profile for ESR 200 Correlation with idm.
   * @param idmServiceVO
   * @throws ServiceException
   */
  public void updateProfileForESRCorrelation(IdmServiceVO idmServiceVO) throws ServiceException;

	/**
	 * Add correlation for preferred facility for a new person
	 * @param person
	 * @param key
	 * @throws ServiceException
	 */
  public FullyQualifiedIdentity addPreferredFacilityCorrelation(Person person) throws ServiceException;


	/**
	 * Add correlation for preferred facility for a person who already has MVI correlations available
	 * @param person
	 * @param list of correlations from MVI
	 * @throws ServiceException
	 */
public FullyQualifiedIdentity addPreferredFacilityCorrelation(Person person, List<PatientIdentifier> correlations) throws ServiceException;

  /**
   * For Add a Person, add a person stub into the database with default veteran indicator = true
   * @param vpid
   * @return
   * @throws ServiceException
   */
  public PersonEntityKey addPersonStub(VPIDEntityKey vpid) throws ServiceException;

  /**
   * For Add a Person, add a person stub into the database with a veteran indicator
   * @param vpid
   * @return
   * @throws ServiceException
   */
  public PersonEntityKey addPersonStub(VPIDEntityKey vpid, Boolean isVeteran, Character voaInd) throws ServiceException;

  /**
   * Find the parent sending site among the preferred facility or its parent/ancestors
   * @param preferredFacility
   * @return
   */
  public VAFacility getParentSite(VAFacility preferredFacility) throws ServiceException;

  /**
   * Get the person with person traits of 200ESR Correlation view
   * @param key
   * @return Person
   */
  public Person get200ESRPerson(PersonEntityKey key) throws ServiceException;


  /**
   * given vpid, get the person with composite call
   * @param key
   * @return
   * @throws ServiceException
   */
  public Person getPersonWithCompositeCall(VPIDEntityKey key) throws ServiceException;

  /**
   * given a person, converts it to a IdmService VO object
   * @param source
   * @return
   * @throws ServiceException
   */
  //CR 11776 - Replacing EJB calls
  //This method is added to this interface as used by various classes
 /* public IdmServiceVO convertPersonToIdmServiceVO(Person source)
	    	throws ServiceException;*/

  //CR 11776 - Replacing EJB calls
  //This method is added to this interface as used by various classes
  /*public Person convertTraitsToPerson(PersonIdentityTraits source)
	    	throws ServiceException;*/

  /**
   * Persist the VOA Indicator
   * @param  VOA Indicator
   * @throws ServiceException Thrown if failed to persist a person
   */
  public void updateVOAIndicator (Person person) throws ServiceException;

	/**
	 * Add ESR Correlation in PSIM
	 *
	 * @param onFile Person on file contains ESR Correlation flag indicator to determine if ESR Correlation Add is needed
	 * @param incomeing Person contains the address populated so that Add ESR Correlation will send address to PSIM
	 *
	 * @return onFile Person
	 */
  public Person addESRCorrelation(Person onFile, Person incoming) throws ServiceException;

}