/********************************************************************
 * Copyright  2004 VHA. All rights reserved
 ********************************************************************/
package gov.va.med.esr.service;

// Java Classes
import java.util.Set;
import java.util.List;
import java.math.BigDecimal;

// Framework Classes
import gov.va.med.fw.model.EntityKey;
import gov.va.med.fw.service.ServiceException;
import gov.va.med.fw.service.pagination.PaginatedSearchService;
import gov.va.med.ps.model.PatientIdentifier;

// ESR Classes
import gov.va.med.esr.common.model.ee.IncomingMessageInfo;
import gov.va.med.esr.common.model.ee.VerificationInfo;
import gov.va.med.esr.common.model.lookup.VAFacility;
import gov.va.med.esr.common.model.messaging.MessageLogEntry;
import gov.va.med.esr.common.model.messaging.MessageLogEntryLite;
import gov.va.med.esr.common.model.messaging.SiteIdentity;
import gov.va.med.esr.common.model.person.Person;
import gov.va.med.esr.common.model.person.id.PersonEntityKey;
import gov.va.med.esr.service.impl.AddProxyPreferredFacilityException;

/**
 * @author Carlos Ruiz
 * @version 1.0
 */
public interface MessagingService extends PaginatedSearchService {

	public static final String VOA_SUBMISSION = "VOA_SUBMISSION";
	public static final String MSDS_QUERY_RESPONSE = "MSDS_QUERY_RESPONSE";
	public static final String VBA_QUERY_RESPONSE = "VBA_QUERY_RESPONSE";
    /**
     * Method to generate a control identifier that is used to uniquely
     * identify a message.
     *
     * @return The control identifier for the message.
     * @throws ServiceException
     */
    public String generateControlIdentifier() throws ServiceException;

    /**
    * @param person
    * @param vaFacility
    * @return
    * @throws ServiceException
    */
   public SiteIdentity getIdentity(Person person, VAFacility vaFacility) throws ServiceException;
   /**
    * Get Matching Person Identity Traits from PSIM
    * @param identityTraits
    * @return
    * @throws ServiceException
    */
   public Set getMatchingTraits(PersonIdentityTraits identityTraits) throws ServiceException;
   /**
    * @param identifier
    * @return
    * @throws ServiceException
    */
   public MessageLogEntry getMessageLogEntryById(BigDecimal identifier) throws ServiceException;


   /**
    * @param identifier
    * @param shouldPopulatePerson
    * @return
    * @throws ServiceException
    */
   public MessageLogEntryLite getMessageLogEntryLiteById(BigDecimal identifier) throws ServiceException;


    /**
    * @param controlIdentifier
    * @return
    * @throws ServiceException
    */
   public MessageLogEntry getMessageLogEntry(String controlIdentifier) throws ServiceException;


   /**
    * @param controlIdentifier
    * @return
    * @throws ServiceException
    */
   public MessageLogEntry getMessageLogEntryByBatchControlNumber(String controlIdentifier) throws ServiceException;

    /**
     * Gets a list of SiteIdentity records for a Person.
     * @param person The person
     * @return The sites for the person or an empty set if no sites were found
     * @throws ServiceException If any problems were encountered
     */
    public Set findSitesOfRecord(Person person) throws ServiceException;

    /**
    * @param entry
    * @throws ServiceException
    */
   public void logMessage(MessageLogEntry entry) throws ServiceException;


    /**
     * Process the person created from GUI.
     *
     * @param person
     * @param sendingFacility
     * @param verificationInfo
     * @return
     * @throws ServiceException
     */
    public Person processAddPerson(Person person, VAFacility sendingFacility, VerificationInfo verificationInfo) throws ServiceException;
    /**
     * Process the person created from ORUZ07 message.
     *
     * @param person
     * @param sendingFacility
     * @param verificationInfo
     * @return
     * @throws ServiceException
     */
    public Person processZ07(Person person, VAFacility sendingFacility, VerificationInfo verificationInfo) throws ServiceException;

    /**
     * Process the person created from a ORUZ05 message.
     *
     * @param person
     * @return person
     * @throws ServiceException
     */
    public Person processORUZ05(Person person) throws ServiceException;

    /**
     * Process the person created from a ORUZ04 message.
     *
     * @param person
     * @return person
     * @throws ServiceException
     */
    public Person processORUZ04(Person person) throws ServiceException;

    /**
     * Process Z12 message.
     *
     * @param person
     * @param sendingFacility
     * @return
     * @throws ServiceException
     */
    public Person processORUZ12(Person person, VAFacility sendingFacility) throws ServiceException;

    /**
     * Process Z13 message.
     *
     * @param person
     * @param sendingFacility
     * @return
     * @throws ServiceException
     */
    public Person processORUZ13(Person person, VAFacility sendingFacility) throws ServiceException;

    /**
     * Process the person created from an inbound non-query Z11 message (eg, ORUZ11, ORFZ11)
     *
     * @param person
     * @return person
     * @throws ServiceException
     */
    public Person processNonQueryZ11(Person person, VerificationInfo verificationInfo) throws ServiceException;

    /**
     * Process QRYZ11 message.
     *
     * @param person
     * @param incomingMessageInfo
     * @throws ServiceException
     */
    public void processQRYZ11(Person person,  IncomingMessageInfo incomingMessageInfo) throws ServiceException;


     /**
     * Process QRYZ10 messsage
     *
     * @param person
     * @param incomeYear
     * @param incomingMessageInfo
     * @throws ServiceException
     */
    public void processQRYZ10(Person person, Integer incomeYear, IncomingMessageInfo incomingMessageInfo) throws ServiceException;

    /**
     * Trigger a QRYZ07 message instantly.
     *
     * @param person The person whose data is being queries
     * @param facility The facility to send the message to
     * @param incomeYear The income year to query on
     * @throws ServiceException
     */
    public void triggerQRYZ07(Person person, VAFacility facility, Integer incomeYear) throws ServiceException;

    /**
     * Trigger a QRYZ07 message
     *
     * @param person The person whose data is being queries
     * @param facility The facility to send the message to
     * @param incomeYear The income year to query on
     * @param handleAfterCommit Determines if the event should be handled after the current TX commits
     * @throws ServiceException
     */
    public void triggerQRYZ07(Person person, VAFacility facility, Integer incomeYear, boolean handleAfterCommit) throws ServiceException;

	/**
	 * Find the Facility from whih the last message has been received and successfully processed
	 *
	 * @param personId
	 * @return
	 * @throws ServiceException
	 */
	public VAFacility findSiteLastTransmittedMsg(EntityKey personId) throws ServiceException;


    /**
     * Retransmits a message.
     *
     * @param identifier message log entry id.
     * @throws ServiceException
     */
    public void triggerRetransmission(BigDecimal identifier) throws ServiceException;

	/** Specific check to determine if already processed an inbound message */
	public boolean hasProcessedInboundMessage(String messageControlNumber, String stationNumber) throws ServiceException;

    /**
     * find message log entries for VOA that has AA Ack sent, given a person id, in decending datetime order (latest first)
     *
     * @param identifier person id.
     * @return list of log entries
     * @throws ServiceException
     */
	public List findVoaAaAckLogEntryByPersonId(EntityKey personId) throws ServiceException;

	//MSDS Seeding Changes

	/**
     * Trigger a QRYZ21 message
     *
     * @param key The person key whose data is being queries
     * @param key The vpid of the person being queried
     * @throws ServiceException
     */
    public void triggerQRYZ21(PersonEntityKey key, String VPID) throws ServiceException;

    /**
     * Process K21 message.
     *
     * @param person
     * @param sendingFacility
     * @return
     * @throws ServiceException
     */
    public Person processK21(Person person, VAFacility sendingFacility, VerificationInfo verificationInfo) throws ServiceException;

    /**
     * Trigger a Broker message instantly.
     *
     * @param person The person whose data is being queries
     * @throws ServiceException
     */
    public void processTriggerBroker(Person person) throws ServiceException;


	/**
	 *
	 * @param incoming
	 * @param sendingFacility
	 * @param verificationInfo
	 * @param isAnonymousLOA1
	 * @param correlations List of correlations associated with this person
	 * @return
	 * @throws ServiceException
	 * @throws AddProxyPreferredFacilityException
	 */
    public Person processVOA(Person incoming, VAFacility sendingFacility, VerificationInfo verificationInfo, boolean isAnonymousLOA1) throws ServiceException;

    /**
     * This method does CD conversion and handles triggering Z11.
     *
     *
     * @param person
     * @return true if converted to descriptors otherwise false
     * @throws ServiceException
     */
    public boolean processCDConversion(Person person) throws ServiceException;


    /**
     * Only convert the CD to use descriptors. Can be used for "on the fly" conversion.
     *
     * @param person
     * @return
     * @throws ServiceException
     */
    public boolean convertCDOnly(Person person) throws ServiceException;


}
