/********************************************************************
 * Copyright  2004-2005 VHA. All rights reserved
 ********************************************************************/
package gov.va.med.esr.service;

// Java classes
import java.math.BigDecimal;
import java.util.List;
import java.util.Map;

import gov.va.med.esr.common.model.financials.IncomeThreshold;
import gov.va.med.esr.common.model.lookup.*;
import gov.va.med.esr.common.model.party.Address;
import gov.va.med.fw.model.lookup.AbstractCode;
import gov.va.med.fw.model.lookup.AbstractLookup;
import gov.va.med.fw.report.ReportExportedType;
import gov.va.med.fw.service.ServiceException;

/**
 * Class that provides services related to a person's enrollment. This service
 * should be accessed
 *
 * @author Martin Francisco
 * @version 1.0
 */
public interface LookupService {

    /**
     * Returns all the predefined lookup objects of a particular type.
     *
     * @param lookupType The type of lookup to search for.
     * @return All the predefined lookup objects of a particular type.
     * @throws UnknownLookupTypeException if the <code>lookupType</code> is unknown.
     */
    public List findAll(Class lookupType) throws UnknownLookupTypeException;

    /**
     * Returns the predefined lookup object with the specified code.
     *
     * @param lookupType The type of lookup to search for.
     * @param code The unique code to search.
     * @return The predefined lookup object.
     * @throws UnknownLookupTypeException if the <code>lookupType</code> is unknown.
     * @throws UnknownLookupCodeException if the <code>code</code> is unknown for the type.
     */
    public AbstractLookup getByCode(Class lookupType, String code)
        throws UnknownLookupTypeException, UnknownLookupCodeException;
    /**
     * Returns the lookup object by matching the description with like sql clause
     * @param lookupType
     * @param description
     * @return
     * @throws UnknownLookupTypeException
     * @throws UnknownLookupCodeException
     */
    public AbstractLookup getByDescription(Class lookupType, String description)
    	throws UnknownLookupTypeException, UnknownLookupCodeException;
    /**
     * Returns the lookup object by matching the name with like sql clause
     * @param lookupType
     * @param name
     * @return
     * @throws UnknownLookupTypeException
     * @throws UnknownLookupCodeException
     */
    public AbstractLookup getByName(Class lookupType, String name)
	throws UnknownLookupTypeException, UnknownLookupCodeException;

    /**
     * Returns the predefined lookup object with the specified code.
     *
     * @param lookupType The type of lookup to search for.
     * @param code The unique code to search.
     * @return The predefined lookup object.
     * @throws UnknownLookupTypeException if the <code>lookupType</code> is unknown.
     * @throws UnknownLookupCodeException if the <code>code</code> is unknown for the type.
     */
    public AbstractLookup getByCode(Class lookupType, AbstractCode code)
        throws UnknownLookupTypeException, UnknownLookupCodeException;


    /**
     * Find all AacIndicator
     *
     * @return List a list of all AacIndicator
     */
    public  List getAllAACIndicators()
            throws UnknownLookupTypeException, UnknownLookupCodeException;

    /**
     * Get AacIndicator by code
     *
     * @return AacIndicator A corresponding AacIndicator by code
     */
    public  AACIndicator getAACIndicatorByCode(String code)
            throws UnknownLookupTypeException, UnknownLookupCodeException;

    /**
     * Find all AddressChangeSource
     *
     * @return List a list of all AddressChangeSource
     */
    public  List getAllAddressChangeSources()
            throws UnknownLookupTypeException, UnknownLookupCodeException;

    /**
     * Get AddressChangeSource by code
     *
     * @return AddressChangeSource A corresponding AddressChangeSource by code
     */
    public  AddressChangeSource getAddressChangeSourceByCode(String code)
            throws UnknownLookupTypeException, UnknownLookupCodeException;

    /**
     * Find all PhoneSourceOfChange
     *
     * @return List a list of all PhoneSourceOfChange
     */
    public  List getAllPhoneSourcesOfChange()
            throws UnknownLookupTypeException, UnknownLookupCodeException;

    /**
     * Get PhoneSourceOfChange by code
     *
     * @return PhoneSourceOfChange A corresponding PhoneSourceOfChange by code
     */
    public  PhoneSourceOfChange getPhoneSourceOfChangeByCode(String code)
            throws UnknownLookupTypeException, UnknownLookupCodeException;


    /**
     * Find all EmailSourceOfChange
     *
     * @return List a list of all EmailSourceOfChange
     */
    public  List getAllEmailSourcesOfChange()
            throws UnknownLookupTypeException, UnknownLookupCodeException;

    /**
     * Get EmailSourceOfChange by code
     *
     * @return EmailSourceOfChange A corresponding EmailSourceOfChange by code
     */
    public  EmailSourceOfChange getEmailSourceOfChangeByCode(String code)
            throws UnknownLookupTypeException, UnknownLookupCodeException;

    /**
     * Find all AckTypes
     *
     * @return List a list of all AckTypes
     */
    public  List getAllAckTypes()
            throws UnknownLookupTypeException, UnknownLookupCodeException;

    /**
     * Get AckType by code
     *
     * @return AckType A corresponding AckType by code
     */
    public  AckType getAckTypeByCode(AckType.Code code)
            throws UnknownLookupTypeException, UnknownLookupCodeException;

    /**
     * Find all AddressType
     *
     * @return List a list of all AddressType
     */
    public  List getAllAddressTypes()
            throws UnknownLookupTypeException, UnknownLookupCodeException;

    /**
     * Get AddressType by code
     *
     * @return AddressType A corresponding AddressType by code
     */
    public  AddressType getAddressTypeByCode(String code)
            throws UnknownLookupTypeException, UnknownLookupCodeException;

    /**
     * Find all AffectedExtremity
     *
     * @return List a list of all AffectedExtremity
     */
    public  List getAllAffectedExtremitys()
            throws UnknownLookupTypeException, UnknownLookupCodeException;

    /**
     * Get AffectedExtremity by code
     *
     * @return AffectedExtremity A corresponding AffectedExtremity by code
     */
    public  AffectedExtremity getAffectedExtremityByCode(String code)
            throws UnknownLookupTypeException, UnknownLookupCodeException;

    /**
     * Find all AlliedCountry
     *
     * @return List a list of all AlliedCountry
     */
    public  List getAllAlliedCountries()
            throws UnknownLookupTypeException, UnknownLookupCodeException;

    /**
     * Get AlliedCountry by code
     *
     * @return AlliedCountry A corresponding AlliedCountry by code
     */
    public  AlliedCountry getAlliedCountryByCode(String code)
            throws UnknownLookupTypeException, UnknownLookupCodeException;

    /**
     * Find all FederalAgency
     *
     * @return List a list of all FederalAgency
     */
    public  List getAllFederalAgencies()
            throws UnknownLookupTypeException, UnknownLookupCodeException;

    /**
     * Get FederalAgency by code
     *
     * @return FederalAgency A corresponding FederalAgency by code
     */
    public  FederalAgency getFederalAgencyByCode(String code)
            throws UnknownLookupTypeException, UnknownLookupCodeException;

    /**
     * Find all FeeBasisProgram
     *
     * @return List a list of all FeeBasisProgram
     */
    public  List getAllFeeBasisPrograms()
            throws UnknownLookupTypeException, UnknownLookupCodeException;

    /**
     * Get FeeBasisProgram by code
     *
     * @return FeeBasisProgram A corresponding FeeBasisProgram by code
     */
    public  FeeBasisProgram getFeeBasisProgramByCode(
            String code) throws UnknownLookupTypeException,
            UnknownLookupCodeException;

    /**
     * Find all FeeBasisTreatmentType
     *
     * @return List a list of all FeeBasisTreatmentType
     */
    public  List getAllFeeBasisTreatmentTypes()
            throws UnknownLookupTypeException, UnknownLookupCodeException;

    /**
     * Get FeeBasisTreatmentType by code
     *
     * @return FeeBasisTreatmentType A corresponding FeeBasisTreatmentType by code
     */
    public  FeeBasisTreatmentType getFeeBasisTreatmentTypeByCode(
            String code) throws UnknownLookupTypeException,
            UnknownLookupCodeException;

    /**
     * Find all AgentOrangeExposureLocation
     *
     * @return List a list of all AgentOrangeExposureLocation
     */
    public  List getAllAgentOrangeExposureLocations()
            throws UnknownLookupTypeException, UnknownLookupCodeException;

    /**
     * Get AgentOrangeExposureLocation by code
     *
     * @return AgentOrangeExposureLocation A corresponding
     *         AgentOrangeExposureLocation by code
     */
    public  AgentOrangeExposureLocation getAgentOrangeExposureLocationByCode(
            String code) throws UnknownLookupTypeException,
            UnknownLookupCodeException;

    /**
     * Find all AssetType
     *
     * @return List a list of all AssetType
     */
    public  List getAllAssetTypes() throws UnknownLookupTypeException,
            UnknownLookupCodeException;

    /**
     * Get AssetType by code
     *
     * @return AssetType A corresponding AssetType by code
     */
    public  AssetType getAssetTypeByCode(String code)
            throws UnknownLookupTypeException, UnknownLookupCodeException;

    /**
     * Find all AssociationType
     *
     * @return List a list of all AssociationType
     */
    public  List getAllAssociationTypes()
            throws UnknownLookupTypeException, UnknownLookupCodeException;

    /**
     * Get AssociationType by code
     *
     * @return AssociationType A corresponding AssociationType by code
     */
    public  AssociationType getAssociationTypeByCode(String code)
            throws UnknownLookupTypeException, UnknownLookupCodeException;

    /**
     * Find all CdDeterminationMethod
     *
     * @return List a list of all CdDeterminationMethod
     */
    public  List getAllCDDeterminationMethods()
            throws UnknownLookupTypeException, UnknownLookupCodeException;

    /**
     * Get CdDeterminationMethod by code
     *
     * @return CdDeterminationMethod A corresponding CdDeterminationMethod by
     *         code
     */
    public  CDDeterminationMethod getCdDeterminationMethodByCode(
            String code) throws UnknownLookupTypeException,
            UnknownLookupCodeException;

    /**
     * Find all CancelDeclineNotProcessedReason
     *
     * @return List a list of all CancelDeclineNotProcessedReason
     */
    public  List getAllCancelDeclineNotProcessedReasons()
            throws UnknownLookupTypeException, UnknownLookupCodeException;

    /**
     * Get CancelDeclineNotProcessedReason by code
     *
     * @return CancelDeclineNotProcessedReason A corresponding
     *         CancelDeclineNotProcessedReason by code
     */
    public  CancelDeclineNotProcessedReason getCancelDeclineNotProcessedReasonByCode(
            String code) throws UnknownLookupTypeException,
            UnknownLookupCodeException;

    /**
     * Find all CancelDeclineReason
     *
     * @return List a list of all CancelDeclineReason
     */
    public  List getAllCancelDeclineReasons()
            throws UnknownLookupTypeException, UnknownLookupCodeException;

    /**
     * Get CancelDeclineReason by code
     *
     * @return CancelDeclineReason A corresponding CancelDeclineReason by code
     */
    public  CancelDeclineReason getCancelDeclineReasonByCode(String code)
            throws UnknownLookupTypeException, UnknownLookupCodeException;

    /**
     * Find all ComLetterTemplateType
     *
     * @return List a list of all ComLetterTemplateType
     */
    public  List getAllComLetterTemplateTypes() throws UnknownLookupTypeException,
            UnknownLookupCodeException;


    /**
     * Get ComLetterTemplateType by code
     *
     * @return ComLetterTemplateType A corresponding ComLetterTemplateType by code
     */
    public  ComLetterTemplateType getComLetterTemplateTypeByCode(String code)
            throws UnknownLookupTypeException, UnknownLookupCodeException;

    /**
     * Find all Condition
     *
     * @return List a list of all Condition
     */
    public  List getAllConditions() throws UnknownLookupTypeException,
            UnknownLookupCodeException;

    /**
     * Get Condition by code
     *
     * @return Condition A corresponding Condition by code
     */
    public  Condition getConditionByCode(String code)
            throws UnknownLookupTypeException, UnknownLookupCodeException;

    /**
     * Find all Conflict
     *
     * @return List a list of all Conflict
     */
    public  List getAllConflicts() throws UnknownLookupTypeException,
            UnknownLookupCodeException;

    /**
     * Get Conflict by code
     *
     * @return Conflict A corresponding Conflict by code
     */
    public  ConflictLocation getConflictByCode(String code)
            throws UnknownLookupTypeException, UnknownLookupCodeException;

    /**
     * Find all Country
     *
     * @return List a list of all Country
     */
    public  List getAllCountrys() throws UnknownLookupTypeException,
            UnknownLookupCodeException;

    /**
     * Get Country by code
     *
     * @return Country A corresponding Country by code
     */
    public  Country getCountryByCode(String code)
            throws UnknownLookupTypeException, UnknownLookupCodeException;

    /**
     * Get MsdsCountry by code
     *
     * @return Country A corresponding Country by code
     */
    public  MsdsCountry getMsdsCountryByCode(String code)
            throws UnknownLookupTypeException, UnknownLookupCodeException;


    /**
     * Find all County
     *
     * @return List a list of all County
     */
    public  List getAllCountys() throws UnknownLookupTypeException,
            UnknownLookupCodeException;

    /**
     * Get County by code
     *
     * @return County A corresponding County by code
     */
    public  County getCountyByCode(String code)
            throws UnknownLookupTypeException, UnknownLookupCodeException;

    /**
     * Get County by County name and State Code
     * e.g County = "Montgomery", State Postal Code = "OH"
     *
     * Returns null if there is no match.
     *
     * @return County A corresponding County by County name and state code
     */
    public  County getCountyByCountyNameAndStateCode(String countyName, String stateCode)
            throws UnknownLookupTypeException, UnknownLookupCodeException;

    public County getCountyByAddress(Address address) throws UnknownLookupTypeException, UnknownLookupCodeException;

    /**
     * Find all DataSource
     *
     * @return List a list of all DataSource
     */
    public  List getAllDataSources() throws UnknownLookupTypeException,
            UnknownLookupCodeException;

    /**
     * Get DataSource by code
     *
     * @return DataSource A corresponding DataSource by code
     */
    public  DataSource getDataSourceByCode(String code)
            throws UnknownLookupTypeException, UnknownLookupCodeException;

    /**
     * Find all DeathDataSource
     *
     * @return List a list of all DeathDataSource
     */
    public  List getAllDeathDataSources()
            throws UnknownLookupTypeException, UnknownLookupCodeException;

    /**
     * Get DeathDataSource by code
     *
     * @return DeathDataSource A corresponding DeathDataSource by code
     */
    public  DeathDataSource getDeathDataSourceByCode(String code)
            throws UnknownLookupTypeException, UnknownLookupCodeException;

    /**
     * Find all DecorationStatus
     *
     * @return List a list of all DecorationStatus
     */
    public  List getAllDecorationStatuss()
            throws UnknownLookupTypeException, UnknownLookupCodeException;

    /**
     * Get DecorationStatus by code
     *
     * @return DecorationStatus A corresponding DecorationStatus by code
     */
    public  DecorationStatus getDecorationStatusByCode(String code)
            throws UnknownLookupTypeException, UnknownLookupCodeException;

    /**
     * Find all Diagnosis
     *
     * @return List a list of all Diagnosis
     */
    public  List getAllDiagnosiss() throws UnknownLookupTypeException,
            UnknownLookupCodeException;

    /**
     * Get Diagnosis by code
     *
     * @return Diagnosis A corresponding Diagnosis by code
     */
    public  Diagnosis getDiagnosisByCode(String code)
            throws UnknownLookupTypeException, UnknownLookupCodeException;

    /**
     * Find all Disability
     *
     * @return List a list of all Disability
     */
    public  List getAllDisabilitys() throws UnknownLookupTypeException,
            UnknownLookupCodeException;

    /**
     * Get Disability by code
     *
     * @return Disability A corresponding Disability by code
     */
    public  Disability getDisabilityByCode(String code)
            throws UnknownLookupTypeException, UnknownLookupCodeException;

    /**
     * Find all DischargeType
     *
     * @return List a list of all DischargeType
     */
    public  List getAllDischargeTypes()
            throws UnknownLookupTypeException, UnknownLookupCodeException;

    /**
     * Get DischargeType by code
     *
     * @return DischargeType A corresponding DischargeType by code
     */
    public  DischargeType getDischargeTypeByCode(String code)
            throws UnknownLookupTypeException, UnknownLookupCodeException;

    /**
     * Find all EgtSettingType
     *
     * @return List a list of all EgtSettingType
     */
    public  List getAllEgtSettingTypes()
            throws UnknownLookupTypeException, UnknownLookupCodeException;

    /**
     * Get EgtSettingType by code
     *
     * @return EgtSettingType A corresponding EgtSettingType by code
     */
    public  EGTSettingType getEgtSettingTypeByCode(String code)
            throws UnknownLookupTypeException, UnknownLookupCodeException;

    /**
     * Find all EmailType
     *
     * @return List a list of all EmailType
     */
    public  List getAllEmailTypes()
            throws UnknownLookupTypeException, UnknownLookupCodeException;

    /**
     * Get EmailType by code
     *
     * @return EmailType A corresponding EmailType by
     *         code
     */
    public  EmailType getEmailTypeByCode(
            String code) throws UnknownLookupTypeException,
            UnknownLookupCodeException;

    /**
     * Find all EligibilityStatus
     *
     * @return List a list of all EligibilityStatus
     */
    public  List getAllEligibilityStatuss()
            throws UnknownLookupTypeException, UnknownLookupCodeException;

    /**
     * Get EligibilityStatus by code
     *
     * @return EligibilityStatus A corresponding EligibilityStatus by code
     */
    public  EligibilityStatus getEligibilityStatusByCode(String code)
            throws UnknownLookupTypeException, UnknownLookupCodeException;
    
    /**
     * Find all EligibilityStatusPendVerfiReason 
     *
     * @return List a list of all EligibilityStatusPendVerfiReason 
     */
    public  List getAllEligibilityStatusReasons()
            throws UnknownLookupTypeException, UnknownLookupCodeException;

    /**
     * Get EligibilityStatusPendVerfiReason  by code
     *
     * @return EligibilityStatusPendVerfiReason  A corresponding EligibilityStatusPendVerfiReason  by code
     */
    public  EligibilityStatusPendVerfiReason  getEligibilityStatusReasonByCode(String code)
            throws UnknownLookupTypeException, UnknownLookupCodeException;

    /**
     * Find all EligibilityType
     *
     * @return List a list of all EligibilityType
     */
    public  List getAllEligibilityTypes()
            throws UnknownLookupTypeException, UnknownLookupCodeException;

    /**
     * Get EligibilityType by code
     *
     * @return EligibilityType A corresponding EligibilityType by code
     */
    public  EligibilityType getEligibilityTypeByCode(String code)
            throws UnknownLookupTypeException, UnknownLookupCodeException;

    /**
     * Get EligibilityFactor by code
     *
     * @return EligibilityFactor A corresponding EligibilityFactor by code
     */
    public  EligibilityFactor getEligibilityFactorByCode(String code)
            throws UnknownLookupTypeException, UnknownLookupCodeException;

    /**
     * Find all EligibilityVerificationSource
     *
     * @return List a list of all EligibilityVerificationSource
     */
    public  List getAllEligibilityVerificationSources()
            throws UnknownLookupTypeException, UnknownLookupCodeException;

    /**
     * Get EligibilityVerificationSource by code
     *
     * @return EligibilityVerificationSource A corresponding
     *         EligibilityVerificationSource by code
     */
    public  EligibilityVerificationSource getEligibilityVerificationSourceByCode(
            String code) throws UnknownLookupTypeException,
            UnknownLookupCodeException;

    /**
     * Find all EmploymentStatus
     *
     * @return List a list of all EmploymentStatus
     */
    public  List getAllEmploymentStatuss()
            throws UnknownLookupTypeException, UnknownLookupCodeException;

    /**
     * Get EmploymentStatus by code
     *
     * @return EmploymentStatus A corresponding EmploymentStatus by code
     */
    public  EmploymentStatus getEmploymentStatusByCode(String code)
            throws UnknownLookupTypeException, UnknownLookupCodeException;

    /**
     * Find all EnrollmentCategory
     *
     * @return List a list of all EnrollmentCategory
     */
    public  List getAllEnrollmentCategorys()
            throws UnknownLookupTypeException, UnknownLookupCodeException;

    /**
     * Get EnrollmentCategory by code
     *
     * @return EnrollmentCategory A corresponding EnrollmentCategory by code
     */
    public  EnrollmentCategory getEnrollmentCategoryByCode(String code)
            throws UnknownLookupTypeException, UnknownLookupCodeException;

    /**
     * Find all EnrollmentOverrideReason
     *
     * @return List a list of all EnrollmentOverrideReason
     */
    public  List getAllEnrollmentOverrideReasons()
            throws UnknownLookupTypeException, UnknownLookupCodeException;

    /**
     * Get EnrollmentOverrideReason by code
     *
     * @return EnrollmentOverrideReason A corresponding EnrollmentOverrideReason by code
     */
    public  EnrollmentOverrideReason getEnrollmentOverrideReasonByCode(String code)
            throws UnknownLookupTypeException, UnknownLookupCodeException;


    /**
     * Find all EnrollmentPriorityGroup
     *
     * @return List a list of all EnrollmentPriorityGroup
     */
    public  List getAllEnrollmentPriorityGroups()
            throws UnknownLookupTypeException, UnknownLookupCodeException;

    /**
     * Get EnrollmentPriorityGroup by code
     *
     * @return EnrollmentPriorityGroup A corresponding EnrollmentPriorityGroup by
     *         code
     */
    public  EnrollmentPriorityGroup getEnrollmentPriorityGroupByCode(
            String code) throws UnknownLookupTypeException,
            UnknownLookupCodeException;

    /**
     * Find all EnrollmentPrioritySubGroup
     *
     * @return List a list of all EnrollmentPrioritySubGroup
     */
    public  List getAllEnrollmentPrioritySubGroups()
            throws UnknownLookupTypeException, UnknownLookupCodeException;

    /**
     * Get EnrollmentPrioritySubGroup by code
     *
     * @return EnrollmentPrioritySubGroup A corresponding
     *         EnrollmentPrioritySubGroup by code
     */
    public  EnrollmentPrioritySubGroup getEnrollmentPrioritySubGroupByCode(
            String code) throws UnknownLookupTypeException,
            UnknownLookupCodeException;

    /**
     * Find all EnrollmentSource
     *
     * @return List a list of all EnrollmentSource
     */
    public  List getAllEnrollmentSources()
            throws UnknownLookupTypeException, UnknownLookupCodeException;

    /**
     * Get EnrollmentSource by code
     *
     * @return EnrollmentSource A corresponding EnrollmentSource by code
     */
    public  EnrollmentSource getEnrollmentSourceByCode(String code)
            throws UnknownLookupTypeException, UnknownLookupCodeException;

    /**
     * Find all EnrollmentStatus
     *
     * @return List a list of all EnrollmentStatus
     */
    public  List getAllEnrollmentStatuss()
            throws UnknownLookupTypeException, UnknownLookupCodeException;

    /**
     * Get EnrollmentStatus by code
     *
     * @return EnrollmentStatus A corresponding EnrollmentStatus by code
     */
    public  EnrollmentStatus getEnrollmentStatusByCode(String code)
            throws UnknownLookupTypeException, UnknownLookupCodeException;

    /**
     * Find all ExpenseType
     *
     * @return List a list of all ExpenseType
     */
    public  List getAllExpenseTypes()
            throws UnknownLookupTypeException, UnknownLookupCodeException;

    /**
     * Get ExpenseType by code
     *
     * @return ExpenseType A corresponding ExpenseType by code
     */
    public  ExpenseType getExpenseTypeByCode(String code)
            throws UnknownLookupTypeException, UnknownLookupCodeException;

    /**
     * Find all FilipinoVeteranProof
     *
     * @return List a list of all FilipinoVeteranProof
     */
    public  List getAllFilipinoVeteranProofs()
            throws UnknownLookupTypeException, UnknownLookupCodeException;

    /**
     * Get FilipinoVeteranProof by code
     *
     * @return FilipinoVeteranProof A corresponding FilipinoVeteranProof by code
     */
    public  FilipinoVeteranProof getFilipinoVeteranProofByCode(
            String code) throws UnknownLookupTypeException,
            UnknownLookupCodeException;

    /**
     * Find all Gender
     *
     * @return List a list of all Gender
     */
    public  List getAllGenders() throws UnknownLookupTypeException,
            UnknownLookupCodeException;

    /**
     * Get Gender by code
     *
     * @return Gender A corresponding Gender by code
     */
    public  Gender getGenderByCode(String code)
            throws UnknownLookupTypeException, UnknownLookupCodeException;

    /**
     * Find all IncomeTestSource
     *
     * @return List a list of all IncomeTestSource
     */
    public  List getAllIncomeTestSources()
            throws UnknownLookupTypeException, UnknownLookupCodeException;

    public  Indicator getIndicatorByCode(Indicator.Code code)
    throws UnknownLookupTypeException, UnknownLookupCodeException;

    public  CombatPayType getCombatPayTypeByCode(CombatPayType.Code code)
    throws UnknownLookupTypeException, UnknownLookupCodeException;


    /**
     * Get IncomeTestSource by code
     *
     * @return IncomeTestSource A corresponding IncomeTestSource by code
     */
    public  IncomeTestSource getIncomeTestSourceByCode(String code)
            throws UnknownLookupTypeException, UnknownLookupCodeException;

    /**
     * Find all IncomeTestType
     *
     * @return List a list of all IncomeTestType
     */
    public  List getAllIncomeTestTypes()
            throws UnknownLookupTypeException, UnknownLookupCodeException;

    /**
     * Get IncomeTestType by code
     *
     * @return IncomeTestType A corresponding IncomeTestType by code
     */
    public  IncomeTestType getIncomeTestTypeByCode(String code)
            throws UnknownLookupTypeException, UnknownLookupCodeException;

    /**
     * Find all IncomeType
     *
     * @return List a list of all IncomeType
     */
    public  List getAllIncomeTypes() throws UnknownLookupTypeException,
            UnknownLookupCodeException;

    /**
     * Get IncomeType by code
     *
     * @return IncomeType A corresponding IncomeType by code
     */
    public  IncomeType getIncomeTypeByCode(String code)
            throws UnknownLookupTypeException, UnknownLookupCodeException;

    /**
     * Find all InsurancePlanType
     *
     * @return List a list of all InsurancePlanType
     */
    public  List getAllInsurancePlanTypes()
            throws UnknownLookupTypeException, UnknownLookupCodeException;

    /**
     * Get InsurancePlanType by code
     *
     * @return InsurancePlanType A corresponding InsurancePlanType by code
     */
    public  InsurancePlanType getInsurancePlanTypeByCode(String code)
            throws UnknownLookupTypeException, UnknownLookupCodeException;

    /**
     * Find all InsuredRelationship
     *
     * @return List a list of all InsuredRelationship
     */
    public  List getAllInsuredRelationships()
            throws UnknownLookupTypeException, UnknownLookupCodeException;

    /**
     * Get InsuredRelationship by code
     *
     * @return InsuredRelationship A corresponding InsuredRelationship by code
     */
    public  InsuredRelationship getInsuredRelationshipByCode(String code)
            throws UnknownLookupTypeException, UnknownLookupCodeException;

    /**
     * Find all MaritalStatus
     *
     * @return List a list of all MaritalStatus
     */
    public  List getAllMaritalStatuss()
            throws UnknownLookupTypeException, UnknownLookupCodeException;

    /**
     * Get MaritalStatus by code
     *
     * @return MaritalStatus A corresponding MaritalStatus by code
     */
    public  MaritalStatus getMaritalStatusByCode(String code)
            throws UnknownLookupTypeException, UnknownLookupCodeException;

    /**
     * Find all MeansTestStatus
     *
     * @return List a list of all MeansTestStatus
     */
    public  List getAllMeansTestStatuss()
            throws UnknownLookupTypeException, UnknownLookupCodeException;

    /**
     * Get MeansTestStatus by code
     *
     * @return MeansTestStatus A corresponding MeansTestStatus by code
     */
    public  MeansTestStatus getMeansTestStatusByCode(String code)
            throws UnknownLookupTypeException, UnknownLookupCodeException;

    /**
     * Find all MessageStatus
     *
     * @return List a list of all MessageStatus
     */
    public  List getAllMessageStatuss()
            throws UnknownLookupTypeException, UnknownLookupCodeException;

    /**
     * Get MessageStatus by code
     *
     * @return MessageStatus A corresponding MessageStatus by code
     * @deprecated
     */
    public  MessageStatus getMessageStatusByCode(String code)
            throws UnknownLookupTypeException, UnknownLookupCodeException;

    /**
     * Get MessageStatus by code
     *
     * @return MessageStatus A corresponding MessageStatus by code
     */
    public  MessageStatus getMessageStatusByCode(MessageStatus.Code code)
            throws UnknownLookupTypeException, UnknownLookupCodeException;

    /**
     * Find all MessageType
     *
     * @return List a list of all MessageType
     */
    public  List getAllMessageTypes()
            throws UnknownLookupTypeException, UnknownLookupCodeException;

    /**
     * Get MessageType by code
     *
     * @return MessageType A corresponding MessageType by code
     */
    public  MessageType getMessageTypeByCode(String code)
            throws UnknownLookupTypeException, UnknownLookupCodeException;

    /**
     * Find all MilitarySexualTraumaStatus
     *
     * @return List a list of all MilitarySexualTraumaStatus
     */
    public  List getAllMilitarySexualTraumaStatuss()
            throws UnknownLookupTypeException, UnknownLookupCodeException;

    /**
     * Get MilitarySexualTraumaStatus by code
     *
     * @return MilitarySexualTraumaStatus A corresponding
     *         MilitarySexualTraumaStatus by code
     */
    public  MilitarySexualTraumaStatus getMilitarySexualTraumaStatusByCode(
            String code) throws UnknownLookupTypeException,
            UnknownLookupCodeException;

    /**
     * Find all MonetaryBenefitType
     *
     * @return List a list of all MonetaryBenefitType
     */
    public  List getAllMonetaryBenefitTypes()
            throws UnknownLookupTypeException, UnknownLookupCodeException;

    /**
     * Get MonetaryBenefitType by code
     *
     * @return MonetaryBenefitType A corresponding MonetaryBenefitType by code
     */
    public  MonetaryBenefitType getMonetaryBenefitTypeByCode(String code)
            throws UnknownLookupTypeException, UnknownLookupCodeException;

    /**
     * Find all NameType
     *
     * @return List a list of all NameType
     */
    public  List getAllNameTypes() throws UnknownLookupTypeException,
            UnknownLookupCodeException;

    /**
     * Get NameType by code
     *
     * @return NameType A corresponding NameType by code
     */
    public  NameType getNameTypeByCode(String code)
            throws UnknownLookupTypeException, UnknownLookupCodeException;

    /**
     * Find all MHDataSources
     *
     * @return List a list of all MHDataSources
     */
    public  List getAllMHDataSources() throws UnknownLookupTypeException,
            UnknownLookupCodeException;

    /**
     * Get MHDataSource by code
     *
     * @return MHDataSource A corresponding MHDataSource by code
     */
    public  MHDataSource getMHDataSourceByCode(String code)
            throws UnknownLookupTypeException, UnknownLookupCodeException;

    /**
     * Find all PhoneType
     *
     * @return List a list of all PhoneType
     */
    public  List getAllPhoneTypes() throws UnknownLookupTypeException,
            UnknownLookupCodeException;

    /**
     * Get PhoneType by code
     *
     * @return PhoneType A corresponding PhoneType by code
     */
    public  PhoneType getPhoneTypeByCode(String code)
            throws UnknownLookupTypeException, UnknownLookupCodeException;

    /**
     * Find all PrisonerOfWarLocation
     *
     * @return List a list of all PrisonerOfWarLocation
     */
    public  List getAllPrisonerOfWarLocations()
            throws UnknownLookupTypeException, UnknownLookupCodeException;

    /**
     * Get PrisonerOfWarLocation by code
     *
     * @return PrisonerOfWarLocation A corresponding PrisonerOfWarLocation by
     *         code
     */
    public  PrisonerOfWarLocation getPrisonerOfWarLocationByCode(
            String code) throws UnknownLookupTypeException,
            UnknownLookupCodeException;

    /**
     * Find all Procedure
     *
     * @return List a list of all Procedure
     */
    public  List getAllProcedures() throws UnknownLookupTypeException,
            UnknownLookupCodeException;

    /**
     * Get Procedure by code
     *
     * @return Procedure A corresponding Procedure by code
     */
    public  Procedure getProcedureByCode(String code)
            throws UnknownLookupTypeException, UnknownLookupCodeException;

    /**
     * Find all EGTStatus
     *
     * @return List a list of all EGTStatus
     */
    public  List getAllEGTStatuss()
            throws UnknownLookupTypeException, UnknownLookupCodeException;

    /**
     * Get EGTStatus by code
     *
     * @return EGTStatus A corresponding EGTStatus by code
     */
    public  EGTStatus getEGTStatusByCode(String code)
            throws UnknownLookupTypeException, UnknownLookupCodeException;

    /**
     * Find all RadiationExposureMethod
     *
     * @return List a list of all RadiationExposureMethod
     */
    public  List getAllRadiationExposureMethods()
            throws UnknownLookupTypeException, UnknownLookupCodeException;

    /**
     * Get RadiationExposureMethod by code
     *
     * @return RadiationExposureMethod A corresponding RadiationExposureMethod by
     *         code
     */
    public  RadiationExposureMethod getRadiationExposureMethodByCode(
            String code) throws UnknownLookupTypeException,
            UnknownLookupCodeException;

    /**
     * Find all RejectionRemark
     *
     * @return List a list of all RejectionRemark
     */
    public  List getAllRejectionRemarks()
            throws UnknownLookupTypeException, UnknownLookupCodeException;

    /**
     * Get RejectionRemark by code
     *
     * @return RejectionRemark A corresponding RejectionRemark by code
     */
    public  RejectionRemark getRejectionRemarkByCode(String code)
            throws UnknownLookupTypeException, UnknownLookupCodeException;

    /**
     * Find all Relationship
     *
     * @return List a list of all Relationship
     */
    public  List getAllRelationships()
            throws UnknownLookupTypeException, UnknownLookupCodeException;

     /**
     * Get Relationship by code
     *
     * @return Relationship A corresponding Relationship by code
     */
    public  Relationship getRelationshipByCode(String code)
            throws UnknownLookupTypeException, UnknownLookupCodeException;

    /**
     * Get RegistryType by code
     *
     * @return RegistryType
     */
    public  RegistryType getRegistryTypeByCode(String code)
            throws UnknownLookupTypeException, UnknownLookupCodeException;

    /**
     * Find all Religion
     *
     * @return List a list of all Religion
     */
    public  List getAllReligions() throws UnknownLookupTypeException,
            UnknownLookupCodeException;

    /**
     * Get Religion by code
     *
     * @return Religion A corresponding Religion by code
     */
    public  Religion getReligionByCode(String code)
            throws UnknownLookupTypeException, UnknownLookupCodeException;

    /**
     * Find all ServiceBranch
     *
     * @return List a list of all ServiceBranch
     */
    public  List getAllServiceBranchs()
            throws UnknownLookupTypeException, UnknownLookupCodeException;

    /**
     * Get ServiceBranch by code
     *
     * @return ServiceBranch A corresponding ServiceBranch by code
     */
    public  ServiceBranch getServiceBranchByCode(String code)
            throws UnknownLookupTypeException, UnknownLookupCodeException;

    /**
     * Find all ServicePeriod
     *
     * @return List a list of all ServicePeriod
     */
    public  List getAllServicePeriods()
            throws UnknownLookupTypeException, UnknownLookupCodeException;

    /**
     * Get ServicePeriod by code
     *
     * @return ServicePeriod A corresponding ServicePeriod by code
     */
    public  ServicePeriod getServicePeriodByCode(String code)
            throws UnknownLookupTypeException, UnknownLookupCodeException;

    /**
     * Find all State
     *
     * @return List a list of all State
     */
    public  List getAllStates() throws UnknownLookupTypeException,
            UnknownLookupCodeException;

    /**
     * Get State by code
     *
     * @return State A corresponding State by code
     */
    public  State getStateByCode(String code)
            throws UnknownLookupTypeException, UnknownLookupCodeException;

    /**
     * Get State by State name
     *
     * @return State A corresponding State by name
     */
    public  State getStateByName(String name)
            throws UnknownLookupTypeException, UnknownLookupCodeException;

    /**
     * Find all VaFacilitys
     *
     * @return List a list of all VaFacility
     */
    public  List getAllVaFacilitys()
            throws UnknownLookupTypeException, UnknownLookupCodeException;

    /**
     * Gets the list of all MFN-ZEG receipients.  Each recipient is a VAFacility whose MFN_ZEG_RECIPIENT
     * flag is set.
     * @return the list of MFN-ZEG receipient.
     * @throws ServiceException if any problems were encountered
     */
    public List getAllMfnzegRecipients() throws ServiceException;

    /**
     * Gets a list of all VAFacility objects whose type is a medical treating facility
     * @return the list of VAFAcility objects.
     * @throws ServiceException if any problems were encountered.
     */
    public List getAllDeathNotificationSites() throws ServiceException;

    /**
     * Get VaFacility by code
     *
     * @return VaFacility A corresponding VaFacility by code
     */
    public  VAFacility getVaFacilityByCode(String code)
            throws UnknownLookupTypeException, UnknownLookupCodeException;

    /**
     * Gets the VISNs vs Facilities map.
     *
     * @return The list of objects.
     * @throws gov.va.med.esr.service.UnknownLookupTypeException
     * @throws ClassNotFoundException
     */
    public Map getVISNSiteMap() throws UnknownLookupTypeException, ClassNotFoundException;

    /**
     * Get VaFacility by station number. This is just a helper method, It gets the VAFacility
     * by code as code and station number are same.
     *
     * @return VaFacility A corresponding VaFacility by station number
     */
    public  VAFacility getVaFacilityByStationNumber(String stationNum)
            throws UnknownLookupTypeException, UnknownLookupCodeException;


    /**
     * Get VaFacility by id
     *
     * @return VaFacility A corresponding VaFacility by id
     */
    public  VAFacility getVAFacilityById(BigDecimal identifier)
            throws UnknownLookupTypeException, UnknownLookupCodeException;

    /**
     * Find all VaFacilityType
     *
     * @return List a list of all VaFacilityType
     */
    public  List getAllVaFacilityTypes()
            throws UnknownLookupTypeException, UnknownLookupCodeException;

    /**
     * Get VaFacilityType by code
     *
     * @return VaFacilityType A corresponding VaFacilityType by code
     */
    public  VAFacilityType getVaFacilityTypeByCode(String code)
            throws UnknownLookupTypeException, UnknownLookupCodeException;

    /**
     * Find all NTRVerificationMethod
     *
     * @return List a list of all NTRVerificationMethod
     */
    public  List getAllNTRVerificationMethods()
            throws UnknownLookupTypeException, UnknownLookupCodeException;

    /**
     * Get ParentVISNSiteMap
     *
     * @return ParentVISNSiteMap
     */
    public Map getParentVISNSiteMap()throws UnknownLookupTypeException, ClassNotFoundException,ServiceException;

    /**
     * Get NTRVerificationMethod by code
     *
     * @return NTRVerificationMethod A corresponding NTRVerificationMethod by code
     */
    public  NTRVerificationMethod getNTRVerificationMethodByCode(String code)
            throws UnknownLookupTypeException, UnknownLookupCodeException;

    /**
     * Find all ZipCode
     *
     * @return List a list of all ZipCode
     */
    public  List getAllZipCodes() throws UnknownLookupTypeException,
            UnknownLookupCodeException;

    /**
     * Get ZipCode by code
     *
     * @return ZipCode A corresponding ZipCode by code
     */
    public  ZipCode getZipCodeByCode(String code)
            throws UnknownLookupTypeException, UnknownLookupCodeException;

    /**
     * Gets a partially filled in zip code object by code with the following attributes filled in:
     *
     * ZipCode.id
     * ZipCode.zipCode
     * ZipCode.state.name
     * ZipCode.cities.name
     * ZipCode.county.name
     *
     * @return The partially filled in zip code object.
     */
    public ZipCode getPartialZipCodeByCode(String code) throws UnknownLookupTypeException;

    /**
     * Gets a list of zip code objects that has only the following attributes filled in:
     * <p>
     * ZipCode.id
     * ZipCode.code
     * <p>
     * If no records are found, an empty list is returned.
     *
     * @param cityName The name of the city to query on
     * @param stateCode The state code to query on
     *
     * @return The zip code object
     */
    public List getPartialZipCodeByCityState(String cityName, String stateCode) throws UnknownLookupTypeException;

    /**
     * Gets a list of zip code objects that has only the following attributes filled in:
     * <p>
     * ZipCode.id
     * ZipCode.code
     * <p>
     * If no records are found, an empty list is returned.
     *
     * @param cityName The name of the city to query on
     *
     * @return The zip code object
     */
    public List getPartialZipCodeByCity(String cityName) throws UnknownLookupTypeException;

    /**
     * Find all BadAddressReason
     *
     * @return List a list of all BadAddressReason
     */
    public  List getAllBadAddressReasons() throws UnknownLookupTypeException,
            UnknownLookupCodeException;

    /**
     * Get BadAddressReason by code
     *
     * @return BadAddressReason A corresponding BadAddressReason by code
     */
    public  BadAddressReason getBadAddressReasonByCode(String code)
            throws UnknownLookupTypeException, UnknownLookupCodeException;

    /**
     * Find all MeansTestNotRequiredReason
     *
     * @return List a list of all MeansTestNotRequiredReason
     */
    public  List getAllMeansTestNotRequiredReasons() throws UnknownLookupTypeException,
            UnknownLookupCodeException;

    /**
     * Get MeansTestNotRequiredReason by code
     *
     * @return MeansTestNotRequiredReason A corresponding MeansTestNotRequiredReason by code
     */
    public  MeansTestNotRequiredReason getMeansTestNotRequiredReasonByCode(String code)
            throws UnknownLookupTypeException, UnknownLookupCodeException;

    /**
     * Find all InsuranceReportSource
     *
     * @return List a list of all InsuranceReportSource
     */
    public  List getAllInsuranceReportSources() throws UnknownLookupTypeException,
            UnknownLookupCodeException;

    /**
     * Get InsuranceReportSource by code
     *
     * @return InsuranceReportSource A corresponding InsuranceReportSource by code
     */
    public  InsuranceReportSource getInsuranceReportSourceByCode(String code)
            throws UnknownLookupTypeException, UnknownLookupCodeException;

    /**
     * Find all SensitivityChangeSource
     *
     * @return List a list of all SensitivityChangeSource
     */
    public  List getAllSensitivityChangeSources()
            throws UnknownLookupTypeException, UnknownLookupCodeException;

    /**
     * Get SensitivityChangeSource by code
     *
     * @return SensitivityChangeSource A corresponding SensitivityChangeSource by code
     */
    public  SensitivityChangeSource getSensitivityChangeSourceByCode(String code)
            throws UnknownLookupTypeException, UnknownLookupCodeException;

    /**
     * Find all SSNType
     *
     * @return List a list of all SSNType
     */
    public  List getAllSSNTypes() throws UnknownLookupTypeException,
            UnknownLookupCodeException;

    /**
     * Get SSNType by code
     *
     * @return SSNType A corresponding SSNType by code
     */
    public  SSNType getSSNTypeByCode(String code)
            throws UnknownLookupTypeException, UnknownLookupCodeException;

    /**
     * Find all SSNChangeSource
     *
     * @return List a list of all SSNChangeSource
     */
    public  List getAllSSNChangeSources() throws UnknownLookupTypeException,
            UnknownLookupCodeException;

    /**
     * Get SSNChangeSource by code
     *
     * @return SSNChangeSource A corresponding SSNChangeSource by code
     */
    public  SSNChangeSource getSSNChangeSourceByCode(String code)
            throws UnknownLookupTypeException, UnknownLookupCodeException;

    /**
     * Find all SSAVerificationStatus
     *
     * @return List a list of all SSAVerificationStatus
     */
    public  List getAllSSAVerificationStatuss() throws UnknownLookupTypeException,
            UnknownLookupCodeException;

    /**
     * Get SSAVerificationStatus by code
     *
     * @return SSAVerificationStatus A corresponding SSAVerificationStatus by code
     */
    public  SSAVerificationStatus getSSAVerificationStatusByCode(String code)
            throws UnknownLookupTypeException, UnknownLookupCodeException;

    /**
     * Find all SSAMessage
     *
     * @return List a list of all SSAMessage
     */
    public  List getAllSSAMessages() throws UnknownLookupTypeException,
            UnknownLookupCodeException;

    /**
     * Get SSAMessage by code
     *
     * @return SSAMessage A corresponding SSAMessage by code
     */
    public  SSAMessage getSSAMessageByCode(String code)
            throws UnknownLookupTypeException, UnknownLookupCodeException;


    /**
     * Find all PseudoSSNReason
     *
     * @return List a list of all PseudoSSNReason
     */
    public  List getAllPseudoSSNReasons() throws UnknownLookupTypeException,
            UnknownLookupCodeException;

    /**
     * Get PseudoSSNReason by code
     *
     * @return PseudoSSNReason A corresponding PseudoSSNReason by code
     */
    public  PseudoSSNReason getPseudoSSNReasonByCode(String code)
            throws UnknownLookupTypeException, UnknownLookupCodeException;

    /**
     * Find all NcoaDeliveryCode
     *
     * @return List a list of all NcoaDeliveryCode
     */
    public  List getAllNcoaDeliveryCodes() throws UnknownLookupTypeException,
            UnknownLookupCodeException;

    /**
     * Get NcoaDeliveryCode by code
     *
     * @return NcoaDeliveryCode A corresponding NcoaDeliveryCode by code
     */
    public  NcoaDeliveryCode getNcoaDeliveryCodeByCode(String code)
            throws UnknownLookupTypeException, UnknownLookupCodeException;

    /**
     * Find all NcoaStatus
     *
     * @return List a list of all NcoaStatus
     */
    public  List getAllNcoaStatuses() throws UnknownLookupTypeException,
            UnknownLookupCodeException;

    /**
     * Get NcoaStatus by code
     *
     * @return NcoaStatus A corresponding NcoaStatus by code
     */
    public  NcoaStatus getNcoaStatusByCode(String code)
            throws UnknownLookupTypeException, UnknownLookupCodeException;

    /**
    * Find all NcoaZipMatch
    *
    * @return List a list of all NcoaZipMatch
    */
   public  List getAllNcoaZipMatches() throws UnknownLookupTypeException,
           UnknownLookupCodeException;

   /**
    * Get NcoaZipMatch by code
    *
    * @return NcoaZipMatch A corresponding NcoaZipMatch by code
    */
   public  NcoaZipMatch getNcoaZipMatchByCode(String code)
           throws UnknownLookupTypeException, UnknownLookupCodeException;

    /**
     * Find all ZipPlus4NoMatchReason
     *
     * @return List a list of all ZipPlus4NoMatchReason
     */
    public  List getAllZipPlus4NoMatchReasons() throws UnknownLookupTypeException,
            UnknownLookupCodeException;

    /**
     * Get ZipPlus4NoMatchReason by code
     *
     * @return ZipPlus4NoMatchReason A corresponding ZipPlus4NoMatchReason by code
     */
    public  ZipPlus4NoMatchReason getZipPlus4NoMatchReasonByCode(String code)
            throws UnknownLookupTypeException, UnknownLookupCodeException;

    /**
     * Find all ComAACErrorType
     *
     * @return List a list of all ComAACErrorType
     */
    public  List getAllComAACErrorTypes() throws UnknownLookupTypeException,
            UnknownLookupCodeException;

    /**
     * Get ComAACErrorType by code
     *
     * @return ComAACErrorType A corresponding ComAACErrorType by code
     */
    public  ComAACErrorType getComAACErrorTypeByCode(String code)
            throws UnknownLookupTypeException, UnknownLookupCodeException;

    /**
     * Find all ComAACRejectReasonType
     *
     * @return List a list of all ComAACRejectReasonType
     */
    public  List getAllComAACRejectReasonTypes() throws UnknownLookupTypeException,
            UnknownLookupCodeException;

    /**
     * Get ComAACRejectReasonType by code
     *
     * @return ComAACRejectReasonType A corresponding ComAACRejectReasonType by code
     */
    public  ComAACRejectReasonType getComAACRejectReasonTypeByCode(String code)
            throws UnknownLookupTypeException, UnknownLookupCodeException;

    /**
     * Find all ComLetterGeneratedType
     *
     * @return List a list of all ComLetterGeneratedType
     */
    public  List getAllComLetterGeneratedTypes() throws UnknownLookupTypeException,
            UnknownLookupCodeException;

    /**
     * Get ComLetterGeneratedType by code
     *
     * @return ComLetterGeneratedType A corresponding ComLetterGeneratedType by code
     */
    public  ComLetterGeneratedType getComLetterGeneratedTypeByCode(String code)
            throws UnknownLookupTypeException, UnknownLookupCodeException;

    /**
     * Find all ComLetterType
     *
     * @return List a list of all ComLetterType
     */
    public  List getAllComLetterTypes() throws UnknownLookupTypeException,
            UnknownLookupCodeException;

    /**
     * Get ComLetterType by code
     *
     * @return ComLetterType A corresponding ComLetterType by code
     */
    public  ComLetterType getComLetterTypeByCode(String code)
            throws UnknownLookupTypeException, UnknownLookupCodeException;

    /**
     * Find all ComMailingStatusType
     *
     * @return List a list of all ComMailingStatusType
     */
    public  List getAllComMailingStatusTypes() throws UnknownLookupTypeException,
            UnknownLookupCodeException;

    /**
     * Find all ComMailingTriggerType
     *
     * @return List a list of all ComMailingStatusType
     */
    public  List getAllComMailingTriggerTypes() throws UnknownLookupTypeException,
            UnknownLookupCodeException;

    /**
     * Get ComMailingStatusType by code
     *
     * @return ComMailingStatusType A corresponding ComMailingStatusType by code
     */
    public  ComMailingStatusType getComMailingStatusTypeByCode(String code)
            throws UnknownLookupTypeException, UnknownLookupCodeException;

    /**
     * Get ComMailingTriggerType by code
     *
     * @return ComMailingStatusType A corresponding ComMailingStatusType by code
     */
    public  ComMailingTriggerType getComMailingTriggerTypeByCode(String code)
            throws UnknownLookupTypeException, UnknownLookupCodeException;

    /**
     * Find all WkfCaseActivityType
     *
     * @return List a list of all WkfCaseActivityType
     */
    public  List getAllWkfCaseActivityTypes() throws UnknownLookupTypeException,
            UnknownLookupCodeException;

    /**
     * Get WkfCaseActivityType by code
     *
     * @return WkfCaseActivityType A corresponding WkfCaseActivityType by code
     */
    public  WkfCaseActivityType getWkfCaseActivityTypeByCode(String code)
            throws UnknownLookupTypeException, UnknownLookupCodeException;

    /**
     * Find all WkfCaseRequestType
     *
     * @return List a list of all WkfCaseRequestType
     */
    public  List getAllWkfCaseRequestTypes() throws UnknownLookupTypeException,
            UnknownLookupCodeException;

    /**
     * Get WkfCaseRequestType by code
     *
     * @return WkfCaseRequestType A corresponding WkfCaseRequestType by code
     */
    public  WkfCaseRequestType getWkfCaseRequestTypeByCode(String code)
            throws UnknownLookupTypeException, UnknownLookupCodeException;

    /**
     * Find all WkfCaseStatusType
     *
     * @return List a list of all WkfCaseStatusType
     */
    public  List getAllWkfCaseStatusTypes() throws UnknownLookupTypeException,
            UnknownLookupCodeException;

    /**
     * Get WkfCaseStatusType by code
     *
     * @return WkfCaseStatusType A corresponding WkfCaseStatusType by code
     */
    public  WkfCaseStatusType getWkfCaseStatusTypeByCode(String code)
            throws UnknownLookupTypeException, UnknownLookupCodeException;

    /**
     * Find all WkfCaseType
     *
     * @return List a list of all WkfCaseType
     */
    public  List getAllWkfCaseTypes() throws UnknownLookupTypeException,
            UnknownLookupCodeException;

    /**
     * Get WkfCaseType by code
     *
     * @return WkfCaseType A corresponding WkfCaseType by code
     */
    public  WkfCaseType getWkfCaseTypeByCode(String code)
            throws UnknownLookupTypeException, UnknownLookupCodeException;

    /**
     * Find all WkfIssueType
     *
     * @return List a list of all WkfIssueType
     */
    public  List getAllWkfIssueTypes() throws UnknownLookupTypeException,
            UnknownLookupCodeException;

    /**
     * Get WkfIssueType by code
     *
     * @return WkfIssueType A corresponding WkfIssueType by code
     */
    public  WkfIssueType getWkfIssueTypeByCode(String code)
            throws UnknownLookupTypeException, UnknownLookupCodeException;

    /**
     * Get the GMTThreshold by income year and fips code.
     * @param year
     * @param fipsCode
     * @return List of GMTThreshold Objects.
     * @throws ServiceException
     */
    public List getGMTThreshold(Integer year, String fipsCode)
    throws ServiceException;

    /**
     * Get the GMTThreshold by income year, fips code and msa code.
     *
     * @param year
     * @param fipsCode
     * @param msaCode
     * @return List of GMTThreshold Objects.
     * @throws ServiceException
     */
    public List getGMTThreshold(Integer year, String fipsCode, String msaCode)
    throws ServiceException;


    public String getFipsCode(String zipCode)
    throws ServiceException;


    public String getMsaCode(String zipCode)
    throws ServiceException;

    /**
     * Get the IncomeThreshold by income year.
     *
     * @param year
     * @return List of IncomeThreshold Objects.
     * @throws ServiceException
     */
    public IncomeThreshold getIncomeThreshold(Integer year) throws ServiceException;

    /**
     * Find all IncomeThreshold
     *
     * @return List a list of all WkfIssueType
     * @throws ServiceException
     */
    public  List getAllIncomeThresholds() throws ServiceException;

    /**
     * Find all SystemParameterType
     *
     * @return List a list of all SystemParameterType
     */
    public List getAllSystemParameterTypes() throws UnknownLookupTypeException,
            UnknownLookupCodeException ;

    /**
     * Get SystemParameterType by code
     *
     * @return ParameterType A corresponding SystemParameterType by code
     */
    public SystemParameterType getSystemParameterTypeByCode(String code)
            throws UnknownLookupTypeException, UnknownLookupCodeException ;
    /**
     * Find all SystemParameterUnit
     *
     * @return List a list of all SystemParameterType
     */
    public List getAllSystemParameterUnits() throws UnknownLookupTypeException,
            UnknownLookupCodeException ;

    /**
     * Get SystemParameterUnit by code
     *
     * @return ParameterUnit A corresponding SystemParameterUnit by code
     */
    public SystemParameterUnit getSystemParameterUnitByCode(String code)
            throws UnknownLookupTypeException, UnknownLookupCodeException ;
    /**
     * Find all Capabilities
     *
     * @return List a list of all Capabilities
     */
    public List getAllCapabilities() throws UnknownLookupTypeException,
            UnknownLookupCodeException ;

    /**
     * Get Capability by code
     *
     * @return Capability
     */
    public Capability getCapabilityByCode(String code)
            throws UnknownLookupTypeException, UnknownLookupCodeException ;
    /**
     * Find all FunctionalGroups
     *
     * @return List a list of all FunctionalGroups
     */
    public List getAllFunctionalGroups() throws UnknownLookupTypeException,
            UnknownLookupCodeException ;

    /**
     * Get FunctionalGroup by code
     *
     * @return FunctionalGroup
     */
    public FunctionalGroup getFunctionalGroupByCode(String code)
            throws UnknownLookupTypeException, UnknownLookupCodeException ;

    /**
     * Find all standard reports
     *
     * @return List a list of all StandardReports
     */
    public List getAllStandardReports() throws UnknownLookupTypeException,
            UnknownLookupCodeException ;

    /**
     * Get StandardReport by code
     *
     * @return StandardReport
     */
    public StandardReport getStandardReportByCode(String code)
            throws UnknownLookupTypeException, UnknownLookupCodeException ;

    /**
     * Get ReportType by code
     *
     * @return ReportType
     */
    public ReportType getReportTypeByCode(String code)
            throws UnknownLookupTypeException, UnknownLookupCodeException ;

    /**
     * Find all report format
     *
     * @return List a list of all ReportFormat
     */
    public List getAllReportFormats() throws UnknownLookupTypeException,
            UnknownLookupCodeException ;

    /**
     * Get ReportFormat by code
     *
     * @return ReportFormat
     */
    public ReportFormat getReportFormatByCode(String code)
            throws UnknownLookupTypeException, UnknownLookupCodeException ;

    /**
     * Find all report exported types
     *
     * @return List a list of all ReportExportedType
     */
    public List getAllReportExportedTypes() throws UnknownLookupTypeException,
            UnknownLookupCodeException ;

    /**
     * Get ReportExportedType by code
     *
     * @return ReportExportedType
     */
    public ReportExportedType getReportExportedTypeByCode(String code)
            throws UnknownLookupTypeException, UnknownLookupCodeException ;

    /**
     * Find all report divisions
     *
     * @return List a list of all ReportDivision
     */
    public List getAllReportDivisions() throws UnknownLookupTypeException,
            UnknownLookupCodeException ;

    /**
     * Get ReportDivision by code
     *
     * @return ReportDivision
     */
    public ReportDivision getReportDivisionByCode(String code)
            throws UnknownLookupTypeException, UnknownLookupCodeException ;

    /**
     * Find all report period types
     *
     * @return List a list of all ReportPeriodTypes
     */
    public List getAllReportPeriodTypes() throws UnknownLookupTypeException,
            UnknownLookupCodeException ;

    /**
     * Get ReportPeriodType by code
     *
     * @return ReportPeriodType
     */
    public ReportPeriodType getReportPeriodTypeByCode(String code)
            throws UnknownLookupTypeException, UnknownLookupCodeException ;

    /**
     * Find all report period types
     *
     * @return List a list of all ReportEEDWeeklyTypes
     */
    public List getAllReportEEDWeeklyTypes() throws UnknownLookupTypeException,
            UnknownLookupCodeException ;

    /**
     * Get ReportEEDWeeklyType by code
     *
     * @return ReportPeriodType
     */
    public ReportEEDWeeklyType getReportEEDWeeklyTypeByCode(String code)
            throws UnknownLookupTypeException, UnknownLookupCodeException ;

    /**
     * Find all quarters
     *
     * @return List a list of all Quarter
     */
    public List getAllQuarters() throws UnknownLookupTypeException,
            UnknownLookupCodeException ;

    /**
     * Get Quarter by code
     *
     * @return Quarter
     */
    public Quarter getQuarterByCode(String code)
            throws UnknownLookupTypeException, UnknownLookupCodeException ;

    /**
     * Get EmergencyResponse by code
     *
     * @return EmergencyResponse
     */
	public EmergencyResponse getEmergencyResponseByCode(String code)
		throws UnknownLookupTypeException, UnknownLookupCodeException;

    /**
     * Find all emergencyResponses
     *
     * @return List a list of all EmergencyResponses
     */
	public List getAllEmergencyResponses()
		throws UnknownLookupTypeException, UnknownLookupCodeException;

    /**
     * Get EthnicityType by code
     *
     * @return EthnicityType
     */
    public EthnicityType getEthnicityTypeByCode(String code)
        throws UnknownLookupTypeException, UnknownLookupCodeException;

    /**
     * Find all ethnicityTypes
     *
     * @return List a list of all EthnicityType
     */
    public List getAllEthnicityTypes()
        throws UnknownLookupTypeException, UnknownLookupCodeException;

    /**
     * Get RaceType by code
     *
     * @return RaceType
     */
    public RaceType getRaceTypeByCode(String code)
        throws UnknownLookupTypeException, UnknownLookupCodeException;

    /**
     * Find all raceTypes
     *
     * @return List a list of all RaceType
     */
    public List getAllRaceTypes()
        throws UnknownLookupTypeException, UnknownLookupCodeException;

    /**
     * Get SpinalCordInjuryType by code
     *
     * @return SpinalCordInjuryType
     */
    public SpinalCordInjuryType getSpinalCordInjuryTypeByCode(String code)
        throws UnknownLookupTypeException, UnknownLookupCodeException;

    /**
     * Find all spinalCordInjuryTypes
     *
     * @return List a list of all SpinalCordInjuryType
     */
    public List getAllSpinalCordInjuryTypes()
        throws UnknownLookupTypeException, UnknownLookupCodeException;

    /**
     * Get BenefitType by code
     *
     * @return BenefitType
     */
    public BenefitType getBenefitTypeByCode(String code)
        throws UnknownLookupTypeException, UnknownLookupCodeException;

    /**
     * Find all benefitTypes
     *
     * @return List a list of all BenefitType
     */
    public List getAllBenefitTypes()
        throws UnknownLookupTypeException, UnknownLookupCodeException;

    /**
     * Get CollectionMethod by code
     *
     * @return CollectionMethod
     */
    public CollectionMethod getCollectionMethodByCode(String code)
        throws UnknownLookupTypeException, UnknownLookupCodeException;

    /**
     * Find all collectionMethods
     *
     * @return List a list of all CollectionMethod
     */
    public List getAllCollectionMethods()
        throws UnknownLookupTypeException, UnknownLookupCodeException;
    /**
     * Get NamedContingency by code
     *
     * @return NamedContingency
     */
	public NamedContingency getNamedContingencyByCode(String code)
		throws UnknownLookupTypeException, UnknownLookupCodeException;

    /**
     * Find all NamedContingencys
     *
     * @return List a list of all NamedContingencys
     */
	public List getAllNamedContingencys()
		throws UnknownLookupTypeException, UnknownLookupCodeException;

    /**
     * Get MilitaryServiceQueryStatus by code
     *
     * @return MilitaryServiceQueryStatus
     */
	public MilitaryServiceQueryStatus getMilitaryServiceQueryStatusByCode(String code)
		throws UnknownLookupTypeException, UnknownLookupCodeException;

    /**
     * Find all MilitaryServiceQueryStatuses
     *
     * @return List a list of all MilitaryServiceQueryStatuses
     */
	public List getAllMilitaryServiceQueryStatuses()
		throws UnknownLookupTypeException, UnknownLookupCodeException;

    /**
     * Get SpecialOperation by code
     *
     * @return SpecialOperation
     */
	public SpecialOperation getSpecialOperationByCode(String code)
		throws UnknownLookupTypeException, UnknownLookupCodeException;

    /**
     * Find all SpecialOperations
     *
     * @return List a list of all SpecialOperations
     */
	public List getAllSpecialOperations()
		throws UnknownLookupTypeException, UnknownLookupCodeException;

    /**
     * Get TerminationReason by code
     *
     * @return TerminationReason
     */
	public TerminationReason getTerminationReasonByCode(String code)
		throws UnknownLookupTypeException, UnknownLookupCodeException;

    /**
     * Find all TerminationReasons
     *
     * @return List a list of all TerminationReasons
     */
	public List getAllTerminationReasons()
		throws UnknownLookupTypeException, UnknownLookupCodeException;


    /**
     * Get MHDocumentType by code
     *
     * @return MHDocumentType
     */
	public MHDocumentType getMHDocumentTypeByCode(String code)
		throws UnknownLookupTypeException, UnknownLookupCodeException;

    /**
     * Find all MHDocumentTypes
     *
     * @return List a list of all MHDocumentTypes
     */
	public List getAllMHDocumentTypes()
		throws UnknownLookupTypeException, UnknownLookupCodeException;

	/**
	 * Find all CombatServiceSources
	 *
	 * @return
	 * @throws UnknownLookupTypeException
	 * @throws UnknownLookupCodeException
	 */
	public List getAllCombatServiceSources()
		throws UnknownLookupTypeException, UnknownLookupCodeException;

	/**
	 * CombatServiceSource
	 *
	 * @param code
	 * @return
	 * @throws UnknownLookupTypeException
	 * @throws UnknownLookupCodeException
	 */
	public CombatServiceSource getCombatServiceSourceByCode(String code)
		throws UnknownLookupTypeException, UnknownLookupCodeException;

	/**
	 * EEServiceGroup
	 * @param code
	 * @return
	 * @throws UnknownLookupTypeException
	 * @throws UnknownLookupCodeException
	 */
	public EEServiceGroup getEEServiceGroupByCode(String code)
    throws UnknownLookupTypeException, UnknownLookupCodeException ;

    /**
     * Get EEServiceGroup by code
     *
     * @return Gender A corresponding Gender by code
     */
    public EEServiceField getEEServiceFieldByCode(String code)
            throws UnknownLookupTypeException, UnknownLookupCodeException;

    /**
     * Get PensioReasonCode by code
     *
     * @return PensionReasonCode
     */
    public PensionReasonCode getPensionReasonByCode(String code)
		throws UnknownLookupTypeException, UnknownLookupCodeException;

    /**
     * Get SeedStatus by code
     *
     * @return SeedStatus A corresponding seeding status by code
     */
    public  SeedStatus getSeedStatusByCode(String code)
            throws UnknownLookupTypeException, UnknownLookupCodeException;

    /**
     * Find all Source Designations
     *
     * @return List a list of all SourceDesignations
     */
    public List getAllSourceDesignations() throws UnknownLookupTypeException,
            UnknownLookupCodeException ;

    /**
     * Get SourceDesignation by code
     *
     * @return SourceDesignation
     */
    public SourceDesignation getSourceDesignationByCode(String code)
            throws UnknownLookupTypeException, UnknownLookupCodeException ;

    /**
     * Get TeamPurpose by code
     *
     * @return TeamPurpose
     */
    public TeamPurpose getTeamPurposeByCode(String code)
            throws UnknownLookupTypeException, UnknownLookupCodeException ;

    /**
     * Get ApplicationMethod by code
     */
    public ApplicationMethod getApplicationMethodByCode(String code)
    	throws UnknownLookupTypeException, UnknownLookupCodeException;

    /**
     * Get ApplicationStatus by code
     */
    public ApplicationStatus getApplicationStatusByCode(String code)
    	throws UnknownLookupTypeException, UnknownLookupCodeException;

    public List getAllHandBookBatchRequestStatus() throws UnknownLookupTypeException,
    UnknownLookupCodeException;

    public HandBookRequestStatusType getHandBookRequestStatusTypeByCode(String code)
    throws UnknownLookupTypeException, UnknownLookupCodeException;

    /**
     * Get HandBookMailStatusType by code
     *
     * @return HandBookMailStatusType A corresponding HandBookMailStatusType by code
     */
    public  HandBookMailStatusType getHandBookMailingStatusTypeByCode(String code)
            throws UnknownLookupTypeException, UnknownLookupCodeException;

	/**
	 * Find all DeliveryPreferenceType
	 *
	 * @return a list of all DeliveryPreferenceType
	 */
	List getAllDeliveryPreferenceTypes() throws UnknownLookupTypeException,
			UnknownLookupCodeException;

	/**
	 * Find all DeliveryPreferenceSourceOfChange
	 *
	 * @return a list of all DeliveryPreferenceSourceOfChange
	 */
	List getAllDeliveryPreferenceSourceOfChangeTypes()
			throws UnknownLookupTypeException, UnknownLookupCodeException;

	/**
	 * Get DeliveryPreferenceType by code
	 *
	 * @return DeliveryPreferenceType A corresponding DeliveryPreferenceType by
	 *         code
	 */
	DeliveryPreferenceType getDeliveryPreferenceTypeByCode(String code)
			throws UnknownLookupTypeException, UnknownLookupCodeException;

	/**
	 * Get DeliveryPreferenceSourceOfChange by code
	 *
	 * @return DeliveryPreferenceSourceOfChange A corresponding
	 *         DeliveryPreferenceSourceOfChange by code
	 */
	DeliveryPreferenceSourceOfChange getDeliveryPreferenceSourceOfChangeByCode(
			String code) throws UnknownLookupTypeException,
			UnknownLookupCodeException;

	/**
	 * Find all AuthenticationLevel
	 *
	 * @return a list of all AuthenticationLevel
	 */
	List getAllAuthenticationLevels()
			throws UnknownLookupTypeException, UnknownLookupCodeException;

	/**
	 * Get AuthenticationLevel by code
	 *
	 * @return AuthenticationLevel A corresponding AuthenticationLevel by
	 *         code
	 */
	AuthenticationLevel getAuthenticationLevelByCode(String code)
			throws UnknownLookupTypeException, UnknownLookupCodeException;


	/**
	 * Find all VOAFormTypes
	 *
	 * @return a list of all VOAFormTypes
	 */
	List getAllVOAFormTypes()
			throws UnknownLookupTypeException, UnknownLookupCodeException;

	/**
	 * Get VOAFormType by code
	 *
	 * @return VOAFormType A corresponding VOAFormType by
	 *         code
	 */
	VOAFormType getVOAFormTypeByCode(String code)
			throws UnknownLookupTypeException, UnknownLookupCodeException;

	/**
	 * Find all VOAApplicationStatuses
	 *
	 * @return a list of all VOAApplicationStatuses
	 */
	List getAllVOAApplicationStatuses()
			throws UnknownLookupTypeException, UnknownLookupCodeException;

	/**
	 * Get VOAApplicationStatus by code
	 *
	 * @return VOAApplicationStatus A corresponding VOAApplicationStatus by
	 *         code
	 */
	VOAApplicationStatus getVOAApplicationStatusByCode(String code)
			throws UnknownLookupTypeException, UnknownLookupCodeException;

	/**
     * Find all EligibilityDocTypes
     *
     * @return List a list of all EligibilityDocType
     */
    public  List getAllEligibilityDocTypes()
            throws UnknownLookupTypeException, UnknownLookupCodeException;

    /**
     * Get EligibilityDocType by code
     *
     * @return EligibilityDocType A corresponding EligibilityDocType by code
     */
    public  EligibilityDocType getEligibilityDocTypeByCode(String code)
            throws UnknownLookupTypeException, UnknownLookupCodeException;

	/**
     * Find all VeteranIdentifierTypes
     *
     * @return List a list of all VeteranIdentifierType
     */
    public  List getAllVeteranIdentifierTypes()
            throws UnknownLookupTypeException, UnknownLookupCodeException;

    /**
     * Get VeteranIdentifierType by code
     *
     * @return VeteranIdentifierType A corresponding VeteranIdentifierType by code
     */
    public  VeteranIdentifierType getVeteranIdentifierTypeByCode(String code)
            throws UnknownLookupTypeException, UnknownLookupCodeException;


	/**
     * Find all HealthBenefitPlanTypes
     *
     * @return List a list of all HealthBenefitPlanTypes
     */
    public  List getAllHealthBenefitPlanTypes()
            throws UnknownLookupTypeException, UnknownLookupCodeException;

    /**
     * Get HealthBenefitPlanType by code
     *
     * @return HealthBenefitPlanType A corresponding HealthBenefitPlanType by code
     */
    public  HealthBenefitPlanType getHealthBenefitPlanTypeByCode(String code)
            throws UnknownLookupTypeException, UnknownLookupCodeException;

	/**
     * Find all DataChangeSources
     *
     * @return List a list of all DataChangeSources
     */
    public  List getAllDataChangeSources()
            throws UnknownLookupTypeException, UnknownLookupCodeException;

    /**
     * Get DataChangeSource by code
     *
     * @return DataChangeSource A corresponding DataChangeSource by code
     */
    public  DataChangeSource getDataChangeSourceByCode(String code)
            throws UnknownLookupTypeException, UnknownLookupCodeException;

    /**
     * Get DescriptorType by code
     *
     * @param code
     * @return
     * @throws UnknownLookupTypeException
     * @throws UnknownLookupCodeException
     */
    public  DescriptorType getDescriptorTypeByCode(String code) throws UnknownLookupTypeException, UnknownLookupCodeException;

    /**
     * Find all Languages.
     *
     * @return List a list of all Language
     */
    public List getAllLanguages() throws UnknownLookupTypeException,
            UnknownLookupCodeException;

    /**
     * Get Language by code.
     *
     * @param code
     * @return Language A corresponding Language by code
     * @throws UnknownLookupTypeException Thrown if the type of lookup cannot be found.
     * @throws UnknownLookupCodeException Thrown if the specified code cannot be found.
     */
    public Language getLanguageByCode(String code)
            throws UnknownLookupTypeException, UnknownLookupCodeException;

    /**
     * Find all HealthBenefitPlanCategory.
     *
     * @return List a list of all HealthBenefitPlanCategory
     */
    public List getAllHealthBenefitPlanCategories() throws UnknownLookupTypeException,
            UnknownLookupCodeException;

    /**
     * Get HealthBenefitPlanCategory by code.
     *
     * @param code
     * @return HealthBenefitPlanCategory A corresponding HealthBenefitPlanCategory by code
     * @throws UnknownLookupTypeException Thrown if the type of lookup cannot be found.
     * @throws UnknownLookupCodeException Thrown if the specified code cannot be found.
     */
    public HealthBenefitPlanCategory getHealthBenefitPlanCategoryByCode(String code)
            throws UnknownLookupTypeException, UnknownLookupCodeException;

    //CCR 13856
    public boolean getDODSharingInd() throws ServiceException;

    /**
     * Get CampLejeuneVerificationType by code.
     *
     * @param code
     * @return CampLejeuneVerificationType A corresponding CampLejeuneVerificationType by code
     * @throws UnknownLookupTypeException Thrown if the type of lookup cannot be found.
     * @throws UnknownLookupCodeException Thrown if the specified code cannot be found.
     */

    public CampLejeuneVerificationType getCampLejeuneVerificationTypeByCode(String code)
    		throws UnknownLookupTypeException, UnknownLookupCodeException;

    /**
     * Get CampLejeuneChangeSource by code.
     *
     * @param code
     * @return CampLejeuneChangeSource A corresponding CampLejeuneChangeSource by code
     * @throws UnknownLookupTypeException Thrown if the type of lookup cannot be found.
     * @throws UnknownLookupCodeException Thrown if the specified code cannot be found.
     */

    public CampLejeuneChangeSource getCampLejeuneChangeSourceByCode(String code)
    		throws UnknownLookupTypeException, UnknownLookupCodeException;

}
