/********************************************************************
 * Copyright  2006 VHA. All rights reserved
 ********************************************************************/

/*
 * Created on Apr 17, 2006
 *
 */
package gov.va.med.esr.service;

// Java classes
import gov.va.med.esr.common.model.financials.IncomeTest;
import gov.va.med.esr.common.model.ivmdm.IVMClearLoadErrorStatistics;
import gov.va.med.esr.common.model.ivmdm.IVMDMExtractBatch;
import gov.va.med.esr.common.model.ivmdm.IVMMigration;
import gov.va.med.esr.common.model.person.id.PersonEntityKey;
import gov.va.med.esr.common.model.person.id.VPIDEntityKey;
import gov.va.med.fw.service.ServiceException;

import java.io.Serializable;
import java.math.BigDecimal;
import java.util.Date;
import java.util.Set;
import gov.va.med.esr.common.model.lookup.MessageStatus;
import gov.va.med.esr.service.external.person.IVMCandidateInfo;
import gov.va.med.esr.service.external.person.IvmLetterCandidateInfo;
/**
 * @author Yi He
 * @author DNS   MANSOG
 * @version 1.0
 */
public interface IVMDMService extends Serializable {
    /**
     * Find the IVMMigration Object for a given person entity key and income year.
     * 
     * @param personKey
     * @param incomeYear
     * @return IVMMigration or null if no record found
     * @throws ServiceException
     */
    public IVMMigration findIVMMigration(PersonEntityKey personKey, Integer incomeYear) throws ServiceException;
    
    /**
     * CCR 12540 support IVM explicit delete
     * Find the IVMMigration Object for a given person entity key and income year and marked for delete.
     * 
     * @param personKey
     * @param incomeYear
     * @return IVMMigration or null if no record found
     * @throws ServiceException
     */
    public IVMMigration findIVMMigrationMarkedForDelete(PersonEntityKey personKey, Integer incomeYear) throws ServiceException;
    
    
    /**
     * Find the IVMMigration for a given ICN and income year.
     * 
     * @param icn - a unique ICN number of a person
     * @param incomeYear
     * @return IVMMigration or null if no record found
     * @throws ServiceException
     */
    public IVMMigration findIVMMigration(String icn, Integer incomeYear) throws ServiceException;
        
    /**
     * Save the IVMMigration Object.
     * @param ivmMigration
     * @throws ServiceException
     */
    public void saveIVMMigration(IVMMigration ivmMigration) throws ServiceException;

    /**
     * CCR 11892 new ivm selection
     * 
     * @param ivmMigration
     * @param status
     * @throws ServiceException
     */
    public void saveIVMMigration(IVMMigration ivmMigration, MessageStatus.Code status) throws ServiceException;    
    
    /**
     * Save the IVMMigration Object.
     * @param extractStartTime
     * @param personKey
     * @param icn
     * @param year
     * @throws ServiceException
     */
    public void saveIVMMigration(Date extractStartTime, PersonEntityKey personKey, String icn, Integer year) throws ServiceException;
    
    /**
     * Save the IVMDMExtractBatch Object and its childs.
     * Persist the IVM DM Extract Batch.
     * @param ivmdmExtract
     */
    public void saveIVMDMExtractBatch(IVMDMExtractBatch ivmdmExtract) throws ServiceException;
    
    
    /**
     * Save the clear load error statistics object.
     * @param ivmCLEStats
     * @throws ServiceException
     */
    public void saveClearLoadErrorStatistics(IVMClearLoadErrorStatistics ivmCLEStats) throws ServiceException;
    
    /**
     * Check whether the given person matches the IVM selection criteria to extract records or not.
     * 
     * @param personKey
     * @param icn
     * @param incomeYear
     * @return True if the person with the specified icn meets the ivm selection criteria for the specified year
     * @throws ServiceException
     */
    public boolean meetSelectionCriteria(PersonEntityKey personKey, String icn, Integer incomeYear) throws ServiceException;

    /**
     * Update the synch error reason data in the database.
     * 
     * @param incomeYear - The income year
     * @param icn - Unique ICN number for the person
     * @param date - Date string 
     * @param errorReason - Error reason
     * @return true if data is updated successfully, false otherwise.
     * @throws ServiceException - if there is any error saving the IVM Migration
     */
    public boolean updateErrorReasonData(Integer incomeYear, String icn, String date, String errorReason) throws ServiceException;

    /**
     * Update the clear load error process data (clear error reasons previously loaded) in the database.

     * @param icn - Unique ICN of the person
     * @param incomeYear - The income year
     * @param loadErrorStats - IVMClearLoadErrorStatistics to update the statistics.
     * @throws ServiceException - if there is any error saving the IVM Migration
     */
    public void updateClearLoadError(String icn, String incomeYear, IVMClearLoadErrorStatistics loadErrorStats) throws ServiceException;

    /**
     * This method check whether the given person should be migrated or not.
     * If the given income year is null, it would check the criteria against current & next
     * income year as per in the requirements.
     * 
     * Returns the List of Integer (income years) to be migrated. If migrated is not allowed for the given income year
     * returns empty array.
     * @param personKey - A PersonEntityKey
     * @param incomeYear - income year
     * @param vpidKey
     * @return Set of income years (as Integers).
     * @throws ServiceException 
     */
    public Set findExtractIncomeYears(PersonEntityKey personKey,Integer incomeYear, VPIDEntityKey vpidKey) throws ServiceException;
    
    public Set findIVMMigrationData(Date date) throws ServiceException;
    
    public Set findIVMLetterStatusData() throws ServiceException;
    
    public IVMMigration findIVMMigration(BigDecimal identifier)
	throws ServiceException;
    
    public int getCurrentActiveIVMIncomeYear() throws ServiceException; 
    public int getNextVMIncomeYear() throws ServiceException;

    /**
	 * IVM ONLY: The most recent IY, for which the Veteran has been previously transferred
	 * without an EDB Load Failure Reason,
	 * 
	 * @param person
	 * @return
	 * @throws ServiceException 
	 */
    public Integer getMostRecentIVMMigratedYear(PersonEntityKey person) throws ServiceException; 
    /**
     * Find whether there is an IVMMigration with a particular status. If muliple, it only returns ONE.
     * Handy for making sure NOT Awaiting Transmission
     * 
     * @param personKey
     * @param incomeYear
     * @param status
     * @return
     * @throws ServiceException
     */
    public IVMMigration findIVMMigration(PersonEntityKey personKey, Integer incomeYear, MessageStatus.Code status) throws ServiceException;
    
    public Integer getLatestHECLegacyYear(PersonEntityKey person) throws ServiceException;    
    
    public Set findIVMMigrationsWithCompleteOrPendingStatus(PersonEntityKey person) throws ServiceException;  
    
    public  IVMCandidateInfo[] updateIVMCandidateRetrieval(Set set) throws ServiceException ;
    
    public boolean updateIVMStatus(BigDecimal transactionId, String status) throws ServiceException;
    
    public  IvmLetterCandidateInfo[] updateIVMLetterCandidateRetrieval(Set set) throws ServiceException ;    
    
	public void updateCommLogEntriesSentToIVMFlag(Set letterCandidates)	throws ServiceException;
}

