/********************************************************************
 * Copyright  2004 VHA. All rights reserved
 ********************************************************************/
package gov.va.med.esr.service;

// Java classes
import java.io.Serializable;
import java.util.Set;
import java.util.List;

import gov.va.med.fw.model.EntityKey;
import gov.va.med.fw.service.ServiceException;

import gov.va.med.esr.common.model.financials.DependentFinancials;
import gov.va.med.esr.common.model.financials.GMTThreshold;
import gov.va.med.esr.common.model.financials.IncomeThreshold;
import gov.va.med.esr.common.model.financials.SpouseFinancials;
import gov.va.med.esr.common.model.financials.IncomeTest;
import gov.va.med.esr.common.model.lookup.MeansTestStatus;
import gov.va.med.esr.common.model.person.Person;
import gov.va.med.esr.common.model.person.id.PersonEntityKey;
import gov.va.med.esr.service.impl.FinancialsChangeEvent;
import gov.va.med.esr.service.impl.HistoricalInfo;

/**
 * Financials Business Service to provide methods related to veteran's income test and financial statements
 * of veteran and dependents. 

 * Project: Common
 * @author DNS   LEV
 * @author DNS   MANSOG
 * @version 1.0
 */
public interface FinancialsService
    extends Serializable 
{
    /**
     * Update the spouse financials information.
     * @param incomeYear
     * @param incoming
     * @return
     * @throws ServiceException
     */
    public Person updateSpouseFinancials(Integer incomeYear, SpouseFinancials spouseFinancials, Person incoming)
    throws ServiceException;
    
    /**
     * Activate spouse.
     * @param incomeYear
     * @param spouseFinancials
     * @param incoming
     * @return
     * @throws ServiceException
     */
    public Person activateSpouse(Integer incomeYear, SpouseFinancials spouseFinancials, Person incoming)
    throws ServiceException;
    
    /**
     * Updates the dependent financials information.
     * @param incomeYear
     * @param dependentFinancials
     * @param incoming
     * @return
     * @throws ServiceException
     */
    public Person updateDependentFinancials(Integer incomeYear, DependentFinancials dependentFinancials, Person incoming)
    throws ServiceException;
    
    /**
     * Manage financial assessment of veteran and dependents. This include processing of 
     * income test and financial statements for veteran and all the dependents.
     * 
     * @param incomeYear
     * @param incoming
     * @return
     * @throws ServiceException
     */
    public Person updateFinancialAssessment(Integer incomeYear, Person incoming)
            throws ServiceException;

    /**
     * Updates the list of dependents for the specified income year.  This is in support of the
     * "Copy Dependents" functionality where all the dependents from a previous year are copied to the
     * currently selected year.
     *
     * @param incomeYear The income year to update the dependents on
     * @param incoming The Person who contains the updated dependents.
     * @return The updated person
     * @throws ServiceException if any problems were encountered.
     */
    public Person updateDependents(Integer incomeYear, Person incoming) throws ServiceException;

    /**
     * assessFinancialImpact
     * @param incomeYear
     * @param incoming
     * @return
     * @throws ServiceException
     */
    public Person assessFinancialImpact(Integer incomeYear, Person incoming) throws ServiceException;

    /**
     * Save the In-Process financial data.
     * @param incomeYear
     * @param person
     */
    public Person updateInProcess(Integer incomeYear, Person person) throws ServiceException;
    
    /**
     * Save the In-Process financial data.
     * @param incomeYear
     * @param incoming
     * @param savePerson
     * @return
     * @throws ServiceException
     */
    public Person updateInProcess(Integer incomeYear, Person incoming, boolean savePerson) throws ServiceException;
    
    /**
     * Returns whether a financials data can added based on the incoming financial data.
     * 
     * @param incomeYear
     * @param person
     * @return
     * @throws ServiceException
     */
    public boolean canFinancialsBeAdded(Integer incomeYear, Person person)
            throws ServiceException;
    
    /**
     * Returns whether a financials data can be edited based on the incoming financial data.
     * 
     * @param incomeYear
     * @param person
     * @return
     * @throws ServiceException
     */
    public boolean canFinancialsBeUpdated(Integer incomeYear, Person person)
    	throws ServiceException;

    /**
     * Returns whether a networth data can be edited based on the income year in incoming financial data.
     * @param incomeYear
     * @param person
     * @return
     * @throws ServiceException
     */
    public boolean canNetWorthBeUpdated(Integer incomeYear, Person person)
		throws ServiceException;
    
    /**
     * Get the GMT Threshold information given a income year and zip code and number of dependents.
     * This method is same as calling getGMTThreshold method with 0 extra dependent.
     * This method call will return a GMT Threshold Object that will contain data for up to 
     * seven dependents.
     * 
     * This method should be use instead of getting GMTThreshold  Object directly from the look up service
     * because this service will calculate data that may not be available in case GMTThreshold Object 

     * @param incomeYear
     * @param zipCode
     * @return GMTThreshold
     * @throws ServiceException
     */
    public GMTThreshold getGMTThreshold(Integer incomeYear, String zipCode)
	throws ServiceException;

    /**
     * Get the GMT Threshold information given a income year and zip code and extra dependents.
     * This method is same as calling getGMTThreshold method with 0 extra dependent.
     * Extra dependents is a number of dependents more than 7. For example, if the total number of 
     * dependents are 10, this method should be called with extraDependent = 3.
     * If extra dependent is 0, GMTThreshold Object will be returned which contain threshold information
     * for up to seven dependents.
     * 
     * This method should be use instead of getting GMTThreshold  Object directly from the look up service
     * because this service will calculate data that may not be available in case GMTThreshold Object 
     * 
     * @param incomeYear
     * @param zipCode
     * @param numberOfDependents
     * @return GMTThreshold
     * @throws ServiceException
     */
    public GMTThreshold getGMTThreshold(Integer incomeYear, String zipCode, int extraDependents)
	throws ServiceException;

    /**
     * Get the income threshold for a given income year.
     * This method should be use instead of getting IncomeThreshold Object directly from the look up service
     * because this service will calculate data that may not be available in case IncomeThreshold Object 
     * is loaded directly.
     * 
     * @param incomeYear
     * @return IncomeThreshold
     * @throws ServiceException
     */
    public IncomeThreshold getIncomeThreshold(Integer incomeYear) 
    throws ServiceException;

    /**
     * Get the income threshold for a given income year.
     * This method is same as calling getIncomeThreshold method with 0 extra dependent.
     * Extra dependents is a number of dependents more than 7. For example, if the total number of 
     * dependents are 10, this method should be called with extraDependent = 3.
     * If extra dependent is 0, IncomeThreshold Object will be returned which contain threshold information
     * for up to seven dependents.

     * This method should be use instead of getting IncomeThreshold Object directly from the look up service
     * because this service will calculate data that may not be available in case IncomeThreshold Object 
     * is loaded directly.
     * 
     * @param incomeYear
     * @return IncomeThreshold
     * @throws ServiceException
     */
    public IncomeThreshold getIncomeThreshold(Integer incomeYear, int extraDependents) 
    throws ServiceException;
	
	//Financials History methods
    public Person getHistoricalFeeBasisInfo(EntityKey key)
        throws ServiceException;

	public Set getFinancialAssessmentHistoryChangeTimes(EntityKey entityKey, Integer incomeYear) 
		throws ServiceException;

	public HistoricalInfo getFinancialAssessmentHistoryByChangeTime(FinancialsChangeEvent changeEvent) 
		throws ServiceException;

    public Set getDependentHistoryChangeTimes(EntityKey entityKey, Integer incomeYear)
    	throws ServiceException;

    public HistoricalInfo getDependentHistoryByChangeTime(FinancialsChangeEvent changeEvent)
            throws ServiceException;

    /**
     * Update the person signature for a person.
     * 
     * @param person
     * @return
     * @throws ServiceException
     */
    public Person updatePersonSignature(Person person) throws ServiceException;    
    
    /**
     * Process and persist the person's signature images submitted by a batch process.
     * 
     * @param person
     * @return
     * @throws ServiceException
     */
    public Person processPersonSignature(Person person) throws ServiceException;    
    
    /**
     * Adjudicate Income test
     * @param incomeYear
     * @param incoming
     * @param status
     * @return
     * @throws ServiceException
     */
    public Person updatePendingAdjudicationStatus(Integer incomeYear, Person incoming, MeansTestStatus status) throws ServiceException;

    public Person processFutureDatedTestFlag(PersonEntityKey key, Integer incomeYear) throws ServiceException;
    
    public Person convertOrReverseESRIncomeTest(Person incoming, IVMFinancialInfo ivmFinancialInfo) throws ServiceException;
    public IncomeTest findConversionBaseTest(IncomeTest incometest) throws ServiceException;
    public List getIncomeTestStatusesForIVMReversal(Serializable key) throws ServiceException;
    
    //CCR12575
    public void validateFinancialsUpdate(Integer incomeYear, Person onFile) throws ServiceException;
}