/********************************************************************
 * Copyright  2004-2005 VHA. All rights reserved
 ********************************************************************/
package gov.va.med.esr.service;

// Java classes
import java.io.Serializable;
import java.util.Date;
import java.util.List;
import java.util.Map;
import java.util.Set;

import gov.va.med.fw.model.EntityKey;
import gov.va.med.fw.service.ServiceException;

import gov.va.med.esr.common.model.ee.EGTProcessStatistic;
import gov.va.med.esr.common.model.ee.EGTSetting;
import gov.va.med.esr.service.impl.ChangeEvent;
import gov.va.med.esr.service.impl.HistoricalInfo;

/**
 * Services related to EGT settings.
 * <p>
 * Project: Common
 *
 * @author DNS   LEV
 * @author Andrew Pach
 * @version 3.0
 */
public interface EGTService extends Serializable
{
    /**
     * Gets the currently active EGT setting.
     * @return The currently active EGT setting.
     * @throws ServiceException if any problems were encountered.
     */
    public EGTSetting getCurrentEGTSetting() throws ServiceException;

    /**
     * Gets a set of all future EGT settings.  These settings may be enabled or disabled.
     * @return The set of future EGT settings.
     * @throws ServiceException if any problems were encountered.
     */
    public Set getFutureEGTSettings() throws ServiceException;

    /**
     * Gets the EGTSetting that was active as of the given date.
     */
    public EGTSetting getActiveEGTSetting(Date asOf) throws ServiceException;
    
    /**
     * Gets a specific EGT setting based on an entity key.
     * @param key The key of the EGT setting to retrieve.
     * @return The EGT setting based on the specified entity key.
     * @throws ServiceException if any problems were encountered.
     */
    public EGTSetting getEGTSetting(EntityKey key) throws ServiceException;

    /**
     * Adds or updates an EGT setting.
     *
     * @param egt The EGT setting to add or update.
     * @return The added or updated EGT setting.
     * @throws ServiceException if any problems were encountered.
     */
    public EGTSetting updateEGTSetting(EGTSetting egt) throws ServiceException;

    /**
     * Stops a currently executing EGT batch job.
     * @return The current EGT setting that contains the statistical records associated with the
     * previously scheduled or currently executing batch job.
     * @throws ServiceException if any problems were encountered.
     */
    public EGTSetting stopEGTProcessing() throws ServiceException;

    /**
     * Gets a list of history change times for EGT setting records.
     * @return The list of change times.
     * @throws ServiceException if any problems were encountered.
     */
    public Set getEGTSettingHistoryChangeTimes() throws ServiceException;

    /**
     * Gets EGT setting historical information for a specific change event.  The change event should
     * be obtained using the getEGTSettingHistoryChangeTimes method call.
     * @param changeEvent The change event used to determine the point of time to retrieve historical
     * information for.
     * @return The historical information for the specified change time.
     * @throws ServiceException if any problems were encountered.
     */
    public HistoricalInfo getEGTSettingHistoryByChangeTime(ChangeEvent changeEvent) throws ServiceException;
    
    /**
     * Process a new EGTSetting. The list of veteran id's meet
     * the criteria for having enrollment recalculated.
     * 
     * @param veteranIds
     * @param newSetting
     * @throws ServiceException if any problems were encountered.      
     */
    public void updateEnrollmentWithNewEGTSetting(List veteranIds, EGTSetting newSetting, Map siteStats) throws ServiceException;
    
    /**
     * Persists the given EGTProcessStatistic.
     * 
     * @param processStatistic
     * @return EGTSetting - egt setting of the saved statistics
     * @throws ServiceException
     */
    public void saveEGTProcessStatistics(EGTProcessStatistic processStatistic) throws ServiceException;
    
    /**
     * Save the given EGT Setting as a current EGTSetting.
     * To make it current, existing current EGTSetting is disabled and not current by
     * setting its current property to false and disable property to true.
     * This method also sets the current property on a given EGTSetting Object to true
     * and disable property to false.
     * Reference to the existing current EGTSetting Object is set in this given EGTSetting Object.
     * 
     * @param egt - The EGTSetting Object to be current
     * @return EGTSetting - The newly persisted current EGTSetting 
     * @throws ServiceException
     */
    public EGTSetting saveAsCurrentEGTSetting(EGTSetting egt) throws ServiceException;
}