package gov.va.med.esr.service;

import java.io.Serializable;
import java.math.BigDecimal;
import java.util.ArrayList;

import gov.va.med.esr.common.model.comms.CommsTemplate;
import gov.va.med.esr.common.model.comms.Correspondence;
import gov.va.med.esr.common.model.lookup.ComLetterTemplateType;
import gov.va.med.esr.common.model.lookup.ComLetterTemplateType.Code;
import gov.va.med.esr.common.model.lookup.CorrespondenceStatus;
import gov.va.med.esr.common.model.person.Person;
import gov.va.med.fw.service.ServiceException;



public interface CorrespondenceService extends Serializable {

	/**
	 * Inserts new CMS mail entry in pending mail queue for form number identified
	 *
     * @param correspondence, the person correspondence entry
     * @param formNumber, the letter template code to be mailed
     */
	public void triggerNewCMSCorrespondence(Correspondence correspondence, ComLetterTemplateType.Code formNumber) throws ServiceException;


	/**
	 * Updates the correspondence entry to status indicated
	 * and error message if necessary
	 * only use for ACA letters
     */
	public void updateACACorrespondence(Correspondence correspondence, CorrespondenceStatus.Code status, String errMsg) throws ServiceException;


	/**
	 * Finds all records for specified person that are pending mail correspondence to CMS for specified formNumber
	 *
     */
	public ArrayList<Correspondence> getPendingPersonCorrespondence(BigDecimal personId, Code formNumber) throws ServiceException;

	/**
	 * Finds all records for specified person that are in sent status to CMS for specified formNumber
	 *
     */
	public ArrayList<Correspondence> getSentPersonCorrespondence(BigDecimal personId, Code formNumber) throws ServiceException;


	/**
	 * Gets all correspondence for person, in all status, for specified list of form numbers
	 *
     */
	public ArrayList<Correspondence> getPersonCorrespondence(BigDecimal personId, ArrayList<Code> formNumbers) throws ServiceException;


	/**
	 * Utility method to lookup the correspondence letter form number by code
	 *
     */
	public CommsTemplate getTemplateByForm(Code formNumber) throws ServiceException;


	/**
	 * Returns true if mail should be not be auto-triggered baesd on communications business rules
	 * Mail generation jobs should set status to EnrollmentReject if True for specified person
	 *
     */
	public boolean shouldSuppressMailCorrespondence(Person person) throws ServiceException;


	/**
	 * Adds the appropriate letter type to accompany IRS 1095B mailing
	 * Letter is associated at time of mail generation of the tax form
	 *
     */
	public void associateACALetterType(Correspondence correspondence, ComLetterTemplateType.Code formNumber) throws ServiceException;

	/**
	 * Remail utility to put previous form back in pending mail queue
	 *
     */
	public void remailCorrespondence(BigDecimal correspondenceId) throws ServiceException;


	/**
	 * Retrieves correspondence detail record including historical address and historical mail status
	 * For display on UI detail
	 * Do not use this method for bulk processing of correspondence entries
	 *
     */
	public Correspondence getCorrespondenceWithHistory(BigDecimal correspondenceId) throws ServiceException;



	/**
	 * Cancel pending mail
	 * Triggered by UI before the records is sent to CMS
	 *
     */
	public void cancelCorrespondence(BigDecimal correspondenceId) throws ServiceException;

}
