package gov.va.med.esr.service;

// Java Classes
import java.util.List;

// Framework Classes
import gov.va.med.fw.service.ServiceException;
import gov.va.med.fw.service.pagination.PaginatedSearchService;
import gov.va.med.fw.service.pagination.SearchQueryInfo;
import gov.va.med.fw.model.EntityKey;

// ESR Classes
import gov.va.med.esr.common.clock.Clock.Type;
import gov.va.med.esr.common.model.comms.CommsLogEntry;
import gov.va.med.esr.common.model.comms.UndeliverableMailInfo;
import gov.va.med.esr.common.model.person.Person;

/**
 * The Communications Log Service.
 * <pre>
 * NOTE: the service is obtained from spring framework by calling
 *          broker.getComponent(<b>"commsLogService"</b>);
 *       Please see comms/business_services.xml for details
 * </pre>
 *
 * @author George Tsai & Andrew Pach
 * @version 3.0
 */
// 3.6 10960 Removed all references to Handbook. They have been moved to the Handbook Service
public interface CommsLogService extends PaginatedSearchService
{
    /**
     * Gets a specific CommsLogEntry based on an entity key.
     *
     * @param key The key of the CommsLogEntry to retrieve.
     *
     * @return The CommsLogEntry based on the specified entity key.
     * @throws ServiceException if any problems were encountered.
     */
    public CommsLogEntry getCommsLogEntry(EntityKey key) throws ServiceException;

    /**
     * Find a letter log entry by barcode id (log id)
     *
     * @param barcode the barcode id (log id)
     *
     * @return a CommsLogEntry
     */
    public CommsLogEntry findByBarcode(String barcode) throws ServiceException;

    /**
     * Update letter log entry
     *
     * @param log
     */
    public void update(CommsLogEntry log) throws ServiceException;

    /**
     * Insert a new letter log entry
     *
     * @param log the letter log entry
     */
    public void insert(CommsLogEntry log) throws ServiceException;

    /**
     * find a list of letter log entries by person id
     *
     * @param personId the person identifier
     *
     * @return a list of CommsLogEntry
     */
    public List findLogEntriesByPersonId(String personId) throws ServiceException;
    /**
     * Gets an UndeliverableMailInfo object based on a barcode String.
     *
     * @param barcode The scanned or manually entered barcode from an undeliverable letter.
     *
     * @return The UndeliverableMailInfo
     * @throws ServiceException if any problems were encountered.
     */
    public UndeliverableMailInfo findUndeliverableMailByBarcode(String barcode)
        throws ServiceException;

    /**
     * Searchs and returns a list of UndeliverableMailInfo objects based on the passed in search
     * criteria.
     *
     * @param searchQueryInfo The search criteria
     *
     * @return The list of UndeliverableMailInfo objects
     * @throws ServiceException if any problems were encountered.
     */
    public List search(SearchQueryInfo searchQueryInfo)
        throws ServiceException;

    /**
     * Processes the undeliverable mail based on a list of CommsLogEntry objects.
     *
     * @param commsLogEntries The list of CommsLogEntry objects to process.
     *
     * @throws ServiceException if any problems were encountered.
     */
    public void processUndeliverableMail(List commsLogEntries)
        throws ServiceException;

    /**
     * Business service for Manage Communications Log use case. Processes
     * changes to comms status and comments. Saves log.
     * 
     * @param log
     * @return commsLogEntry
     * @throws ServiceException
     */
    public CommsLogEntry updateCommunicationLog(CommsLogEntry log) throws ServiceException;
    
    /**
     * This is entry point for Manage Mailing Responses use case.
     * 
     * @param responses
     * @throws ServiceException
     */
    public void  handleMailingResponses(List responses) throws ServiceException;  
    
    public void  handleMailingResponses(CommsLogEntry log) throws ServiceException;    
    /**
     * Trigger automatic letter requests
     * 
     * @param person
     * @param clockType
     * @param subject
     * @throws ServiceException
     */
    public void processAutomaticLetterRequests(Person person, Type clockType, String subject) throws ServiceException;
}
