/********************************************************************
 * Copyright  2004 VHA. All rights reserved
 ********************************************************************/
package gov.va.med.esr.service;

// Java Classes
import java.util.Set;
import java.util.List;
import java.math.BigDecimal;

// Framework Classes
import gov.va.med.fw.model.EntityKey;
import gov.va.med.fw.service.ServiceException;
import gov.va.med.fw.service.pagination.PaginatedSearchService;

// ESR Classes
import gov.va.med.esr.common.model.ee.IncomingMessageInfo;
import gov.va.med.esr.common.model.ee.VerificationInfo;
import gov.va.med.esr.common.model.lookup.VAFacility;
import gov.va.med.esr.common.model.messaging.ArchivedMessageLogEntry;
import gov.va.med.esr.common.model.messaging.ArchivedMessageLogEntryLite;
import gov.va.med.esr.common.model.messaging.MessageLogEntry;
import gov.va.med.esr.common.model.messaging.MessageLogEntryLite;
import gov.va.med.esr.common.model.messaging.SiteIdentity;
import gov.va.med.esr.common.model.person.Person;
import gov.va.med.esr.common.model.person.id.PersonEntityKey;

/**
 * @author Carlos Ruiz
 * @version 1.0
 */
public interface ArchivedMessagingService extends PaginatedSearchService {
    
    /**
     * Method to generate a control identifier that is used to uniquely
     * identify a message. 
     * 
     * @return The control identifier for the message.
     * @throws ServiceException
     */
    public String generateControlIdentifier() throws ServiceException;

    /**
    * @param person
    * @param vaFacility
    * @return
    * @throws ServiceException
    */
   public SiteIdentity getIdentity(Person person, VAFacility vaFacility) throws ServiceException;
   /**
    * Get Matching Person Identity Traits from PSIM
    * @param identityTraits
    * @return
    * @throws ServiceException
    */
   public Set getMatchingTraits(PersonIdentityTraits identityTraits) throws ServiceException;
   /**
    * @param identifier
    * @return
    * @throws ServiceException
    */
   public ArchivedMessageLogEntry getMessageLogEntryById(BigDecimal identifier) throws ServiceException;
   
   
   /**
    * @param identifier
    * @param shouldPopulatePerson
    * @return
    * @throws ServiceException
    */
   public ArchivedMessageLogEntryLite getMessageLogEntryLiteById(BigDecimal identifier) throws ServiceException;

   
    /**
    * @param controlIdentifier
    * @return
    * @throws ServiceException
    */
   public ArchivedMessageLogEntry getMessageLogEntry(String controlIdentifier) throws ServiceException;

   
   /**
    * @param controlIdentifier
    * @return
    * @throws ServiceException
    */
   public ArchivedMessageLogEntry getMessageLogEntryByBatchControlNumber(String controlIdentifier) throws ServiceException;

    /**
     * Gets a list of SiteIdentity records for a Person.
     * @param person The person
     * @return The sites for the person or an empty set if no sites were found
     * @throws ServiceException If any problems were encountered
     */
    public Set findSitesOfRecord(Person person) throws ServiceException;

    /**
    * @param entry
    * @throws ServiceException
    */
   public void logMessage(MessageLogEntry entry) throws ServiceException;


    /**
     * Trigger a QRYZ07 message instantly.
     *
     * @param person The person whose data is being queries
     * @param facility The facility to send the message to
     * @param incomeYear The income year to query on
     * @throws ServiceException
     */
    public void triggerQRYZ07(Person person, VAFacility facility, Integer incomeYear) throws ServiceException;

    /**
     * Trigger a QRYZ07 message
     *
     * @param person The person whose data is being queries
     * @param facility The facility to send the message to
     * @param incomeYear The income year to query on
     * @param handleAfterCommit Determines if the event should be handled after the current TX commits
     * @throws ServiceException
     */
    public void triggerQRYZ07(Person person, VAFacility facility, Integer incomeYear, boolean handleAfterCommit) throws ServiceException;
    
	/**
	 * Find the Facility from whih the last message has been received and successfully processed
	 *
	 * @param personId
	 * @return
	 * @throws ServiceException
	 */
	public VAFacility findSiteLastTransmittedMsg(EntityKey personId) throws ServiceException;
    
    
    /**
     * Retransmits a message.
     * 
     * @param identifier message log entry id.
     * @throws ServiceException
     */
    public void triggerRetransmission(BigDecimal identifier) throws ServiceException;
    
	/** Specific check to determine if already processed an inbound message */
	public boolean hasProcessedInboundMessage(String messageControlNumber, String stationNumber) throws ServiceException;  
	
    /**
     * find message log entries for VOA that has AA Ack sent, given a person id, in decending datetime order (latest first)
     * 
     * @param identifier person id.
     * @return list of log entries
     * @throws ServiceException
     */	
	public List findVoaAaAckLogEntryByPersonId(EntityKey personId) throws ServiceException; 	
	
}
