/********************************************************************
 * Copyright  2005 VHA. All rights reserved
 ********************************************************************/
package gov.va.med.esr.common.util;

// Java Classes
import java.util.Iterator;
import java.util.Set;

// Library Classes

// Framework Classes

// ESR Classes
import gov.va.med.esr.common.model.lookup.NameType;
import gov.va.med.esr.common.model.person.Name;
import gov.va.med.esr.common.model.person.Person;
import gov.va.med.fw.util.StringUtils;
import gov.va.med.fw.security.UserPrincipal;

/**
 * @author DNS   chenb
 * @author Andrew Pach
 * @version 3.0
 *          <p/>
 *          Helper class for working with Names.
 */
public class CommonNameHelper
{
    /**
     * Gets the legal name from the person.
     * @param person The person
     * @return The legal name for the person or null if no legal name was found.
     */
    public static Name getLegalName(Person person)
    {
        // If no person was passed in, just return null
        if (person == null)
        {
            return null;
        }

        // Get the list of names and search through them
        Set names = person.getNames();
        if (names != null && names.size() > 0)
        {
            for (Iterator iter = names.iterator(); iter.hasNext();)
            {
                Name name = (Name)iter.next();
                NameType type = name.getType();
                String code = (type != null) ? type.getCode() : null;

                if (NameType.LEGAL_NAME.getName().equalsIgnoreCase(code))
                {
                    // We found the legal name so return it
                    return name;
                }
            }
        }

        // If no name was found, return null
        return null;
    }

    /**
     * Formats a name in the format of "lastName suffix, prefix firstName middleName".
     *
     * @param principal The user principal to format
     *
     * @return The formatted name
     */
    public static String formatNameLastNameFirst(UserPrincipal principal)
    {
        String formattedName = formatNameLastNameFirst(null, principal.getGivenName(), principal.getMiddleName(),
            principal.getFamilyName(), null);
        return StringUtils.isBlank(formattedName) ? principal.getName() : formattedName;
    }

    /**
     * Formats a name in the format of "lastName suffix, prefix firstName middleName".
     *
     * @param name The name to format
     *
     * @return The formatted name
     */
    public static String formatNameLastNameFirst(Name name)
    {
        return formatNameLastNameFirst(name.getPrefix(), name.getGivenName(), name.getMiddleName(),
            name.getFamilyName(), name.getSuffix());
    }

    /**
     * Formats a name in the format of "lastName suffix, prefix firstName middleName".
     *
     * @param prefix The name prefix
     * @param firstName The first name
     * @param middleName The middle name0
     * @param lastName The last name
     * @param suffix The name suffix
     *
     * @return The formatted name
     */
    public static String formatNameLastNameFirst(String prefix, String firstName, String middleName, String lastName,
        String suffix)
    {
        // Build "lastName suffix"
        StringBuffer lastPart = new StringBuffer();
        appendNamePiece(lastPart, lastName);
        appendNamePiece(lastPart, suffix);

        // Build "prefix firstName middleName"
        StringBuffer firstPart = new StringBuffer();
        appendNamePiece(firstPart, prefix);
        appendNamePiece(firstPart, firstName);
        appendNamePiece(firstPart, middleName);

        // Create result from lastPart, firstPart
        StringBuffer result = new StringBuffer(lastPart.toString());
        if ((result.length() > 0) && (firstPart.length() > 0))
        {
            result.append(", " + firstPart.toString());
        }

        // Return the result
        return result.toString();
    }

    /**
     * Formats a name in the format of "prefix firstName middleName lastName suffix".
     *
     * @param principal The user principal to format
     *
     * @return The formatted name
     */
    public static String formatNameFirstNameFirst(UserPrincipal principal)
    {
        String formattedName = formatNameFirstNameFirst(null, principal.getGivenName(), principal.getMiddleName(),
            principal.getFamilyName(), null);
        return StringUtils.isBlank(formattedName) ? principal.getName() : formattedName;
    }

    /**
     * Formats a name in the format of "prefix firstName middleName lastName suffix".
     *
     * @param name The name to format
     *
     * @return The formatted name
     */
    public static String formatNameFirstNameFirst(Name name)
    {
        return formatNameFirstNameFirst(name.getPrefix(), name.getGivenName(), name.getMiddleName(),
            name.getFamilyName(), name.getSuffix());
    }

    /**
     * Formats a name in the format of "prefix firstName middleName lastName suffix".
     *
     * @param prefix The name prefix
     * @param firstName The first name
     * @param middleName The middle name0
     * @param lastName The last name
     * @param suffix The name suffix
     *
     * @return The formatted name
     */
    public static String formatNameFirstNameFirst(String prefix, String firstName, String middleName, String lastName,
        String suffix)
    {
        // Build the name
        StringBuffer result = new StringBuffer();
        appendNamePiece(result, prefix);
        appendNamePiece(result, firstName);
        appendNamePiece(result, middleName);
        appendNamePiece(result, lastName);
        appendNamePiece(result, suffix);

        // Return the result
        return result.toString();
    }

    /**
     * Appends a piece of the name to the buffer.  If the buffer already contains part of the name,
     * a space is added before appending the namePiece.
     *
     * @param buffer The current buffer
     * @param namePiece The part of the name to add
     */
    public static void appendNamePiece(StringBuffer buffer, String namePiece)
    {
        if (!StringUtils.isBlank(namePiece))
        {
            if (buffer.length() > 0)
            {
                buffer.append(" ");
            }
            buffer.append(namePiece);
        }
    }
}
