package gov.va.med.esr.common.util;

// Java Classes
import java.text.SimpleDateFormat;
import java.util.Calendar;
import java.util.GregorianCalendar;
import java.util.Date;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

// Framework Classes
import gov.va.med.fw.util.StringUtils;

// ESR Classes
import gov.va.med.esr.common.infra.ImpreciseDate;
import gov.va.med.esr.common.infra.ImpreciseDateUtils;
import org.apache.commons.lang.Validate;
import org.apache.commons.lang.time.DateUtils;
/**
 * @author Muddaiah ranga
 * @author Andrew Pach
 * @version 3.0
 */
public class CommonDateUtils
{
    // Various date formatting constants
    public final static String MM = "MM";
    public final static String MMDD = "MM/dd";
    public final static String YYYY = "yyyy";
    public final static String MMYYYY = "MM/yyyy";
    public final static String MMDDYYYY = "MM/dd/yyyy";
    public final static String MMDDYYYYHHMM = "MM/dd/yyyy h:mma";
    public final static String MMDDYYYYHHMMSS = "MM/dd/yyyy h:mm:ssa";
    public final static String MMDDYYYYHHMM_EDIT = "MM/dd/yyyy HH:mm";
    public final static String MMDDYYYYHHMMSS_EDIT = "MM/dd/yyyy HH:mm:ss";

    // Pattern Matching constants for parsing dates
    private static final String PATTERN_MM = "([0-9]{1,2})()()()()()"; // MM
    private static final String PATTERN_MMDD = "([0-9]{1,2})/([0-9]{1,2})()()()()"; // MM/dd
    private static final String PATTERN_YYYY = "()()([0-9]{4})()()()"; // yyyy
    private static final String PATTERN_MMYYYY = "([0-9]{1,2})/()([0-9]{4})()()()"; // MM/yyyy
    private static final String PATTERN_MMDDYYYY =
        "([0-9]{1,2})/([0-9]{1,2})/([0-9]{4})()()()"; // MM/dd/yyyy
    private static final String PATTERN_MMDDYYYYHHMM =
        "([0-9]{1,2})/([0-9]{1,2})/([0-9]{4})\\s([0-9]{1,2}):([0-9]{1,2})()"; // MM/dd/yyyy HH:mm
    private static final String PATTERN_MMDDYYYYHHMMSS =
        "([0-9]{1,2})/([0-9]{1,2})/([0-9]{4})\\s([0-9]{1,2}):([0-9]{1,2}):([0-9]{1,2})"; // MM/dd/yyyy HH:mm:ss
    private static final String[] IMPRECISE_DATE_PATTERNS =
        new String[]{PATTERN_MM, PATTERN_MMDD, PATTERN_YYYY, PATTERN_MMYYYY, PATTERN_MMDDYYYY,
            PATTERN_MMDDYYYYHHMM, PATTERN_MMDDYYYYHHMMSS};
    private static final String[] PRECISE_DATE_PATTERNS =
        new String[]{PATTERN_MMDDYYYY, PATTERN_MMDDYYYYHHMM, PATTERN_MMDDYYYYHHMMSS};
    private static final int PATTERN_GROUP_MONTH = 1;
    private static final int PATTERN_GROUP_DAY = 2;
    private static final int PATTERN_GROUP_YEAR = 3;
    private static final int PATTERN_GROUP_HOUR = 4;
    private static final int PATTERN_GROUP_MINUTE = 5;
    private static final int PATTERN_GROUP_SECOND = 6;

    //VFA discontinue Net Worth on and after Income Year 2009
	private static Integer YEAR_2009= new Integer("2009"); //VFA
    
    public static boolean isIncomeYear2009OrLatter(Integer incomeYear)
    {
    	if (incomeYear != null &&
    		incomeYear.compareTo(YEAR_2009) >= 0)
    	{
    		return true;
    	}
    	return false;
    }
    /**
     * Formats a date with the specified format or a default format if null is passed in for the
     * format. The default format does not include a time.
     *
     * @param date The date to format.
     * @param format An optional format.
     *
     * @return The formatted date.
     */
    public static String format(Date date, String format)
    {
        if (date == null)
        {
            return null;
        }
        if (StringUtils.isBlank(format))
        {
            format = MMDDYYYY;
        }
        SimpleDateFormat sdf = new SimpleDateFormat(format);
        sdf.setLenient(false);
        return sdf.format(date);
    }

    /**
     * Formats a date with the specified format or a default format if null is passed in for the
     * format. The default format includes a time.
     *
     * @param date The date to format.
     * @param format An optional format.
     *
     * @return The formatted date.
     */
    public static String formatWithTime(Date date, String format)
    {
        if (StringUtils.isBlank(format))
        {
            format = MMDDYYYYHHMMSS;
        }
        return format(date, format);
    }

    /**
     * Formats an imprecise date with the specified format or a default format if null is passed in
     * for the format. The default format does not include a time.
     *
     * @param date The date to format.
     * @param format An optional format.
     *
     * @return The formatted date.
     */
    public static String format(ImpreciseDate date, String format)
    {
        if (date == null)
        {
            return null;
        }
        if (date.isPrecise())
        {
            return format(date.getDate(), (StringUtils.isEmpty(format)) ? MMDDYYYY : format);
        }
        else
        {
            String string = date.getString();
            String year = string.substring(0, 4);
            String month = date.isMonthPrecise() ? string.substring(4, 6) : null;
            String day = date.isDayPrecise() ? string.substring(6, 8) : null;
            if (month != null)
            {
                if (day != null)
                {
                    return month + "/" + day + "/" + year;
                }
                else
                {
                    return month + "/" + year;
                }
            }
            else
            {
                return year;
            }

        }
    }

    /**
     * Formats an imprecise date with the specified format or a default format if null is passed in
     * for the format. The default format includes a time if the date is precise.  Otherwise, no
     * time will be included.
     *
     * @param date The date to format.
     * @param format An optional format.
     *
     * @return The formatted date.
     */
    public static String formatWithTime(ImpreciseDate date, String format)
    {
    	if (date == null) {
    		return null;
    	}
    	
        if (date.isPrecise())
        {
            return formatWithTime(date.getDate(), format);
        }
        return format(date, format);
    }

    public static String formatForDisplayWithTime(ImpreciseDate date, String format)
    {
    	if (date == null) {
    		return null;
    	}
    	
        if (StringUtils.isBlank(format))
        {
            if (date.isTimePresent())
            {
            	if (date.isSecPresent()) {
            		format = MMDDYYYYHHMMSS;
            	}
            	else {
            		format = MMDDYYYYHHMM;
            	}
            }
            else
            {
                format = MMDDYYYY;
            }
        }
        return formatWithTime(date, format);
    }
    
    /**
     * Formats an imprecise date with the specified format or a default "editable" format if null is
     * passed in for the format.  The default format includes a time if the date is precise.
     * Otherwise, no time will be included.
     *
     * @param date The date to format.
     * @param format An optional format.
     *
     * @return The formatted date.
     */
    public static String formatForEdit(ImpreciseDate date, String format)
    {
    	if (date == null) {
    		return null;
    	}
    	
        if (StringUtils.isBlank(format))
        {
            if (date.isTimePresent())
            {
            	if (date.isSecPresent()) {
            		format = MMDDYYYYHHMMSS_EDIT;
            	}
            	else {
            		format = MMDDYYYYHHMM_EDIT;
            	}
            }
            else
            {
                format = MMDDYYYY;
            }
        }
        return formatWithTime(date, format);
    }

    /**
     * Formats a precise date with the specified format or a default "editable" format if null is
     * passed in for the format.  The default format includes a time if the time portion of the date
     * includes any part that is not 0.  Otherwise, no time will be included.
     *
     * @param date The date to format.
     * @param format An optional format.
     *
     * @return The formatted date.
     */
    public static String formatForEdit(Date date, String format)
    {
        if (StringUtils.isBlank(format))
        {
            if (isTimePresent(date))
            {
                format = MMDDYYYYHHMM_EDIT;
            }
            else
            {
                format = MMDDYYYY;
            }
        }
        return formatWithTime(date, format);
    }

    /**
     * Determines if a time is present on a date.  A time is considered present if any portion
     * of the time isn't 0.
     *
     * @param date The date to check.
     * @return True if a time is present on the date or false if not.
     */
    public static boolean isTimePresent(Date date)
    {
        Calendar cal = Calendar.getInstance();
        cal.setTime(date);
        if ((cal.get(Calendar.HOUR_OF_DAY) == 0) &&
            (cal.get(Calendar.MINUTE) == 0) &&
            (cal.get(Calendar.SECOND) == 0))
        {
            return false;
        }
        else
        {
            return true;
        }
    }

    /**
     * Determines if a date to check is between two other dates (inclusive).
     *
     * @param dateToCheck The date to check
     * @param lowerDate The lower date to check against.  If null is passed, no lower bound will be
     * used.
     * @param upperDate The upper date to check against.  If null is passed, no upper bound will be
     * used.
     *
     * @return True if the date to check is between the other two dates or false if not.
     */
    public static boolean isDateBetween(Date dateToCheck, Date lowerDate, Date upperDate)
    {
        // Get the time to check
        long checkTime = dateToCheck.getTime();

        // Check the lower date if present
        if (lowerDate != null)
        {
            long lowTime = lowerDate.getTime();
            if (checkTime < lowTime)
            {
                return false;
            }
        }

        // Check the upper date is present
        if (upperDate != null)
        {
            long upperTime = upperDate.getTime();
            if (checkTime > upperTime)
            {
                return false;
            }
        }

        // The date is between the other dates
        return true;
    }

    /**
     * Determines if a date to check is between two other dates (inclusive).
     *
     * @param dateToCheck The date to check
     * @param lowerDate The lower date to check against.  If null is passed, no lower bound will be
     * used.
     * @param upperDate The upper date to check against.  If null is passed, no upper bound will be
     * used.
     *
     * @return True if the date to check is between the other two dates or false if not.
     */
    public static boolean isDateBetween(ImpreciseDate dateToCheck, ImpreciseDate lowerDate,
        ImpreciseDate upperDate)
    {
        // Check the lower date if present
        if (lowerDate != null)
        {
            if (dateToCheck.compareTo(lowerDate) < 0)
            {
                return false;
            }
        }

        // Check the upper date is present
        if (upperDate != null)
        {
            if (dateToCheck.compareTo(upperDate) > 0)
            {
                return false;
            }
        }

        // The date is between the other dates
        return true;
    }

    /**
     * Determines if a date is after another date.
     *
     * @param dateToCheck The date to check
     * @param compareToDate The date to compare to
     *
     * @return True if the date to check is after the compare to date.  Otherwise, false will be returned.
     * If a compare to date is null, false is returned.
     */
    public static boolean isDateAfter(ImpreciseDate dateToCheck, ImpreciseDate compareToDate)
    {
        if (compareToDate != null)
        {
            if (dateToCheck.compareTo(compareToDate) > 0)
            {
                return true;
            }
        }
        return false;
    }

    /**
     * Determines if a date is before another date.
     *
     * @param dateToCheck The date to check
     * @param compareToDate The date to compare to
     *
     * @return True if the date to check is before the compare to date.  Otherwise, false will be returned.
     * If a compare to date is null, false is returned.
     */
    public static boolean isDateBefore(ImpreciseDate dateToCheck, ImpreciseDate compareToDate)
    {
        if (compareToDate != null)
        {
            if (dateToCheck.compareTo(compareToDate) < 0)
            {
                return true;
            }
        }
        return false;
    }

    /**
     * Gets the date that would be at the beginning of the day based on the passed in date.  This is done
     * by setting the time to 00:00:00.
     *
     * @param date the date
     * @return the date at the beginning of the day
     */
    public static ImpreciseDate getBeginningOfDay(ImpreciseDate date)
    {
        if (date == null)
        {
            return null;
        }
        Calendar beginDate = date.getCalendar();
        beginDate.set(Calendar.HOUR_OF_DAY, 0);
        beginDate.set(Calendar.MINUTE, 0);
        beginDate.set(Calendar.SECOND, 0);
        return new ImpreciseDate(beginDate);
    }

    /**
     * Gets the date that would be at the end of the day based on the passed in date.  This is done
     * by setting the time to 23:59:59.
     *
     * @param date the date
     * @return the date at the end of the day
     */
    public static ImpreciseDate getEndOfDay(ImpreciseDate date)
    {
        if (date == null)
        {
            return null;
        }
        Calendar endDate = date.getCalendar();
        endDate.set(Calendar.HOUR_OF_DAY, 23);
        endDate.set(Calendar.MINUTE, 59);
        endDate.set(Calendar.SECOND, 59);
        return new ImpreciseDate(endDate);
    }

    /**
     * Gets a precise date from the passed in date as a String.
     *
     * @param dateString The date in the format of a string.
     * @param timePermitted If true, an optional time component is considered valid.
     *
     * @return The parsed date or null if no date was passed in.
     * @throws IllegalArgumentException if the date is invalid.
     */
    public static Date getDate(String dateString, boolean timePermitted)
    {
        // Check for empty string
        if (StringUtils.isEmpty(dateString))
        {
            return null;
        }

        // Get a calendar representing the date
        Calendar calendar = getCalendar(dateString, PRECISE_DATE_PATTERNS, false);

        // Check the time portion
        if ((!timePermitted) &&
            ((calendar.isSet(Calendar.HOUR_OF_DAY)) || (calendar.isSet(Calendar.MINUTE))))
        {
            throw new IllegalArgumentException(
                "The date can not contain hours or minutes: " + dateString);
        }

        // Return the newly created date based on the calendar
        return calendar.getTime();
    }

    /**
     * Gets a precise date from the passed in date as a String.  A time portion is not permitted.
     *
     * @param dateString The date in the format of a string.
     *
     * @return The parsed date or null if no date was passed in.
     * @throws IllegalArgumentException if the date is invalid.
     */
    public static Date getDate(String dateString)
    {
        return CommonDateUtils.getDate(dateString, false);
    }

    /**
     * Gets days between two dates (assumes imprecise which will handle precise and
     * imprecise). Returns 1 day for same date.
     *
     * @param fromDateString The from date
     * @param toDateString The to date
     *
     * @return The number of days.
     * @throws IllegalArgumentException if the date is invalid.
     */
    public static Long getDays(String fromDateString, String toDateString)
    {
        return getDays(fromDateString, toDateString, false);
    }

    /**
     * Gets days between two imprecise dates. Returns 1 day for same date.
     *
     * @param fromDateString The from date
     * @param toDateString The to date
     * @param precise if true, only calculate when both are precise
     *
     * @return The number of days.
     * @throws IllegalArgumentException if the date is null.
     */
    public static Long getDays(String fromDateString, String toDateString, boolean precise)
    {
        if (StringUtils.isEmpty(fromDateString))
        {
            throw new IllegalArgumentException("The from date can not be null.");
        }
        if (StringUtils.isEmpty(toDateString))
        {
            throw new IllegalArgumentException("The to date can not be null.");
        }

        // Assume imprecise date since an imprecise date can be precise
        ImpreciseDate fromDateImprecise = getImpreciseDate(fromDateString);
        ImpreciseDate toDateImprecise = getImpreciseDate(toDateString);

        // Return the days in between the two imprecise dates
        return getDays(fromDateImprecise, toDateImprecise, precise);
    }
    
    /**
     * Gets days between two imprecise dates. Returns 1 day for same date.
     *
     * @param fromDate The from date
     * @param toDate The to date
     *
     * @return The number of days.
     * @throws IllegalArgumentException if the date is null.
     */
    public static Long getDays(ImpreciseDate fromDate, ImpreciseDate toDate)
    {
    	return getDays(fromDate, toDate, false);
    }
    /**
     * Gets days between two imprecise dates. Returns 1 day for same date.
     *
     * @param fromDate The from date
     * @param toDate The to date
     * @param precise if true, only calculate when both are precise
     *
     * @return The number of days.
     * @throws IllegalArgumentException if the date is null.
     */
    public static Long getDays(ImpreciseDate fromDate, ImpreciseDate toDate, boolean precise)
    {
        Validate.notNull(fromDate, "The from date can not be null.");
        Validate.notNull(toDate, "The to date can not be null.");

        if (precise && (fromDate.isImprecise() || toDate.isImprecise()))
        	return null;
        
        // Convert the imprecise dates to precise dates where defaults will be filled in
        // as necessary for the month and day.
        Date fromDatePrecise = ImpreciseDateUtils.getDateWithDefault(fromDate);
        Date toDatePrecise = ImpreciseDateUtils.getDateWithDefault(toDate);

        // Return the days in between the two precise dates
        return getDays(fromDatePrecise, toDatePrecise);
    }

    public static Long getDays(Date fromDate, Date toDate)
    {
        if (toDate == null || fromDate == null)
            return null;

        Calendar fromCalenda = new GregorianCalendar();
        fromCalenda.setTime(fromDate);
        Calendar toCalendar = new GregorianCalendar();
        toCalendar.setTime(toDate);

        return getDays(fromCalenda, toCalendar);
    }

    public static Long getDays(Calendar fromCalenda, Calendar toCalendar)
    {
        if (toCalendar == null || fromCalenda == null)
            return null;

        Calendar calendar = Calendar.getInstance();
        calendar.clear();
        calendar.set(fromCalenda.get(Calendar.YEAR), fromCalenda.get(Calendar.MONTH),
            fromCalenda.get(Calendar.DAY_OF_MONTH));
        long fromDateMilliseconds = calendar.getTime().getTime();

        calendar.clear();
        calendar.set(toCalendar.get(Calendar.YEAR), toCalendar.get(Calendar.MONTH),
            toCalendar.get(Calendar.DAY_OF_MONTH));
        long toDateMilliseconds = calendar.getTime().getTime();

        long msPerDay = 1000 * 60 * 60 * 24;
        long result = (toDateMilliseconds - fromDateMilliseconds) / msPerDay;

        return new Long(result + 1);  //return 1 for same date.
    }

    /**
     * Gets an imprecise date from the passed in date as a String.  A timestamp is not permitted and
     * the current year will not be substituted.
     *
     * @param dateString The date in the format of a String.
     *
     * @return The parsed date or null if no date was passed in.
     * @throws IllegalArgumentException if the date is invalid.
     */
    public static ImpreciseDate getImpreciseDate(String dateString)
    {
        return getImpreciseDate(dateString, false, false);
    }

    /**
     * Gets an imprecise date from the passed in date as a String.  The current year will not be
     * substituted.
     *
     * @param dateString The date in the format of a String.
     * @param timePermitted If true, the hours and minutes are permitted.  Otherwise, not.
     *
     * @return The parsed date or null if no date was passed in.
     * @throws IllegalArgumentException if the date is invalid.
     */
    public static ImpreciseDate getImpreciseDate(String dateString, boolean timePermitted)
    {
        return getImpreciseDate(dateString, timePermitted, false);
    }

    /**
     * Gets an imprecise date from the passed in date as a String.
     *
     * @param dateString The date in the format of a String.
     * @param timePermitted If true, the hours and minutes are permitted.  Otherwise, not.
     * @param setCurrentYearAsDefault If true, the current year will be set on the date if none is
     * present.
     *
     * @return The parsed date or null if no date was passed in.
     * @throws IllegalArgumentException if the date is invalid.
     */
    public static ImpreciseDate getImpreciseDate(String dateString, boolean timePermitted,
        boolean setCurrentYearAsDefault)
    {
        // Check for empty string
        if ((StringUtils.isEmpty(dateString)) )
        {
            return null;
        }

        // Get a calendar representing the date
        Calendar calendar =
            getCalendar(dateString, IMPRECISE_DATE_PATTERNS, setCurrentYearAsDefault);

        // Check the time portion
        ImpreciseDate returnDate = new ImpreciseDate(calendar);
        if ((!timePermitted) && (returnDate.isTimePresent()))
        {
            throw new IllegalArgumentException(
                "The imprecise date can not contain hours or minutes: " + dateString);
        }

        // Return the newly built imprecise date
        return returnDate;
    }

    /**
     * Gets an imprecise date from the passed in date as a String.
     *
     * @param dateString The date in the format of a String.
     * @param datePatterns An array of valid date patterns.
     * @param setCurrentYearAsDefault If true, the current year will be set on the date if none is
     * present.
     *
     * @return The parsed calendar or null if no date was passed in.
     * @throws IllegalArgumentException if the date is invalid.
     */
    private static Calendar getCalendar(String dateString, String[] datePatterns,
        boolean setCurrentYearAsDefault)
    {
        // Check for empty string
        if (StringUtils.isEmpty(dateString))
        {
            if (setCurrentYearAsDefault)
            {
                dateString = String.valueOf(getCurrentYear());
            }
            else
            {
                return null;
            }
        }

        // Iterate through all the supported date formats until one is found that matches the input string.
        String trimmedDateString = dateString.trim();
        for (int i = 0; (i < datePatterns.length); i++)
        {
            String datePatternString = (String)datePatterns[i];
            Matcher matcher = Pattern.compile(datePatternString).matcher(trimmedDateString);

            if (matcher.matches())
            {
                String year = matcher.group(PATTERN_GROUP_YEAR);
                String month = matcher.group(PATTERN_GROUP_MONTH);
                String day = matcher.group(PATTERN_GROUP_DAY);
                String hour = matcher.group(PATTERN_GROUP_HOUR);
                String minute = matcher.group(PATTERN_GROUP_MINUTE);
                String second = matcher.group(PATTERN_GROUP_SECOND);

                Calendar calendar = Calendar.getInstance();
                calendar.setLenient(false);
                calendar.clear();

                if (!StringUtils.isEmpty(year))
                {
                    calendar.set(Calendar.YEAR, convertToInt(year));
                }
                else
                {
                    if (setCurrentYearAsDefault)
                    {
                        calendar.set(Calendar.YEAR, getCurrentYear());
                    }
                }
                if (!StringUtils.isEmpty(month))
                {
                    calendar.set(Calendar.MONTH, convertToInt(month) - 1);
                }
                if (!StringUtils.isEmpty(day))
                {
                    calendar.set(Calendar.DAY_OF_MONTH, convertToInt(day));
                }
                if (!StringUtils.isEmpty(hour))
                {
                    calendar.set(Calendar.HOUR_OF_DAY, convertToInt(hour));
                }

                if (!StringUtils.isEmpty(minute))
                {
                    calendar.set(Calendar.MINUTE, convertToInt(minute));
                }
                
                if (!StringUtils.isEmpty(second))
                {
                    calendar.set(Calendar.SECOND, convertToInt(second));
                }

                // Return the newly built calendar
                return calendar;
            }
        }

        // If we get here, we couldn't find a valid format so throw an exception
        throw new IllegalArgumentException("Invalid date format: " + dateString);
    }

    /**
     * Returns a date based on the passed in Imprecise Date formatted String with the number of
     * 'years' added to it. Note that the passed in date will have a default month and/or year set
     * to it if those fieldds are not present.
     *
     * @param dateStr The string format of an Imprecise Date
     * @param years The number of years to add to the date.
     *
     * @return The final date
     */
    public static Date getDatePlusYears(String dateStr, int years)
    {
        ImpreciseDate impDate = CommonDateUtils.getImpreciseDate(dateStr);
        if (impDate == null)
        {
            return null;
        }
        else
        {
            Date date = ImpreciseDateUtils.getDateWithDefault(impDate);
            Calendar cal = Calendar.getInstance();
            cal.clear();
            cal.setTime(date);
            cal.add(Calendar.YEAR, years);
            return cal.getTime();
        }
    }

    /**
     * Returns the current year.
     *
     * @return The current year.
     */
    public static int getCurrentYear()
    {
        return Calendar.getInstance().get(Calendar.YEAR);
    }

    /**
     * Returns date without time portion of it
     *
     * @return the current date with no time filled in
     */
    public static Date getCurrentDate()
    {
        Calendar calendar = Calendar.getInstance();
        int year = calendar.get(Calendar.YEAR);
        int month = calendar.get(Calendar.MONTH);
        int day = calendar.get(Calendar.DATE);
        calendar.clear();
        calendar.set(year, month, day);
        return calendar.getTime();
    }

    /**
     * Returns date without time portion of it
     *
     * @return The current date with the time filled in
     */
    public static Date getCurrentDateTime()
    {
        Calendar calendar = Calendar.getInstance();
        return calendar.getTime();
    }
    
    /**
     * Determines if two date objects are on the same day ignoring time.
     * 
     * Uses apache commons DateUtils.isSameDay which was not null safe.
     * 
     * @param date1 The date to compare
     * @param date2 The date to compare
     *
     * @return TRUE: if both of the dates are not null and both date objects are on the same day ignoring time.
     */
    public static boolean isSameDay(Date date1, Date date2)
    {
        if (date1 != null && date2 != null)
        {
            return DateUtils.isSameDay(date1, date2);
        }
        
        return false;
    }

    /**
     * Converts the passed in string to an integer.
     *
     * @param string The string to convert.
     *
     * @return The string as an integer.
     */
    private static int convertToInt(String string)
    {
        return (new Integer(string)).intValue();
    }
}
