/********************************************************************
 * Copyright  2005 VHA. All rights reserved
 ********************************************************************/
package gov.va.med.esr.common.ui.conversion;

// Java Classes

// Library Classes

// Framework Classes
import gov.va.med.esr.common.model.financials.Asset;
import gov.va.med.esr.common.model.financials.DependentFinancials;
import gov.va.med.esr.common.model.financials.Expense;
import gov.va.med.esr.common.model.financials.FinancialStatement;
import gov.va.med.esr.common.model.financials.Income;
import gov.va.med.esr.common.model.financials.IncomeTest;
import gov.va.med.esr.common.model.financials.SpouseFinancials;
import gov.va.med.esr.common.model.lookup.AssetType;
import gov.va.med.esr.common.model.lookup.ExpenseType;
import gov.va.med.esr.common.model.lookup.IncomeType;
import gov.va.med.esr.common.model.lookup.NameType;
import gov.va.med.esr.common.model.person.Name;
import gov.va.med.esr.common.model.person.Person;
import gov.va.med.esr.common.ui.form.DependentFinancialInfoBean;
import gov.va.med.esr.common.ui.form.FinancialDetailsForm;
import gov.va.med.esr.common.ui.form.SpouseFinancialInfoBean;
import gov.va.med.esr.common.ui.conversion.CommonConversionServiceImpl;
import gov.va.med.esr.common.util.CommonJspUtils;
import gov.va.med.esr.common.util.CommonNameHelper;

import gov.va.med.fw.conversion.ConversionServiceException;
import gov.va.med.fw.ui.UIConstants;
import gov.va.med.fw.util.StringUtils;

import java.util.ArrayList;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Set;

/**
 * An implementation of the conversion service in the common package layer with only the personToForm conversion.
 * 
 * @author Priya R.
 * @version 1.0
 */
public class CommonFinancialDetailsConversionService extends CommonConversionServiceImpl {

    private CommonFinancialsDependentConversionService dependentConversionService;
    
    public CommonFinancialDetailsConversionService() {}

    /**
     * Converts between a Person object that contains the financial information
     * and the FinancialDetailsForm.
     * 
     * @param source
     *            the source object
     * @param target
     *            the target object
     * 
     * @throws gov.va.med.fw.conversion.ConversionServiceException
     *             if any errors were encountered during the conversion.
     */
    protected void convertBean(Object source, Object target) throws ConversionServiceException {
        // TODO need to account for pre and post 2005 formats when converting
        // the financial amounts

        Person person = null;
        FinancialDetailsForm form = null;

        if (source instanceof Person) {
            person = (Person) source;
            if (target instanceof FinancialDetailsForm) {
                form = (FinancialDetailsForm) target;
            } else {
                throw new ConversionServiceException("Target object is of type "
                        + target.getClass().getName() + " and must be of type FinancialDetailsForm.");
            }

            // Convert from the Person to the form
            convertPersonToForm(person, form);
        } else {
                 throw new ConversionServiceException("Source object is of type "
                        + source.getClass().getName()
                        + " and must be of type Person.");
        }
    }

    /**
     * Converts the financial data from the Person object to the form.
     * 
     * @param person
     *            The person
     * @param form
     *            The form
     * 
     * @throws ConversionServiceException
     *             If there were any problems converting the person to the form.
     */
    protected void convertPersonToForm(Person person, FinancialDetailsForm form)
            throws ConversionServiceException {
        try {
        	
        	boolean isAddAPerson = form.isAddAPerson();
            
            String year = form.getIncomeYear();
            // TODO: Don't hard code the year here. If this is a scenario we
            // shouldn't hit since we are using a
            // combo box on the UI, just let it through an exception when year
            // is null.
            if (year == null || year.equals("null") || year.equals(""))
                throw new ConversionServiceException(
                        "Unable to convert Person to FinancialDetailsForm, no IncomeYear selected");

            Set otherNames = new HashSet(person.getNames());
            Name name = Name.getNameOfType(otherNames, NameType.LEGAL_NAME.getName());

            if (name != null) {
                form.setName(CommonNameHelper.formatNameFirstNameFirst(name));
            }

            // reset the test indicators
            form.setIncomeTestAvaliableInd(false);
            form.setPost2005formatInd(true);

            IncomeTest test = person.getIncomeTest(new Integer(year));
            // CCR 9552
            // For VOA Z07, the income test can be newly added, so evaluate the newIncomeTestInd
            if(test != null && test.getEntityKey() == null) {
                form.setNewIncomeTestInd(true);
            } else {
                form.setNewIncomeTestInd(false);                
            }
                

            if (test != null) {
                form.setIncomeTestAvaliableInd(true);
                form.setPendingAdjudication(test.isPendingAdjudication());
                
                form.setBtFinancialInd(test.getBtFinancialInd());
                // test effective date
                if (test.getEffectiveDate() != null)
                    form.setTestEffDate(CommonJspUtils.displayValue(test.getEffectiveDate(),
                            CommonJspUtils.STANDARD_DATE_FORMAT));

                form.setSendForAdjudication(getYesNoNoData(test.getSendForAdjudication()));
                if (test.getSendForAdjudication() != null) {
                    form.setAskAdjudicationQuestion(true);
                    //VFA-SP1 TODO
                    // form.setSendForAdjudication(getYesNoNoData(Boolean.FALSE));
                }
                form.setAgreeToPayDeductible(getYesNoNoData(test.getAgreesToPayDeductible()));
                form.setDiscloseFinIncome(getYesNoNoData(test.getDiscloseFinancialInformation()));
                form.setSourceOfTest(getLookupCode(test.getSource()));
            }

            FinancialStatement ftmt = person.getFinancialStatement(new Integer(year));

            if (ftmt != null) {
                if (ftmt.getIsPost2005Format() != null)
                    form.setPost2005formatInd(ftmt.getIsPost2005Format().booleanValue());

                form.setMarriedLastCalendarYear(getYesNoNoData(ftmt.getMarriedLastCalendarYear()));

                // veteran financial info
                convertVeteranFinancialAmountsToForm(ftmt, form);
                
                //spouse financials (all)
                int spouseCount = 0; // inactive spouse count
                Set spouseFinancials = ftmt.getSpouseFinancials();
                SpouseFinancials sf = ftmt.getRecentSpouseFinancials();
                for (Iterator i= spouseFinancials.iterator(); i.hasNext();){
                    SpouseFinancials spFin = (SpouseFinancials) i.next();                    
                    SpouseFinancialInfoBean sbean = new SpouseFinancialInfoBean();
                    convertSpouseFinancialsToBean(spFin, sbean);
                    convertSpouseFinancialAmountsToBean(spFin, sbean);
                    //If the SSN field is not present set it as unknown for displaying the link body.
                    if(StringUtils.isEmpty(sbean.getDisplaySSN())) sbean.setDisplaySSN(UIConstants.UNKNOWN);
                    if (isAddAPerson){
                    	  addSpouseToForm(spouseCount,form, sbean);
	                      spouseCount++;
	                      form.setSpouseInfo(sbean);
                    }
                    else {
	                    if (sf != null) {
	                        // CCR9552 -- if it's VOA Z07, there may not be financial data on file, so entityKey could be null
	                        if (sf.getEntityKey()!= null) {
	                            if (sf.getEntityKey().equals(spFin.getEntityKey())) {
	                                form.setSpouseInfo(sbean);
	                            } 
	                        } else if (sf.equals(spFin)) {
	                            form.setSpouseInfo(sbean);
	                        }                    
	                    }  
	                    else //Inactive list
	                    {
	                        addSpouseToForm(spouseCount,form, sbean);
	                        spouseCount++;
	                    }
                    }
                }                

                int depCount = 0;
                Set dependentFinancials = ftmt.getDependentFinancials();
                for (Iterator iter = dependentFinancials.iterator(); iter.hasNext();) {
                    DependentFinancials df = (DependentFinancials) iter.next();
                    DependentFinancialInfoBean bean = new DependentFinancialInfoBean();
                    convertDependentFinancialsToBean(df, bean);
                    convertDependentFinancialAmountsToBean(df, bean);
//                  If the SSN field is not present set it as unknown for displaying the link body.
                    if(StringUtils.isEmpty(bean.getDisplaySSN())) bean.setDisplaySSN(UIConstants.UNKNOWN);
                    addDependentToForm(depCount, form, bean);
                    depCount++;
                    // depInfoList.add(bean);
                }
                List depInfoList = new ArrayList();                
                form.setDependentInfoList(depInfoList);
            }

        } catch (Exception ex) {
            throw new ConversionServiceException("Unable to convert Person to FinancialDetailsForm.", ex);
        }
    }

    // --------------- Person to Form
    // ----------------------------------------------------------------
    private void convertVeteranFinancialAmountsToForm(FinancialStatement ftmt, FinancialDetailsForm form) {
        // financial info - income amts, asset etc
        for (Iterator i = ftmt.getIncome().values().iterator(); i.hasNext();) {
            Income e = (Income) i.next();
            if (e.getAmount() != null) {
                if (StringUtils.equals(e.getType().getCode(),
                        IncomeType.INCOME_TYPE_TOTAL_INCOME_FROM_EMPLOYMENT.getName()))
                    form.setTotalEmploymentIncome(e.getAmount().toString());
                else if (StringUtils.equals(e.getType().getCode(),
                        IncomeType.INCOME_TYPE_TOTAL_ALL_OTHER_INCOME.getName()))
                    form.setOtherIncome(e.getAmount().toString());
                else if (StringUtils.equals(e.getType().getCode(),
                        IncomeType.INCOME_TYPE_FARM_RANCH_PROPERTY_OR_BUSINESS_INCOME.getName()))
                    form.setNetIncomeFromFarmRanchOrBusiness(e.getAmount().toString());
                else if (StringUtils.equals(e.getType().getCode(), IncomeType.INCOME_TYPE_CIVIL_SERVICE
                        .getName()))
                    form.setUSCivilServiceIncome(e.getAmount().toString());
                else if (StringUtils.equals(e.getType().getCode(),
                        IncomeType.INCOME_TYPE_INTEREST_DIVIDEND_ANNUITY.getName()))
                    form.setInterestDividentAnuityIncome(e.getAmount().toString());
                else if (StringUtils.equals(e.getType().getCode(),
                        IncomeType.INCOME_TYPE_UNEMPLOYMENT_COMPENSATION.getName()))
                    form.setUnemploymentIncome(e.getAmount().toString());
                else if (StringUtils.equals(e.getType().getCode(), IncomeType.INCOME_TYPE_SOCIAL_SECURITY
                        .getName()))
                    form.setSocialSecurityIncome(e.getAmount().toString());
                else if (StringUtils.equals(e.getType().getCode(), IncomeType.INCOME_TYPE_MILITARY_RETIREMENT
                        .getName()))
                    form.setMilitaryRetirementIncome(e.getAmount().toString());
                else if (StringUtils.equals(e.getType().getCode(), IncomeType.INCOME_TYPE_RAILROAD_RETIREMENT
                        .getName()))
                    form.setRailroadRetirementIncome(e.getAmount().toString());
                else if (StringUtils.equals(e.getType().getCode(), IncomeType.INCOME_TYPE_OTHER_RETIREMENT
                        .getName()))
                    form.setOtherRetirementIncome(e.getAmount().toString());
                else if (StringUtils.equals(e.getType().getCode(),
                        IncomeType.INCOME_TYPE_WORKERS_COMP_BLACK_LUNG.getName()))
                    form.setWorkersCompIncome(e.getAmount().toString());

            }
        }

        for (Iterator i = ftmt.getExpenses().values().iterator(); i.hasNext();) {
            Expense e = (Expense) i.next();
            if (e.getAmount() != null) {
                if (StringUtils.equals(e.getType().getCode(), ExpenseType.EXPENSE_TYPE_COLLEGE_AND_VOCATION
                        .getName()))
                    form.setEducationExpense(e.getAmount().toString());
                if (StringUtils.equals(e.getType().getCode(), ExpenseType.EXPENSE_TYPE_NON_REIMBURSED_MEDICAL
                        .getName()))
                    form.setNonReimbMedicalExpense(e.getAmount().toString());
                if (StringUtils.equals(e.getType().getCode(), ExpenseType.EXPENSE_TYPE_FUNERAL_AND_BURIAL
                        .getName()))
                    form.setFuneralBurialExpense(e.getAmount().toString());
                if (StringUtils.equals(e.getType().getCode(), ExpenseType.EXPENSE_TYPE_ADJUSTED_MEDICAL
                        .getName()))
                    form.setAdjustedMedicalExpense(e.getAmount().toString());
            }
        }

        for (Iterator i = ftmt.getAssets().values().iterator(); i.hasNext();) {
            Asset e = (Asset) i.next();
            if (e.getAmount() != null) {
                if (StringUtils.equals(e.getType().getCode(), AssetType.CODE_CASH.getName()))
                    form.setCashAndBankAccountAsset(e.getAmount().toString());
                else if (StringUtils.equals(e.getType().getCode(), AssetType.CODE_REAL_ESTATE.getName()))
                    form.setRealPropertyAsset(e.getAmount().toString());
                else if (StringUtils.equals(e.getType().getCode(), AssetType.CODE_OTHER.getName()))
                    form.setOtherPropertyAsset(e.getAmount().toString());
                else if (StringUtils.equals(e.getType().getCode(), AssetType.CODE_STOCKS_AND_BONDS.getName()))
                    form.setStocksAsset(e.getAmount().toString());
            }
        }

        if (ftmt.getDebt() != null && ftmt.getDebt().getAmount() != null) {
            form.setDebt(ftmt.getDebt().getAmount().toString());
        }

    }

    private void convertDependentFinancialAmountsToBean(DependentFinancials ftmt,
            DependentFinancialInfoBean bean) {
        // financial info - income amts, asset etc
        for (Iterator i = ftmt.getIncome().values().iterator(); i.hasNext();) {
            Income e = (Income) i.next();
            if (e.getAmount() != null) {
                if (StringUtils.equals(e.getType().getCode(),
                        IncomeType.INCOME_TYPE_TOTAL_INCOME_FROM_EMPLOYMENT.getName()))
                    bean.setTotalEmploymentIncome(e.getAmount().toString());
                else if (StringUtils.equals(e.getType().getCode(),
                        IncomeType.INCOME_TYPE_TOTAL_ALL_OTHER_INCOME.getName()))
                    bean.setOtherIncome(e.getAmount().toString());
                else if (StringUtils.equals(e.getType().getCode(),
                        IncomeType.INCOME_TYPE_FARM_RANCH_PROPERTY_OR_BUSINESS_INCOME.getName()))
                    bean.setNetIncomeFromFarmRanchOrBusiness(e.getAmount().toString());
                else if (StringUtils.equals(e.getType().getCode(), IncomeType.INCOME_TYPE_CIVIL_SERVICE
                        .getName()))
                    bean.setUSCivilServiceIncome(e.getAmount().toString());
                else if (StringUtils.equals(e.getType().getCode(),
                        IncomeType.INCOME_TYPE_INTEREST_DIVIDEND_ANNUITY.getName()))
                    bean.setInterestDividentAnuityIncome(e.getAmount().toString());
                else if (StringUtils.equals(e.getType().getCode(),
                        IncomeType.INCOME_TYPE_UNEMPLOYMENT_COMPENSATION.getName()))
                    bean.setUnemploymentIncome(e.getAmount().toString());
                else if (StringUtils.equals(e.getType().getCode(), IncomeType.INCOME_TYPE_SOCIAL_SECURITY
                        .getName()))
                    bean.setSocialSecurityIncome(e.getAmount().toString());
                else if (StringUtils.equals(e.getType().getCode(), IncomeType.INCOME_TYPE_MILITARY_RETIREMENT
                        .getName()))
                    bean.setMilitaryRetirementIncome(e.getAmount().toString());
                else if (StringUtils.equals(e.getType().getCode(), IncomeType.INCOME_TYPE_RAILROAD_RETIREMENT
                        .getName()))
                    bean.setRailroadRetirementIncome(e.getAmount().toString());
                else if (StringUtils.equals(e.getType().getCode(), IncomeType.INCOME_TYPE_OTHER_RETIREMENT
                        .getName()))
                    bean.setOtherRetirementIncome(e.getAmount().toString());
                else if (StringUtils.equals(e.getType().getCode(),
                        IncomeType.INCOME_TYPE_WORKERS_COMP_BLACK_LUNG.getName()))
                    bean.setWorkersCompIncome(e.getAmount().toString());

            }
        }

        for (Iterator i = ftmt.getExpenses().values().iterator(); i.hasNext();) {
            Expense e = (Expense) i.next();
            if (e.getAmount() != null) {
                if (StringUtils.equals(e.getType().getCode(),
                        ExpenseType.EXPENSE_TYPE_EDUCATIONAL_EXPENSES_BY_DEPENDENT.getName()))
                    bean.setEducationExpense(e.getAmount().toString());
            }
        }

        for (Iterator i = ftmt.getAssets().values().iterator(); i.hasNext();) {
            Asset e = (Asset) i.next();
            if (e.getAmount() != null) {
                if (StringUtils.equals(e.getType().getCode(), AssetType.CODE_CASH.getName()))
                    bean.setCashAndBankAccountAsset(e.getAmount().toString());
                else if (StringUtils.equals(e.getType().getCode(), AssetType.CODE_REAL_ESTATE.getName()))
                    bean.setRealPropertyAsset(e.getAmount().toString());
                else if (StringUtils.equals(e.getType().getCode(), AssetType.CODE_OTHER.getName()))
                    bean.setOtherPropertyAsset(e.getAmount().toString());
                else if (StringUtils.equals(e.getType().getCode(), AssetType.CODE_STOCKS_AND_BONDS.getName()))
                    bean.setStocksAsset(e.getAmount().toString());
            }
        }

        /*
         * if(ftmt.getDebt() != null && ftmt.getDebt().getAmount() != null) {
         * bean.setDebt(ftmt.getDebt().getAmount().toString()); }
         */

    }

    private void convertSpouseFinancialAmountsToBean(SpouseFinancials ftmt, SpouseFinancialInfoBean bean)
            throws Exception {
        // financial info - income amts, asset etc
        for (Iterator i = ftmt.getIncome().values().iterator(); i.hasNext();) {
            Income e = (Income) i.next();
            if (e.getAmount() != null) {
                if (StringUtils.equals(e.getType().getCode(),
                        IncomeType.INCOME_TYPE_TOTAL_INCOME_FROM_EMPLOYMENT.getName()))
                    bean.setTotalEmploymentIncome(e.getAmount().toString());
                else if (StringUtils.equals(e.getType().getCode(),
                        IncomeType.INCOME_TYPE_TOTAL_ALL_OTHER_INCOME.getName()))
                    bean.setOtherIncome(e.getAmount().toString());
                else if (StringUtils.equals(e.getType().getCode(),
                        IncomeType.INCOME_TYPE_FARM_RANCH_PROPERTY_OR_BUSINESS_INCOME.getName()))
                    bean.setNetIncomeFromFarmRanchOrBusiness(e.getAmount().toString());
                else if (StringUtils.equals(e.getType().getCode(), IncomeType.INCOME_TYPE_CIVIL_SERVICE
                        .getName()))
                    bean.setUSCivilServiceIncome(e.getAmount().toString());
                else if (StringUtils.equals(e.getType().getCode(),
                        IncomeType.INCOME_TYPE_INTEREST_DIVIDEND_ANNUITY.getName()))
                    bean.setInterestDividentAnuityIncome(e.getAmount().toString());
                else if (StringUtils.equals(e.getType().getCode(),
                        IncomeType.INCOME_TYPE_UNEMPLOYMENT_COMPENSATION.getName()))
                    bean.setUnemploymentIncome(e.getAmount().toString());
                else if (StringUtils.equals(e.getType().getCode(), IncomeType.INCOME_TYPE_SOCIAL_SECURITY
                        .getName()))
                    bean.setSocialSecurityIncome(e.getAmount().toString());
                else if (StringUtils.equals(e.getType().getCode(), IncomeType.INCOME_TYPE_MILITARY_RETIREMENT
                        .getName()))
                    bean.setMilitaryRetirementIncome(e.getAmount().toString());
                else if (StringUtils.equals(e.getType().getCode(), IncomeType.INCOME_TYPE_RAILROAD_RETIREMENT
                        .getName()))
                    bean.setRailroadRetirementIncome(e.getAmount().toString());
                else if (StringUtils.equals(e.getType().getCode(), IncomeType.INCOME_TYPE_OTHER_RETIREMENT
                        .getName()))
                    bean.setOtherRetirementIncome(e.getAmount().toString());
                else if (StringUtils.equals(e.getType().getCode(),
                        IncomeType.INCOME_TYPE_WORKERS_COMP_BLACK_LUNG.getName()))
                    bean.setWorkersCompIncome(e.getAmount().toString());

            }
        }

        for (Iterator i = ftmt.getAssets().values().iterator(); i.hasNext();) {
            Asset e = (Asset) i.next();
            if (e.getAmount() != null) {
                if (StringUtils.equals(e.getType().getCode(), AssetType.CODE_CASH.getName()))
                    bean.setCashAndBankAccountAsset(e.getAmount().toString());
                else if (StringUtils.equals(e.getType().getCode(), AssetType.CODE_REAL_ESTATE.getName()))
                    bean.setRealPropertyAsset(e.getAmount().toString());
                else if (StringUtils.equals(e.getType().getCode(), AssetType.CODE_OTHER.getName()))
                    bean.setOtherPropertyAsset(e.getAmount().toString());
                else if (StringUtils.equals(e.getType().getCode(), AssetType.CODE_STOCKS_AND_BONDS.getName()))
                    bean.setStocksAsset(e.getAmount().toString());
            }
        }

        if (ftmt.getDebt() != null && ftmt.getDebt().getAmount() != null) {
            bean.setDebt(ftmt.getDebt().getAmount().toString());
        }
    }

    private void addSpouseToForm(int depIndex, FinancialDetailsForm form, SpouseFinancialInfoBean bean) {
        form.addSpouse();
        form.setSpouseActive(depIndex, bean.isActive() ? "true" : "false");
        form.setSpouseFinacialsId(depIndex, bean.getFinancialsId());
        form.setSpouseName(depIndex, bean.getName());
        form.setSpouseSSN(depIndex, bean.getSsn());
        form.setSpouseRelationshipCode(depIndex, bean.getRelationshipCode());
        form.setSpouseRelationship(depIndex, bean.getRelationship());
        form.setSpouseGenderCode(depIndex, bean.getGenderCode());
        form.setSpouseGender(depIndex, bean.getGender());
        form.setSpouseMarriageDate(depIndex, bean.getEffectiveDate());
        form.setSpouseValid(depIndex, bean.getValidDependent());
        form.setSpouseInactiveDate(depIndex, bean.getInactiveDate());
        form.setSpouseLivedWithPatient(depIndex, bean.getLivedWithVeteran());
        form.setSpouseContributedToSupport(depIndex, bean.getContributedToSupport());
        form.setSpouseTotalIncome(depIndex, bean.getTotalEmploymentIncome());
        form.setSpouseIncomeFromRanch(depIndex, bean.getNetIncomeFromFarmRanchOrBusiness());
        form.setSpouseOtherIncome(depIndex, bean.getOtherIncome());
        form.setSpouseEduExpenses(depIndex, bean.getEducationExpense());
        form.setSpouseCashAsset(depIndex, bean.getCashAndBankAccountAsset());
        form.setSpouseRealAsset(depIndex, bean.getRealPropertyAsset());
        form.setSpouseOtherAsset(depIndex, bean.getOtherPropertyAsset());
        form.setSpouseStocksAsset(depIndex, bean.getStocksAsset());
        form.setSpouseUSCivilServiceIncome(depIndex, bean.getUSCivilServiceIncome());
        form.setSpouseInterestDividentAnuityIncome(depIndex, bean.getInterestDividentAnuityIncome());
        form.setSpouseUnemploymentIncome(depIndex, bean.getUnemploymentIncome());
        form.setSpouseSocialSecurityIncome(depIndex, bean.getSocialSecurityIncome());
        form.setSpouseMilitaryRetirementIncome(depIndex, bean.getMilitaryRetirementIncome());
        form.setSpouseRailroadRetirementIncome(depIndex, bean.getRailroadRetirementIncome());
        form.setSpouseOtherRetirementIncome(depIndex, bean.getOtherRetirementIncome());
        form.setSpouseWorkersCompIncome(depIndex, bean.getWorkersCompIncome());
        form.setSpouseDisplaySSN(depIndex, bean.getDisplaySSN());
        form.setSpouseDebt(depIndex, bean.getDebt());        
        form.setContributedToSpousalSupport(depIndex, bean.getContributedToSpouseInd());
    }    
    private void addDependentToForm(int depIndex, FinancialDetailsForm form, DependentFinancialInfoBean bean) {
        form.addDependent();
        form.setDepFinacialsId(depIndex, bean.getFinancialsId());
        form.setDepName(depIndex, bean.getName());
        form.setDepSSN(depIndex, bean.getSsn());
        form.setDepRelationshipCode(depIndex, bean.getRelationshipCode());
        form.setDepRelationship(depIndex, bean.getRelationship());
        form.setDepGenderCode(depIndex, bean.getGenderCode());
        form.setDepGender(depIndex, bean.getGender());
        form.setDepEffectiveDate(depIndex, bean.getEffectiveDate());
        form.setDepValid(depIndex, bean.getValidDependent());
        form.setDepInactiveDate(depIndex, bean.getInactiveDate());
        form.setDepLivedWithPatient(depIndex, bean.getLivedWithVeteran());
        form.setDepContributedToSupport(depIndex, bean.getContributedToSupport());
        form.setDepAmtContributed(depIndex, bean.getAmtContributed());
        form.setDepHasIncome(depIndex, bean.getHasIncome());
        form.setDepIncomeAvailableToPatient(depIndex, bean.getIncomeAvailableToPatient());
        form.setDepAttendedSchool(depIndex, bean.getAttendedSchool());
        form.setDepIncapableOfSelfSupport(depIndex, bean.getIncapableOfSelfSupport());
        form.setDepTotalIncome(depIndex, bean.getTotalEmploymentIncome());
        form.setDepIncomeFromRanch(depIndex, bean.getNetIncomeFromFarmRanchOrBusiness());
        form.setDepOtherIncome(depIndex, bean.getOtherIncome());
        form.setDepEduExpenses(depIndex, bean.getEducationExpense());
        form.setDepCashAsset(depIndex, bean.getCashAndBankAccountAsset());
        form.setDepRealAsset(depIndex, bean.getRealPropertyAsset());
        form.setDepOtherAsset(depIndex, bean.getOtherPropertyAsset());
        form.setDepStocksAsset(depIndex, bean.getStocksAsset());
        form.setDepUSCivilServiceIncome(depIndex, bean.getUSCivilServiceIncome());
        form.setDepInterestDividentAnuityIncome(depIndex, bean.getInterestDividentAnuityIncome());
        form.setDepUnemploymentIncome(depIndex, bean.getUnemploymentIncome());
        form.setDepSocialSecurityIncome(depIndex, bean.getSocialSecurityIncome());
        form.setDepMilitaryRetirementIncome(depIndex, bean.getMilitaryRetirementIncome());
        form.setDepRailroadRetirementIncome(depIndex, bean.getRailroadRetirementIncome());
        form.setDepOtherRetirementIncome(depIndex, bean.getOtherRetirementIncome());
        form.setDepWorkersCompIncome(depIndex, bean.getWorkersCompIncome());
        form.setDepDisplaySSN(depIndex, bean.getDisplaySSN());
    }

    public List getDependentInfoList(FinancialStatement ftmt) throws Exception {
        List depInfoList = new ArrayList();
        Set spouseFinancials = ftmt.getSpouseFinancials();
        for(Iterator iter=spouseFinancials.iterator(); iter.hasNext();) {
            SpouseFinancials sf = (SpouseFinancials)iter.next();
            SpouseFinancialInfoBean sbean = new SpouseFinancialInfoBean();
            convertSpouseFinancialsToBean(sf, sbean);
            //If the SSN field is not present set it as unknown for displaying the link body.
            if(StringUtils.isEmpty(sbean.getDisplaySSN())) sbean.setDisplaySSN(UIConstants.UNKNOWN);
            depInfoList.add(sbean);
        }

        Set dependentFinancials = ftmt.getDependentFinancials();
        for(Iterator iter = dependentFinancials.iterator(); iter.hasNext();) {
            DependentFinancials df = (DependentFinancials) iter.next();
            DependentFinancialInfoBean bean = new DependentFinancialInfoBean();
            convertDependentFinancialsToBean(df, bean);
            //If the SSN field is not present set it as unknown for displaying the link body.
            if(StringUtils.isEmpty(bean.getDisplaySSN())) bean.setDisplaySSN(UIConstants.UNKNOWN);
            depInfoList.add(bean);
        }
        return depInfoList;
    }

    public CommonFinancialsDependentConversionService getDependentConversionService() {
        return dependentConversionService;
    }

    public void setDependentConversionService(
            CommonFinancialsDependentConversionService dependentConversionService) {
        this.dependentConversionService = dependentConversionService;
    }
    
    public void convertSpouseFinancialsToBean(SpouseFinancials spFins, SpouseFinancialInfoBean bean) {
        getDependentConversionService().convertSpouseFinancialsToBean(spFins, bean);
    }

    public void convertDependentFinancialsToBean(DependentFinancials depFin, DependentFinancialInfoBean bean) {
        getDependentConversionService().convertDependentFinancialsToBean(depFin, bean);
       
    }
}