/*******************************************************************************
 * Copyright  2004 VHA. All rights reserved
 ******************************************************************************/
package gov.va.med.esr.common.rule.service.impl;

// Framework classes
import java.util.List;

import org.apache.commons.lang.Validate;

import gov.va.med.fw.rule.AbstractRuleValidationService;
import gov.va.med.fw.rule.RuleDataAware;
import gov.va.med.fw.rule.RuleParameters;
import gov.va.med.fw.rule.SimpleValidatableRule;
import gov.va.med.fw.service.ServiceException;
import gov.va.med.fw.validation.ValidationMessages;

// ESR classes
import gov.va.med.esr.common.model.ee.CombatEpisode;
import gov.va.med.esr.common.model.ee.ConflictExperience;
import gov.va.med.esr.common.model.ee.Document;
import gov.va.med.esr.common.model.ee.EGTSetting;
import gov.va.med.esr.common.model.ee.MilitaryService;
import gov.va.med.esr.common.model.ee.MilitaryServiceEpisode;
import gov.va.med.esr.common.model.ee.POWEpisode;
import gov.va.med.esr.common.model.ee.PurpleHeartDocument;
import gov.va.med.esr.common.model.financials.DependentFinancials;
import gov.va.med.esr.common.model.financials.SpouseFinancials;
import gov.va.med.esr.common.model.insurance.InsurancePolicy;
import gov.va.med.esr.common.model.party.Address;
import gov.va.med.esr.common.model.person.Association;
import gov.va.med.esr.common.model.person.Person;
import gov.va.med.esr.common.model.registry.Registry;
import gov.va.med.esr.common.model.comms.CommsLogEntry;
import gov.va.med.esr.common.rule.data.AddressInputData;
import gov.va.med.esr.common.rule.data.AssociationInputData;
import gov.va.med.esr.common.rule.data.CommsInputData;
import gov.va.med.esr.common.rule.data.DependentFinancialsInputData;
import gov.va.med.esr.common.rule.data.EgtInputData;
import gov.va.med.esr.common.rule.data.FinancialInputData;
import gov.va.med.esr.common.rule.data.InsuranceInputData;
import gov.va.med.esr.common.rule.data.MilitaryServiceInputData;
import gov.va.med.esr.common.rule.data.PersonInputData;
import gov.va.med.esr.common.rule.data.POWInputData;
import gov.va.med.esr.common.rule.data.PHInputData;
import gov.va.med.esr.common.rule.data.RegistryInputData;
import gov.va.med.esr.common.rule.data.SpouseFinancialsInputData;
import gov.va.med.esr.common.rule.service.RuleValidationService;
import gov.va.med.esr.service.trigger.LetterTriggerEvent;
import gov.va.med.esr.service.trigger.LetterTriggerEvent.MailType;
import gov.va.med.esr.common.model.lookup.ComLetterTemplateType;
import gov.va.med.esr.common.model.lookup.RegistryType;

/**
 * Provides methods to validate domain object entities based on business rules.
 * Project: Common</br> Created on: 10:20:11 AM </br>
 * 
 * @author DNS   LEV
 */
public class RuleValidationServiceImpl extends AbstractRuleValidationService
		implements RuleValidationService {

	/**
	 * An instance of serialVersionUID
	 */
	private static final long serialVersionUID = 3587897087412351790L;

	/**
	 * An instance of demographicRuleParameters
	 */
	private String demographicRuleParameters = null;

	/**
	 * An instance of insuranceRuleParameters
	 */
	private String insuranceRuleParameters = null;

	/**
	 * An instance of insuranceRuleParameters
	 */
	private String associationRuleParameters = null;

	/**
	 * An instance of addressRuleParameters
	 */
	private String addressRuleParameters = null;

	/**
	 * An instance of msValidationRuleParameters
	 */
	private String msValidationRuleParameters = null;
	
	/**
	 * An instance of mseValidationRuleParameters
	 */
	private String mseValidationRuleParameters = null;

    /**
     * An instance of combatEpisodeValidationRuleParameters
     */
    private String  combatEpisodeValidationRuleParameters = null;
    
	/**
	 * An instance of conflictExperienceValidationRuleParameters
	 */
	private String conflictExperienceValidationRuleParameters = null;
	
	/**
	 * An instance of VeteranFinancialValidationRuleParameters
	 */
	private String veteranFinancialValidationRuleParameters = null;
	 
	/**
	 * An instance of SpouseValidationRuleParameters
	 */
	private String spouseValidationRuleParameters = null;
	
	/**
	 * An instance of DependentValidationRuleParameters
	 */
	private String dependentValidationRuleParameters = null;
    
    /**
     * An instance of EnrollmentValidationRuleParameters
     */
    private String enrollmentValidationRuleParameters = null;    

    /**
     * An instance of IneligibilityValidationRuleParameters
     */
    private String ineligibilityValidationRuleParameters = null;    

    /**
     * An instance of powValidationRuleParameters
     */
    private String powValidationRuleParameters = null;    
    
    /**
     * An instance of letterMailingValidationRuleParameters
     */
    private String letterMailingValidationRuleParameters = null;
    
    /**
     * An instance of registryValidationRuleParameters
     */
    private String registryValidationRuleParameters = null;
    
    /**
     * An instance of registryDocumentValidationRuleParameters
     */
    private String registryDocumentValidationRuleParameters = null;
    
    /**
     * An instance of processLetterSendRequestRuleParameters
     */
    private String processLetterSendRequestRuleParameters = null;    
    
    /**
     * An instance of processLetterSendRequestRuleParameters
     */    
    private String purpleHeartValidationRuleParameters = null;
    
    private String egtValidationRuleParameters = null;
    
    private String identityTraitsValidationRuleParameters = null;
    
    
	/**
	 * A default constructor
	 */
	public RuleValidationServiceImpl() {
		super();
	}

	/**
	 * Validates demographic personal information.
	 * 
	 * @param person
	 *           incoming person object
	 * @param isDataFromGUI
	 *           indicator flag (UI)
	 * @return return ValidationMessages object.
	 */
	public ValidationMessages validateDemographic( Person person,
																  boolean isDataFromGUI ) 
		throws ServiceException {
		
		return this.validate( this.createDemographicInput(person, isDataFromGUI ),
									 this.getRuleParameters( getDemographicRuleParameters() ),
									 isDataFromGUI );
	}

	/**
	 * @see gov.va.med.esr.common.rule.service.RuleValidationService#validateAssociation(gov.va.med.esr.common.model.person.Association, boolean)
	 */
	public ValidationMessages validateAssociation( Association association,
            Person onFile, Person pristine,
																  boolean isDataFromGUI ) throws ServiceException {
		
		return this.validate( this.createAssociationInput(association, onFile, pristine, isDataFromGUI), 
									 this.getRuleParameters( getAssociationRuleParameters() ),
									 isDataFromGUI );
	}

    
	/**
	 * @see gov.va.med.esr.common.rule.service.RuleValidationService#validateConflictExperience(gov.va.med.esr.common.model.ee.ConflictExperience)
	 */
	public ValidationMessages validateCombatEpisode( CombatEpisode ce,Person onFile,Person pristine,boolean isDataFromGUI ) 
    throws ServiceException {
		return this.validate( this.createCombatEpisodeInput(ce, onFile, pristine ), this.getRuleParameters( getCombatEpisodeValidationRuleParameters() ),isDataFromGUI );
	}
    
    /**
     * @see gov.va.med.esr.common.rule.service.RuleValidationService#validateConflictExperience(gov.va.med.esr.common.model.ee.ConflictExperience)
     */
    public ValidationMessages validateConflictExperience( ConflictExperience ce,
                                                          Person onFile, 
                                                          Person pristine,
                                                          boolean isDataFromGUI ) throws ServiceException {
        
        return this.validate( this.createConflictExperienceInput(ce, onFile, pristine ),this.getRuleParameters( getConflictExperienceValidationRuleParameters() ),isDataFromGUI );
    }

	/**
	 * Validates the veterans financials.
	 */
	public ValidationMessages validateVeteranFinancials(Integer incomeYear, Person incoming, Person onFile,Person pristine,boolean isDemographicUpdate,boolean isDataFromGUI) 
	throws ServiceException
	{
	    return this.validate( this.createFinancialsInput(incomeYear,incoming,onFile,pristine,isDemographicUpdate,isDataFromGUI),
				                 this.getRuleParameters( getVeteranFinancialValidationRuleParameters() ),
                             isDataFromGUI);
	}

	/** 
	 * Validates the dependent financials.
	 */
	public ValidationMessages validateSpouseFinancials(Integer incomeYear, SpouseFinancials spouseFinancials, Person incoming, Person onFile,boolean isDemographicUpdate,boolean isDataFromGUI)
	throws ServiceException {
	    // ES 4.2_CodeCR13338_CR_Rule Layer Modifications for Decommissioning Spousal/Dependent Support
	    // Spouse rules were not getting the income year passed to parameter.
	    SpouseFinancialsInputData data = new SpouseFinancialsInputData(spouseFinancials,incoming,onFile,isDemographicUpdate);
	    data.setIncomingIncomeYear(incomeYear);
        return this.validate(data, this.getRuleParameters( getSpouseValidationRuleParameters() ), isDataFromGUI);
	}
	
	/**
	 * Validates the dependent financials.
	 */
	public ValidationMessages validateDependentFinancials(Integer incomeYear,DependentFinancials depenentFinancials,Person incoming, Person onFile,boolean isDemographicUpdate, boolean isDataFromGUI)
	throws ServiceException
	{
	    return this.validate(this.createDependentFinancialsInput(depenentFinancials,incoming,onFile,isDemographicUpdate),
				 this.getRuleParameters( getDependentValidationRuleParameters() ),isDataFromGUI);
	}

	/**
	 * @see gov.va.med.esr.common.rule.service.RuleValidationService#validateInsurance(gov.va.med.esr.common.model.insurance.InsurancePolicy)
	 */
	public ValidationMessages validateInsurancePolicy( InsurancePolicy insurancePolicy, 
																		Person result,
																		Person pristine,
																		boolean isDataFromGUI )
			throws ServiceException {
		
		return this.validate( this.createInsurancePolicyInput(insurancePolicy, result, pristine ), 
									 this.getRuleParameters( getInsuranceRuleParameters() ),
									 isDataFromGUI );
	}

	public ValidationMessages validateMS(MilitaryService ms,
										 Person onFile, 
										 Person pristine,
										 boolean isDataFromGUI,
                                         String useCaseName) throws ServiceException {
		return this.validate(this.createMilitaryServiceInput(ms,onFile,pristine,isDataFromGUI,useCaseName), 
							 this.getRuleParameters( getMsValidationRuleParameters() ),
							 isDataFromGUI);
	}
	
	/**
	 * @see gov.va.med.esr.common.rule.service.RuleValidationService#validateMSE(gov.va.med.esr.common.model.ee.MilitaryServiceEpisode,
	 *      gov.va.med.esr.common.model.person.Person)
	 */
	public ValidationMessages validateMSE( MilitaryServiceEpisode mse,
														Person onFile, 
														Person pristine,
														boolean isDataFromGUI ) throws ServiceException {
		return this.validate( this.createMilitaryServiceInput(mse, onFile, pristine ), 
									 this.getRuleParameters( getMseValidationRuleParameters() ),
									 isDataFromGUI );
	}

    /**
     * @see gov.va.med.esr.common.rule.service.RuleValidationService#validateReceivedEnrollment(gov.va.med.esr.common.model.person.Person, boolean)
     */
	public ValidationMessages validateReceivedEnrollment(Person incoming, Person onFile, Person pristine, boolean isDataFromGUI) throws ServiceException {
        PersonInputData personInputData = new PersonInputData(incoming,onFile,pristine,isDataFromGUI);
        return this.validate(personInputData,
                             this.getRuleParameters( getEnrollmentValidationRuleParameters() ),
                             isDataFromGUI );
    }
    
    /**
     * @see gov.va.med.esr.common.rule.service.RuleValidationService#validateIneligibility(gov.va.med.esr.common.model.person.Person, boolean)
     */
    public ValidationMessages validateIneligibility(Person incoming, boolean isDataFromGUI) throws ServiceException {
        return this.validate( this.createPersonInput(incoming, isDataFromGUI ), 
                this.getRuleParameters( getIneligibilityValidationRuleParameters() ),
                isDataFromGUI );
    }

    /**
     * @see gov.va.med.esr.common.rule.service.RuleValidationService#validatePOW(gov.va.med.esr.common.model.ee.POWEpisode, gov.va.med.esr.common.model.person.Person)
     */
    public ValidationMessages validatePOW(POWEpisode powEpisode, Person incoming, boolean isDataFromGUI) throws ServiceException {
      return this.validate( this.createPOWInput(powEpisode, incoming, isDataFromGUI ),    
                            this.getRuleParameters( getPowValidationRuleParameters() ), 
                            isDataFromGUI );
    }    
    
    /**
     * @see gov.va.med.esr.common.rule.service.RuleValidationService#validateLetterMailing(gov.va.med.fw.model.AbstractKeyedEntity, gov.va.med.esr.common.model.party.Address, gov.va.med.esr.common.model.person.Person, boolean)
     */
    public ValidationMessages validateLetterMailing(CommsLogEntry letterMailing, Person incoming) throws ServiceException {
        Validate.notNull( letterMailing, "A letter mailing must not be null" );        
        return this.validate( this.createCommsInput(letterMailing, letterMailing.getAddress(), incoming), 
                              this.getRuleParameters( getLetterMailingValidationRuleParameters() ), 
                              true);
    }


    /**
     * @see gov.va.med.esr.common.rule.service.RuleValidationService#validateLetterSendRequest(gov.va.med.esr.common.model.person.Person, gov.va.med.esr.common.model.lookup.ComLetterTemplateType, java.lang.String, gov.va.med.esr.service.trigger.LetterTriggerEvent.MailType, java.lang.Boolean, java.util.List, java.lang.String, java.lang.String)
     */
    public ValidationMessages validateLetterSendRequest(Person person, ComLetterTemplateType formNumber, String workflowCaseId, MailType mailType, Boolean override, List logs, String categoryType, String context) throws ServiceException {
        return this.validate( this.createCommsInput(person, formNumber, workflowCaseId, mailType, override, logs, categoryType, context), 
                this.getRuleParameters( getProcessLetterSendRequestRuleParameters() ), 
                true);
    }

    public ValidationMessages validateRegistry(Registry registry,boolean isDataFromGUI) throws ServiceException {
        return this.validate( this.createRegistryInput(registry,isDataFromGUI),
                              this.getRuleParameters( getRegistryValidationRuleParameters() ), 
                              isDataFromGUI);
    }
    
    public ValidationMessages validateRegistryDocument(Registry registry,Document document,boolean isDataFromGUI) throws ServiceException {
        return this.validate( this.createRegistryInput(registry,document,isDataFromGUI),
                              this.getRuleParameters( getRegistryDocumentValidationRuleParameters() ),
                              isDataFromGUI);
    }
    
    /**
     * @see gov.va.med.esr.common.rule.service.RuleValidationService#validatePurpleHeartDocument(gov.va.med.esr.common.model.ee.PurpleHeartDocument, gov.va.med.esr.common.model.person.Person, boolean)
     */
    public ValidationMessages validatePurpleHeartDocument(PurpleHeartDocument purpleHeartDocument, Person incoming, boolean isDataFromGUI) throws ServiceException {
        return this.validate( this.createPHInput(incoming, purpleHeartDocument, isDataFromGUI),
                              this.getRuleParameters( getPurpleHeartValidationRuleParameters() ),
                              isDataFromGUI );
    }
    
    /**
     * @see gov.va.med.esr.common.rule.service.RuleValidationService#validateEgtSetting(gov.va.med.esr.common.model.ee.EGTSetting, gov.va.med.esr.common.model.ee.EGTSetting)
     */
    public ValidationMessages validateEgtSetting(EGTSetting egtSetting, EGTSetting onFile) throws ServiceException {
        return this.validate( this.createEgtInput(egtSetting, onFile),
                              this.getRuleParameters( getEgtValidationRuleParameters() ), 
                              true);
    }
    
    /**
     * @see gov.va.med.esr.common.rule.service.RuleValidationService#validateIdentityTraits(gov.va.med.esr.common.model.person.Person, boolean)
     */
    public ValidationMessages validateIdentityTraits(Person person, boolean isDataFromGUI) throws ServiceException {
        return this.validate( this.createIdentityTraitsInput(person, isDataFromGUI ),
                 this.getRuleParameters( this.getIdentityTraitsValidationRuleParameters() ),
                 isDataFromGUI );
    }

    protected RuleDataAware createPHInput(Person incoming, PurpleHeartDocument purpleHeartDocument, boolean isUpdateFromGUI) {
        return new PHInputData(incoming, purpleHeartDocument, isUpdateFromGUI);
    }
    
    protected RuleDataAware createAddressInput(Address address) {
		RuleDataAware data = new AddressInputData(address);
		return data;
	}

	protected RuleDataAware createInsurancePolicyInput( InsurancePolicy insurancePolicy, 
																		 Person result, 
																		 Person pristine )
			throws ServiceException {
		RuleDataAware data = new InsuranceInputData( insurancePolicy );
		data.setIncomingData( insurancePolicy.getPerson() );
		data.setResultData( result );
		data.setPristineData( pristine );
		return data;
	}

	protected RuleDataAware createMilitaryServiceInput(MilitaryService ms, 
			 										   Person onFile,
			 										   Person pristine,
			 										   boolean isDataFromGUI,
                                                       String useCaseName) throws ServiceException {
        MilitaryServiceInputData data = new MilitaryServiceInputData(ms, onFile, pristine, isDataFromGUI );
        data.setUseCaseName(useCaseName);
	    return data;
	}
	
	protected RuleDataAware createMilitaryServiceInput( MilitaryServiceEpisode mse, 
																		 Person onFile,
																		 Person pristine ) throws ServiceException {
		RuleDataAware data = new MilitaryServiceInputData(mse, onFile, pristine );
		return data;
	}

    protected RuleDataAware createCombatEpisodeInput(CombatEpisode ce,Person onFile,Person pristine ) 
    throws ServiceException {
        RuleDataAware data = new MilitaryServiceInputData(ce, onFile, pristine );
        return data;
    }
    
	protected RuleDataAware createConflictExperienceInput(ConflictExperience ce,Person onFile, Person pristine ) 
    throws ServiceException {
		RuleDataAware data = new MilitaryServiceInputData(ce, onFile, pristine );
		return data;
	}

	protected RuleDataAware createAssociationInput(Association association,
            Person onFile, Person pristine,
			boolean isDataFromGUI) throws ServiceException {
		RuleDataAware data = new AssociationInputData(association, onFile, pristine, isDataFromGUI, null);
		return data;
	}

	protected RuleDataAware createDemographicInput(Person incoming,
			boolean isDataFromGUI) throws ServiceException {
		return new PersonInputData(incoming, isDataFromGUI);
	}
    
    protected RuleDataAware createIdentityTraitsInput(Person incoming,
            boolean isDataFromGUI) throws ServiceException {
        return new PersonInputData(incoming, isDataFromGUI);
    }    
    
    protected RuleDataAware createPersonInput(Person incoming, boolean isDataFromGUI) throws ServiceException {
        return new PersonInputData(incoming, isDataFromGUI);
    }
    
    protected RuleDataAware createCommsInput(CommsLogEntry letterMailing, Address mailingAddress, Person incoming) throws ServiceException {
        return new CommsInputData(incoming, letterMailing, mailingAddress);
    }
    
    protected CommsInputData createCommsInput(Person person,
            ComLetterTemplateType formNumber, String workflowCaseId, LetterTriggerEvent.MailType mailType,
            Boolean override, List logs, String categoryType, String context) throws ServiceException {
        CommsInputData cid = new CommsInputData(person, formNumber, workflowCaseId, mailType,
                override);
        cid.setCommsLogEntries(logs);
        cid.setSubject(categoryType);   
        cid.setRuleContext(context);
        return cid;
    }      
    
    protected RuleDataAware createPOWInput(POWEpisode powEpisode, Person incoming, boolean isDataFromGUI) throws ServiceException {
        return new POWInputData(powEpisode, incoming, isDataFromGUI);
    }     

	protected RuleDataAware createFinancialsInput(Integer incomeYear,Person incoming,Person onFile,Person pristine,boolean isDemographicUpdate,boolean isDataFromGUI) 
	throws ServiceException 
	{
		return new FinancialInputData(incoming,onFile,pristine,incomeYear,isDemographicUpdate,isDataFromGUI); 
	}
	
    protected RuleDataAware createSpouseFinancialsInput(SpouseFinancials spouseFinancials,Person incoming,Person onFile,boolean isDemographicUpdate) 
    throws ServiceException 
    {
        return new SpouseFinancialsInputData(spouseFinancials,incoming,onFile,isDemographicUpdate);
    }
    
	protected RuleDataAware createDependentFinancialsInput(DependentFinancials dependentFinancials,Person incoming,Person onFile,boolean isDemographicUpdate) 
	throws ServiceException 
	{
		return new DependentFinancialsInputData(dependentFinancials,incoming,onFile,isDemographicUpdate);
	}
    
    protected RuleDataAware createRegistryInput(Registry incoming,boolean isDataFromGUI) 
        throws ServiceException {
    	RegistryType regTYpe=incoming.getRegistryTrait()!=null?incoming.getRegistryTrait().getRegistryType():null;
        return new RegistryInputData(incoming,isDataFromGUI,regTYpe);
    }
    
    protected RuleDataAware createRegistryInput(Registry incoming,Document document,boolean isDataFromGUI) 
    throws ServiceException {
    RegistryType regType=incoming.getRegistryTrait()!=null?incoming.getRegistryTrait().getRegistryType():null;	
    return new RegistryInputData(incoming,document,isDataFromGUI,regType);
}

    
    private EgtInputData createEgtInput(EGTSetting incoming, EGTSetting onFile) throws ServiceException {
        return new EgtInputData(incoming, onFile);
    }
    
	protected ValidationMessages validate( RuleDataAware data,
														RuleParameters parameters,
														boolean isDataFromGUI ) throws ServiceException {

		String mode = (isDataFromGUI == true) ? SimpleValidatableRule.EXHAUSTIVE : SimpleValidatableRule.ELIMINATION;
		return validate(new SimpleValidatableRule( mode, parameters, data ));
	}
	
    public String getDemographicRuleParameters() {
        return demographicRuleParameters;
    }

    public void setDemographicRuleParameters(
          String demographicRuleParameters) {
        this.demographicRuleParameters = demographicRuleParameters;
    }

    public String getInsuranceRuleParameters() {
        return insuranceRuleParameters;
    }

    public void setInsuranceRuleParameters(String insuranceRuleParameters) {
        this.insuranceRuleParameters = insuranceRuleParameters;
    }

    public String getAssociationRuleParameters() {
        return associationRuleParameters;
    }

    public void setAssociationRuleParameters(
          String associationRuleParameters) {
        this.associationRuleParameters = associationRuleParameters;
    }

    public String getAddressRuleParameters() {
        return addressRuleParameters;
    }

    public void setAddressRuleParameters(String addressRuleParameters) {
        this.addressRuleParameters = addressRuleParameters;
    }

	/**
	 * @return Returns the msValidationRuleParameters.
	 */
	public String getMsValidationRuleParameters() {
		return msValidationRuleParameters;
	}

	/**
	 * @param mseValidationRuleParameters
	 *           The msValidationRuleParameters to set.
	 */
	public void setMsValidationRuleParameters(String msValidationRuleParameters) {
		this.msValidationRuleParameters = msValidationRuleParameters;
	}
	
	/**
	 * @return Returns the mseValidationRuleParameters.
	 */
	public String getMseValidationRuleParameters() {
		return mseValidationRuleParameters;
	}

	/**
	 * @param mseValidationRuleParameters
	 *           The mseValidationRuleParameters to set.
	 */
	public void setMseValidationRuleParameters(String mseValidationRuleParameters) {
		this.mseValidationRuleParameters = mseValidationRuleParameters;
	}

	public String getCombatEpisodeValidationRuleParameters() {
        return combatEpisodeValidationRuleParameters;
    }

    public void setCombatEpisodeValidationRuleParameters(
            String combatEpisodeValidationRuleParameters) {
        this.combatEpisodeValidationRuleParameters = combatEpisodeValidationRuleParameters;
    }

    /**
	 * @return Returns the conflictExperienceValidationRuleParameters.
	 */
	public String getConflictExperienceValidationRuleParameters() {
		return conflictExperienceValidationRuleParameters;
	}

	/**
	 * @param conflictExperienceValidationRuleParameters
	 *           The conflictExperienceValidationRuleParameters to set.
	 */
	public void setConflictExperienceValidationRuleParameters(
         String conflictExperienceValidationRuleParameters) {
		this.conflictExperienceValidationRuleParameters = conflictExperienceValidationRuleParameters;
	}
	
    public String getVeteranFinancialValidationRuleParameters() {
        return veteranFinancialValidationRuleParameters;
    }

    public void setVeteranFinancialValidationRuleParameters(
          String veteranFinancialValidationRuleParameters) {
        this.veteranFinancialValidationRuleParameters = veteranFinancialValidationRuleParameters;
    }
    
    public String getSpouseValidationRuleParameters() {
        return spouseValidationRuleParameters;
    }

    public void setSpouseValidationRuleParameters(
          String spouseValidationRuleParameters) {
        this.spouseValidationRuleParameters = spouseValidationRuleParameters;
    }
    
    public String getDependentValidationRuleParameters() {
        return dependentValidationRuleParameters;
    }
    
    public void setDependentValidationRuleParameters(
          String dependentValidationRuleParameters) {
        this.dependentValidationRuleParameters = dependentValidationRuleParameters;
    }

    /**
     * @return Returns the enrollmentValidationRuleParameters.
     */
    public String getEnrollmentValidationRuleParameters() {
        return enrollmentValidationRuleParameters;
    }

    /**
     * @param enrollmentValidationRuleParameters The enrollmentValidationRuleParameters to set.
     */
    public void setEnrollmentValidationRuleParameters(
          String enrollmentValidationRuleParameters) {
        this.enrollmentValidationRuleParameters = enrollmentValidationRuleParameters;
    }

    /**
     * @return Returns the ineligibilityValidationRuleParameters.
     */
    public String getIneligibilityValidationRuleParameters() {
        return ineligibilityValidationRuleParameters;
    }

    /**
     * @param ineligibilityValidationRuleParameters The ineligibilityValidationRuleParameters to set.
     */
    public void setIneligibilityValidationRuleParameters(
          String ineligibilityValidationRuleParameters) {
        this.ineligibilityValidationRuleParameters = ineligibilityValidationRuleParameters;
    }

    /**
     * @return Returns the powValidationRuleParameters.
     */
    public String getPowValidationRuleParameters() {
        return powValidationRuleParameters;
    }

    /**
     * @param powValidationRuleParameters The powValidationRuleParameters to set.
     */
    public void setPowValidationRuleParameters(
          String powValidationRuleParameters) {
        this.powValidationRuleParameters = powValidationRuleParameters;
    }

    /**
     * @return Returns the letterMailingValidationRuleParameters.
     */
    public String getLetterMailingValidationRuleParameters() {
        return letterMailingValidationRuleParameters;
    }

    /**
     * @param letterMailingValidationRuleParameters The letterMailingValidationRuleParameters to set.
     */
    public void setLetterMailingValidationRuleParameters(
          String letterMailingValidationRuleParameters) {
        this.letterMailingValidationRuleParameters = letterMailingValidationRuleParameters;
    }

    /**
     * @return Returns the registryValidationRuleParameters.
     */
    public String getRegistryValidationRuleParameters() {
        return registryValidationRuleParameters;
    }

    /**
     * @param registryValidationRuleParameters The registryValidationRuleParameters to set.
     */
    public void setRegistryValidationRuleParameters(
          String registryValidationRuleParameters) {
        this.registryValidationRuleParameters = registryValidationRuleParameters;
    }

    /**
     * @return Returns the registryDocumentValidationRuleParameters.
     */
    public String getRegistryDocumentValidationRuleParameters() {
        return registryDocumentValidationRuleParameters;
    }

    /**
     * @param registryDocumentValidationRuleParameters The registryDocumentValidationRuleParameters to set.
     */
    public void setRegistryDocumentValidationRuleParameters(
          String registryDocumentValidationRuleParameters) {
        this.registryDocumentValidationRuleParameters = registryDocumentValidationRuleParameters;
    }

    /**
     * @return Returns the processLetterSendRequestRuleParameters.
     */
    public String getProcessLetterSendRequestRuleParameters() {
        return processLetterSendRequestRuleParameters;
    }

    /**
     * @param processLetterSendRequestRuleParameters The processLetterSendRequestRuleParameters to set.
     */
    public void setProcessLetterSendRequestRuleParameters(
          String processLetterSendRequestRuleParameters) {
        this.processLetterSendRequestRuleParameters = processLetterSendRequestRuleParameters;
    }

    /**
     * @return Returns the purpleHeartValidationRuleParameters.
     */
    public String getPurpleHeartValidationRuleParameters() {
        return purpleHeartValidationRuleParameters;
    }

    /**
     * @param purpleHeartValidationRuleParameters The purpleHeartValidationRuleParameters to set.
     */
    public void setPurpleHeartValidationRuleParameters(
          String purpleHeartValidationRuleParameters) {
        this.purpleHeartValidationRuleParameters = purpleHeartValidationRuleParameters;
    }

    /**
     * @return Returns the egtValidationRuleParameters.
     */
    public String getEgtValidationRuleParameters() {
        return egtValidationRuleParameters;
    }

    /**
     * @param egtValidationRuleParameters The egtValidationRuleParameters to set.
     */
    public void setEgtValidationRuleParameters(
            String egtValidationRuleParameters) {
        this.egtValidationRuleParameters = egtValidationRuleParameters;
    }

    /**
     * @return Returns the identityTraitsValidationRuleParameters.
     */
    public String getIdentityTraitsValidationRuleParameters() {
        return identityTraitsValidationRuleParameters;
    }

    /**
     * @param identityTraitsValidationRuleParameters The identityTraitsValidationRuleParameters to set.
     */
    public void setIdentityTraitsValidationRuleParameters(
            String identityTraitsValidationRuleParameters) {
        this.identityTraitsValidationRuleParameters = identityTraitsValidationRuleParameters;
    }
}