/********************************************************************
 * Copyright  2004 VHA. All rights reserved
 ********************************************************************/
package gov.va.med.esr.common.rule.service.impl;

// Java classes
import java.util.Iterator;
import java.util.Set;

// Library classes
import org.apache.commons.lang.Validate;

// Framework classes
import gov.va.med.fw.service.ServiceException;
import gov.va.med.fw.validation.ValidationMessages;
import gov.va.med.fw.validation.ValidationServiceException;

// EDB classes
import gov.va.med.esr.common.model.ee.PrisonerOfWar;
import gov.va.med.esr.common.model.ee.POWEpisode;
import gov.va.med.esr.common.model.lookup.VAFacility;
import gov.va.med.esr.common.model.person.Person;
import gov.va.med.esr.common.rule.data.POWInputData;
import gov.va.med.esr.common.rule.service.POWRuleService;

/**
 * Project: Common
 * @author DNS   LEV
 * @version 1.0
 */
public class POWRuleServiceImpl extends AbstractRuleValidationServiceAwareImpl implements POWRuleService {

   /**
	 * An instance of serialVersionUID
	 */
	private static final long serialVersionUID = 2593357573979544087L;
	
	/**
	 * An instance of powRuleParameters
	 */
	private String powRuleParameters = null;
	
	/**
    * 
    */
   public POWRuleServiceImpl() {
      super();
   }
 
   /**
    * 
    */
   public void processPOW(Person incoming, Person onFile)
            throws ServiceException {
       this.doProcessPOW(incoming, onFile, false, null);
    }
   public void processPOW(Person incoming, Person onFile, VAFacility sendingFacility)
   throws ServiceException {
this.doProcessPOW(incoming, onFile, false, sendingFacility);
}
   
	/**
     * @see gov.va.med.esr.common.rule.service.POWRuleService#managePOW(gov.va.med.esr.common.model.person.Person,
     *      gov.va.med.esr.common.model.person.Person)
     */
    public void managePOW(Person incoming, Person onFile)
            throws ServiceException {
        this.doProcessPOW(incoming, onFile, true, null);
    }
    
    /**
	 * @see gov.va.med.esr.common.rule.service.impl.AbstractRuleValidationServiceAwareImpl#afterPropertiesSet()
	 */
	public void afterPropertiesSet() throws Exception {
		super.afterPropertiesSet();
		Validate.notNull( this.powRuleParameters, "powRuleParameters property is required");
	}

   /**
	 * @return Returns the powRuleParameters.
	 */
	public String getPowRuleParameters() {
		return powRuleParameters;
	}
	/**
	 * @param powRuleParameters The powRuleParameters to set.
	 */
	public void setPowRuleParameters(String powRuleParameters) {
		this.powRuleParameters = powRuleParameters;
	}
    
    private void doProcessPOW(Person incoming, Person onFile, boolean ui, VAFacility sendingFacility)
            throws ServiceException {
        Validate.notNull(onFile, "A veteran on file must not be null");
        Validate.notNull(incoming, "The incoming must not be null");

        Person pristine = this.getPristinePerson(onFile);
        if (ui) {
            this.doValidation(incoming, true);            
        }
        
        PrisonerOfWar pow = incoming.getPrisonerOfWar();
        if (pow == null) {
            onFile.setPrisonerOfWar(null);
        }
        else {
            invokeRuleFlow( this.getRuleParameters(getPowRuleParameters()), new POWInputData(incoming, onFile, pristine, ui, sendingFacility));            
        }
    }
    
    private void doValidation(Person incoming, boolean isUpdateFromGUI)
            throws ServiceException {
        PrisonerOfWar pow = incoming.getPrisonerOfWar();
        Set episodes = pow != null ? pow.getEpisodes() : null;
        if (episodes != null) {
            for (Iterator i = episodes.iterator(); i.hasNext();) {
                POWEpisode powe = (POWEpisode) i.next();
                ValidationMessages messages = this.getRuleValidationService().validatePOW(powe, incoming, true);

                if (messages != null && !messages.isEmpty()) {
                    throw new ValidationServiceException(messages);
                }
            }
        }
    }
}