/********************************************************************
 * Copyright  2004 VHA. All rights reserved
 ********************************************************************/
package gov.va.med.esr.common.rule.service;

//Java classes
import java.io.Serializable;

// Library classes

// Framework classes
import gov.va.med.fw.service.ServiceException;

// EDB classes
import gov.va.med.esr.common.batchprocess.LoadVSSCDataResult;
import gov.va.med.esr.common.model.lookup.VAFacility;
import gov.va.med.esr.common.model.person.Person;
import gov.va.med.esr.common.model.ee.MilitaryService;
import gov.va.med.esr.service.MsdsResponseInfo;
import gov.va.med.esr.common.model.ee.VerificationInfo;

/** Provides methods to implement specific business rules regarding to Military Service information
 * 
 * Project: Common
 * @author DNS   LEV
 * @version 1.0
 */
public interface MilitaryRuleService extends Serializable {

   /** Processes a military service data from a message.  This method processes military 
    * service from each site record
    * 
    * @param sendingFacility sending facility
    * @param dfn dfn
    * @param ms A military service to be updated
    * @param onFile The current veteran on file
    * @param string caller
    * @return Person a veteran with updated military service info
    * @throws ServiceException Thrown in case of errors
    */
   public Person processMilitaryService(VAFacility sendingFacility, String dfn,  MilitaryService ms, Person onFile ,String caller) throws ServiceException;

   /** Processes a military service data from a GUI screen.  This method processes military 
    * service from each site record
    * 
    * @param ms A military service to be updated
    * @param onFile The current veteran on file
    * @return Person a veteran with updated military service info
    * @throws ServiceException Thrown in case of errors
    */
   public Person manageMilitaryService( MilitaryService ms, Person onFile ) throws ServiceException;
   
   /**
    * Process Military Service changes. If from Eligibility Edit screen, set isFromEE = true.
    * 
    * @param ms
    * @param onFile
    * @param isFromEE
    * @return
    * @throws ServiceException
    */
   public Person manageMilitaryService( MilitaryService ms, Person onFile, boolean isFromEE ) throws ServiceException;   
   
   /**
    * Process VSSC data (OEF/OIF) combat episodes
    * @param ms
    * @param onFile
    * @param loadVSSCDataResult
    * @return
    * @throws ServiceException
    */
   public Person processVSSCData(MilitaryService ms, Person onFile, LoadVSSCDataResult loadVSSCDataResult) throws ServiceException;
   
   /** Calculates a period of service for a veteran's military service record.
    * 
    * @param incoming An incoming veteran
    * @param onFile A veteran on the syste
    * @param useCaseName use case name
    * @return An updated veteran with updated period of service calculated.
    * @throws ServiceException Thrown in case of errors.
    */
   public Person calculatePeriodOfServiceForUI(Person incoming, Person onFile,String useCaseName) throws ServiceException;
   
   
   /** Calculates a period of service for a veteran's military service record.
    * 
    * @param incoming An incoming veteran
    * @param onFile A veteran on the syste,
    * @param useCaseName use case name
    * @param sendingFacility the sending facility
    * @return An updated veteran with updated period of service calculated.
    * @throws ServiceException Thrown in case of errors.
    */
   public Person calculatePeriodOfService(Person incoming, Person onFile,String useCaseName, VAFacility sendingFacility) throws ServiceException;
   
   /** Validates the current military service.
    * 
    * @param incoming An incoming veteran
    * @param onFile A veteran on the syste,
    * @throws ServiceException Thrown in case of validation errors.
    */
   public void validateMilitaryService(Person incoming,Person onFile) 
   throws ServiceException;

   /**
    * Process the Broker message contents CCR 8395
    * 
    * @param incoming
    * @param onFile
    * @param info
    * @return true=terminated for error false=success
    * @throws ServiceException
    */
   public boolean processMSDSData(Person incoming, Person onFile, MsdsResponseInfo info) throws ServiceException;
   
   /**
	 * Determine if person has mse that qualifies for use in figuring
	 * eligibility status.
	 * 
	 * @param info
	 * @param incoming
	 * @return
	 * @throws ServiceException
	 */
   public boolean hasQualifyingMse(Person person, VerificationInfo info) throws ServiceException;   
   
   /**
    * CCR 8395
    * Evaluate the incoming person against the pristine person to see if there is an improvement
    * with certain factors as outlined by MSDS rules.
    * 
    * @param incoming
    * @param onFile - gets updated when improvement occurs
    * @return true=improved, false=no improvement
    * @throws ServiceException
    */
   public boolean evaluateMSDSImprovement(Person incoming, Person onFile, MsdsResponseInfo info) throws ServiceException;   
   
   //MSDS Seeding Changes
   
   /** Processes a military service data from a message for Seeding purposes.  This method processes military 
    * service from each site record
    * 
    * @param sendingFacility sending facility
    * @param dfn dfn
    * @param ms A military service to be updated
    * @param onFile The current veteran on file
    * @return Person a veteran with updated military service info
    * @throws ServiceException Thrown in case of errors
    */
   public Person processSeedingMilitaryService(MilitaryService ms, Person onFile ) throws ServiceException;
   
   //MSDS create hec record for new veteran upon ending reconciliation process
   
   /** Processes a military service data from a message for creating hec record purposes.  This method processes military 
    * service from the receiving site record
    * 
    * @param sendingFacility sending facility     
    * @param ms A military service to be updated
    * @param onFile The current veteran on file
    * @return Person a veteran with updated military service info
    * @throws ServiceException Thrown in case of errors
    */
   public Person processCreateHECRecord(MilitaryService ms, Person onFile ) throws ServiceException;

   /**
    * Process the Military Service Numbers from Site Military Service Episode and populate it in HEC Military Service Episode.
    * 
    * If a HEC Military Service Episode lacks the Service Number and the site specific record has the same exact Begin Date and End Date and the Service Number, 
    * then the Service Number shall be added to the HEC Military Service Episode.  
    * If the HEC Military Service Episode has a Service Number, it shall not be replaced by a different Service Number for the same Military Service Episode 
    * in the site specific record. 
    * 
    * @param onFile
    * @return resultPerson
    * @throws ServiceException
    */
   public Person processMilitaryServiceNumbers(Person onFile) throws ServiceException;  
}