/********************************************************************
 * Copyright  2004 VHA. All rights reserved
 ********************************************************************/
package gov.va.med.esr.common.rule.service;

// Java classes
import java.io.Serializable;
import java.util.Set;
import java.util.Map;

import gov.va.med.fw.service.ServiceException;

import gov.va.med.esr.common.model.ee.EnrollmentDetermination;
import gov.va.med.esr.common.model.financials.DependentFinancials;
import gov.va.med.esr.common.model.financials.SpouseFinancials;
import gov.va.med.esr.common.model.lookup.MeansTestStatus;
import gov.va.med.esr.common.model.lookup.VAFacility;
import gov.va.med.esr.common.model.person.Person;
import gov.va.med.esr.common.model.person.SignatureImage;
import gov.va.med.esr.service.EEResultInfo;
import gov.va.med.esr.service.IVMFinancialInfo;

/**
 * Project: Common
 * @author DNS   LEV
 * @version 1.0
 */
public interface FinancialInfoRuleService
    extends Serializable {

    /**
     * Manages the financial information of a veteran, spouse and dependents. 
     * 
     * @param incomeYear - income year to be processed, must not be null
     * @param incoming - Incoming Person
     * @param onFile - Person on file
	 * @return Person - Result Person will have updated information as a result of processing
     * @throws ServiceException - if there is any error 
     */
    public Person manageFinancialAssessment(Integer incomeYear, VAFacility fromFacility, Person incoming, Person onFile) throws ServiceException;

    /**
     * Manages the financial information of a veteran, spouse and dependents. Passes
     * parameter  object for EE information.
     * 
     * @param incomeYear
     * @param fromFacility
     * @param incoming
     * @param onFile
     * @param info
     * @return
     * @throws ServiceException
     */
    public Person manageFinancialAssessment(Integer incomeYear, VAFacility fromFacility, Person incoming, Person onFile, EEResultInfo info) throws ServiceException;    

    /**
     * Manage the spouse financials and demographic information.
     * 
     * @param incomeYear - Income year to be processed, must not be null
     * @param dependentFinancials - Incoming dependent financials
     * @param incoming - Incoming Person
     * @param onFile - Person on file
	 * @return Person - Result Person will have updated information as a result of processing
     * @throws ServiceException - if there is any error 
     */
    public Person manageSpouseInfo(Integer incomeYear,SpouseFinancials spouseFinancials, Person incoming, Person onFile) throws ServiceException;

    
    /**
     * Manage the dependent financials and demographic information.
     * 
     * @param incomeYear - Income year to be processed, must not be null
     * @param dependentFinancials - Incoming dependent financials
     * @param incoming - Incoming Person
     * @param onFile - Person on file
	 * @return Person - Result Person will have updated information as a result of processing
     * @throws ServiceException - if there is any error 
     */
    public Person manageDependentFinancialInfo(Integer incomeYear, DependentFinancials dependentFinancials, Person incoming, Person onFile) throws ServiceException;


    /**
	 * Processes the financial information provided by a message. Processing includes
	 * Financial IncomeTest (Means, Pharmacy Copay, LTC type Income test) 
	 * Veteran Financial Statement, 
	 * Spouse Financial Statement,
	 * Dependent Financial Statement,
	 * Spouse Demographics Information
	 * Dependents Demographics Information  
	 * Patient Visit Summary
	 * Beneficiary Travel Information
	 * 
     * @param incomeYear - Income year to be processed, must not be null
     * @param fromFacility - facility where message is received
     * @param incoming - Incoming Person
     * @param onFile - Person on file
	 * @return Person - Result Person will have updated information as a result of processing
	 * @throws ServiceException
     */
    public Person processFinancialInfo(Integer incomeYear, VAFacility fromFacility, Person incoming, Person onFile) throws ServiceException;

    /**
     * Same as above except provides flag indicating if Z10 sent.
     * 
     * @param incomeYear
     * @param fromFacility
     * @param incoming
     * @param onFile
     * @param info
     * @return
     * @throws ServiceException
     */
    public Person processFinancialInfo(Integer incomeYear, VAFacility fromFacility, Person incoming, Person onFile, EEResultInfo info) throws ServiceException;    
    
    public Person processFinancialInfo(Integer incomeYear, VAFacility fromFacility, Person incoming, Person personOnFile,
            EEResultInfo info, boolean isDataFromZ07) throws ServiceException;
    
    /**
     * Determine the means test status based on eligibilities.
     * 
     * @param incoming
     * @param onFile
     * @param pristine
     * @param result
     * @return
     * @throws ServiceException
     */
    public Person determineIncomeTestFromElig(Person incoming, Person onFile, Person pristine, EnrollmentDetermination result) throws ServiceException;

    /**
     * Determine the means test status based on eligibilities.
     * 
     * @param incoming
     * @param onFile
     * @param pristine
     * @param result
     * @param isFromGUI
     * @return
     * @throws ServiceException
     */
    public Person determineIncomeTestFromElig(Person incoming, Person onFile, Person pristine, EnrollmentDetermination result, boolean isFromGUI, VAFacility sendingFacility) throws ServiceException;

    /**
     * Manage the person's signature images from UI.
     * 
     * @param signatures
     * @param onFile
     * @param dfns
     * @return
     * @throws ServiceException
     */
    public Person managePersonSignatures(Set signatures, Person onFile, Map dfns) throws ServiceException;
    
    /**
     * Manage a person's signature image from UI.
     * 
     * @param signature
     * @param onFile
     * @param dfn
     * @return signatureImage
     * @throws ServiceException
     */
    public SignatureImage managePersonSignature(SignatureImage signature, Person onFile, String dfn) throws ServiceException;
    
    /**
     * Process person's signature images from messaging.
     * 
     * @param signatures
     * @param onFile
     * @param dfns
     * @return
     * @throws ServiceException
     */
    public Person processPersonSignatures(Set signatures, Person onFile, Map dfns) throws ServiceException;
    
    /**
     * Process a person's signature image from messaging.
     * 
     * @param signature
     * @param onFile
     * @param dfn
     * @return signatureImage
     * @throws ServiceException
     */
    public SignatureImage processPersonSignature(SignatureImage signature, Person onFile, String dfn) throws ServiceException;     

    
    /**
     * Process a fee basis program of the veteran from a given site.
     * 
     * @param fromFacility
     * @param incoming - incoming person
     * @param onFile - person on file
     * @param boolean isDataFromGUI
     * @return Person - a result Person
     * @throws ServiceException
     */
    public Person processFeeBasis(VAFacility fromFacility, Person incoming, Person onFile,boolean isDataFromGUI) throws ServiceException;

    /**
     * Adjudicate means test status by setting a new status when the current status is 
     * pending adjudication.
     * 
     * @param incomeYear
     * @param incoming
     * @param onFile
     * @param status
     * @return - updated person
     */
    public Person managePendingAdjudicationStatus(Integer incomeYear, Person incoming, Person onFile, MeansTestStatus status) throws ServiceException;
    
    public Person processIVMFinancials(Person incoming, Person onFile, IVMFinancialInfo ivmFinancialInfo) throws ServiceException;    
}