/*******************************************************************************
 * Copyright  2004 VHA. All rights reserved
 ******************************************************************************/
package gov.va.med.esr.common.rule.service;

// Java classes
import java.io.Serializable;

// Library classes

// Framework classes
import gov.va.med.fw.service.ServiceException;

// EDB Classes
import gov.va.med.esr.common.clock.Clock.Type;
import gov.va.med.esr.common.model.lookup.VAFacility;
import gov.va.med.esr.common.model.person.Person;
import gov.va.med.esr.common.model.ee.Application;
import gov.va.med.esr.common.model.ee.CancelDecline;
import gov.va.med.esr.common.model.ee.EGTSetting;
import gov.va.med.esr.common.model.ee.EnrollmentDetermination;
import gov.va.med.esr.common.model.ee.ReceivedEnrollment;
import gov.va.med.esr.common.model.ee.VerificationInfo;

/**
 * Project: Common
 *
 * @author DNS   LEV
 * @version 1.0
 */
public interface EligibilityEnrollmentRuleService extends Serializable {

    /**
     * A method to process a cancel decline request.
     *
     * @param incoming
     * @param onFile
     * @throws ServiceException
     */
    public void processCancelDecline(CancelDecline incoming, Person onFile)
            throws ServiceException;

    /**
     * Calculate the Eligibility and Enrollment for a person.
     *
     * @param onFile
     * @param isDataFromGUI
     * @param isReviewOnly
     * @throws ServiceException
     */
    public void calculateEE(Person onFile, boolean isDataFromGUI,
            boolean isReviewOnly) throws ServiceException;

    public void calculateEE(Person onFile, boolean isDataFromGUI, boolean isReviewOnly, boolean shouldTriggerHBP,
    			Type clockType, VAFacility sendingFacility) throws ServiceException;

    /**
     * Convenience method for a cleanup job that switches on special
     * E&E behavior
     *
     * @param onFile
     */
    public void calculateEEForCleanup(Person onFile) throws ServiceException;;

    public void calculateEE(Person onFile, boolean isDataFromGUI,
            boolean isReviewOnly, Type clockType, VAFacility sendingFacility) throws ServiceException;
    /**
     * Calculate the EE for the Expire MT batch job.
     *
     * @param onFile
     * @param incomeYear
     *            This year will be used to find current test
     * @throws ServiceException
     */
    public void calculateEEForExpireMTBatchJob(Person onFile, Integer incomeYear)
            throws ServiceException;

    /**
     * A method to calculate enrollment priority and enrollment status for a
     * veteran. One potential caller is EGT processing. The result is not linked
     * to person after the calculation.
     *
     * @param incoming
     * @param egtSetting
     * @return
     * @throws ServiceException
     */
    public Person determineEnrollment(Person incoming, EGTSetting egtSetting)
            throws ServiceException;

    /**
     * Applies determine enrollment rules to a veteran, but does not process any
     * letter trigger events generated from processing the rules
     *
     * @param incoming
     *            the veteran
     * @return a person object describing the veteran after rules are applied
     * @throws ServiceException
     */
    public Person determineEnrollmentNoTriggerLetters(Person incoming)
            throws ServiceException;

    /**
     * A method to manage an ineligibility factor from UI.
     *
     * @param incoming
     * @param onFile
     * @throws ServiceException
     */
    public void manageIneligibility(Person incoming, Person onFile)
            throws ServiceException;

    /**
     * Manage the received eligibilities submitted by UI
     *
     * @param incoming
     * @param onFile
     * @throws ServiceException
     */
    public void manageReceivedEligibilities(Person incoming, Person onFile)
            throws ServiceException;

    /**
     * A method to process an ineligibility from messaging.
     *
     * @param incoming
     * @param onFile
     * @param verificationInfo
     * @param  boolean isDataFromGUI
     * @throws ServiceException
     */
    public void processIneligibility(Person incoming, Person onFile, VAFacility sendingFacility,
            VerificationInfo verificationInfo, boolean isDataFromGUI) throws ServiceException;

    /**
     * Process the received eligibilities submitted in HL7 message.
     *
     * @param incoming
     * @param onFile
     * @throws ServiceException
     */
    public void processReceivedEligibilities(Person incoming, Person onFile, VerificationInfo verificationInfo,boolean isUpdateFromGUI)
            throws ServiceException;

    /**
     * A method to process a received enrollment from messaging.
     *
     * @param enrollment
     * @param onFile
     * @throws ServiceException
     */
    public void processReceivedEnrollment(ReceivedEnrollment enrollment,
            Person onFile) throws ServiceException;

    /**
     * A method to process an application.
     *
     * @param application
     * @param onFile
     * @throws ServiceException
     */
    public void processApplication(Application application, Person onFile)
            throws ServiceException;

    /**
     * A method to manage Application from UI
     *
     * @param application
     * @param onFile
     * @throws ServiceException
     */
    public void manageApplication(Application application, Person onFile) throws ServiceException;

    /**
     * A method to process an ineligibility from messaging.
     *
     * @param incoming
     * @param onFile
     * @param enrollment
     * @throws ServiceException
     */
    public void processOtherEligibililities(Person incoming, Person onFile,
            EnrollmentDetermination enrollment) throws ServiceException;

    /**
     * Apply rules for setting service connected indicator
     *
     * @param person
     */
    public void processServiceConnectedIndicator(Person person);

     /**
     * Apply certain rules from Process Person use case on HL7 input;
     * specifically those rules are for Received Enrollment and Application.
     *
     * @param incoming
     * @param onFile
     * @param updateFromGUI
     */
    public void processPerson(Person incoming, Person onFile, VAFacility sendingFacility, VerificationInfo verificationInfo, boolean updateFromGUI)  throws ServiceException;;

    /**
     * Apply handbook trigger rules from Process Person use case on HL7 input;
     * specifically those rules are for Received Enrollment and Application.
     *
     * @param incoming
     * @param onFile
     */
    public void processPersonHandBookTrigger(Person incoming, Person onFile, VAFacility sendingFacility, VerificationInfo verificationInfo)  throws ServiceException;

    /**
     * @see gov.va.med.esr.common.rule.service.EligibilityEnrollmentRuleService#processOtherEligibilities(gov.va.med.esr.common.model.person.Person,
     *      gov.va.med.esr.common.model.person.Person,
     *      gov.va.med.esr.common.model.ee.EnrollmentDetermination)
     */
    public void processHandBookTriggers(Person incoming, Person onFile, VAFacility sendingFacility, VerificationInfo verificationInfo) throws ServiceException;

    public void processHealthBenefitProfile(Person incoming, Person onFile) throws ServiceException;

    /**
     * A method to manage Health Benefit Plan from UI
     *
     * @param incoming
     * @param onFile
     */
    public void manageHealthBenefitProfile(Person incoming, Person onFile) throws ServiceException;

}