/*******************************************************************************
 * Copyright  2004 VHA. All rights reserved
 ******************************************************************************/
package gov.va.med.esr.common.rule.parameter;
 
// Java classes
import java.math.BigDecimal;
import java.util.Calendar;
import java.util.Date;
import java.util.GregorianCalendar;
// Library classes

// Framework classes
import gov.va.med.fw.rule.RuleException;
import gov.va.med.fw.service.ServiceException;
import gov.va.med.fw.util.DateUtils;

// ESR Classes
import gov.va.med.esr.common.model.ee.EnrollmentDetermination;
import gov.va.med.esr.common.model.ee.IncompetenceRuling;
import gov.va.med.esr.common.model.ee.MonetaryBenefit;
import gov.va.med.esr.common.model.ee.Pension;
import gov.va.med.esr.common.model.ee.MonetaryBenefitAward;
import gov.va.med.esr.common.model.person.Employment;
import gov.va.med.esr.common.model.person.Person;
import gov.va.med.esr.common.model.lookup.EligibilityVerificationSource;
import gov.va.med.esr.common.model.lookup.EmploymentStatus;
import gov.va.med.esr.common.model.lookup.Indicator;
import gov.va.med.esr.common.model.lookup.MonetaryBenefitType;
import gov.va.med.esr.common.model.lookup.PensionReasonCode;
import gov.va.med.esr.common.rule.MonetaryAwardInput;

/**
 * Project: Common</br> Created on: 10:31:53 AM </br>
 * 
 * @author DNS   LEV
 */
public class MonetaryAwardInputParameter extends BaseParameter implements
        MonetaryAwardInput {

    private static final long serialVersionUID = 7206910358099240141L;
    private static final String Original_Award_Reason_Code = "0";

    /**
     */
    public MonetaryAwardInputParameter() {
        super();

    }

    /**
     * @see gov.va.med.esr.common.rule.parameter.MonetaryAwardInput#getTotalCheckAmount()
     */
    public BigDecimal getTotalCheckAmount() throws RuleException {
        return getTotalCheckAmount(getIncomingPerson());
    }

    /**
     * @see gov.va.med.esr.common.rule.MonetaryAwardInput#setTotalCheckAmount(java.math.BigDecimal)
     */
    public void setTotalCheckAmount(BigDecimal param) {
        setTotalCheckAmount(param, getResultPerson());
    }

    /**
     * @see gov.va.med.esr.common.rule.MonetaryAwardInput#getVeteranRatedIncompetent()
     */
    public IncompetenceRuling getVeteranRatedIncompetent() {
        return getPristinePerson().getIncompetenceRuling();
    }

    /**
     * @see gov.va.med.esr.common.rule.MonetaryAwardInput#getHouseBound()
     */
    public MonetaryBenefit getHouseBound() throws RuleException {
        return getHouseBound(getIncomingPerson());
    }

    /**
     * @see gov.va.med.esr.common.rule.MonetaryAwardInput#getHouseBound()
     */
    public MonetaryBenefit getHouseBound(Person person) throws RuleException {
        MonetaryBenefitAward mba = person != null ? person
                .getMonetaryBenefitAward() : null;
        return mba != null ? mba.getHouseBound() : null;
    }

    /**
     * @see gov.va.med.esr.common.rule.MonetaryAwardInput#addMonetaryBenefit(gov.va.med.esr.common.model.ee.MonetaryBenefit)
     */
    public void addMonetaryBenefit(MonetaryBenefit benefit)
            throws RuleException {

        Person person = getResultPerson();
        MonetaryBenefitAward mba = benefit != null ? person
                .getMonetaryBenefitAward() : null;
        if (mba != null) {
            try {
                mba.removeMonetaryBenefitByType(benefit.getType());

                MonetaryBenefit mb = new MonetaryBenefit();
                getMergeRuleService().mergeMonetaryBenefit(benefit, mb);
                mb.setType(benefit.getType());
                mb.setMonetaryBenefitIndicator(benefit
                        .getMonetaryBenefitIndicator());

                mba.addMonetaryBenefit(mb);
            } catch (ServiceException e) {
                throw new RuleException(
                        "Failed to add a monetary benefit of type "
                                + benefit.getType().getCode());
            }
        }
    }

    /**
     * @see gov.va.med.esr.common.rule.MonetaryAwardInput#containsMonetaryBenefits()
     */
    public boolean containsMonetaryBenefits() throws RuleException {
        Person person = getIncomingPerson();
        MonetaryBenefitAward mba = person != null ? person
                .getMonetaryBenefitAward() : null;
        return mba != null ? mba.containsMonetaryBenefits() : false;
    }

    /**
     * @see gov.va.med.esr.common.rule.MonetaryAwardInput#getVADisability()
     */
    public MonetaryBenefit getVADisability() throws RuleException {
        return getVADisability(getIncomingPerson());
    }

    /**
     * @see gov.va.med.esr.common.rule.MonetaryAwardInput#getVADisability()
     */
    public MonetaryBenefit getVADisability(Person person) throws RuleException {
        MonetaryBenefitAward mba = person != null ? person
                .getMonetaryBenefitAward() : null;
        return mba != null ? mba.getDisabilityCompensation() : null;
    }

    /**
     * @see gov.va.med.esr.common.rule.MonetaryAwardInput#getVAPension()
     */
    public MonetaryBenefit getVAPension() throws RuleException {
        return getVAPension(getIncomingPerson());
    }

    /**
     * @see gov.va.med.esr.common.rule.MonetaryAwardInput#getVAPension()
     */
    public MonetaryBenefit getVAPension(Person person) throws RuleException {
        MonetaryBenefitAward mba = person != null ? person
                .getMonetaryBenefitAward() : null;
        return mba != null ? mba.getVAPension() : null;
    }

    /**
     * @see gov.va.med.esr.common.rule.MonetaryAwardInput#getVAPensionIndicator()
     */
    public Boolean getVAPensionIndicator() throws RuleException {
        return getVAPensionIndicator(getIncomingPerson());
    }

    /**
     * @see gov.va.med.esr.common.rule.MonetaryAwardInput#getVAPensionIndicator()
     */
    public Boolean getVAPensionIndicator(Person person) throws RuleException {
        MonetaryBenefitAward mba = person != null ? person
                .getMonetaryBenefitAward() : null;
        MonetaryBenefit mb = mba != null ? mba.getVAPension() : null;
        return convertMBIndicatorToBoolean(mb);
    }

    /**
     * @see gov.va.med.esr.common.rule.MonetaryAwardInput#getVAPensionIndicator()
     */
    public void setVAPensionIndicator(Boolean flag) throws RuleException {

        try {
            Person updated = getResultPerson();
            MonetaryBenefit mb = getVAPension(updated);
            if (mb == null) {
                mb = getHelperService().addVAPension(updated);     
            }
            setMonetaryBenefitIndicator(mb, flag);
        } catch (ServiceException e) {
            throw new RuleException("Failed to set a VA pension indicator", e);
        }
    }

    /**
     * @see gov.va.med.esr.common.rule.MonetaryAwardInput#getRatedIncompetentOfVerifiedSite()
     */
    public IncompetenceRuling getRatedIncompetentOfVerifiedSite()
            throws RuleException {
        Person verified = getVerifiedSiteData();
        return verified != null ? verified.getIncompetenceRuling() : null;
    }

    /**
     * @see gov.va.med.esr.common.rule.MonetaryAwardInput#getAidAttendance()
     */
    public MonetaryBenefit getAidAttendance() throws RuleException {
        return getAidAttendance(getIncomingPerson());
    }

    /**
     * @see gov.va.med.esr.common.rule.MonetaryAwardInput#getAidAttendance(gov.va.med.esr.common.model.person.Person)
     */
    public MonetaryBenefit getAidAttendance(Person person) throws RuleException {
        MonetaryBenefitAward mba = person != null ? person
                .getMonetaryBenefitAward() : null;
        return mba != null ? mba.getAidAndAttendance() : null;
    }

    /**
     * @see gov.va.med.esr.common.rule.MonetaryAwardInput#getAidAttendanceIndicator()
     */
    public Boolean getAidAttendanceIndicator() throws RuleException {
        return getAidAttendanceIndicator(getIncomingPerson());
    }

    /**
     * @see gov.va.med.esr.common.rule.MonetaryAwardInput#getAidAttendanceIndicator(gov.va.med.esr.common.model.person.Person)
     */
    public Boolean getAidAttendanceIndicator(Person person)
            throws RuleException {

        MonetaryBenefitAward mba = person != null ? person
                .getMonetaryBenefitAward() : null;
        MonetaryBenefit mb = mba != null ? mba.getAidAndAttendance() : null;
        return convertMBIndicatorToBoolean(mb);
    }

    /**
     * @see gov.va.med.esr.common.rule.MonetaryAwardInput#getVAPensionIndicator()
     */
    public void setAidAttendanceIndicator(Boolean flag) throws RuleException {

        try {
            Person updated = getResultPerson();
            MonetaryBenefit mb = getAidAttendance(updated);
            if (mb == null) {
                mb = getHelperService().addAABenefit(updated);
            }
            setMonetaryBenefitIndicator(mb, flag);
        } catch (ServiceException e) {
            throw new RuleException("Failed to set a House bound indicator", e);
        }
    }

    /**
     * @see gov.va.med.esr.common.rule.MonetaryAwardInput#getHouseBoundIndicator()
     */
    public Boolean getHouseBoundIndicator() throws RuleException {
        return getHouseBoundIndicator(getIncomingPerson());
    }

    /**
     * @see gov.va.med.esr.common.rule.MonetaryAwardInput#getHouseBoundIndicator()
     */
    public Boolean getHouseBoundIndicator(Person person) throws RuleException {
        MonetaryBenefitAward mba = person != null ? person
                .getMonetaryBenefitAward() : null;
        MonetaryBenefit mb = mba != null ? mba.getHouseBound() : null;
        return convertMBIndicatorToBoolean(mb);
    }

    /**
     * @see gov.va.med.esr.common.rule.MonetaryAwardInput#setHouseBoundIndicator(java.lang.Boolean)
     */
    public void setHouseBoundIndicator(Boolean flag) throws RuleException {

        try {
            Person updated = getResultPerson();
            MonetaryBenefit mb = getHouseBound(updated);
            if (mb == null) {
                mb = getHelperService().addHouseboundBenefit(updated);
            }
            setMonetaryBenefitIndicator(mb, flag);
        } catch (ServiceException e) {
            throw new RuleException("Failed to set a House bound indicator", e);
        }
    }

    /**
     * @see gov.va.med.esr.common.rule.MonetaryAwardInput#getVADisabilityIndicator()
     */
    public Boolean getVADisabilityIndicator() throws RuleException {
        return getVADisabilityIndicator(getIncomingPerson());
    }

    /**
     * @see gov.va.med.esr.common.rule.MonetaryAwardInput#getVADisabilityIndicator(gov.va.med.esr.common.model.person.Person)
     */
    public Boolean getVADisabilityIndicator(Person person) throws RuleException {

        MonetaryBenefitAward mba = person != null ? person
                .getMonetaryBenefitAward() : null;
        MonetaryBenefit mb = mba != null ? mba.getDisabilityCompensation()
                : null;
        return convertMBIndicatorToBoolean(mb);
    }

    /**
     * @see gov.va.med.esr.common.rule.MonetaryAwardInput#setVADisabilityIndicator(java.lang.Boolean)
     */
    public void setVADisabilityIndicator(Boolean flag) throws RuleException {

        try {
            Person updated = getResultPerson();
            MonetaryBenefit mb = getVADisability(updated);
            if (mb == null) {
                mb = getHelperService().addVADisability(updated);
            }
            setMonetaryBenefitIndicator(mb, flag);
        } catch (ServiceException e) {
            throw new RuleException("Failed to set a VA pension indicator", e);
        }
    }

    public BigDecimal getTotalCheckAmount(Person person) throws RuleException {
        MonetaryBenefitAward mba = (person != null) ? person
                .getMonetaryBenefitAward() : null;
        BigDecimal result = (mba != null) ? mba.getCheckAmount() : BigDecimal
                .valueOf(0);
        logger.debug(result);
        return result;
    }

    private void setTotalCheckAmount(BigDecimal amount, Person person) {

        MonetaryBenefitAward monetaryBenefitAward = (person != null) ? person
                .getMonetaryBenefitAward() : null;
        if (monetaryBenefitAward != null) {
            monetaryBenefitAward.setCheckAmount(amount);
        } else {
            monetaryBenefitAward = new MonetaryBenefitAward();
            monetaryBenefitAward.setCheckAmount(amount);
            person.setMonetaryBenefitAward(monetaryBenefitAward);
        }
    }

    private Boolean convertMBIndicatorToBoolean(MonetaryBenefit mb) {
        return (mb != null && mb.getMonetaryBenefitIndicator() != null) ? mb
                .getMonetaryBenefitIndicator().toBoolean() : null;
    }

    /**
     * Set an indicator for a monetary benefit
     * 
     * @param benefit
     */
    private void setMonetaryBenefitIndicator(MonetaryBenefit benefit,
            Boolean flag) throws RuleException {

        if (benefit != null) {
            try {
                Indicator.Code code = Boolean.TRUE.equals(flag) ? Indicator.YES
                        : Indicator.NO;
                benefit.setMonetaryBenefitIndicator(getLookupService()
                        .getIndicatorByCode(code));
            } catch (Exception e) {
                throw new RuleException(
                        "Failed to set a monetary benefit indicator", e);
            }
        }
    }
    
    // new methods for pension data sharing CR4510 CCR10352
    public Pension getPension() throws RuleException {
        return getPension(getIncomingPerson());
    }
    private Pension getPension(Person person) throws RuleException {
    	/*
    	MonetaryBenefitAward mba = person != null ? person
                .getMonetaryBenefitAward() : null;
        MonetaryBenefit mb = mba != null ? mba.getVAPension() : null;
        */
        // 
        MonetaryBenefit mb = this.getVAPension(person);
        return mb!= null? mb.getPension() : null;
    }
    // new method for pension status  for Handbook_CodeCR10386
  public boolean getPensionStatus(Person person) throws RuleException {
	  
	  MonetaryBenefitAward mba = person != null ? person
              .getMonetaryBenefitAward() : null;
      MonetaryBenefit pension = mba != null ? mba.getMonetaryBenefitByType(MonetaryBenefitType.CODE_VA_PENSION) : null;
      if(pension !=null  && pension.getMonetaryBenefitIndicator() !=null &&
    		  pension.getMonetaryBenefitIndicator().toBoolean() !=null ){
    		return pension.getMonetaryBenefitIndicator().toBoolean().booleanValue();
        } else{
        	return false;
        }   
  }
  public boolean isPensionStatusChanged()throws RuleException {
     boolean result=getPensionStatus(getResultPerson());
     boolean prior=getPensionStatus(getPristinePerson());
     if (prior !=result ) {
         return true;
     }
     return false;
  }
    public Date getPristinePensionAwardEffectiveDate() throws RuleException {
        return getPensionAwardEffectiveDate(getPristinePerson());
    }
    public Date getPensionAwardEffectiveDate() throws RuleException {
        return getPensionAwardEffectiveDate(getIncomingPerson());
    }
    private Date getPensionAwardEffectiveDate(Person person) throws RuleException {
        Pension pension = this.getPension(person);
        return pension != null? pension.getEffectiveDate(): null;  
    }
    
    public Date getPensionAwardTerminationDate() throws RuleException {
        return getPensionAwardTerminationDate(getIncomingPerson());
    }
    private Date getPensionAwardTerminationDate(Person person) throws RuleException {
        Pension pension = this.getPension(person);
        return pension != null? pension.getTerminationDate(): null;  
    }
    
    public String getPensionAwardReasonCode() throws RuleException {
        return getPensionAwardReasonCode(getIncomingPerson());
    }
    private String getPensionAwardReasonCode(Person person) throws RuleException {
         Pension pension = this.getPension(person);
    	 PensionReasonCode prcode = pension !=null ? pension.getAwardReasonCode():null;
    	 return prcode != null ? prcode.getCode():null;
    }
    
    public String getPensionAwardTerminationReasonCode1() throws RuleException {
        return getPensionAwardTerminationReasonCode1(getIncomingPerson());
    }
    private String getPensionAwardTerminationReasonCode1(Person person) throws RuleException {
         Pension pension = this.getPension(person);
    	 PensionReasonCode prcode = pension !=null ? pension.getTerminationReasonCode1():null;
    	 return prcode != null ? prcode.getCode():null;
    }
    
    public String getPensionAwardTerminationReasonCode2() throws RuleException {
        return getPensionAwardTerminationReasonCode2(getIncomingPerson());
    }
    private String getPensionAwardTerminationReasonCode2(Person person) throws RuleException {
         Pension pension = this.getPension(person);
    	 PensionReasonCode prcode = pension !=null ? pension.getTerminationReasonCode2():null;
    	 return prcode != null ? prcode.getCode():null;
    }
    
    public String getPensionAwardTerminationReasonCode3() throws RuleException {
        return getPensionAwardTerminationReasonCode3(getIncomingPerson());
    }
    private String getPensionAwardTerminationReasonCode3(Person person) throws RuleException {
         Pension pension = this.getPension(person);
    	 PensionReasonCode prcode = pension !=null ? pension.getTerminationReasonCode3():null;
    	 return prcode != null ? prcode.getCode():null;
    }
    
    public String getPensionAwardTerminationReasonCode4() throws RuleException {
        return getPensionAwardTerminationReasonCode4(getIncomingPerson());
    }
    private String getPensionAwardTerminationReasonCode4(Person person) throws RuleException {
         Pension pension = this.getPension(person);
    	 PensionReasonCode prcode = pension !=null ? pension.getTerminationReasonCode4():null;
    	 return prcode != null ? prcode.getCode():null;
    }
    
    
    public Pension getResultPension() throws RuleException {
        try {
            Person updated = getResultPerson();
            MonetaryBenefit mb = getVAPension(updated);
            Pension pension = getPension(updated);
            if (mb == null) {
                mb = getHelperService().addVAPension(updated); 
            }
            if (pension == null ){
            	pension = new Pension();
            	mb.setPension(pension);
            }
           return pension;  	 
        } catch (ServiceException e) {
            throw new RuleException("Failed to get a pension", e);
        }
    }

    public void setPensionAwardEffectiveDate(Date effDate) throws RuleException {
           // Pension resultPension = this.getResultPension();
          //  resultPension.setEffectiveDate(effDate);
            this.getResultPension().setEffectiveDate(effDate);
    }
 
    public void setPensionAwardTerminationDate(Date termDate) throws RuleException {
        this.getResultPension().setTerminationDate(termDate);
   }

    public void setPensionAwardReasonCode(String awaCode) throws RuleException {
    	try {
    		PensionReasonCode reasonCode = null;
    		if (awaCode != null)
    			reasonCode = getLookupService().getPensionReasonByCode(awaCode);
    		this.getResultPension().setAwardReasonCode(reasonCode);
    	 } catch (Exception e) {
             throw new RuleException(
                     "Failed to set a pension reason code", e);
         }
   }
   
    public void setPensionAwardTerminationReasonCode1(String termCode) throws RuleException {
    	try {
    		PensionReasonCode reasonCode = null;
    		if (termCode != null)
    			reasonCode = getLookupService().getPensionReasonByCode(termCode);
    		this.getResultPension().setTerminationReasonCode1(reasonCode);
    	 } catch (Exception e) {
             throw new RuleException(
                     "Failed to set a pension reason code", e);
         }
   }
    
    public void setPensionAwardTerminationReasonCode2(String termCode) throws RuleException {
    	try {
    		PensionReasonCode reasonCode = null;
    		if (termCode != null)
    			reasonCode = getLookupService().getPensionReasonByCode(termCode);
    		this.getResultPension().setTerminationReasonCode2(reasonCode);
    	 } catch (Exception e) {
             throw new RuleException(
                     "Failed to set a pension reason code", e);
         }
   }
    
    public void setPensionAwardTerminationReasonCode3(String termCode) throws RuleException {
    	try {
    		PensionReasonCode reasonCode = null;
    		if (termCode != null)
    			reasonCode = getLookupService().getPensionReasonByCode(termCode);
    		this.getResultPension().setTerminationReasonCode3(reasonCode);
    	 } catch (Exception e) {
             throw new RuleException(
                     "Failed to set a pension reason code", e);
         }
   }
    
    public void setPensionAwardTerminationReasonCode4(String termCode) throws RuleException {
    	try {
    		PensionReasonCode reasonCode = null;
    		if (termCode != null)
    			reasonCode = getLookupService().getPensionReasonByCode(termCode);
    		this.getResultPension().setTerminationReasonCode4(reasonCode);
    	 } catch (Exception e) {
             throw new RuleException(
                     "Failed to set a pension reason code", e);
         }
   }
     
    public void setPensionAwardTerminationReasonCodes() throws RuleException {
    		setPensionAwardTerminationReasonCode1(this.getPensionAwardTerminationReasonCode1());
    		setPensionAwardTerminationReasonCode2(this.getPensionAwardTerminationReasonCode2());
    		setPensionAwardTerminationReasonCode3(this.getPensionAwardTerminationReasonCode3());
    		setPensionAwardTerminationReasonCode4(this.getPensionAwardTerminationReasonCode4());		
    }
    
    public void setPensionAwardTerminationReasonCodesToNull() throws RuleException {
        		setPensionAwardTerminationReasonCode1(null);
        		setPensionAwardTerminationReasonCode2(null);
        		setPensionAwardTerminationReasonCode3(null);
        		setPensionAwardTerminationReasonCode4(null);
    }
    
    public boolean isIncomingPensionAwardEffectiveDateGreaterThanToday() throws RuleException {
    	Date effDate = this.getPensionAwardEffectiveDate();
    	/*
    	GregorianCalendar cal = new GregorianCalendar();
    	Date todayDt = cal.getTime();
    	if ( effDate != null && effDate.compareTo(todayDt) > 0 )
    		return true;
    	else
    		return false;
    	*/
    	Date today = DateUtils.getCurrentDate();
        if (effDate != null && effDate.after(today)) 
        {
            return true;
        }
        else {
            return false;
        }	
    }
    
    public boolean isIncomingPensionAwardTerminationDateGreaterThanToday()throws RuleException {
    	
    	Date termDate = this.getPensionAwardTerminationDate();
    	/*
    	GregorianCalendar cal = new GregorianCalendar();
    	Date todayDt = cal.getTime();
    	if ( termDate != null && termDate.compareTo(todayDt) > 0 )
    		return true;
    	else
    		return false;
    	*/
    	Date today = DateUtils.getCurrentDate(); // not considering time
        if (termDate != null && termDate.after(today)) 
        {
            return true;
        }
        else {
            return false;
        }
    }
    
    public boolean isPensionAwardReasonCodeEqualToZero()throws RuleException {
    	String pawdCode = getPensionAwardReasonCode(getIncomingPerson());
    	if ( pawdCode!=null && Original_Award_Reason_Code.equalsIgnoreCase(pawdCode))
    		return true;
    	else 
    		return false;
    }
    
    public boolean isDateInFuture(Date adate) {
		Date today = Calendar.getInstance().getTime();
		return isAfter(adate, today);
	}
    
    public boolean isPensionAwardReasonCodesDiff() throws RuleException{
    	boolean diff = false;
    	String incomingPenAwardCode = getPensionAwardReasonCode(getIncomingPerson());
    	String resultPenAwardCode = getPensionAwardReasonCode(getResultPerson());
    	
    	if(incomingPenAwardCode != resultPenAwardCode){    		
    		diff = true;    		
    	}
    	
    	return diff;
    }
    
    public boolean isPensionAwardEffDateDiff() throws RuleException{
    	boolean diff = false;
    	Date incomingPenAwardCode = getPensionAwardEffectiveDate(getIncomingPerson());
    	Date resultPenAwardCode = getPensionAwardEffectiveDate(getResultPerson());
    	
    	if(resultPenAwardCode.compareTo(incomingPenAwardCode)>0 ||resultPenAwardCode.compareTo(incomingPenAwardCode)==0){
    		diff = true;    		
    	}
    	
    	return diff;
    }
    
    public Boolean isPensionAwardReasonCodeNull(Person person) throws RuleException{
    	boolean isNull = false;    	
    	String penAwardCode = getPensionAwardReasonCode(person);
    	
    	if(penAwardCode == null){
    		isNull = true;    		
    	}    	
    	
    	return new Boolean(isNull);
    }
    
    public Boolean isPensionAwardEffDateNull(Person person) throws RuleException{
    	boolean isNull = false;    	
    	
    	Date penAwardEffDate = getPensionAwardEffectiveDate(person);
    	
    	if(penAwardEffDate == null){
    		isNull = true;    		
    	}
    	
    	return new Boolean(isNull);
    }   
    
    public boolean getServiceIndicator() throws ServiceException{
		return super.getSystemParameterService().getVBADataSharingIndicator().booleanValue();
		
	}
    
    public void pensionIndicatorReset() throws RuleException, ServiceException{
    	MonetaryBenefit pensionMB = getVAPension(getIncomingPerson());
    	
    	if(getServiceIndicator() && pensionMB != null && getVAPensionIndicator() != null){    		
    		
    		if(Boolean.TRUE.equals(getVAPensionIndicator())){
    			setMonetaryBenefitIndicator(pensionMB, Boolean.FALSE); 
    			if(isCheckSumDueToPension()){
    				getIncomingPerson().getMonetaryBenefitAward().setCheckAmount(new BigDecimal(0));    				
    			}    			
    		}    		
    	}    	
    }
    
    private boolean isCheckSumDueToPension() throws RuleException {    	

		if ((getVADisabilityIndicator() != null && Boolean.TRUE.equals(getVADisabilityIndicator()))
				|| (getHouseBoundIndicator() != null && Boolean.TRUE.equals(getHouseBoundIndicator()))
				|| (getAidAttendanceIndicator() != null && Boolean.TRUE.equals(getAidAttendanceIndicator()))) {
			return false;

		}
		return true;
	}
    
    
    
}