/*******************************************************************************
 * Copyright  2004 VHA. All rights reserved
 ******************************************************************************/
package gov.va.med.esr.common.rule.parameter;

// Java classes
import java.util.Date;

// Library classes

// Framework classes
import gov.va.med.fw.rule.RuleException;
import gov.va.med.fw.service.ServiceException;

// ESR classes
import gov.va.med.esr.common.model.ee.IneligibilityFactor;
import gov.va.med.esr.common.model.lookup.VAFacility;
import gov.va.med.esr.common.model.person.Person;
import gov.va.med.esr.common.rule.IneligibleInput;

/**
 * This class implements an interface which gets mapped to a virtual class in
 * ILOG.
 * 
 * @author Carlos Ruiz
 * @version 1.0
 */
public class IneligibleInputParameter extends BaseParameter implements
		IneligibleInput {

	/**
	 * An instance of serialVersionUID
	 */
	private static final long serialVersionUID = -1069379818694573999L;

	private Date ineligibilityDate = null;

	private String ineligibilityReason = null;
   
	public IneligibleInputParameter() {
		super();
	}

	/**
	 * @see gov.va.med.esr.common.rule.parameter.IneligibleInput#getIneligibilityDate()
	 */
	public Date getIneligibilityDate() throws RuleException {
		if( ineligibilityDate == null ) {
			ineligibilityDate = getIneligibilityDate( getIncomingPerson() );
		}
		return ineligibilityDate;
	}

   /**
    * @see gov.va.med.esr.common.rule.IneligibleInput#getIneligibilityDate(gov.va.med.esr.common.model.person.Person)
    */
   public Date getIneligibilityDate( Person person ) throws RuleException {
      IneligibilityFactor ief = getIneligibilityFactor( person );
      return (ief != null) ? ief.getIneligibleDate() : null;
   }
   
   /**
    * @see gov.va.med.esr.common.rule.IneligibleInput#getIneligibilityFactor(gov.va.med.esr.common.model.person.Person)
    */
   public IneligibilityFactor getIneligibilityFactor() throws RuleException {
      return getIneligibilityFactor( this.getIncomingPerson() );
   }
   
   /**
    * @see gov.va.med.esr.common.rule.IneligibleInput#getIneligibilityFactor(gov.va.med.esr.common.model.person.Person)
    */
   public IneligibilityFactor getIneligibilityFactor( Person person ) throws RuleException {
      return person != null ? person.getIneligibilityFactor() : null;
   }
   
	/**
	 * @see gov.va.med.esr.common.rule.parameter.IneligibleInput#getIneligibilityReason()
	 */
	public String getIneligibilityReason()  throws RuleException {
		if( ineligibilityReason == null ) {
			ineligibilityReason = getIneligibilityReason( getIncomingPerson() );
		}
		return this.ineligibilityReason;
	}

   public String getIneligibilityReason( Person person ) throws RuleException {
      IneligibilityFactor ief = getIneligibilityFactor( person );
      return (ief != null) ? ief.getReason() : null;
   }

	/**
	 * @see gov.va.med.esr.common.rule.parameter.IneligibleInput#isUpdateFromSiteViaZ07()
	 */
	public boolean isUpdateFromSiteViaZ07() throws RuleException {
		IneligibilityFactor ineligibilityFactor = this
				.getIneligibilityFactor(this.getIncomingPerson());
		return ineligibilityFactor != null
				&& ineligibilityFactor.getFacilityReceived() != null
				&& !(ineligibilityFactor.getFacilityReceived().getStationNumber()
						.equals(VAFacility.CODE_HEC.getName()));
	}

	/**
	 * @see gov.va.med.esr.common.rule.parameter.IneligibleInput#updateIneligibility()
	 */
	public void updateIneligibility() throws RuleException {
		this.setIneligible( this.getIneligibilityFactor( this.getIncomingPerson() ) );
	}

	/**
	 * @see gov.va.med.esr.common.rule.parameter.IneligibleInput#removeIneligibility()
	 */
	public void removeIneligibility() {
		// Assumes you want to REMOVE an ineligibility
		if (this.getResultPerson().getIneligibilityFactor() != null) {
			this.getResultPerson().setIneligibilityFactor(null);
		}
	}

	/**
	 * @see gov.va.med.esr.common.rule.parameter.IneligibleInput#getIneligibleDateReceivedFrom()
	 */
	public String getIneligibleDateReceivedFrom() throws RuleException {
		IneligibilityFactor ineligibilityFactor = this
				.getIneligibilityFactor(this.getIncomingPerson());
		String stationNumber = (ineligibilityFactor != null && ineligibilityFactor
				.getFacilityReceived() != null) ? ineligibilityFactor
				.getFacilityReceived().getStationNumber() : null;
		return stationNumber;
	}

	/**
	 * @see gov.va.med.esr.common.rule.parameter.IneligibleInput#isIneligibleDateUpdatedByVAMC()
	 */
	public boolean isIneligibleDateUpdatedByVAMC() throws RuleException {
		IneligibilityFactor ineligibilityFactor = this
				.getIneligibilityFactor(this.getIncomingPerson());
		String stationNumber = (ineligibilityFactor != null && ineligibilityFactor
				.getFacilityReceived() != null) ? ineligibilityFactor
				.getFacilityReceived().getStationNumber() : null;
		return (stationNumber != null) ? !stationNumber
				.equals(VAFacility.CODE_HEC.getName()) : false;
	}

	/**
	 * @see gov.va.med.esr.common.rule.IneligibleInput#setIneligible()
	 */
	public void setIneligible(IneligibilityFactor incoming) throws RuleException {
		
		try {
			this.getMergeRuleService().mergeIneligibilityFactor( incoming, this.getResultPerson() );
		}
		catch( ServiceException e ) {
			if( logger.isDebugEnabled() ) {
				logger.debug( "Failed to copy ineligibility factor", e );
			}
		}
	}

   /**
    * @see gov.va.med.esr.common.rule.IneligibleInput#setIneligibilityDate(java.util.Date)
    */
   public void setIneligibilityDate(Date date) throws RuleException {
      IneligibilityFactor ief = getIneligibilityFactor( getResultPerson() );
      if( ief == null ) {
         ief = new IneligibilityFactor();
         getResultPerson().setIneligibilityFactor( ief );
      }
      ief.setIneligibleDate( date );
   }

   /**
    * @see gov.va.med.esr.common.rule.IneligibleInput#setIneligibilityReason(java.lang.String)
    */
   public void setIneligibilityReason(String reason) throws RuleException {
      IneligibilityFactor ief = getIneligibilityFactor( getResultPerson() );
      if( ief == null ) {
         ief = new IneligibilityFactor();
         getResultPerson().setIneligibilityFactor( ief );
      }
      ief.setReason( reason );
   }

   /**
    * @see gov.va.med.esr.common.rule.IneligibleInput#getIneligibilityVARODecision()
    */
   public String getIneligibilityVARODecision() throws RuleException {
      return this.getIneligibilityVARODecision( this.getIncomingPerson() );
   }

   /**
    * @see gov.va.med.esr.common.rule.IneligibleInput#getIneligibilityVARODecision(gov.va.med.esr.common.model.person.Person)
    */
   public String getIneligibilityVARODecision(Person person) throws RuleException {
      IneligibilityFactor ief = getIneligibilityFactor( person );
      return ief != null ? ief.getVaroDecision() : null;
   }

   /**
    * @see gov.va.med.esr.common.rule.IneligibleInput#setIneligibilityVARODecision(java.lang.String)
    */
   public void setIneligibilityVARODecision(String decision) throws RuleException {
      IneligibilityFactor ief = getIneligibilityFactor( getResultPerson() );
      if( ief == null ) {
         ief = new IneligibilityFactor();
         getResultPerson().setIneligibilityFactor( ief );
      }
      ief.setVaroDecision( decision );
   }
}