/********************************************************************
 * Copyright  2004 VHA. All rights reserved
 ********************************************************************/
package gov.va.med.esr.common.rule.data;

// Java classes

// Library classes
import java.util.Set;

import gov.va.med.esr.common.model.ee.VerificationInfo;
import gov.va.med.esr.common.model.insurance.InsurancePolicy;
import gov.va.med.esr.common.model.lookup.VAFacility;
import gov.va.med.esr.common.model.party.Address;
import gov.va.med.esr.common.model.person.Association;
import gov.va.med.esr.common.model.person.Person;
import gov.va.med.esr.common.rule.data.accessor.DemographicAccessor;
import gov.va.med.esr.common.rule.data.accessor.PropertyAccessor.DataType;


/**
 * Project: Common
 * 
 * @author DNS   LEV
 * @version 1.0
 */
public class AddressInputData extends BaseData implements DemographicAccessor {

    /**
	 * An instance of serialVersionUID
	 */
	private static final long serialVersionUID = -8430111235679047716L;

	private Set updatedVeteranAddresses = null;

    private Address pristineAddress = null;
    
    private Address resultAddress = null;
    
    private Address incomingAddress = null;    
    
    private Association onFileAssociation = null;
    
    //CCR11898 Added VrificationInfo
    private VerificationInfo verificationInfo = null;
    
    /**
     * Encapsulate data coming from an incoming address
     * 
     * @param incoming
     */
    public AddressInputData(Address incoming) {
        super(incoming.getPerson());
        
        this.setIncomingAddress(incoming);
        this.setResultAddress(null);
        this.setPristineAddress(null);
    }

    /**
     * Encapsulates data from an incoming address and the current person in a
     * system.
     * 
     * @param incoming
     *            An incoming address
     * @param onFile
     *            A person on file
     */
    public AddressInputData(Address incoming, Person onFile, Person pristine) {
        super(incoming.getPerson(), onFile, pristine);

        if (incoming.getType() != null) {
	        // Set address data
	        Address found = Address.getAddressOfType(pristine.getAddresses(), incoming.getType().getCode());
	       
	        this.setIncomingAddress(incoming);
	        this.setPristineAddress(found);
	        this.setResultAddress(null);
	
	        // Set the veteran addresses
	        this.setUpdatedVeteranAddresses(onFile.getAddresses());
        }
    }

    // Use for validating/processing address from insurance
    public AddressInputData(Address incoming, InsurancePolicy onFilePolicy) {
        super(onFilePolicy.getPerson());
        
        this.setIncomingAddress(incoming);
    }    

    // Use for validating/processing address from association
    public AddressInputData(Address incoming, Association onFileAssociation) {
        super(incoming.getAssociation().getPerson(), onFileAssociation.getPerson());
        
        this.setIncomingAddress(incoming);
        this.setResultAddress((Address)onFileAssociation.getAddress());
        // Effectively the same as result in this context
        this.setPristineAddress((Address)onFileAssociation.getAddress());
        this.setOnFileAssociation(onFileAssociation);
    }
    
    public AddressInputData(Address incoming, Person onFile, Person pristine, VAFacility sendingFacility) {
        this(incoming, onFile, pristine);        
        setSendingFacility(sendingFacility);
       
    }  
    
    //CCR11898 added VerificationInfo 
    public AddressInputData(Address incoming, Person onFile, Person pristine, VAFacility sendingFacility, VerificationInfo veriInfo) {
        this(incoming, onFile, pristine);        
        setSendingFacility(sendingFacility);
        setVerificationInfo(veriInfo);
       
    }    
    
    /**
     * @return Returns the incomingAddress.
     */
    public Address getIncomingAddress() {
        return incomingAddress;
    }

    /**
     * @param incomingAddress The incomingAddress to set.
     */
    public void setIncomingAddress(Address incomingAddress) {
        this.incomingAddress = incomingAddress;
    }

    /**
     * @return Returns the pristineAddress.
     */
    public Address getPristineAddress() {
        return pristineAddress;
    }

    /**
     * @param pristineAddress The pristineAddress to set.
     */
    public void setPristineAddress(Address pristineAddress) {
        this.pristineAddress = pristineAddress;
    }

    /**
     * @return Returns the resultAddress.
     */
    public Address getResultAddress() {
        return resultAddress;
    }

    /**
     * @param resultAddress The resultAddress to set.
     */
    public void setResultAddress(Address resultAddress) {
        this.resultAddress = resultAddress;
    }

    /**
     * @return Returns the updatedVeteranAddresses.
     */
    public Set getUpdatedVeteranAddresses() {
        return updatedVeteranAddresses;
    }

    /**
     * @param updatedVeteranAddresses The updatedVeteranAddresses to set.
     */
    public void setUpdatedVeteranAddresses(Set updatedVeteranAddresses) {
        this.updatedVeteranAddresses = updatedVeteranAddresses;
    }

    /**
     * @return Returns the onFileAssociation.
     */
    public Association getOnFileAssociation() {
        return onFileAssociation;
    }

    /**
     * @param onFileAssociation The onFileAssociation to set.
     */
    public void setOnFileAssociation(Association onFileAssociation) {
        this.onFileAssociation = onFileAssociation;
    }

	/* (non-Javadoc)
	 * @see gov.va.med.esr.common.rule.data.accessor.DemographicAccessor#getAddress(gov.va.med.esr.common.rule.data.accessor.PropertyAccessor.DataType)
	 */
	public Address getAddress(DataType dataType) {
		if(dataType == null)
			return null;
		
		Address add = null;
		if(dataType.isIncoming())
			add = getIncomingAddress();
		else if(dataType.isPristine())
			add = getPristineAddress();
		else if(dataType.isResult())
			add = getResultAddress();
		
		return add;
	}
   
	//CCR11898
	public VerificationInfo getVerificationInfo() {
		return verificationInfo;
	}

	public void setVerificationInfo(VerificationInfo verificationInfo) {
		this.verificationInfo = verificationInfo;
	}

   
}